#include "Result.h"

#include <stdlib.h>
#include <string.h>

#include "FruitBase.h"

#if defined(_WIN32) || defined(_WIN64)

  extern "C"
  {
    #include "MPIWrapper.h"
  }

#else // assume POSIX

  extern "C"
  {
    #include "MPIWrapper.h"
  }

#endif

#include "YBWCManager.h"
extern YBWCManager* g_pYBWCManagerInstance;

Result* Result_Construct(Result* pThis)
{
  pThis->pNext                    = NULL;
  pThis->nSourceId                = -1;
  pThis->nDestId                  = -1;
  pThis->pParameterBuffer[0]      = '\0'; // kh 03.08.04 defensive
  pThis->mpiRequest               = MPI_REQUEST_NULL;
  pThis->nPosition                = 0;

  pThis->bSendSuppressed          = FALSE;

  pThis->nType                    = RESULT_TYPE_UNDEFINED;

// kh 07.08.06 may be done only in debug mode for a better performance
  pThis->szTypeToString[0]        = '\0';

  pThis->sProcessorName[0]        = '\0';

  pThis->nBestMove                = 0;
  pThis->nBestValue               = 0;
  pThis->nFlags                   = 0;
  pThis->nDepth                   = 0;
  pThis->nMaxDepth                = 0;

// kh 05.09.06 first element only for a better performance
  pThis->nPVLength                = 0; 
  pThis->PV[0]                    = 0; 

  pThis->pPV                      = NULL; // kh 12.09.06 used in Result_FillParameterIntoBuffer(...

  pThis->bPonderMoveOk            = FALSE;
  pThis->dTime                    = 0.0;
  pThis->dSpeed                   = 0.0;
  pThis->dCPU                     = 0.0;
  pThis->nNodes                   = 0;

  pThis->nSyncPositionId          = -2;

  pThis->nWorkId                  = -2;

  pThis->nStackIndex              = -1;
  pThis->nMoveListIndex           = -1;
  pThis->nCallType                = CALL_TYPE_UNDEFINED;
  pThis->nReturnPosition          = RETURN_POSITION_UNDEFINED;
  pThis->nMove                    = 0;
//pThis->nDepth                   = 0;
  pThis->nNewDepth                = 0;
  pThis->nHeight                  = 0;
  pThis->bExtended                = FALSE;
  pThis->bInPV                    = FALSE;
  pThis->nPlayedNb                = 0;
  pThis->bAlphaBoundIsEstimated   = FALSE;
  pThis->nHistoryValue            = 0;
  pThis->nAlpha                   = 0;
  pThis->nBeta                    = 0;
  pThis->nTransDate               = 0;
  pThis->bGoodCap                 = FALSE;
  pThis->nMoveAheadListCount      = -1;
  pThis->MoveAheadList[0]         = 0;
  pThis->pMoveAheadList           = NULL;
  pThis->nKillerListCount         = -1;
  pThis->KillerList[0]            = 0;
  pThis->pKillerList              = NULL;

  pThis->nTotalNodes              = 0;

  pThis->nHistoryTableSize        = -1;
//pThis->HistoryTable[0]          = 0;

  pThis->nHistTotTableSize        = -1;
//pThis->HistTotTable[0]          = 0;

  pThis->nHistHitTableSize        = -1;
//pThis->HistHitTable[0]          = 0;

  return pThis;
}

void Result_Destruct(Result* pThis)
{
  if(pThis)
  {
    free(pThis);
  }
}

char* Result_TypeToString(Result* pThis)
{
  pThis->szTypeToString[0] = '\0';

  switch(pThis->nType)
  {
  case RESULT_TYPE_UNDEFINED:
    strncat(pThis->szTypeToString, "RESULT_TYPE_UNDEFINED",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_NON_ROOT_MASTER_LOGON:
    strncat(pThis->szTypeToString, "RESULT_TYPE_NON_ROOT_MASTER_LOGON",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_NON_ROOT_MASTER_LOGOFF:
    strncat(pThis->szTypeToString, "RESULT_TYPE_NON_ROOT_MASTER_LOGOFF",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_NON_ROOT_MASTER_SHUTDOWN:
    strncat(pThis->szTypeToString, "RESULT_TYPE_NON_ROOT_MASTER_SHUTDOWN",  MAX_RESULT_TYPE_TO_STRING);
    break;
    
  case RESULT_TYPE_NON_ROOT_MASTER_REMOTE_TEST:
    strncat(pThis->szTypeToString, "RESULT_TYPE_NON_ROOT_MASTER_REMOTE_TEST",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_NON_ROOT_MASTER_SYNC_NEW_GAME:
    strncat(pThis->szTypeToString, "RESULT_TYPE_NON_ROOT_MASTER_SYNC_NEW_GAME",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_NON_ROOT_MASTER_SYNC_SET_OPTION:
    strncat(pThis->szTypeToString, "RESULT_TYPE_NON_ROOT_MASTER_SYNC_SET_OPTION",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_MASTER_NO_WORK_AVAILABLE:
    strncat(pThis->szTypeToString, "RESULT_TYPE_MASTER_NO_WORK_AVAILABLE",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_MASTER_WORK_AVAILABLE:
    strncat(pThis->szTypeToString, "RESULT_TYPE_MASTER_WORK_AVAILABLE",  MAX_RESULT_TYPE_TO_STRING);
    break;

  case RESULT_TYPE_SYNC_AFTER_BROADCAST:
    strncat(pThis->szTypeToString, "RESULT_TYPE_SYNC_AFTER_BROADCAST",  MAX_RESULT_TYPE_TO_STRING);
    break;

  default:
    strncat(pThis->szTypeToString, "<undefined>",  MAX_RESULT_TYPE_TO_STRING);
    break;
  }

  return pThis->szTypeToString;
}

int Result_FillParameterIntoBuffer(Result* pThis)
{
  int nResult;
  int nProcessornameLength;

  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                pThis->nType,
                                pThis->pParameterBuffer,
                                MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                &pThis->nPosition,
                                MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    switch(pThis->nType)
    {
    case RESULT_TYPE_UNDEFINED:
      break;

    case RESULT_TYPE_NON_ROOT_MASTER_LOGON:
      nResult = MPIWrapper_Get_processor_name(MPIWrapper_Instance(), 
                                              pThis->sProcessorName, 
                                              &nProcessornameLength);
      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_STR(MPIWrapper_Instance(),
                                      pThis->sProcessorName, 
                                      pThis->pParameterBuffer, 
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_STR(... pThis->sProcessorName..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Get_processor_name(..., errorcode = %d\n", nResult);
        }
      }
      break;

    case RESULT_TYPE_NON_ROOT_MASTER_LOGOFF:
      nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                      g_pYBWCManagerInstance->nTotalNodes, 
                                      pThis->pParameterBuffer, 
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at g_pYBWCManagerInstance->nTotalNodes(... ., errorcode = %d\n", nResult);
        }
      }

      break;

    case RESULT_TYPE_NON_ROOT_MASTER_SHUTDOWN:
      break;

    case RESULT_TYPE_NON_ROOT_MASTER_REMOTE_TEST:
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    pThis->nBestMove, 
                                    pThis->pParameterBuffer, 
                                    MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                    &pThis->nPosition, 
                                    MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                      pThis->nBestValue, 
                                      pThis->pParameterBuffer, 
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                        pThis->nFlags, 
                                        pThis->pParameterBuffer, 
                                        MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
            nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                          pThis->nDepth, 
                                          pThis->pParameterBuffer, 
                                          MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                          &pThis->nPosition, 
                                          MPI_COMM_WORLD);
            if(nResult == MPI_SUCCESS)
            {
              nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                            pThis->nMaxDepth, 
                                            pThis->pParameterBuffer, 
                                            MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                            &pThis->nPosition, 
                                            MPI_COMM_WORLD);
              if(nResult == MPI_SUCCESS)
              {
                nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                        pThis->nPVLength,
                                                        pThis->pPV, 
                                                        pThis->pParameterBuffer, 
                                                        MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                        &pThis->nPosition, 
                                                        MPI_COMM_WORLD);
                if(nResult == MPI_SUCCESS)
                {
                  nResult = MPIWrapper_Pack_BOOL(MPIWrapper_Instance(),
                                                pThis->bPonderMoveOk, 
                                                pThis->pParameterBuffer, 
                                                MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                &pThis->nPosition, 
                                                MPI_COMM_WORLD);
                  if(nResult == MPI_SUCCESS)
                  {
                    nResult = MPIWrapper_Pack_DOUBLE(MPIWrapper_Instance(),
                                                      pThis->dTime, 
                                                      pThis->pParameterBuffer, 
                                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                      &pThis->nPosition, 
                                                      MPI_COMM_WORLD);
                    if(nResult == MPI_SUCCESS)
                    {
                      nResult = MPIWrapper_Pack_DOUBLE(MPIWrapper_Instance(),
                                                      pThis->dSpeed, 
                                                      pThis->pParameterBuffer, 
                                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                      &pThis->nPosition, 
                                                      MPI_COMM_WORLD);
                      if(nResult == MPI_SUCCESS)
                      {
                        nResult = MPIWrapper_Pack_DOUBLE(MPIWrapper_Instance(),
                                                        pThis->dCPU, 
                                                        pThis->pParameterBuffer, 
                                                        MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                        &pThis->nPosition, 
                                                        MPI_COMM_WORLD);
                        if(nResult == MPI_SUCCESS)
                        {
                          nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                                          pThis->nNodes, 
                                                          pThis->pParameterBuffer, 
                                                          MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                          &pThis->nPosition, 
                                                          MPI_COMM_WORLD);
                          if(nResult == MPI_SUCCESS)
                          {
                          }
                          else
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->nNodes..., errorcode = %d\n", nResult);
                            }
                          }
                        }
                        else
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_DOUBLE(... pThis->dCPU..., errorcode = %d\n", nResult);
                          }
                        }
                      }
                      else
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_DOUBLE(... pThis->dSpeed..., errorcode = %d\n", nResult);
                        }
                      }
                    }
                    else
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_DOUBLE(... pThis->dTime..., errorcode = %d\n", nResult);
                      }
                    }
                  }
                  else
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_BOOL(... pThis->bPonderMoveOk..., errorcode = %d\n", nResult);
                    }
                  }
                }
                else
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... pThis->nPVLength, pThis->pPV..., errorcode = %d\n", nResult);
                  }
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nMaxDepth..., errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nDepth..., errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nFlags..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nBestValue..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nBestMove..., errorcode = %d\n", nResult);
        }
      }
      break;

    case RESULT_TYPE_NON_ROOT_MASTER_SYNC_NEW_GAME:
      break;
    
    case RESULT_TYPE_NON_ROOT_MASTER_SYNC_SET_OPTION:
      break;

    case RESULT_TYPE_MASTER_NO_WORK_AVAILABLE:
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    pThis->nSyncPositionId, 
                                    pThis->pParameterBuffer, 
                                    MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                    &pThis->nPosition, 
                                    MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nSyncPositionId..., errorcode = %d\n", nResult);
        }
      }

      break;

    case RESULT_TYPE_MASTER_WORK_AVAILABLE:
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    g_pYBWCManagerInstance->nSyncPositionId, 
                                    pThis->pParameterBuffer, 
                                    MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                    &pThis->nPosition, 
                                    MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                        pThis->nWorkId, 
                                        pThis->pParameterBuffer, 
                                        MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                        pThis->nStackIndex, 
                                        pThis->pParameterBuffer, 
                                        MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
            nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                          pThis->nMoveListIndex,
                                          pThis->pParameterBuffer, 
                                          MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                          &pThis->nPosition, 
                                          MPI_COMM_WORLD);
            if(nResult == MPI_SUCCESS)
            {
              nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                            pThis->nCallType,
                                            pThis->pParameterBuffer, 
                                            MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                            &pThis->nPosition, 
                                            MPI_COMM_WORLD);
              if(nResult == MPI_SUCCESS)
              {
                nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                              pThis->nReturnPosition,
                                              pThis->pParameterBuffer, 
                                              MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                              &pThis->nPosition, 
                                              MPI_COMM_WORLD);
                if(nResult == MPI_SUCCESS)
                {
                  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                pThis->nMove,
                                                pThis->pParameterBuffer, 
                                                MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                &pThis->nPosition, 
                                                MPI_COMM_WORLD);
                  if(nResult == MPI_SUCCESS)
                  {
                    nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                  pThis->nDepth,
                                                  pThis->pParameterBuffer, 
                                                  MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                  &pThis->nPosition, 
                                                  MPI_COMM_WORLD);
                    if(nResult == MPI_SUCCESS)
                    {
                      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                    pThis->nNewDepth,
                                                    pThis->pParameterBuffer, 
                                                    MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                    &pThis->nPosition, 
                                                    MPI_COMM_WORLD);
                      if(nResult == MPI_SUCCESS)
                      {
                        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                      pThis->nHeight,
                                                      pThis->pParameterBuffer, 
                                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                      &pThis->nPosition, 
                                                      MPI_COMM_WORLD);
                        if(nResult == MPI_SUCCESS)
                        {
                          nResult = MPIWrapper_Pack_BOOL(MPIWrapper_Instance(),
                                                         pThis->bExtended,
                                                         pThis->pParameterBuffer, 
                                                         MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                         &pThis->nPosition, 
                                                         MPI_COMM_WORLD);
                          if(nResult == MPI_SUCCESS)
                          {
                            nResult = MPIWrapper_Pack_BOOL(MPIWrapper_Instance(),
                                                          pThis->bInPV,
                                                          pThis->pParameterBuffer, 
                                                          MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                          &pThis->nPosition, 
                                                          MPI_COMM_WORLD);
                            if(nResult == MPI_SUCCESS)
                            {
                              nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                            pThis->nPlayedNb,
                                                            pThis->pParameterBuffer, 
                                                            MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                            &pThis->nPosition, 
                                                            MPI_COMM_WORLD);
                              if(nResult == MPI_SUCCESS)
                              {
                                nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                              pThis->nHistoryValue,
                                                              pThis->pParameterBuffer, 
                                                              MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                              &pThis->nPosition, 
                                                              MPI_COMM_WORLD);
                                if(nResult == MPI_SUCCESS)
                                {
                                  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                                pThis->nAlpha,
                                                                pThis->pParameterBuffer, 
                                                                MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                &pThis->nPosition, 
                                                                MPI_COMM_WORLD);
                                  if(nResult == MPI_SUCCESS)
                                  {
                                    nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                                  pThis->nBeta,
                                                                  pThis->pParameterBuffer, 
                                                                  MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                  &pThis->nPosition, 
                                                                  MPI_COMM_WORLD);
                                    if(nResult == MPI_SUCCESS)
                                    {
                                      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                                    pThis->nTransDate,
                                                                    pThis->pParameterBuffer, 
                                                                    MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                    &pThis->nPosition, 
                                                                    MPI_COMM_WORLD);
                                      if(nResult == MPI_SUCCESS)
                                      {
                                        nResult = MPIWrapper_Pack_BOOL(MPIWrapper_Instance(),
                                                                       pThis->bGoodCap,
                                                                       pThis->pParameterBuffer, 
                                                                       MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                       &pThis->nPosition, 
                                                                       MPI_COMM_WORLD);
                                        if(nResult == MPI_SUCCESS)
                                        {
                                          nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                                                pThis->nMoveAheadListCount,
                                                                                pThis->pMoveAheadList, 
                                                                                pThis->pParameterBuffer, 
                                                                                MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                                &pThis->nPosition, 
                                                                                MPI_COMM_WORLD);
                                          if(nResult == MPI_SUCCESS)
                                          {
                                            nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                                                  pThis->nKillerListCount,
                                                                                  pThis->pKillerList, 
                                                                                  pThis->pParameterBuffer, 
                                                                                  MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                                  &pThis->nPosition, 
                                                                                  MPI_COMM_WORLD);
                                            if(nResult == MPI_SUCCESS)
                                            {
                                              if(g_pYBWCManagerInstance->pFruitConfiguration->bTransmitHistoryTable)
                                              {
                                                nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                                                      HistorySize,
                                                                                      gpHistory->History, 
                                                                                      pThis->pParameterBuffer, 
                                                                                      MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                                      &pThis->nPosition, 
                                                                                      MPI_COMM_WORLD);
                                                if(nResult == MPI_SUCCESS)
                                                {
                                                }
                                                else
                                                {
                                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                                  {
                                                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... HistorySize, gpHistory->History..., errorcode = %d\n", nResult);
                                                  }
                                                }
                                              }

                                              if(nResult == MPI_SUCCESS)
                                              {
                                                if(g_pYBWCManagerInstance->pFruitConfiguration->bTransmitHistTotTable)
                                                {
                                                  nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                                                        HistorySize,
                                                                                        gpHistory->HistTot, 
                                                                                        pThis->pParameterBuffer, 
                                                                                        MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                                        &pThis->nPosition, 
                                                                                        MPI_COMM_WORLD);
                                                  if(nResult == MPI_SUCCESS)
                                                  {
                                                    nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                                                          HistorySize,
                                                                                          gpHistory->HistHit, 
                                                                                          pThis->pParameterBuffer, 
                                                                                          MAX_RESULT_PARAMETER_BUFFER_SIZE, 
                                                                                          &pThis->nPosition, 
                                                                                          MPI_COMM_WORLD);
                                                    if(nResult == MPI_SUCCESS)
                                                    {
                                                    }
                                                    else
                                                    {
                                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                                      {
                                                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... HistorySize, gpHistory->HistHit..., errorcode = %d\n", nResult);
                                                      }
                                                    }
                                                  }
                                                  else
                                                  {
                                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                                    {
                                                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... HistorySize, gpHistory->HistTot..., errorcode = %d\n", nResult);
                                                    }
                                                  }
                                                }
                                              }
                                            }
                                            else
                                            {
                                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                              {
                                                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... pThis->nKillerListCount, pThis->pKillerList..., errorcode = %d\n", nResult);
                                              }
                                            }
                                          }
                                          else
                                          {
                                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                            {
                                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... pThis->nMoveAheadListCount, pThis->pMoveAheadList..., errorcode = %d\n", nResult);
                                            }
                                          }
                                        }
                                        else
                                        {
                                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                          {
                                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->bGoodCap..., errorcode = %d\n", nResult);
                                          }
                                        }
                                      }
                                      else
                                      {
                                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                        {
                                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nTransDate..., errorcode = %d\n", nResult);
                                        }
                                      }
                                    }
                                    else
                                    {
                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                      {
                                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nBeta..., errorcode = %d\n", nResult);
                                      }
                                    }
                                  }
                                  else
                                  {
                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                    {
                                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nAlpha..., errorcode = %d\n", nResult);
                                    }
                                  }
                                }
                                else
                                {
                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                  {
                                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nHistoryValue..., errorcode = %d\n", nResult);
                                  }
                                }
                              }
                              else
                              {
                                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                {
                                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nPlayedNb..., errorcode = %d\n", nResult);
                                }
                              }
                            }
                            else
                            {
                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                              {
                                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_BOOL(... pThis->bInPV..., errorcode = %d\n", nResult);
                              }
                            }
                          }
                          else
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_BOOL(... pThis->bExtended..., errorcode = %d\n", nResult);
                            }
                          }
                        }
                        else
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nHeight..., errorcode = %d\n", nResult);
                          }
                        }
                      }
                      else
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nNewDepth..., errorcode = %d\n", nResult);
                        }
                      }
                    }
                    else
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nDepth..., errorcode = %d\n", nResult);
                      }
                    }
                  }
                  else
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nMove..., errorcode = %d\n", nResult);
                    }
                  }
                }
                else
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nReturnPosition..., errorcode = %d\n", nResult);
                  }
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nCallType..., errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nMoveListIndex..., errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nStackIndex..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->nWorkId..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... g_pYBWCManagerInstance->nSyncPositionId..., errorcode = %d\n", nResult);
        }
      }
      break;

    case RESULT_TYPE_SYNC_AFTER_BROADCAST:
      break;

    default:
      ASSERT(false);
      break;
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nType..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitNonRootMasterLogon(Result* pThis)
{
  int nResult;

//pThis->nSourceId                = g_pYBWCManagerInstance->nRank;  // kh 01.08.06 implicitly set by MPIWrapper_Isend(...
//pThis->nDestId                  = nDestId;                        // kh 26.09.06 already implicitly set by ProcessorHandler_CreateResult(...

  pThis->nType                    = RESULT_TYPE_NON_ROOT_MASTER_LOGON;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitNonRootMasterLogonInvitation(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitNonRootMasterLogoff(Result* pThis)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_NON_ROOT_MASTER_LOGOFF;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitNonRootMasterLogoff(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitNonRootMasterShutdown(Result* pThis)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_NON_ROOT_MASTER_SHUTDOWN;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitNonRootMasterShutdown(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitNonRootMasterRemoteTest(Result* pThis,
                                       int     nBestMove,
                                       int     nBestValue,
                                       int     nFlags,
                                       int     nDepth,
                                       int     nMaxDepth,
                                       int     nPVLength,
                                       uint16  PV[],
                                       BOOL    bPonderMoveOk,
                                       double  dTime,
                                       double  dSpeed,
                                       double  dCPU,
                                       sint64  nNodes)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_NON_ROOT_MASTER_REMOTE_TEST;

  pThis->nBestMove                = nBestMove;
  pThis->nBestValue               = nBestValue;
  pThis->nFlags                   = nFlags;
  pThis->nDepth                   = nDepth;
  pThis->nMaxDepth                = nMaxDepth;
  pThis->nPVLength                = nPVLength,
  pThis->pPV                      = PV;
  pThis->bPonderMoveOk            = bPonderMoveOk;
  pThis->dTime                    = dTime;
  pThis->dSpeed                   = dSpeed;
  pThis->dCPU                     = dCPU;
  pThis->nNodes                   = nNodes;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitNonRootMasterRemoteTest(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitNonRootMasterSyncNewGame(Result*  pThis)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_NON_ROOT_MASTER_SYNC_NEW_GAME;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitNonRootMasterSyncNewGame(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitSyncAfterBroadcast(Result*  pThis)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_SYNC_AFTER_BROADCAST;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitSyncAfterBroadcast(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitNonRootMasterSyncSetOption(Result*  pThis)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_NON_ROOT_MASTER_SYNC_SET_OPTION;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitNonRootMasterSyncSetOption(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitMasterNoWorkAvailable(Result* pThis,
                                     int     nSlaveSyncPositionId)
{
  int nResult;

  pThis->nType                    = RESULT_TYPE_MASTER_NO_WORK_AVAILABLE;

// kh 20.09.06 send the sync position id of the request for work back to the slave 
// to be able to assert that the sync position id of the slave has not changed in between
// (may be done only in debug mode for a better performance)
  pThis->nSyncPositionId          = nSlaveSyncPositionId; 

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitMasterNoWorkAvailable(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_InitMasterWorkAvailable(Result*  pThis,
                                   int      nStackIndex,
                                   int      nMoveListIndex,
                                   int      nCallType, 
                                   int      nReturnPosition,
                                   int      nMove, 
                                   int      nDepth,
                                   int      nNewDepth,
                                   int      nHeight,
                                   BOOL     bInPV,
                                   int      nPlayedNb,
                                   int      nHistoryValue,
                                   int      nAlpha,
                                   int      nBeta,
                                   int      nTransDate,
                                   BOOL     bGoodCap,
                                   int      nMoveAheadListCount,
                                   uint16   MoveAheadList[],
                                   int      nKillerListCount,
                                   uint16   KillerList[])
{
  int nResult;

  g_pYBWCManagerInstance->nWorkId++;
  pThis->nWorkId                  = g_pYBWCManagerInstance->nWorkId;

  pThis->nStackIndex              = nStackIndex;
  pThis->nMoveListIndex           = nMoveListIndex;
  pThis->nCallType                = nCallType;
  pThis->nReturnPosition          = nReturnPosition;
  pThis->nMove                    = nMove,
  pThis->nDepth                   = nDepth;
  pThis->nNewDepth                = nNewDepth;
  pThis->nHeight                  = nHeight;
  pThis->bInPV                    = bInPV;
  pThis->nPlayedNb                = nPlayedNb;
  pThis->nHistoryValue            = nHistoryValue;
  pThis->nAlpha                   = nAlpha,
  pThis->nBeta                    = nBeta;
  pThis->nTransDate               = nTransDate;
  pThis->bGoodCap                 = bGoodCap;
  pThis->nMoveAheadListCount      = nMoveAheadListCount,
  pThis->pMoveAheadList           = MoveAheadList;
  pThis->nKillerListCount         = nKillerListCount,
  pThis->pKillerList              = KillerList;

  pThis->nType                    = RESULT_TYPE_MASTER_WORK_AVAILABLE;

  nResult = Result_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_InitMasterNoWorkAvailable(... failed at Result_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Result_Send(Result* pThis)
{
  int               nResult;

  ProcessorHandler* pProcessorHandler;

  ASSERT(pThis->mpiRequest == MPI_REQUEST_NULL);
  nResult = MPIWrapper_Isend(MPIWrapper_Instance(),
                             pThis->pParameterBuffer, 
                             pThis->nPosition, 
                             MPI_PACKED, 
                             pThis->nDestId, 
                             YBWC_MANAGER_RESULT_TAG, 
                             MPI_COMM_WORLD,
                             &pThis->mpiRequest);

  if(nResult == MPI_SUCCESS)
  {

// kh 14.12.06 tuning 
    pProcessorHandler = g_pYBWCManagerInstance->processorHandlerPool[pThis->nDestId];

    nResult = ProcessorHandler_ForcedWaitForAllSendsCompleted(pProcessorHandler);

    if(nResult == MPI_SUCCESS)
    {
    }
    else
    {
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_Send(... failed at ProcessorHandler_ForcedWaitForAllSendsCompleted(..., errorcode = %d\n", nResult);
      }
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_Send(... failed at MPIWrapper_Isend(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

BOOL Result_IsSendPending(Result* pThis)
{
  BOOL        bResult;
  int         nResult;

  int         nFlag;
  MPI_Status  mpiStatus;

  bResult = FALSE;

  if(pThis->bSendSuppressed)
  {
  }
  else
  {
    if(pThis->mpiRequest == MPI_REQUEST_NULL)
    {
      bResult = TRUE; // kh 26.09.06 not send yet
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_WARNING_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "WARNING Result_IsSendPending(... pThis->mpiRequest == MPI_REQUEST_NULL\n");
      }
    }
    else
    {
      nResult = MPIWrapper_Test(MPIWrapper_Instance(),
                                &pThis->mpiRequest,
                                &nFlag,
                                &mpiStatus);

      if(nResult == MPI_SUCCESS) 
      {
        if(nFlag)
        {
        }
        else
        {
          bResult = TRUE;

          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_3)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ProcessorHandler <%d> Result_IsSendPending(... send operation is not completed\n", pThis->nDestId);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_IsSendPending(... failed at MPIWrapper_Test(..., errorcode = %d\n", nResult);
        }
      }
    }
  }

  return bResult;
}

int Result_GetParameterFromBuffer(Result* pThis)
{
  int nResult;

// kh 02.08.06 get pResult->nType from receive buffer
  pThis->nPosition = 0;
  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                  pThis->pParameterBuffer,
                                  MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                  &pThis->nPosition,
                                  &pThis->nType,
                                  MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, 
                         "Result received from processor <%d>, Type = %d <%s>\n",
                         pThis->nSourceId,
                         pThis->nType,
                         Result_TypeToString(pThis));
    }

    switch(pThis->nType)
    {
    case RESULT_TYPE_UNDEFINED:
      break;

    case RESULT_TYPE_NON_ROOT_MASTER_LOGON:
      nResult = MPIWrapper_Unpack_STR(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      pThis->sProcessorName,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "Processor name is <%s>\n", pThis->sProcessorName);
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_STR(... pThis->sProcessorName..., errorcode = %d\n", nResult);
        }
      }

      break;

    case RESULT_TYPE_NON_ROOT_MASTER_LOGOFF:
      nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        &pThis->nTotalNodes,
                                        MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nTotalNodes <" F12INT64d ">\n", pThis->nTotalNodes);
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nTotalNodes... errorcode = %d\n", nResult);
        }
      }

      break;

    case RESULT_TYPE_NON_ROOT_MASTER_SHUTDOWN:
      break;

    case RESULT_TYPE_NON_ROOT_MASTER_REMOTE_TEST:
      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      &pThis->nBestMove,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nBestMove = <%d>\n", pThis->nBestMove);
        }

        nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        &pThis->nBestValue,
                                        MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "nBestValue = <%d>\n", pThis->nBestValue);
          }

          nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &pThis->nFlags,
                                          MPI_COMM_WORLD);

          if(nResult == MPI_SUCCESS)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "nFlags = <%d>\n", pThis->nFlags);
            }

            nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                            pThis->pParameterBuffer,
                                            MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                            &pThis->nPosition,
                                            &pThis->nDepth,
                                            MPI_COMM_WORLD);

            if(nResult == MPI_SUCCESS)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "nDepth = <%d>\n", pThis->nDepth);
              }

              nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                              pThis->pParameterBuffer,
                                              MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                              &pThis->nPosition,
                                              &pThis->nMaxDepth,
                                              MPI_COMM_WORLD);

              if(nResult == MPI_SUCCESS)
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "nMaxDepth = <%d>\n", pThis->nMaxDepth);
                }

                nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                         pThis->pParameterBuffer,
                                                         MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                         &pThis->nPosition,
                                                         &pThis->nPVLength,
                                                         pThis->PV,
                                                         MPI_COMM_WORLD);

                if(nResult == MPI_SUCCESS)
                {
                  ASSERT(pThis->nPVLength < MAX_PV_LENGTH);
                  pThis->PV[pThis->nPVLength] = MoveNone; // kh 26.09.06 mark the end of the list

                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "nPVLength = <%d>\n", pThis->nPVLength);

// kh 05.09.06 future expansion (fruit functions may be helpful here)
//                  YBWCManager_Printf(g_pYBWCManagerInstance, "PV = <n * %d>\n", pThis->PV);
                  }

                  nResult = MPIWrapper_Unpack_BOOL(MPIWrapper_Instance(), 
                                                    pThis->pParameterBuffer,
                                                    MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                    &pThis->nPosition,
                                                    &pThis->bPonderMoveOk,
                                                    MPI_COMM_WORLD);

                  if(nResult == MPI_SUCCESS)
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "bPonderMoveOk = <%d>\n", pThis->bPonderMoveOk);
                    }

                    nResult = MPIWrapper_Unpack_DOUBLE(MPIWrapper_Instance(), 
                                                       pThis->pParameterBuffer,
                                                       MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                       &pThis->nPosition,
                                                       &pThis->dTime,
                                                       MPI_COMM_WORLD);

                    if(nResult == MPI_SUCCESS)
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "dTime <%9.2f>\n", pThis->dTime);
                      }

                      nResult = MPIWrapper_Unpack_DOUBLE(MPIWrapper_Instance(), 
                                                         pThis->pParameterBuffer,
                                                         MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                         &pThis->nPosition,
                                                         &pThis->dSpeed,
                                                         MPI_COMM_WORLD);

                      if(nResult == MPI_SUCCESS)
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "dSpeed <%9.2f>\n", pThis->dSpeed);
                        }

                        nResult = MPIWrapper_Unpack_DOUBLE(MPIWrapper_Instance(), 
                                                           pThis->pParameterBuffer,
                                                           MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                           &pThis->nPosition,
                                                           &pThis->dCPU,
                                                           MPI_COMM_WORLD);

                        if(nResult == MPI_SUCCESS)
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "dCPU <%7.2f>\n", pThis->dCPU);
                          }

                          nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                                            pThis->pParameterBuffer,
                                                            MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                            &pThis->nPosition,
                                                            &pThis->nNodes,
                                                            MPI_COMM_WORLD);

                          if(nResult == MPI_SUCCESS)
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "nNodes <" F12INT64d ">\n", pThis->nNodes);
                            }
                          }
                          else
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nNodes... errorcode = %d\n", nResult);
                            }
                          }
                        }
                        else
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_DOUBLE(..., &pThis->dCPU... errorcode = %d\n", nResult);
                          }
                        }
                      }
                      else
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_DOUBLE(..., &pThis->dSpeed... errorcode = %d\n", nResult);
                        }
                      }
                    }
                    else
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_DOUBLE(..., &pThis->dTime... errorcode = %d\n", nResult);
                      }
                    }
                  }
                  else
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Pack_BOOL(..., &pThis->bPonderMoveOk... errorcode = %d\n", nResult);
                    }
                  }
                }
                else
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nPVLength, pThis->PV... errorcode = %d\n", nResult);
                  }
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nMaxDepth... errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nDepth... errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nFlags... errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nBestValue... errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nBestMove... errorcode = %d\n", nResult);
        }
      }
      break;

    case RESULT_TYPE_MASTER_NO_WORK_AVAILABLE:
      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      &pThis->nSyncPositionId,
                                      MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nSyncPositionId... errorcode = %d\n", nResult);
        }
      }

      break;

    case RESULT_TYPE_MASTER_WORK_AVAILABLE:
      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      &pThis->nSyncPositionId,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nSyncPositionId = <%d>\n", pThis->nSyncPositionId);
        }

        nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &pThis->nWorkId,
                                          MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "nWorkId = <" F12INT64d ">\n", pThis->nWorkId);
          }

          nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &pThis->nStackIndex,
                                          MPI_COMM_WORLD);

          if(nResult == MPI_SUCCESS)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "nStackIndex = <%d>\n", pThis->nStackIndex);
            }

            nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                            pThis->pParameterBuffer,
                                            MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                            &pThis->nPosition,
                                            &pThis->nMoveListIndex,
                                            MPI_COMM_WORLD);

            if(nResult == MPI_SUCCESS)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "nMoveListIndex = <%d>\n", pThis->nMoveListIndex);
              }

              nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                              pThis->pParameterBuffer,
                                              MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                              &pThis->nPosition,
                                              &pThis->nCallType,
                                              MPI_COMM_WORLD);

              if(nResult == MPI_SUCCESS)
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "nCallType = <%d>\n", pThis->nCallType);
                }

                nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                pThis->pParameterBuffer,
                                                MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                &pThis->nPosition,
                                                &pThis->nReturnPosition,
                                                MPI_COMM_WORLD);

                if(nResult == MPI_SUCCESS)
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "nReturnPosition = <%d>\n", pThis->nReturnPosition);
                  }

                  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                  pThis->pParameterBuffer,
                                                  MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                  &pThis->nPosition,
                                                  &pThis->nMove,
                                                  MPI_COMM_WORLD);

                  if(nResult == MPI_SUCCESS)
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "nMove = <%d>\n", pThis->nMove);
                    }

                    nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                    pThis->pParameterBuffer,
                                                    MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                    &pThis->nPosition,
                                                    &pThis->nDepth,
                                                    MPI_COMM_WORLD);

                    if(nResult == MPI_SUCCESS)
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "nDepth = <%d>\n", pThis->nDepth);
                      }

                      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                      pThis->pParameterBuffer,
                                                      MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                      &pThis->nPosition,
                                                      &pThis->nNewDepth,
                                                      MPI_COMM_WORLD);

                      if(nResult == MPI_SUCCESS)
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "nNewDepth = <%d>\n", pThis->nNewDepth);
                        }

                        nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                        pThis->pParameterBuffer,
                                                        MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                        &pThis->nPosition,
                                                        &pThis->nHeight,
                                                        MPI_COMM_WORLD);

                        if(nResult == MPI_SUCCESS)
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "nHeight = <%d>\n", pThis->nHeight);
                          }

                          nResult = MPIWrapper_Unpack_BOOL(MPIWrapper_Instance(), 
                                                           pThis->pParameterBuffer,
                                                           MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                           &pThis->nPosition,
                                                           &pThis->bExtended,
                                                           MPI_COMM_WORLD);

                          if(nResult == MPI_SUCCESS)
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "bExtended = <%d>\n", pThis->bExtended);
                            }

                            nResult = MPIWrapper_Unpack_BOOL(MPIWrapper_Instance(), 
                                                             pThis->pParameterBuffer,
                                                             MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                             &pThis->nPosition,
                                                             &pThis->bInPV,
                                                             MPI_COMM_WORLD);

                            if(nResult == MPI_SUCCESS)
                            {
                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                              {
                                YBWCManager_Printf(g_pYBWCManagerInstance, "bInPV = <%d>\n", pThis->bInPV);
                              }

                              nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                              pThis->pParameterBuffer,
                                                              MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                              &pThis->nPosition,
                                                              &pThis->nPlayedNb,
                                                              MPI_COMM_WORLD);

                              if(nResult == MPI_SUCCESS)
                              {
                                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                {
                                  YBWCManager_Printf(g_pYBWCManagerInstance, "nPlayedNb = <%d>\n", pThis->nPlayedNb);
                                }

// kh 17.06.08 sufficient at the moment, no need to really transmit the flag
// used for root moves (of the root master) only at the moment
                                pThis->bAlphaBoundIsEstimated = (pThis->nPlayedNb <= 0); 

                                nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                                pThis->pParameterBuffer,
                                                                MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                &pThis->nPosition,
                                                                &pThis->nHistoryValue,
                                                                MPI_COMM_WORLD);

                                if(nResult == MPI_SUCCESS)
                                {
                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                  {
                                    YBWCManager_Printf(g_pYBWCManagerInstance, "nHistoryValue = <%d>\n", pThis->nHistoryValue);
                                  }

                                  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                                  pThis->pParameterBuffer,
                                                                  MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                  &pThis->nPosition,
                                                                  &pThis->nAlpha,
                                                                  MPI_COMM_WORLD);

                                  if(nResult == MPI_SUCCESS)
                                  {
                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                    {
                                      YBWCManager_Printf(g_pYBWCManagerInstance, "nAlpha = <%d>\n", pThis->nAlpha);
                                    }

                                    nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                                    pThis->pParameterBuffer,
                                                                    MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                    &pThis->nPosition,
                                                                    &pThis->nBeta,
                                                                    MPI_COMM_WORLD);

                                    if(nResult == MPI_SUCCESS)
                                    {
                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                      {
                                        YBWCManager_Printf(g_pYBWCManagerInstance, "nBeta = <%d>\n", pThis->nBeta);
                                      }

                                      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                                      pThis->pParameterBuffer,
                                                                      MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                      &pThis->nPosition,
                                                                      &pThis->nTransDate,
                                                                      MPI_COMM_WORLD);

                                      if(nResult == MPI_SUCCESS)
                                      {
                                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                        {
                                          YBWCManager_Printf(g_pYBWCManagerInstance, "nTransDate = <%d>\n", pThis->nTransDate);
                                        }

                                        nResult = MPIWrapper_Unpack_BOOL(MPIWrapper_Instance(), 
                                                                        pThis->pParameterBuffer,
                                                                        MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                        &pThis->nPosition,
                                                                        &pThis->bGoodCap,
                                                                        MPI_COMM_WORLD);

                                        if(nResult == MPI_SUCCESS)
                                        {
                                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                          {
                                            YBWCManager_Printf(g_pYBWCManagerInstance, "bGoodCap = <%d>\n", pThis->bGoodCap);
                                          }

                                          nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                                                  pThis->pParameterBuffer,
                                                                                  MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                                  &pThis->nPosition,
                                                                                  &pThis->nMoveAheadListCount,
                                                                                  pThis->MoveAheadList,
                                                                                  MPI_COMM_WORLD);

                                          if(nResult == MPI_SUCCESS)
                                          {
                                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                            {
                                              YBWCManager_Printf(g_pYBWCManagerInstance, "nMoveAheadListCount = <%d>\n", pThis->nMoveAheadListCount);

// kh 12.09.06 future expansion (fruit functions may be helpful here)
//                                            YBWCManager_Printf(g_pYBWCManagerInstance, "MoveAheadList = <n * %d>\n", pThis->MoveAheadList);
                                            }

                                            nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                                                    pThis->pParameterBuffer,
                                                                                    MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                                    &pThis->nPosition,
                                                                                    &pThis->nKillerListCount,
                                                                                    pThis->KillerList,
                                                                                    MPI_COMM_WORLD);

                                            if(nResult == MPI_SUCCESS)
                                            {
                                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                              {
                                                YBWCManager_Printf(g_pYBWCManagerInstance, "nKillerListCount = <%d>\n", pThis->nKillerListCount);

// kh 14.02.07 future expansion (fruit functions may be helpful here)
//                                              YBWCManager_Printf(g_pYBWCManagerInstance, "KillerList = <n * %d>\n", pThis->KillerList);
                                              }

                                              if(g_pYBWCManagerInstance->pFruitConfiguration->bTransmitHistoryTable)
                                              {
                                                nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                                                        pThis->pParameterBuffer,
                                                                                        MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                                        &pThis->nPosition,
                                                                                        &pThis->nHistoryTableSize,
                                                                                        gpHistory->History, // pThis->HistoryTable,
                                                                                        MPI_COMM_WORLD);

                                                if(nResult == MPI_SUCCESS)
                                                {
                                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                                  {
                                                    YBWCManager_Printf(g_pYBWCManagerInstance, "nHistoryTableSize = <%d>\n", pThis->nHistoryTableSize);

// kh 29.03.07 future expansion (fruit functions may be helpful here)
//                                                  YBWCManager_Printf(g_pYBWCManagerInstance, "HistoryTable = <n * %d>\n", gpHistory->History);
                                                  }

                                                  ASSERT(pThis->nHistoryTableSize == HistorySize);
                                                }
                                                else
                                                {
                                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                                  {
                                                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetHistoryTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nHistoryTableCount, gpHistory->History... errorcode = %d\n", nResult);
                                                  }
                                                }
                                              }

                                              if(nResult == MPI_SUCCESS)
                                              {
                                                if(g_pYBWCManagerInstance->pFruitConfiguration->bTransmitHistTotTable)
                                                {
                                                  nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                                                          pThis->pParameterBuffer,
                                                                                          MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                                          &pThis->nPosition,
                                                                                          &pThis->nHistTotTableSize,
                                                                                          gpHistory->HistTot, // pThis->HistTotTable,
                                                                                          MPI_COMM_WORLD);

                                                  if(nResult == MPI_SUCCESS)
                                                  {
                                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                                    {
                                                      YBWCManager_Printf(g_pYBWCManagerInstance, "nHistTotTableSize = <%d>\n", pThis->nHistTotTableSize);

// kh 29.03.07 future expansion (fruit functions may be helpful here)
//                                                    YBWCManager_Printf(g_pYBWCManagerInstance, "HistTotTable = <n * %d>\n", gpHistory->HistTot);
                                                    }

                                                    ASSERT(pThis->nHistTotTableSize == HistorySize);

                                                    nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                                                            pThis->pParameterBuffer,
                                                                                            MAX_RESULT_PARAMETER_BUFFER_SIZE,
                                                                                            &pThis->nPosition,
                                                                                            &pThis->nHistHitTableSize,
                                                                                            gpHistory->HistHit, // pThis->HistHitTable,
                                                                                            MPI_COMM_WORLD);

                                                    if(nResult == MPI_SUCCESS)
                                                    {
                                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                                      {
                                                        YBWCManager_Printf(g_pYBWCManagerInstance, "nHistHitTableSize = <%d>\n", pThis->nHistHitTableSize);

// kh 29.03.07 future expansion (fruit functions may be helpful here)
//                                                      YBWCManager_Printf(g_pYBWCManagerInstance, "HistHitTable = <n * %d>\n", gpHistory->HistHit);
                                                      }

                                                      ASSERT(pThis->nHistHitTableSize == HistorySize);

                                                    }
                                                    else
                                                    {
                                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                                      {
                                                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nHistTotTableCount, gpHistory->HistHit... errorcode = %d\n", nResult);
                                                      }
                                                    }
                                                  }
                                                  else
                                                  {
                                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                                    {
                                                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nHistTotTableCount, gpHistory->HistTot... errorcode = %d\n", nResult);
                                                    }
                                                  }
                                                }
                                              }
                                            }
                                            else
                                            {
                                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                              {
                                                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nKillerListCount, pThis->KillerList... errorcode = %d\n", nResult);
                                              }
                                            }
                                          }
                                          else
                                          {
                                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                            {
                                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nMoveAheadListCount, pThis->MoveAheadList... errorcode = %d\n", nResult);
                                            }
                                          }
                                        }
                                        else
                                        {
                                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                          {
                                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_BOOL(..., &pThis->bGoodCap... errorcode = %d\n", nResult);
                                          }
                                        }
                                      }
                                      else
                                      {
                                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                        {
                                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nTransDate... errorcode = %d\n", nResult);
                                        }
                                      }
                                    }
                                    else
                                    {
                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                      {
                                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nBeta... errorcode = %d\n", nResult);
                                      }
                                    }
                                  }
                                  else
                                  {
                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                    {
                                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nAlpha... errorcode = %d\n", nResult);
                                    }
                                  }
                                }
                                else
                                {
                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                  {
                                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nHistoryValue... errorcode = %d\n", nResult);
                                  }
                                }
                              }
                              else
                              {
                                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                {
                                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nPlayedNb... errorcode = %d\n", nResult);
                                }
                              }
                            }
                            else
                            {
                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                              {
                                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_BOOL(..., &pThis->bInPV... errorcode = %d\n", nResult);
                              }
                            }
                          }
                          else
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_BOOL(..., &pThis->bExtended... errorcode = %d\n", nResult);
                            }
                          }
                        }
                        else
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nHeight... errorcode = %d\n", nResult);
                          }
                        }
                      }
                      else
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nNewDepth... errorcode = %d\n", nResult);
                        }
                      }
                    }
                    else
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nDepth... errorcode = %d\n", nResult);
                      }
                    }
                  }
                  else
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nMove... errorcode = %d\n", nResult);
                    }
                  }
                }
                else
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nReturnPosition... errorcode = %d\n", nResult);
                  }
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nCallType... errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nMoveListIndex... errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nStackIndex... errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nWorkId... errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nSyncPositionId... errorcode = %d\n", nResult);
        }
      }
      break;

    case RESULT_TYPE_SYNC_AFTER_BROADCAST:
      break;

    default:
      break;
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Result_GetParameterFromBuffer(... failed at MPIWrapper_Pack_INT(... &pThis->nType..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}
