#include "Request.h"

#include <stdlib.h>

#include <string.h>

#if defined(_WIN32) || defined(_WIN64)

  extern "C"
  {
    #include "MPIWrapper.h"
  }

#else // assume POSIX

  extern "C"
  {
    #include "MPIWrapper.h"
  }

#endif

#include "YBWCManager.h"
extern YBWCManager* g_pYBWCManagerInstance;

Request* Request_Construct(Request* pThis)
{
  pThis->pNext                    = NULL;

  pThis->pBroadcastRequest        = NULL;
  pThis->nReferenceCount          = 0;
  pThis->bPreventDestruct         = FALSE;

  pThis->nSourceId                = -1;
  pThis->nDestId                  = -1;
  pThis->pParameterBuffer[0]      = '\0'; // kh 01.08.06 defensive
  pThis->mpiRequest               = MPI_REQUEST_NULL;
  pThis->nPosition                = 0;

  pThis->bSendSuppressed          = FALSE;

  pThis->bDispatchedFirst         = TRUE;
  pThis->bDelayedDispatch         = FALSE;

  pThis->nType                    = REQUEST_TYPE_UNDEFINED;
  pThis->szTypeToString[0]        = '\0';

  pThis->psSyncPosition           = NULL;
  pThis->nSyncPositionId          = -1;

  pThis->psOptionName             = NULL;
  pThis->psOptionValue            = NULL;

  pThis->psGoCommands             = NULL;

  pThis->nBestMove                = 0;
  pThis->nBestValue               = 0;
  pThis->nFlags                   = 0;
  pThis->nDepth                   = 0;
  pThis->nMaxDepth                = 0;

// kh 11.10.06 first element only for a better performance
  pThis->nPVLength                = 0; 
  pThis->PV[0]                    = 0; 

  pThis->pPV                      = NULL; // kh 11.10.06 used in Request_FillParameterIntoBuffer(...

  pThis->bPonderMoveOk            = FALSE;
  pThis->dTime                    = 0.0;
  pThis->dSpeed                   = 0.0;
  pThis->dCPU                     = 0.0;
  pThis->nNodes                   = 0;
  pThis->nTransUsed               = 0;

  pThis->nWorkId                  = -2;

  pThis->nStackIndex              = -1;
  pThis->nMoveListIndex           = -1;

  pThis->nAlpha                   = 0;
  pThis->nBeta                    = 0;

  pThis->nHistoryTableSize        = -1;
  pThis->HistoryTable[0]          = 0;

  pThis->nHistTotTableSize        = -1;
  pThis->HistTotTable[0]          = 0;

  pThis->nHistHitTableSize        = -1;
  pThis->HistHitTable[0]          = 0;

  pThis->pHashTableBuffer         = NULL;

  return pThis;
}

void Request_Destruct(Request* pThis)
{
  if(pThis)
  {
    if(pThis->psSyncPosition)
    {
      free(pThis->psSyncPosition);
    }

    if(pThis->psOptionName)
    {
      free(pThis->psOptionName);
    }

    if(pThis->psOptionValue)
    {
      free(pThis->psOptionValue);
    }

    if(pThis->psGoCommands)
    {
      free(pThis->psGoCommands);
    }

    if(pThis->pBroadcastRequest)
    {
      ASSERT(pThis->pBroadcastRequest->nReferenceCount > 0);
      pThis->pBroadcastRequest->nReferenceCount--;
      if(pThis->pBroadcastRequest->nReferenceCount == 0) 
      {
        if(pThis->pBroadcastRequest->bPreventDestruct)
        {

          // kh 07.12.06 buffer is still in use and a least one new reference is expected
          // so do not destruct the buffer too early

          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "INFO buffer destruction prevented\n");
          }
        } // if(pThis->pBroadcastRequest->bPreventDestruct)
        else
        {

// kh 06.12.06 there are no more reference to the broadcast buffer 
          Request_Destruct(pThis->pBroadcastRequest); 
        } // !if(pThis->pBroadcastRequest->bPreventDestruct)
      }
    }

    free(pThis);
  }
}

char* Request_TypeToString(Request* pThis)
{
  pThis->szTypeToString[0] = '\0';

  switch(pThis->nType)
  {
  case REQUEST_TYPE_UNDEFINED:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_UNDEFINED",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_LOGON_INVITATION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_LOGON_INVITATION",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_LOGOFF_INVITATION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_LOGOFF_INVITATION",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_SHUTDOWN:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_SHUTDOWN",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_IDLE_INVITATION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_IDLE_INVITATION",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_SYNC_POSITION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_SYNC_POSITION",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_SUPPORT_SEARCH:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_SUPPORT_SEARCH",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_STOP_REMOTE_SEARCH:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_STOP_REMOTE_SEARCH",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_NEW_GAME:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_NEW_GAME",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_SET_OPTION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_SET_OPTION",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_REMOTE_TEST:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_REMOTE_TEST",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_ROOT_MASTER_MACHINE_INFO:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_ROOT_MASTER_MACHINE_INFO",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_NON_ROOT_MASTER_SYNC_AFTER_BROADCAST:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_NON_ROOT_MASTER_SYNC_AFTER_BROADCAST",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_SLAVE_TRY_TO_GET_WORK:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_SLAVE_TRY_TO_GET_WORK",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_MASTER_STOP:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_MASTER_STOP",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_SLAVE_PROCESS_SEARCH_RESULT:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_SLAVE_PROCESS_SEARCH_RESULT",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_MASTER_UPDATE_SEARCH_WINDOW:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_MASTER_UPDATE_SEARCH_WINDOW",  MAX_REQUEST_TYPE_TO_STRING);
    break;

  case REQUEST_TYPE_HASH_TABLE_REPLICATION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_HASH_TABLE_REPLICATION",  MAX_REQUEST_TYPE_TO_STRING);
    break;
   
  case REQUEST_TYPE_HASH_TABLE_BROADCAST_BUFFER:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_HASH_TABLE_BROADCAST_BUFFER",  MAX_REQUEST_TYPE_TO_STRING);
    break;
   
  case REQUEST_TYPE_HISTORY_TABLE_REPLICATION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_HISTORY_TABLE_REPLICATION",  MAX_REQUEST_TYPE_TO_STRING);
    break;
   
  case REQUEST_TYPE_HISTORY_TABLE_BROADCAST_BUFFER:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_HISTORY_TABLE_BROADCAST_BUFFER",  MAX_REQUEST_TYPE_TO_STRING);
    break;
   
  case REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION",  MAX_REQUEST_TYPE_TO_STRING);
    break;
   
  case REQUEST_TYPE_HIST_TOT_TABLE_BROADCAST_BUFFER:
    strncat(pThis->szTypeToString, "REQUEST_TYPE_HIST_TOT_TABLE_BROADCAST_BUFFER",  MAX_REQUEST_TYPE_TO_STRING);
    break;
   
  default:
    strncat(pThis->szTypeToString, "<undefined>",  MAX_REQUEST_TYPE_TO_STRING);
    break;
  }

  return pThis->szTypeToString;
}

int Request_FillHashTableBufferParameterIntoBuffer(Request* pThis)
{
  int             nResult;
  int             nResultTmp;

  int             i;

  HashTableEntry* pHashTableEntryTmp;

  nResultTmp = MPI_SUCCESS;

  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                pThis->pHashTableBuffer->nCount, // kh 15.11.06 pThis->pHashTableBuffer is the global transmit buffer
                                pThis->pParameterBuffer,
                                MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                &pThis->nPosition,
                                MPI_COMM_WORLD);
  if(nResult == MPI_SUCCESS)
  {
    for(i = 0; i < pThis->pHashTableBuffer->nCount; i++)
    {
      pHashTableEntryTmp = &pThis->pHashTableBuffer->hashTableEntry[i];

      nResult = MPIWrapper_Pack_UINT64(MPIWrapper_Instance(),
                                       pHashTableEntryTmp->nKey, 
                                       pThis->pParameterBuffer, 
                                       MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                       &pThis->nPosition, 
                                       MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                      pHashTableEntryTmp->nMove, 
                                      pThis->pParameterBuffer, 
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                        pHashTableEntryTmp->nDepth, 
                                        pThis->pParameterBuffer, 
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
            nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                          pHashTableEntryTmp->nFlags, 
                                          pThis->pParameterBuffer, 
                                          MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                          &pThis->nPosition, 
                                          MPI_COMM_WORLD);
            if(nResult == MPI_SUCCESS)
            {
              nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                            pHashTableEntryTmp->nValue, 
                                            pThis->pParameterBuffer, 
                                            MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                            &pThis->nPosition, 
                                            MPI_COMM_WORLD);
              if(nResult == MPI_SUCCESS)
              {
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pHashTableEntryTmp->nValue..., errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed MPIWrapper_Pack_INT(... pHashTableEntryTmp->nValue..., errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed MPIWrapper_Pack_INT(... pHashTableEntryTmp->nDepth..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed MPIWrapper_Pack_INT(... pHashTableEntryTmp->nMove..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed MPIWrapper_Pack_UINT64(... pHashTableEntryTmp->nKey..., errorcode = %d\n", nResult);
        }
      }

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {

// kh 15.11.06 save first error only
        if(nResultTmp == MPI_SUCCESS)
        {
          nResultTmp = nResult;
        }
      }
    } // for(i = 0; i < pThis->pHashTableBuffer->nCount; i++)

    nResult = nResultTmp;
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pHashTableBuffer->nCount..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_FillMachineInfoParameterIntoBuffer(Request* pThis)
{
  int               nResult;
  int               nResultTmp;

  ProcessorHandler* pProcessorHandler;

  int               i;

  nResultTmp = MPI_SUCCESS;

// kh 13.12.06 the size is known in this special case
/*
  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                pThis-><nAppropriateCount>, 
                                pThis->pParameterBuffer,
                                MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                &pThis->nPosition,
                                MPI_COMM_WORLD);
*/

  nResult = MPI_SUCCESS;
  if(nResult == MPI_SUCCESS)
  {
    for(i = 0; i < g_pYBWCManagerInstance->nSize; i++)
    {
      pProcessorHandler = g_pYBWCManagerInstance->processorHandlerPool[i];

      nResult = MPIWrapper_Pack_STR(MPIWrapper_Instance(),
                                    pProcessorHandler->sProcessorName, 
                                    pThis->pParameterBuffer, 
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                    &pThis->nPosition, 
                                    MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillMachineInfoParameterIntoBuffer(... failed at MPIWrapper_Pack_STR(... pProcessorHandler->sProcessorName..., errorcode = %d\n", nResult);
        }
      }

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {

// kh 13.12.06 save first error only
        if(nResultTmp == MPI_SUCCESS)
        {
          nResultTmp = nResult;
        }
      }
    } // for(i = 0; i < g_pYBWCManagerInstance->nSize; i++)

    nResult = nResultTmp;
  }
  else
  {

// kh 13.12.06 not used at the moment
/*
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillHashTableBufferParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... ..., errorcode = %d\n", nResult);
    }
*/
  }

  return nResult;
}

int Request_FillParameterIntoBuffer(Request* pThis)
{
  int nResult;

  pThis->nPosition = 0;
  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                pThis->nType,
                                pThis->pParameterBuffer,
                                MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                &pThis->nPosition,
                                MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    switch(pThis->nType)
    {
    case REQUEST_TYPE_UNDEFINED:
      break;

    case REQUEST_TYPE_ROOT_MASTER_LOGON_INVITATION:
      break;

    case REQUEST_TYPE_ROOT_MASTER_LOGOFF_INVITATION:
      break;

    case REQUEST_TYPE_ROOT_MASTER_SHUTDOWN:
      break;

    case REQUEST_TYPE_ROOT_MASTER_IDLE_INVITATION:
      break;

    case REQUEST_TYPE_ROOT_MASTER_SYNC_POSITION:
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(), 
                                    g_pYBWCManagerInstance->nSyncPositionId,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_STR(MPIWrapper_Instance(), 
                                      pThis->psSyncPosition,
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_STR(... pThis->psSyncPosition..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... g_pYBWCManagerInstance->nSyncPositionId, errorcode = %d\n", nResult);
        }
      }

      break;

    case REQUEST_TYPE_ROOT_MASTER_SUPPORT_SEARCH:
      break;

    case REQUEST_TYPE_ROOT_MASTER_STOP_REMOTE_SEARCH:
      break;

    case REQUEST_TYPE_ROOT_MASTER_NEW_GAME:
      break;

    case REQUEST_TYPE_ROOT_MASTER_SET_OPTION:
      nResult = MPIWrapper_Pack_STR(MPIWrapper_Instance(), 
                                    pThis->psOptionName,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_STR(MPIWrapper_Instance(), 
                                      pThis->psOptionValue,
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_STR(... pThis->psOptionValue..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_STR(... pThis->psOptionName..., errorcode = %d\n", nResult);
        }
      }
     
      break;

    case REQUEST_TYPE_ROOT_MASTER_REMOTE_TEST:
      nResult = MPIWrapper_Pack_STR(MPIWrapper_Instance(), 
                                    pThis->psGoCommands,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_STR(... pThis->psGoCommands..., errorcode = %d\n", nResult);
        }
      }
     
      break;

    case REQUEST_TYPE_ROOT_MASTER_MACHINE_INFO:
      nResult = Request_FillMachineInfoParameterIntoBuffer(pThis);
      break;

    case REQUEST_TYPE_NON_ROOT_MASTER_SYNC_AFTER_BROADCAST:
      break;
      
    case REQUEST_TYPE_SLAVE_TRY_TO_GET_WORK:
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(), 
                                    pThis->nSyncPositionId,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nSyncPositionId..., errorcode = %d\n", nResult);
        }
      }

      break;

    case REQUEST_TYPE_MASTER_STOP:
      nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                      pThis->nWorkId, 
                                      pThis->pParameterBuffer, 
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->nWorkId..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_SLAVE_PROCESS_SEARCH_RESULT:

// kh 11.10.06 only used for plausibility checks and assertions
// the stack access key information (nWorkId, nStackIndex and nMoveListIndex) is already sufficient
// as a future expansion the nSyncPositionId may e.g. only be transmitted (and received) in debug mode 
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    g_pYBWCManagerInstance->nSyncPositionId, 
                                    pThis->pParameterBuffer, 
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                    &pThis->nPosition, 
                                    MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                        pThis->nWorkId, 
                                        pThis->pParameterBuffer, 
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                        pThis->nStackIndex, 
                                        pThis->pParameterBuffer, 
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
            nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                          pThis->nMoveListIndex,
                                          pThis->pParameterBuffer, 
                                          MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                          &pThis->nPosition, 
                                          MPI_COMM_WORLD);
            if(nResult == MPI_SUCCESS)
            {
              nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                            pThis->nBestMove, 
                                            pThis->pParameterBuffer, 
                                            MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                            &pThis->nPosition, 
                                            MPI_COMM_WORLD);
              if(nResult == MPI_SUCCESS)
              {
                nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                              pThis->nBestValue, 
                                              pThis->pParameterBuffer, 
                                              MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                              &pThis->nPosition, 
                                              MPI_COMM_WORLD);
                if(nResult == MPI_SUCCESS)
                {
                  nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                pThis->nFlags, 
                                                pThis->pParameterBuffer, 
                                                MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                &pThis->nPosition, 
                                                MPI_COMM_WORLD);
                  if(nResult == MPI_SUCCESS)
                  {
                    nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                  pThis->nDepth, 
                                                  pThis->pParameterBuffer, 
                                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                  &pThis->nPosition, 
                                                  MPI_COMM_WORLD);
                    if(nResult == MPI_SUCCESS)
                    {
                      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                                    pThis->nMaxDepth, 
                                                    pThis->pParameterBuffer, 
                                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                    &pThis->nPosition, 
                                                    MPI_COMM_WORLD);
                      if(nResult == MPI_SUCCESS)
                      {
                        nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                              pThis->nPVLength,
                                                              pThis->pPV, 
                                                              pThis->pParameterBuffer, 
                                                              MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                              &pThis->nPosition, 
                                                              MPI_COMM_WORLD);
                        if(nResult == MPI_SUCCESS)
                        {
                          nResult = MPIWrapper_Pack_BOOL(MPIWrapper_Instance(),
                                                        pThis->bPonderMoveOk, 
                                                        pThis->pParameterBuffer, 
                                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                        &pThis->nPosition, 
                                                        MPI_COMM_WORLD);
                          if(nResult == MPI_SUCCESS)
                          {
                            nResult = MPIWrapper_Pack_DOUBLE(MPIWrapper_Instance(),
                                                            pThis->dTime, 
                                                            pThis->pParameterBuffer, 
                                                            MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                            &pThis->nPosition, 
                                                            MPI_COMM_WORLD);
                            if(nResult == MPI_SUCCESS)
                            {
                              nResult = MPIWrapper_Pack_DOUBLE(MPIWrapper_Instance(),
                                                              pThis->dSpeed, 
                                                              pThis->pParameterBuffer, 
                                                              MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                              &pThis->nPosition, 
                                                              MPI_COMM_WORLD);
                              if(nResult == MPI_SUCCESS)
                              {
                                nResult = MPIWrapper_Pack_DOUBLE(MPIWrapper_Instance(),
                                                                pThis->dCPU, 
                                                                pThis->pParameterBuffer, 
                                                                MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                                &pThis->nPosition, 
                                                                MPI_COMM_WORLD);
                                if(nResult == MPI_SUCCESS)
                                {
                                  nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                                                  pThis->nNodes, 
                                                                  pThis->pParameterBuffer, 
                                                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                                  &pThis->nPosition, 
                                                                  MPI_COMM_WORLD);
                                  if(nResult == MPI_SUCCESS)
                                  {
                                    nResult = MPIWrapper_Pack_UINT64(MPIWrapper_Instance(),
                                                                     pThis->nTransUsed, 
                                                                     pThis->pParameterBuffer, 
                                                                     MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                                     &pThis->nPosition, 
                                                                     MPI_COMM_WORLD);
                                    if(nResult == MPI_SUCCESS)
                                    {
                                    }
                                    else
                                    {
                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                      {
                                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->ntransUsed..., errorcode = %d\n", nResult);
                                      }
                                    }
                                  }
                                  else
                                  {
                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                    {
                                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->nNodes..., errorcode = %d\n", nResult);
                                    }
                                  }
                                }
                                else
                                {
                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                  {
                                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_DOUBLE(... pThis->dCPU..., errorcode = %d\n", nResult);
                                  }
                                }
                              }
                              else
                              {
                                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                {
                                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_DOUBLE(... pThis->dSpeed..., errorcode = %d\n", nResult);
                                }
                              }
                            }
                            else
                            {
                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                              {
                                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_DOUBLE(... pThis->dTime..., errorcode = %d\n", nResult);
                              }
                            }
                          }
                          else
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_BOOL(... pThis->bPonderMoveOk..., errorcode = %d\n", nResult);
                            }
                          }
                        }
                        else
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... pThis->nPVLength, pThis->pPV..., errorcode = %d\n", nResult);
                          }
                        }
                      }
                      else
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nMaxDepth..., errorcode = %d\n", nResult);
                        }
                      }
                    }
                    else
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nDepth..., errorcode = %d\n", nResult);
                      }
                    }
                  }
                  else
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nFlags..., errorcode = %d\n", nResult);
                    }
                  }
                }
                else
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nBestValue..., errorcode = %d\n", nResult);
                  }
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nBestMove..., errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nMoveListIndex..., errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nStackIndex..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->nWorkId..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... g_pYBWCManagerInstance->nSyncPositionId..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_MASTER_UPDATE_SEARCH_WINDOW:
      nResult = MPIWrapper_Pack_INT64(MPIWrapper_Instance(),
                                      pThis->nWorkId, 
                                      pThis->pParameterBuffer, 
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                      pThis->nAlpha,
                                      pThis->pParameterBuffer, 
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                      &pThis->nPosition, 
                                      MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                        pThis->nBeta,
                                        pThis->pParameterBuffer, 
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                        &pThis->nPosition, 
                                        MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nBeta..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nAlpha..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT64(... pThis->nWorkId..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_HASH_TABLE_REPLICATION:

// kh 06.12.06 this is done directly now with a special broadcast buffer for a better performance
/*
      nResult = Request_FillHashTableBufferParameterIntoBuffer(pThis); 

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at Request_FillHashTableBufferParameterIntoBuffer(..., errorcode = %d\n", nResult);
        }
      }
*/

      break;

    case REQUEST_TYPE_HASH_TABLE_BROADCAST_BUFFER:

      ASSERT(pThis->pHashTableBuffer->nCount == 0);

// kh 06.12.06 overwrite the type of the broadcast buffer to REQUEST_TYPE_HASH_TABLE_REPLICATION
      pThis->nPosition = 0;
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    REQUEST_TYPE_HASH_TABLE_REPLICATION, // pThis->nType,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                      pThis->pHashTableBuffer->nChunkSizeForFlush, // kh 06.12.06 pack the count in advance, the actual count is 0 yet
//                                    pThis->pHashTableBuffer->nCount,             // kh 15.11.06 pThis->pHashTableBuffer is the global transmit buffer

                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      MPI_COMM_WORLD);
        if(nResult == MPI_SUCCESS)
        {
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->pHashTableBuffer->nChunkSizeForFlush..., errorcode = %d\n", nResult);
          }
        }
        // kh 06.12.06 nothing to do here for the hash table entries
        // the broadcast buffer is filled directly for a better performance with the transposition table entries
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... REQUEST_TYPE_HASH_TABLE_REPLICATION..., errorcode = %d\n", nResult);
        }
      }

      break;

    case REQUEST_TYPE_HISTORY_TABLE_REPLICATION:

// kh 20.02.07 this is done directly now with a special broadcast buffer for a better performance

      break;

    case REQUEST_TYPE_HISTORY_TABLE_BROADCAST_BUFFER:

// kh 20.02.07 overwrite the type of the broadcast buffer to REQUEST_TYPE_HISTORY_TABLE_REPLICATION
      pThis->nPosition = 0;
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    REQUEST_TYPE_HISTORY_TABLE_REPLICATION, // pThis->nType,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(), 
                                      g_pYBWCManagerInstance->nSyncPositionId,
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                 HistorySize,
                                                 g_pYBWCManagerInstance->HistoryTableBuffer, 
                                                 pThis->pParameterBuffer, 
                                                 MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                 &pThis->nPosition, 
                                                 MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... HistorySize, g_pYBWCManagerInstance->HistoryTableBuffer..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... g_pYBWCManagerInstance->nSyncPositionId, errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... REQUEST_TYPE_HISTORY_TABLE_REPLICATION..., errorcode = %d\n", nResult);
        }
      }

      break;

    case REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION:

// kh 22.02.07 this is done directly now with a special broadcast buffer for a better performance

      break;

    case REQUEST_TYPE_HIST_TOT_TABLE_BROADCAST_BUFFER:

// kh 22.02.07 overwrite the type of the broadcast buffer to REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION
      pThis->nPosition = 0;
      nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(),
                                    REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION, // pThis->nType,
                                    pThis->pParameterBuffer,
                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                    &pThis->nPosition,
                                    MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        nResult = MPIWrapper_Pack_INT(MPIWrapper_Instance(), 
                                      g_pYBWCManagerInstance->nSyncPositionId,
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                HistorySize,
                                                g_pYBWCManagerInstance->HistTotTableBuffer, 
                                                pThis->pParameterBuffer, 
                                                MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                &pThis->nPosition, 
                                                MPI_COMM_WORLD);
          if(nResult == MPI_SUCCESS)
          {
            nResult = MPIWrapper_Pack_USHORT_ARRAY(MPIWrapper_Instance(),
                                                  HistorySize,
                                                  g_pYBWCManagerInstance->HistHitTableBuffer, 
                                                  pThis->pParameterBuffer, 
                                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE, 
                                                  &pThis->nPosition, 
                                                  MPI_COMM_WORLD);
            if(nResult == MPI_SUCCESS)
            {
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... HistorySize, g_pYBWCManagerInstance->HistHitTableBuffer..., errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_USHORT_ARRAY(... HistorySize, g_pYBWCManagerInstance->HistTotTableBuffer..., errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... g_pYBWCManagerInstance->nSyncPositionId, errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION..., errorcode = %d\n", nResult);
        }
      }

      break;

    default:
      break;
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_FillParameterIntoBuffer(... failed at MPIWrapper_Pack_INT(... pThis->nType..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterLogonInvitation(Request* pThis)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_LOGON_INVITATION;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterLogonInvitation(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterLogoffInvitation(Request* pThis)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_LOGOFF_INVITATION;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterLogoffInvitation(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterShutdown(Request* pThis)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_SHUTDOWN;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterShutdown(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterIdleInvitation(Request* pThis)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_IDLE_INVITATION;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterIdleInvitation(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterSyncPosition(Request* pThis, char* psSyncPosition)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_SYNC_POSITION;

  pThis->psSyncPosition = (char*)malloc(sizeof(char) * (strlen(psSyncPosition) + 1));
  *pThis->psSyncPosition = '\0';

  strncat(pThis->psSyncPosition, psSyncPosition, MAX_REQUEST_SYNC_POSITION_LENGTH);

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterSyncPosition(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterSupportSearch(Request* pThis)
{
  int nResult;

  pThis->nType = REQUEST_TYPE_ROOT_MASTER_SUPPORT_SEARCH;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterSupportSearch(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterStopRemoteSearch(Request* pThis)
{
  int nResult;

  pThis->nType = REQUEST_TYPE_ROOT_MASTER_STOP_REMOTE_SEARCH;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterStopRemoteSearch(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterNewGame(Request* pThis)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_NEW_GAME;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterNewGame(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterSetOption(Request* pThis, char* psOptionName, char* psOptionValue)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_ROOT_MASTER_SET_OPTION;

  pThis->psOptionName = (char*)malloc(sizeof(char) * (strlen(psOptionName) + 1));
  *pThis->psOptionName= '\0';

  strncat(pThis->psOptionName, psOptionName, MAX_REQUEST_OPTION_NAME_LENGTH);

  pThis->psOptionValue = (char*)malloc(sizeof(char) * (strlen(psOptionValue) + 1));
  *pThis->psOptionValue = '\0';

  strncat(pThis->psOptionValue, psOptionValue, MAX_REQUEST_OPTION_VALUE_LENGTH);

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterSetOption(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitNonRootMasterSyncAfterBroadcast(Request* pThis)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_NON_ROOT_MASTER_SYNC_AFTER_BROADCAST;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitNonRootMasterSyncAfterBroadcast(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterRemoteTest(Request* pThis, char* psGoCommands)
{
  int nResult;
  int nLength;

  pThis->nType        = REQUEST_TYPE_ROOT_MASTER_REMOTE_TEST;

  if(psGoCommands == NULL)
  {
    nLength = 0;
  }
  else
  {
    nLength = strlen(psGoCommands);
  }

  pThis->psGoCommands = (char*)malloc(sizeof(char) * (nLength + 1));
  *pThis->psGoCommands = '\0';

  if(psGoCommands)
  {
    strncat(pThis->psGoCommands, psGoCommands, MAX_REQUEST_GO_COMMANDS_LENGTH);
  }

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterRemoteTest(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitRootMasterMachineInfo(Request* pThis)
{
  int nResult;

  pThis->nType        = REQUEST_TYPE_ROOT_MASTER_MACHINE_INFO;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitRootMasterMachineInfo(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitSlaveTryToGetWork(Request*  pThis,
                                  int       nSlaveSyncPositionId)
{
  int nResult;

  pThis->nType            = REQUEST_TYPE_SLAVE_TRY_TO_GET_WORK;

  pThis->nSyncPositionId  = nSlaveSyncPositionId; 

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitSlaveTryToGetWork(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitMasterStop(Request* pThis,
                           sint64   nWorkId)
{
  int nResult;

  pThis->nType = REQUEST_TYPE_MASTER_STOP;

  pThis->nWorkId = nWorkId;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitMasterStop(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitSlaveProcessSearchResult(Request* pThis,
                                         sint64   nWorkId,
                                         int      nStackIndex,
                                         int      nMoveListIndex,
                                         int      nBestMove,
                                         int      nBestValue,
                                         int      nFlags,
                                         int      nDepth,
                                         int      nMaxDepth,
                                         int      nPVLength,
                                         uint16   PV[],
                                         BOOL     bPonderMoveOk,
                                         double   dTime,
                                         double   dSpeed,
                                         double   dCPU,
                                         sint64   nNodes,
                                         uint64   nTransUsed)
{
  int nResult;

  pThis->nType                    = REQUEST_TYPE_SLAVE_PROCESS_SEARCH_RESULT;

  pThis->nWorkId                  = nWorkId;
  pThis->nStackIndex              = nStackIndex;
  pThis->nMoveListIndex           = nMoveListIndex;
  pThis->nBestMove                = nBestMove;
  pThis->nBestValue               = nBestValue;
  pThis->nFlags                   = nFlags;
  pThis->nDepth                   = nDepth;
  pThis->nMaxDepth                = nMaxDepth;
  pThis->nPVLength                = nPVLength,
  pThis->pPV                      = PV;
  pThis->bPonderMoveOk            = bPonderMoveOk;
  pThis->dTime                    = dTime;
  pThis->dSpeed                   = dSpeed;
  pThis->dCPU                     = dCPU;
  pThis->nNodes                   = nNodes;
  pThis->nTransUsed               = nTransUsed;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitSlaveProcessSearchResult(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitMasterUpdateSearchWindow(Request* pThis,
                                         sint64   nWorkId,
                                         int      nAlpha,
                                         int      nBeta)
{
  int nResult;

  pThis->nType = REQUEST_TYPE_MASTER_UPDATE_SEARCH_WINDOW;

  pThis->nWorkId = nWorkId;

  pThis->nAlpha  = nAlpha;
  pThis->nBeta   = nBeta;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitMasterUpdateSearchWindow(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitHashTableReplication(Request*         pThis,
                                     HashTableBuffer* pHashTableBuffer)
{
  int nResult;

  pThis->nType              = REQUEST_TYPE_HASH_TABLE_REPLICATION;

  pThis->pHashTableBuffer   = pHashTableBuffer; // kh 15.11.06 for better performance do not use a deep copy here 

  ASSERT(pHashTableBuffer->pBroadcastRequestBuffer);

  pThis->pBroadcastRequest  = pHashTableBuffer->pBroadcastRequestBuffer;
  pThis->pBroadcastRequest->nReferenceCount++;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitHashTableReplication(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitHashTableBroadcastBuffer(Request*         pThis,
                                         HashTableBuffer* pHashTableBuffer)
{
  int nResult;

  pThis->nType            = REQUEST_TYPE_HASH_TABLE_BROADCAST_BUFFER;

  pThis->pHashTableBuffer = pHashTableBuffer; // kh 15.11.06 for better performance do not use a deep copy here 

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitHashTableBroadcastBuffer(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitHistoryTableReplication(Request* pThis,
                                        Request* pHistoryTableBroadcastRequestBuffer)
{
  int nResult;

  pThis->nType               = REQUEST_TYPE_HISTORY_TABLE_REPLICATION;

  ASSERT(pHistoryTableBroadcastRequestBuffer);

  pThis->pBroadcastRequest = pHistoryTableBroadcastRequestBuffer;
  pThis->pBroadcastRequest->nReferenceCount++;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitHistoryTableReplication(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitHistoryTableBroadcastBuffer(Request* pThis)
{
  int nResult;

  pThis->nType            = REQUEST_TYPE_HISTORY_TABLE_BROADCAST_BUFFER;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitHistoryTableBroadcastBuffer(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitHistTotTableReplication(Request* pThis,
                                        Request* pHistTotTableBroadcastRequestBuffer)
{
  int nResult;

  pThis->nType               = REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION;

  ASSERT(pHistTotTableBroadcastRequestBuffer);

  pThis->pBroadcastRequest = pHistTotTableBroadcastRequestBuffer;
  pThis->pBroadcastRequest->nReferenceCount++;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitHistTotTableReplication(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_InitHistTotTableBroadcastBuffer(Request* pThis)
{
  int nResult;

  pThis->nType            = REQUEST_TYPE_HIST_TOT_TABLE_BROADCAST_BUFFER;

  nResult = Request_FillParameterIntoBuffer(pThis);

  if(nResult == MPI_SUCCESS)
  {
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_InitHistTotTableBroadcastBuffer(... failed at Request_FillParameterIntoBuffer(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_Send(Request* pThis)
{
  int               nResult;

  ProcessorHandler* pProcessorHandler;

  ASSERT(pThis->mpiRequest == MPI_REQUEST_NULL);
  nResult = MPIWrapper_Isend(MPIWrapper_Instance(),
                             pThis->pParameterBuffer, 
                             pThis->nPosition, 
                             MPI_PACKED, 
                             pThis->nDestId, 
                             YBWC_MANAGER_REQUEST_TAG, 
                             MPI_COMM_WORLD,
                             &pThis->mpiRequest);

  if(nResult == MPI_SUCCESS)
  {

// kh 14.12.06 tuning 
    pProcessorHandler = g_pYBWCManagerInstance->processorHandlerPool[pThis->nDestId];

    nResult = ProcessorHandler_ForcedWaitForAllSendsCompleted(pProcessorHandler);

    if(nResult == MPI_SUCCESS)
    {
    }
    else
    {
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_Send(... failed at ProcessorHandler_ForcedWaitForAllSendsCompleted(..., errorcode = %d\n", nResult);
      }
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_Send(... failed at MPIWrapper_Isend(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_SendBroadcastBuffer(Request* pThis)
{
  int               nResult;

  ProcessorHandler* pProcessorHandler;

  ASSERT(pThis->mpiRequest == MPI_REQUEST_NULL);
  ASSERT(pThis->pBroadcastRequest);

  nResult = MPIWrapper_Isend(MPIWrapper_Instance(),
                             pThis->pBroadcastRequest->pParameterBuffer, 
                             pThis->pBroadcastRequest->nPosition, 
                             MPI_PACKED, 
                             pThis->nDestId, 
                             YBWC_MANAGER_REQUEST_TAG, 
                             MPI_COMM_WORLD,
                             &pThis->mpiRequest);

  if(nResult == MPI_SUCCESS)
  {
    pProcessorHandler = g_pYBWCManagerInstance->processorHandlerPool[pThis->nDestId];

    nResult = ProcessorHandler_ForcedWaitForAllSendsCompleted(pProcessorHandler);

    if(nResult == MPI_SUCCESS)
    {
    }
    else
    {
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_Send(... failed at ProcessorHandler_ForcedWaitForAllSendsCompleted(..., errorcode = %d\n", nResult);
      }
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_SendBroadcastBuffer(... failed at MPIWrapper_Isend(..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

BOOL Request_IsSendPending(Request* pThis)
{
  BOOL        bResult;
  int         nResult;

  int         nFlag;
  MPI_Status  mpiStatus;

  bResult = FALSE;

  if(pThis->bSendSuppressed)
  {
  }
  else
  {
    if(pThis->mpiRequest == MPI_REQUEST_NULL)
    {
      bResult = TRUE; // kh 26.09.06 not send yet
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_WARNING_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "WARNING Request_IsSendPending(... pThis->mpiRequest == MPI_REQUEST_NULL\n");
      }
    }
    else
    {
      nResult = MPIWrapper_Test(MPIWrapper_Instance(),
                                &pThis->mpiRequest,
                                &nFlag,
                                &mpiStatus);

      if(nResult == MPI_SUCCESS) 
      {
        if(nFlag)
        {
        }
        else
        {
          bResult = TRUE;

          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_3)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ProcessorHandler <%d> Request_IsSendPending(... send operation is not completed\n", pThis->nDestId);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_IsSendPending(... failed at MPIWrapper_Test(..., errorcode = %d\n", nResult);
        }
      }
    }
  }

  return bResult;
}

int Request_GetHashTableBufferParameterFromBuffer(Request* pThis)
{
  int     nResult;
  int     nResultTmp;

  int     i;

  int     nCount;
  uint64  nKey;
  int     nMove;
  int     nDepth;
  int     nFlags;
  int     nValue;

  nResultTmp = MPI_SUCCESS;

  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                  pThis->pParameterBuffer,  // kh 15.11.06 pThis->pHashTableBuffer is the global receive buffer
                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                  &pThis->nPosition,
                                  &nCount,
                                  MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "nCount = %d\n", nCount);
    }

    for(i = 0; i < nCount; i++)
    {
      nResult = MPIWrapper_Unpack_UINT64(MPIWrapper_Instance(), 
                                         pThis->pParameterBuffer,
                                         MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                         &pThis->nPosition,
                                         &nKey,
                                         MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
        {
          if(i == 0) // kh 15.11.06 do this just as an example
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "nKey = <" F12INT64u ">\n", nKey);
          }
        }

        nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        &nMove,
                                        MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
          {
            if(i == 0) // kh 15.11.06 do this just as an example
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "nMove = %d\n", nMove);
            }
          }

          nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &nDepth,
                                          MPI_COMM_WORLD);

          if(nResult == MPI_SUCCESS)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
            {
              if(i == 0) // kh 15.11.06 do this just as an example
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "nDepth = %d\n", nDepth);
              }
            }

            nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                            pThis->pParameterBuffer,
                                            MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                            &pThis->nPosition,
                                            &nFlags,
                                            MPI_COMM_WORLD);

            if(nResult == MPI_SUCCESS)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
              {
                if(i == 0) // kh 15.11.06 do this just as an example
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "nFlags = %d\n", nFlags);
                }
              }

              nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                              pThis->pParameterBuffer,
                                              MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                              &pThis->nPosition,
                                              &nValue,
                                              MPI_COMM_WORLD);

              if(nResult == MPI_SUCCESS)
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
                {
                  if(i == 0) // kh 15.11.06 do this just as an example
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "nValue = %d\n", nValue);
                  }
                }

// kh 07.12.06 this is done directly now for a better performance
/*
                HashTableBuffer_AddEntry(pThis->pHashTableBuffer,
                                         nKey,
                                         nMove,
                                         nDepth,
                                         nMinValue,
                                         nMaxValue);
*/

                if(   !g_pYBWCManagerInstance->bLoggedOn 
                    || g_pYBWCManagerInstance->bResizeHashTableActive)
                {
                  // kh 05.02.08 do not access the hash table during shutdown or while resizing it
                }
                else
                {
                  trans_store(Trans, 
                              nKey,
                              nMove,
                              nDepth,
                              nFlags,
                              nValue);
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &nValue... errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &nFlags... errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &nDepth... errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &nMove... errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_UINT64(..., &nKey... errorcode = %d\n", nResult);
        }
      }

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {

// kh 15.11.06 save first error only
        if(nResultTmp == MPI_SUCCESS)
        {
          nResultTmp = nResult;
        }
      }
    } // for(i = 0; i < nCount; i++)

// kh 07.12.06 pHashTableBuffer is not really any longer
//  ASSERT(pThis->pHashTableBuffer->nCount == nCount);

    nResult = nResultTmp;
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetHashTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(... &pThis->pHashTableBuffer->nCount..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_GetHistoryTableBufferParameterFromBuffer(Request* pThis)
{
  int     nResult;
  int     nResultTmp;

  int     i;

  bool    bOverflow;

  nResultTmp = MPI_SUCCESS;


  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                  pThis->pParameterBuffer,
                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                  &pThis->nPosition,
                                  &pThis->nSyncPositionId,
                                  MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "nSyncPositionId = %d\n", pThis->nSyncPositionId);
    }

    nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                             pThis->pParameterBuffer,
                                             MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                             &pThis->nPosition,
                                             &pThis->nHistoryTableSize,
                                             pThis->HistoryTable,
                                             MPI_COMM_WORLD);

    if(nResult == MPI_SUCCESS)
    {
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "nHistoryTableSize = <%d>\n", pThis->nHistoryTableSize);

// kh 20.02.07 future expansion (fruit functions may be helpful here)
//      YBWCManager_Printf(g_pYBWCManagerInstance, "HistoryTable = <n * %d>\n", pThis->HistoryTable);
      }

// kh 05.02.08 only access the history table when not shutdown
      if(g_pYBWCManagerInstance->bLoggedOn)
      {
        ASSERT(pThis->nHistoryTableSize == HistorySize);

        if(g_pYBWCManagerInstance->nSyncPositionId == pThis->nSyncPositionId)
        {

          ASSERT(g_pYBWCManagerInstance->bHistoryTableInitialized);

          if(g_pYBWCManagerInstance->bHistoryTableInitialized)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->bUseSharedMemoryHistoryTable)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->bSynchronizeHistoryTableAccess)
              {
                CriticalSection_Enter(g_pYBWCManagerInstance->pCriticalSectionHistoryTable);
              }
            }

            bOverflow = false;
            for(i = 0; i < HistorySize; i++) 
            {

/*
// kh 26.03.07 test only: prefer local table
              gpHistory->History[i] += (pThis->HistoryTable[i] + 1) / 2;
*/

              gpHistory->History[i] += pThis->HistoryTable[i];
              if(gpHistory->History[i] >= HistoryMax) 
              {
                bOverflow = true;
              }
            }

            if(bOverflow)
            {
              for(i = 0; i < HistorySize; i++) 
              {
                gpHistory->History[i] = (gpHistory->History[i] + 1) / 2;
              }
            }

            if(g_pYBWCManagerInstance->pFruitConfiguration->bUseSharedMemoryHistoryTable)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->bSynchronizeHistoryTableAccess)
              {
                CriticalSection_Leave(g_pYBWCManagerInstance->pCriticalSectionHistoryTable);
              }
            }
          } // if(g_pYBWCManagerInstance->bHistoryTableInitialized)
        } // if(g_pYBWCManagerInstance->nSyncPositionId == pThis->nSyncPositionId)
      } // if(g_pYBWCManagerInstance->bLoggedOn)

      nResult = nResultTmp;
    }
    else
    {
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetHistoryTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nHistoryTableCount, pThis->HistoryTable... errorcode = %d\n", nResult);
      }
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetHistoryTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(... &pThis->nSyncPositionId..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_GetHistTotTableBufferParameterFromBuffer(Request* pThis)
{
  int     nResult;
  int     nResultTmp;

  int     i;

  bool    bOverflow;

  nResultTmp = MPI_SUCCESS;


  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                  pThis->pParameterBuffer,
                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                  &pThis->nPosition,
                                  &pThis->nSyncPositionId,
                                  MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "nSyncPositionId = %d\n", pThis->nSyncPositionId);
    }

    nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                             pThis->pParameterBuffer,
                                             MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                             &pThis->nPosition,
                                             &pThis->nHistTotTableSize,
                                             pThis->HistTotTable,
                                             MPI_COMM_WORLD);

    if(nResult == MPI_SUCCESS)
    {

      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "nHistTotTableSize = <%d>\n", pThis->nHistTotTableSize);

// kh 22.02.07 future expansion (fruit functions may be helpful here)
//      YBWCManager_Printf(g_pYBWCManagerInstance, "HistTotTable = <n * %d>\n", pThis->HistTotTable);
      }

      ASSERT(pThis->nHistTotTableSize == HistorySize);

      nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                              pThis->pParameterBuffer,
                                              MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                              &pThis->nPosition,
                                              &pThis->nHistHitTableSize,
                                              pThis->HistHitTable,
                                              MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nHistHitTableSize = <%d>\n", pThis->nHistHitTableSize);

// kh 22.02.07 future expansion (fruit functions may be helpful here)
//        YBWCManager_Printf(g_pYBWCManagerInstance, "HistHitTable = <n * %d>\n", pThis->HistHitTable);
        }

        ASSERT(pThis->nHistHitTableSize == HistorySize);

        if(g_pYBWCManagerInstance->nSyncPositionId == pThis->nSyncPositionId)
        {

          ASSERT(g_pYBWCManagerInstance->bHistoryTableInitialized);

          if(g_pYBWCManagerInstance->bHistoryTableInitialized)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->bUseSharedMemoryHistoryTable)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->bSynchronizeHistoryTableAccess)
              {
                CriticalSection_Enter(g_pYBWCManagerInstance->pCriticalSectionHistoryTable);
              }
            }

            bOverflow = false;
            for(i = 0; i < HistorySize; i++) 
            {

/*
// kh 26.03.07 test only: prefer local table
              gpHistory->HistHit[i] += (pThis->HistHitTable[i] + 1) / 2;
              gpHistory->HistTot[i] += (pThis->HistTotTable[i] + 1) / 2;
*/


              gpHistory->HistHit[i] += pThis->HistHitTable[i];
              gpHistory->HistTot[i] += pThis->HistTotTable[i];

              if(gpHistory->HistTot[i] >= HistoryMax) 
              {
                bOverflow = true;

                gpHistory->HistHit[i] = (gpHistory->HistHit[i] + 1) / 2;
                gpHistory->HistTot[i] = (gpHistory->HistTot[i] + 1) / 2;
              }
            }

/*
            if(bOverflow)
            {
              for(i = 0; i < HistorySize; i++) 
              {
                gpHistory->HistHit[i] = (gpHistory->HistHit[i] + 1) / 2;
                gpHistory->HistTot[i] = (gpHistory->HistTot[i] + 1) / 2;
              }
            }
*/

            if(g_pYBWCManagerInstance->pFruitConfiguration->bUseSharedMemoryHistoryTable)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->bSynchronizeHistoryTableAccess)
              {
                CriticalSection_Leave(g_pYBWCManagerInstance->pCriticalSectionHistoryTable);
              }
            }
          }
        }

        nResult = nResultTmp;

      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetHistHitTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nHistTotTableCount, pThis->HistHitTable... errorcode = %d\n", nResult);
        }
      }
    }
    else
    {
      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetHistTotTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nHistTotTableCount, pThis->HistTotTable... errorcode = %d\n", nResult);
      }
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetHistTotTableBufferParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(... &pThis->nSyncPositionId..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}

int Request_GetMachineInfoParameterFromBuffer(Request* pThis)
{
  int               nResult;
  int               nResultTmp;

  ProcessorHandler* pProcessorHandler;

  int               i;

  nResultTmp = MPI_SUCCESS;

// kh 13.12.06 size is known in this special case
/*
  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                  pThis->pParameterBuffer,  
                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                  &pThis->nPosition,
                                  &<nAppropriateCount>,
                                  MPI_COMM_WORLD);
*/

  nResult = MPI_SUCCESS;
  if(nResult == MPI_SUCCESS)
  {
    for(i = 0; i < g_pYBWCManagerInstance->nSize; i++)
    {
      pProcessorHandler = g_pYBWCManagerInstance->processorHandlerPool[i];

      nResult = MPIWrapper_Unpack_STR(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      pProcessorHandler->sProcessorName,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, 
                             "Processor name on rank %d is <%s>\n", 
                             i,
                             pProcessorHandler->sProcessorName);
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetMachineInfoParameterFromBuffer(... failed at MPIWrapper_Unpack_STR(... pProcessorHandler->sProcessorName..., errorcode = %d\n", nResult);
        }
      }

      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {

// kh 15.11.06 save first error only
        if(nResultTmp == MPI_SUCCESS)
        {
          nResultTmp = nResult;
        }
      }
    } for(i = 0; i < g_pYBWCManagerInstance->nSize; i++)

    nResult = nResultTmp;
  }
  else
  {

// kh 13.12.06 not used at the moment

  }

  return nResult;
}

int Request_GetParameterFromBuffer(Request* pThis)
{
  int   nResult;

  char  sSyncPosition   [MAX_REQUEST_SYNC_POSITION_LENGTH + 1];
  char  sOptionName     [MAX_REQUEST_OPTION_NAME_LENGTH   + 1];
  char  sOptionValue    [MAX_REQUEST_OPTION_VALUE_LENGTH  + 1];
  char  sGoCommands     [MAX_REQUEST_GO_COMMANDS_LENGTH   + 1];

  bool  bPrint;


// kh 02.08.06 get pRequest->nType from receive buffer
  pThis->nPosition = 0;
  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                  pThis->pParameterBuffer,
                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                  &pThis->nPosition,
                                  &pThis->nType,
                                  MPI_COMM_WORLD);

  if(nResult == MPI_SUCCESS)
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
    {
      bPrint = false;
      switch(pThis->nType)
      {
      case REQUEST_TYPE_HASH_TABLE_REPLICATION: 

// kh 15.11.06 this request is usually received with a very high frequency so printing is only done 
// if the nFruitDebugLevel has an appropriate high level 
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_999)
        {
          bPrint = true;
        }
        break;
  
      default:
        bPrint = true;
        break;
      }

      if(bPrint)
      {
        YBWCManager_Printf(g_pYBWCManagerInstance, 
                           "Request received from processor <%d>, Type = %d <%s>\n",
                           pThis->nSourceId,
                           pThis->nType,
                           Request_TypeToString(pThis));
      }
    }

    switch(pThis->nType)
    {
    case REQUEST_TYPE_UNDEFINED:
      break;

    case REQUEST_TYPE_ROOT_MASTER_LOGON_INVITATION:
      break;

    case REQUEST_TYPE_ROOT_MASTER_LOGOFF_INVITATION:
      break;

    case REQUEST_TYPE_ROOT_MASTER_SHUTDOWN:
      break;

    case REQUEST_TYPE_ROOT_MASTER_IDLE_INVITATION:
      break;

    case REQUEST_TYPE_ROOT_MASTER_SYNC_POSITION:
      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      &pThis->nSyncPositionId,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nSyncPositionId = %d\n", pThis->nSyncPositionId);
        }

        nResult = MPIWrapper_Unpack_STR(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        sSyncPosition,
                                        MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "SyncPosition for <%s>\n", sSyncPosition);
          }

          pThis->psSyncPosition = (char*)malloc(sizeof(char) * (strlen(sSyncPosition) + 1));
          *pThis->psSyncPosition = '\0';

          strncat(pThis->psSyncPosition, sSyncPosition, MAX_REQUEST_SYNC_POSITION_LENGTH);

        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_STR(... sSyncPosition..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(... &pThis->nSyncPositionId..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_ROOT_MASTER_SUPPORT_SEARCH:
      break;

    case REQUEST_TYPE_ROOT_MASTER_STOP_REMOTE_SEARCH:
      break;

    case REQUEST_TYPE_ROOT_MASTER_NEW_GAME:
      break;

    case REQUEST_TYPE_ROOT_MASTER_SET_OPTION:
      nResult = MPIWrapper_Unpack_STR(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      sOptionName,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "sOptionName <%s>\n", sOptionName);
        }

        pThis->psOptionName = (char*)malloc(sizeof(char) * (strlen(sOptionName) + 1));
        *pThis->psOptionName = '\0';

        strncat(pThis->psOptionName, sOptionName, MAX_REQUEST_OPTION_NAME_LENGTH);

        nResult = MPIWrapper_Unpack_STR(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        sOptionValue,
                                        MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "sOptionValue <%s>\n", sOptionValue);
          }

          pThis->psOptionValue = (char*)malloc(sizeof(char) * (strlen(sOptionValue) + 1));
          *pThis->psOptionValue = '\0';

          strncat(pThis->psOptionValue, sOptionValue, MAX_REQUEST_OPTION_VALUE_LENGTH);
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_STR(... sOptionValue..., errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_STR(... sOptionName..., errorcode = %d\n", nResult);
        }
      }
     
      break;

    case REQUEST_TYPE_ROOT_MASTER_REMOTE_TEST:
      nResult = MPIWrapper_Unpack_STR(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      sGoCommands,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "GoCommands <%s>\n", sGoCommands);
        }

        pThis->psGoCommands = (char*)malloc(sizeof(char) * (strlen(sGoCommands) + 1));
        *pThis->psGoCommands = '\0';

        strncat(pThis->psGoCommands, sGoCommands, MAX_REQUEST_GO_COMMANDS_LENGTH);
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_STR(... sGoCommands..., errorcode = %d\n", nResult);
        }
      }
     
      break;

    case REQUEST_TYPE_ROOT_MASTER_MACHINE_INFO:
      nResult = Request_GetMachineInfoParameterFromBuffer(pThis);
      break;

    case REQUEST_TYPE_NON_ROOT_MASTER_SYNC_AFTER_BROADCAST:
      break;

    case REQUEST_TYPE_SLAVE_TRY_TO_GET_WORK:
      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      &pThis->nSyncPositionId,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_2)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nSyncPositionId = %d\n", pThis->nSyncPositionId);
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(... &pThis->nSyncPositionId..., errorcode = %d\n", nResult);
        }
      }

      break;

    case REQUEST_TYPE_MASTER_STOP:
      nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        &pThis->nWorkId,
                                        MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nWorkId = <" F12INT64d ">\n", pThis->nWorkId);
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nWorkId... errorcode = %d\n", nResult);
        }
      }

      break;

    case REQUEST_TYPE_SLAVE_PROCESS_SEARCH_RESULT:
      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                      pThis->pParameterBuffer,
                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                      &pThis->nPosition,
                                      &pThis->nSyncPositionId,
                                      MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nSyncPositionId = <%d>\n", pThis->nSyncPositionId);
        }

        nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &pThis->nWorkId,
                                          MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "nWorkId = <" F12INT64d ">\n", pThis->nWorkId);
          }

          nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &pThis->nStackIndex,
                                          MPI_COMM_WORLD);

          if(nResult == MPI_SUCCESS)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "nStackIndex = <%d>\n", pThis->nStackIndex);
            }

            nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                            pThis->pParameterBuffer,
                                            MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                            &pThis->nPosition,
                                            &pThis->nMoveListIndex,
                                            MPI_COMM_WORLD);

            if(nResult == MPI_SUCCESS)
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "nMoveListIndex = <%d>\n", pThis->nMoveListIndex);
              }

              nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                              pThis->pParameterBuffer,
                                              MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                              &pThis->nPosition,
                                              &pThis->nBestMove,
                                              MPI_COMM_WORLD);

              if(nResult == MPI_SUCCESS)
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "nBestMove = <%d>\n", pThis->nBestMove);
                }

                nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                pThis->pParameterBuffer,
                                                MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                &pThis->nPosition,
                                                &pThis->nBestValue,
                                                MPI_COMM_WORLD);

                if(nResult == MPI_SUCCESS)
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "nBestValue = <%d>\n", pThis->nBestValue);
                  }

                  nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                  pThis->pParameterBuffer,
                                                  MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                  &pThis->nPosition,
                                                  &pThis->nFlags,
                                                  MPI_COMM_WORLD);

                  if(nResult == MPI_SUCCESS)
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "nFlags = <%d>\n", pThis->nFlags);
                    }

                    nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                    pThis->pParameterBuffer,
                                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                    &pThis->nPosition,
                                                    &pThis->nDepth,
                                                    MPI_COMM_WORLD);

                    if(nResult == MPI_SUCCESS)
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "nDepth = <%d>\n", pThis->nDepth);
                      }

                      nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                                      pThis->pParameterBuffer,
                                                      MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                      &pThis->nPosition,
                                                      &pThis->nMaxDepth,
                                                      MPI_COMM_WORLD);

                      if(nResult == MPI_SUCCESS)
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "nMaxDepth = <%d>\n", pThis->nMaxDepth);
                        }

                        nResult = MPIWrapper_Unpack_USHORT_ARRAY(MPIWrapper_Instance(), 
                                                                 pThis->pParameterBuffer,
                                                                 MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                                 &pThis->nPosition,
                                                                 &pThis->nPVLength,
                                                                 pThis->PV,
                                                                 MPI_COMM_WORLD);

                        if(nResult == MPI_SUCCESS)
                        {
                          ASSERT(pThis->nPVLength < MAX_PV_LENGTH);
                          pThis->PV[pThis->nPVLength] = MoveNone; // kh 26.09.06 mark the end of the list

                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "nPVLength = <%d>\n", pThis->nPVLength);

// kh 05.09.06 future expansion (fruit functions may be helpful here)
//                          YBWCManager_Printf(g_pYBWCManagerInstance, "PV = <n * %d>\n", pThis->PV);
                          }

                          nResult = MPIWrapper_Unpack_BOOL(MPIWrapper_Instance(), 
                                                           pThis->pParameterBuffer,
                                                           MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                           &pThis->nPosition,
                                                           &pThis->bPonderMoveOk,
                                                           MPI_COMM_WORLD);

                          if(nResult == MPI_SUCCESS)
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "bPonderMoveOk = <%d>\n", pThis->bPonderMoveOk);
                            }

                            nResult = MPIWrapper_Unpack_DOUBLE(MPIWrapper_Instance(), 
                                                               pThis->pParameterBuffer,
                                                               MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                               &pThis->nPosition,
                                                               &pThis->dTime,
                                                               MPI_COMM_WORLD);

                            if(nResult == MPI_SUCCESS)
                            {
                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                              {
                                YBWCManager_Printf(g_pYBWCManagerInstance, "dTime <%9.2f>\n", pThis->dTime);
                              }

                              nResult = MPIWrapper_Unpack_DOUBLE(MPIWrapper_Instance(), 
                                                                 pThis->pParameterBuffer,
                                                                 MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                                 &pThis->nPosition,
                                                                 &pThis->dSpeed,
                                                                 MPI_COMM_WORLD);

                              if(nResult == MPI_SUCCESS)
                              {
                                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                {
                                  YBWCManager_Printf(g_pYBWCManagerInstance, "dSpeed <%9.2f>\n", pThis->dSpeed);
                                }

                                nResult = MPIWrapper_Unpack_DOUBLE(MPIWrapper_Instance(), 
                                                                   pThis->pParameterBuffer,
                                                                   MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                                   &pThis->nPosition,
                                                                   &pThis->dCPU,
                                                                   MPI_COMM_WORLD);

                                if(nResult == MPI_SUCCESS)
                                {
                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                  {
                                    YBWCManager_Printf(g_pYBWCManagerInstance, "dCPU <%7.2f>\n", pThis->dCPU);
                                  }

                                  nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                                                    pThis->pParameterBuffer,
                                                                    MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                                    &pThis->nPosition,
                                                                    &pThis->nNodes,
                                                                    MPI_COMM_WORLD);

                                  if(nResult == MPI_SUCCESS)
                                  {
                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                    {
                                      YBWCManager_Printf(g_pYBWCManagerInstance, "nNodes <" F12INT64d ">\n", pThis->nNodes);
                                    }

                                    nResult = MPIWrapper_Unpack_UINT64(MPIWrapper_Instance(), 
                                                                       pThis->pParameterBuffer,
                                                                       MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                                                       &pThis->nPosition,
                                                                       &pThis->nTransUsed,
                                                                       MPI_COMM_WORLD);

                                    if(nResult == MPI_SUCCESS)
                                    {
                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
                                      {
                                        YBWCManager_Printf(g_pYBWCManagerInstance, "nTransUSed <" F12INT64d ">\n", pThis->nTransUsed);
                                      }
                                    }
                                    else
                                    {
                                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                      {
                                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nTransUsed... errorcode = %d\n", nResult);
                                      }
                                    }
                                  }
                                  else
                                  {
                                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                    {
                                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nNodes... errorcode = %d\n", nResult);
                                    }
                                  }
                                }
                                else
                                {
                                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                  {
                                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_DOUBLE(..., &pThis->dCPU... errorcode = %d\n", nResult);
                                  }
                                }
                              }
                              else
                              {
                                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                                {
                                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_DOUBLE(..., &pThis->dSpeed... errorcode = %d\n", nResult);
                                }
                              }
                            }
                            else
                            {
                              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                              {
                                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_DOUBLE(..., &pThis->dTime... errorcode = %d\n", nResult);
                              }
                            }
                          }
                          else
                          {
                            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                            {
                              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Pack_BOOL(..., &pThis->bPonderMoveOk... errorcode = %d\n", nResult);
                            }
                          }
                        }
                        else
                        {
                          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                          {
                            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_USHORT_ARRAY(..., &pThis->nPVLength, pThis->PV... errorcode = %d\n", nResult);
                          }
                        }
                      }
                      else
                      {
                        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                        {
                          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nMaxDepth... errorcode = %d\n", nResult);
                        }
                      }
                    }
                    else
                    {
                      if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                      {
                        YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nDepth... errorcode = %d\n", nResult);
                      }
                    }
                  }
                  else
                  {
                    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                    {
                      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nFlags... errorcode = %d\n", nResult);
                    }
                  }
                }
                else
                {
                  if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                  {
                    YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nBestValue... errorcode = %d\n", nResult);
                  }
                }
              }
              else
              {
                if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
                {
                  YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nBestMove... errorcode = %d\n", nResult);
                }
              }
            }
            else
            {
              if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
              {
                YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nMoveListIndex... errorcode = %d\n", nResult);
              }
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nStackIndex... errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nWorkId... errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nSyncPositionId... errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_MASTER_UPDATE_SEARCH_WINDOW:
      nResult = MPIWrapper_Unpack_INT64(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        &pThis->nWorkId,
                                        MPI_COMM_WORLD);

      if(nResult == MPI_SUCCESS)
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "nWorkId = <" F12INT64d ">\n", pThis->nWorkId);
        }

        nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                        pThis->pParameterBuffer,
                                        MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                        &pThis->nPosition,
                                        &pThis->nAlpha,
                                        MPI_COMM_WORLD);

        if(nResult == MPI_SUCCESS)
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "nAlpha = <%d>\n", pThis->nAlpha);
          }

          nResult = MPIWrapper_Unpack_INT(MPIWrapper_Instance(), 
                                          pThis->pParameterBuffer,
                                          MAX_REQUEST_PARAMETER_BUFFER_SIZE,
                                          &pThis->nPosition,
                                          &pThis->nBeta,
                                          MPI_COMM_WORLD);

          if(nResult == MPI_SUCCESS)
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_INFO_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "nBeta = <%d>\n", pThis->nBeta);
            }
          }
          else
          {
            if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
            {
              YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nBeta... errorcode = %d\n", nResult);
            }
          }
        }
        else
        {
          if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
          {
            YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(..., &pThis->nAlpha... errorcode = %d\n", nResult);
          }
        }
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT64(..., &pThis->nWorkId... errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_HASH_TABLE_REPLICATION:
      pThis->pHashTableBuffer = g_pYBWCManagerInstance->pHashTableReceiveBuffer;

      ASSERT(pThis->pHashTableBuffer->nCount == 0);

      pThis->pHashTableBuffer->nCount = 0; // kh 15.11.06 defensive

      nResult = Request_GetHashTableBufferParameterFromBuffer(pThis); 
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at Request_GetHashTableBufferParameterFromBuffer(..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_HASH_TABLE_BROADCAST_BUFFER:
      ASSERT(false); // kh 06.12.06 this request will never be received, it is only used as a broadcast buffer 
      break;

    case REQUEST_TYPE_HISTORY_TABLE_REPLICATION:

      nResult = Request_GetHistoryTableBufferParameterFromBuffer(pThis); 
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at Request_GetHistoryTableBufferParameterFromBuffer(..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_HISTORY_TABLE_BROADCAST_BUFFER:
      ASSERT(false); // kh 20.02.07 this request will never be received, it is only used as a broadcast buffer 
      break;

    case REQUEST_TYPE_HIST_TOT_TABLE_REPLICATION:

      nResult = Request_GetHistTotTableBufferParameterFromBuffer(pThis); 
      if(nResult == MPI_SUCCESS)
      {
      }
      else
      {
        if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
        {
          YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at Request_GetHistTotTableBufferParameterFromBuffer(..., errorcode = %d\n", nResult);
        }
      }
      break;

    case REQUEST_TYPE_HIST_TOT_TABLE_BROADCAST_BUFFER:
      ASSERT(false); // kh 22.02.07 this request will never be received, it is only used as a broadcast buffer 
      break;

    default:
      break;
    }
  }
  else
  {
    if(g_pYBWCManagerInstance->pFruitConfiguration->nFruitDebugLevel >= FRUIT_DEBUG_LEVEL_ERROR_PRIO_1)
    {
      YBWCManager_Printf(g_pYBWCManagerInstance, "ERROR Request_GetParameterFromBuffer(... failed at MPIWrapper_Unpack_INT(... &pThis->nType..., errorcode = %d\n", nResult);
    }
  }

  return nResult;
}
