//    Copyright 2009 Antonio Torrecillas Gonzalez
//
//    This file is part of Rocinante.
//
//    Rocinante is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, either version 3 of the License, or
//    (at your option) any later version.
//
//    Rocinante is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with Rocinante.  If not, see <http://www.gnu.org/licenses/>
//

#include <cassert>
#include <math.h>
#include "TreeNode.h"
#include <stdio.h>

static TreeNode Nodes[MAXNODES];
static TreeNode *RootNode;
static TreeNode *FirstFree;

TreeNode::TreeNode(void)
{
	Reset();
}

TreeNode::~TreeNode(void)
{
}

void TreeNode::InitTree()
{
	int i;
	for(i = MAXNODES-2;i >= 0;i--)
	{
		Nodes[i].Reset();
		Nodes[i].NextSibbling = &Nodes[i+1];
	}
	FirstFree = &Nodes[0];
}
void TreeNode::Reset()
{
	Parent = FirstChild = NextSibbling = 0;
	RealVal = UNDEFINED;
	OptVal = UNDEFINED;
	PessVal = UNDEFINED;
	OptPrb = 0.0;
	MoveCount = 0;
	Color = 0;
	Move = 0;
	fen[0] = '\0';
	MoveStr[0] = '\0';
	SubtreeSize = 0;
}
// Get the first free node
TreeNode *TreeNode::GetFree()
{
	TreeNode *aux = FirstFree;
	if(!FirstFree)
		InitTree();
	aux = FirstFree;
		
	FirstFree = FirstFree->NextSibbling;
	aux->NextSibbling = 0;
	return aux;
}

void TreeNode::SetRootNode(TreeNode *New)
{
	RootNode = New;
}

TreeNode *TreeNode::GetRootNode()
{
	return RootNode;
}

TreeNode *TreeNode::MoveTo(MoveMode mode)
{
	switch(mode)
	{
	case PARENT:
		return Parent;
	case FIRSTCHILD:
		return FirstChild;
	case NEXTSIBBLING:
		return NextSibbling;
	case LASTCHILD:
		TreeNode *aux;
		aux = FirstChild;
		while(aux->NextSibbling)
			aux = aux->NextSibbling;
		return aux;
	}
	return 0;
}

void TreeNode::Add(TreeNode *New)
{
	TreeNode *aux;
	assert(this->Color != New->Color);
	if(!FirstChild)
	{
		FirstChild = New;
	}
	else
	{
		aux = FirstChild;
		while(aux->NextSibbling)
			aux = aux->NextSibbling;
		aux->NextSibbling = New;
	}
	New->Parent = this;
}

void TreeNode::Delete()
{
	// don't delete rootnode subtree
	// recursive on child
	if(!this) return;
	if(FirstChild)
	{
		if(FirstChild == RootNode)
		{
			RootNode->Parent = 0;
			if(RootNode->NextSibbling)
			{
				RootNode->NextSibbling->Delete();
				RootNode->NextSibbling = 0;
			}
		}
		else
		{
			FirstChild->Delete();
		}
	}
	// recursive on Sibblings
	if(NextSibbling)
	{
		if(NextSibbling == RootNode)
		{
			RootNode->Parent = 0;
			if(RootNode->NextSibbling)
			{
				RootNode->NextSibbling->Delete();
				RootNode->NextSibbling = 0;
			}
		}
		else
			NextSibbling->Delete();
	}
	Reset();
	this->NextSibbling = FirstFree;
	FirstFree = this;
}

const int INFINITO = 999999;

TreeNode *TreeNode::SelectBestReal()
{
	int Val;
	TreeNode *aux;
	TreeNode *Sel = 0;
	if(Color == 0)
	{
		// Maximize
		Val = -INFINITO;
		for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			if(aux->RealVal > Val)
			{
				Sel = aux;
				Val = Sel->RealVal;
			}
		}
	}
	else
	{
		// minimize
		Val = INFINITO;
		for(aux = MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			if(aux->RealVal < Val)
			{
				Sel = aux;
				Val = Sel->RealVal;
			}
		}
	}
	return Sel;
}

TreeNode *TreeNode::SelectBestRealNotDraw()
{
	int Val;
	int ValD;
	TreeNode *aux;
	TreeNode *Sel = 0;
	if(Color == 0)
	{
		// Maximize
		Val = -INFINITO;
		for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			ValD = aux->RealVal;
			if(aux->SubtreeSize)
				ValD /= (int)(sqrt((double)aux->SubtreeSize));
			if(ValD > Val)
			{
				Sel = aux;
				Val = ValD;
			}

			//if(aux->RealVal > Val && (aux->RealVal != 0 || aux->OptVal != 0))
			//{
			//	Sel = aux;
			//	Val = Sel->RealVal;
			//}
		}
	}
	else
	{
		// minimize
		Val = INFINITO;
		for(aux = MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			ValD = aux->RealVal;
			if(aux->SubtreeSize)
				ValD /= (int)(sqrt((double)aux->SubtreeSize));
			if(ValD < Val)
			{
				Sel = aux;
				Val = ValD;
			}
			//if(aux->RealVal < Val && (aux->RealVal != 0 || aux->OptVal != 0))
			//{
			//	Sel = aux;
			//	Val = Sel->RealVal;
			//}
		}
	}
	return Sel;
}

TreeNode *TreeNode::SelectBestOpt()
{
	int Val;
	TreeNode *aux;
	TreeNode *Sel = 0;
	if(Color == 0) 
	{
		// Maximize
		Val = -INFINITO;
		for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			if(aux->OptVal != UNDEFINED)
			if(aux->OptVal > Val)
			{
				Sel = aux;
				Val = Sel->OptVal;
			}
		}
	}
	else
	{
		// minimize
		Val = INFINITO;
		for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			if(aux->OptVal != UNDEFINED)
			if(aux->OptVal < Val)
			{
				Sel = aux;
				Val = Sel->OptVal;
			}
		}
	}
	return Sel;
}
TreeNode *TreeNode::SelectBestPess()
{
	int Val;
	TreeNode *aux;
	TreeNode *Sel = 0;
	if(Color== 0) // GetRootNode()->
	{
		// Maximize
		Val = -INFINITO;
		for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			if(aux->PessVal != UNDEFINED)
			if(aux->PessVal > Val)
			{
				Sel = aux;
				Val = Sel->PessVal;
			}
		}
	}
	else
	{
		// minimize
		Val = INFINITO;
		for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
		{
			if(aux->PessVal != UNDEFINED)
			if(aux->PessVal < Val)
			{
				Sel = aux;
				Val = Sel->PessVal;
			}
		}
	}
	return Sel;
}

const int MATE = 30000;

TreeNode *TreeNode::SelectBestOptPrb()
{
	double Val;
	TreeNode *aux;
	TreeNode *Sel = 0;
	double PrbD = 0.0;

	// Maximize
	Val = 0;
	for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
	{
		if(aux->SubtreeSize)
			PrbD = aux->OptPrb  / (sqrt((double)(aux->SubtreeSize)));
		else
			PrbD = aux->OptPrb ;
		if(  PrbD > Val	)
		{
			Sel = aux;
			Val = PrbD; 
		}
	}
	return Sel;
}

TreeNode *TreeNode::SelectBestOptPrbNotDraw()
{
	double Val;
	TreeNode *aux;
	TreeNode *Sel = 0;
	TreeNode *Best = SelectBestReal();

	// Maximize
	Val = 0;
	for(aux = this->MoveTo(FIRSTCHILD);aux; aux = aux->MoveTo(NEXTSIBBLING))
	{
		// si nos dan mate no nos vale
		if((aux->OptPrb) > Val 
			&& (aux->RealVal != 0 || aux->OptVal != 0)
			&& !(Best && aux == Best)
			)
		{
			Sel = aux;
			Val = Sel->OptPrb ; 
		}
	}
	return Sel;
}

TreeNode *TreeNode::TraceDown(int OptReal)
{
	TreeNode *Sel = 0;
	if(!OptReal)
	{	// optimism
		Sel = SelectBestOptPrb();
	}
	else
	{	// real
		Sel = SelectBestReal();
	}
	if(Sel)
		return Sel->TraceDown(!OptReal);
	return this;
}
