/*
    Protector -- a UCI chess engine

    Copyright (C) 2009-2010 Raimund Heid (Raimund_Heid@yahoo.com)

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <math.h>
#include "search.h"
#include "matesearch.h"
#include "io.h"
#include "movegeneration.h"
#include "hash.h"
#include "evaluation.h"
#include "book.h"
#include "coordination.h"

#ifdef INCLUDE_TABLEBASE_ACCESS
#include "tablebase.h"
#endif

/* #define DEBUG_THREAD_COORDINATION       /* */
/* #define CRV_CHECK               /* */
#define QUIESCENCE_FUTILITY_CUT /* */
#define FUTILITY_CUTOFF_HISTORY /* */
/* #define EXTRA_BONUS_BETA_CUT    /* */
#define EXTEND_PIECE_SWAP       /* */
#define AGRESSIVE_HISTORY_PRUNING       /* */
#define EXTEND_ON_THREATS       /* */
#define PASSIVE_DEFENCE_MOVES   /* */
/* #define EXTEND_CONNECTED_MOVES  /* */
#define RAZORING                /* */
/* #define STATIC_NULLMOVE_CUTS    /* */
#define NULLMOVE_EXTRA_REDUCTION        /* */
#define TEST_FOREIGN_KILLERS    /* */
#define SINGLE_MOVE_EXTENSION   /* */
/* #define STANDARD_HISTORY_PRUNING        /* */
/* #define NONULL_SPECIALIZED      /* */
/* #define HASHTABLE_IN_QS         /* */
/* #define ASPIRATION_SEARCH_ROOT  /* */

extern bool resetGlobalHashtable;
const int FAIL_LOW_MARGIN = 50; /* was 20 */
const int FUTILITY_CUT_LIMIT_PCT = 60;  /* was 60 */
int THREAT_LIMIT = DEFAULT_THREAT_LIMIT;
const int HASH_DEPTH_OFFSET = 3;

Move crv[MAX_DEPTH];
UINT64 crvHash;
INT32 checkTimeCount = 0;

int quietMoveCountLimit[32];    /* number of quiet moves to be examined @ specific restDepth */
int quietPvMoveReduction[64][64];       /* [restDepth][moveCount] */
int quietMoveReduction[64][64]; /* [restDepth][moveCount] */
int futilityMargin[16][64];     /* [restDepth][moveCount] */

static int searchBest(Variation * variation, int alpha, int beta,
                      const int ply, const int restDepth, const bool pvNode,
                      const bool reduction, PrincipalVariation * pv,
                      Move excludeMove);

INLINE static void checkTime(Variation * variation)
{
   variation->timestamp = getTimestamp();

   if (variation->timeLimit != 0 &&
       variation->timestamp - variation->startTime >= variation->timeLimit &&
       variation->searchStatus == SEARCH_STATUS_RUNNING &&
       variation->ponderMode == FALSE)
   {
      variation->searchStatus = SEARCH_STATUS_TERMINATE;

#ifdef DEBUG_THREAD_COORDINATION
      logDebug
         ("Time limit reached (time=%lu ms, limit=%lu ms)).\n",
          variation->timestamp - variation->startTime, variation->timeLimit);
#endif
   }

   if ((checkTimeCount++ % 10) == 0 && variation->handleSearchEvent != 0)
   {
      variation->handleSearchEvent(SEARCHEVENT_STATISTICS_UPDATE, variation);
   }
}

INLINE static void initPlyInfo(PlyInfo * info)
{
   info->quietMove = FALSE;
   info->staticValueAvailable = FALSE;
}

INLINE static bool hasDangerousPawns(const Position * position,
                                     const Color color)
{
   if (color == WHITE)
   {
      return (bool)
         ((position->piecesOfType[WHITE_PAWN] & squaresOfRank[RANK_7]) !=
          EMPTY_BITBOARD);
   }
   else
   {
      return (bool)
         ((position->piecesOfType[BLACK_PAWN] & squaresOfRank[RANK_2]) !=
          EMPTY_BITBOARD);
   }
}

static INLINE int getStaticValue(Variation * variation, const int ply)
{
   PlyInfo *pi = &variation->plyInfo[ply];

   if (pi->staticValueAvailable == FALSE)
   {
      EvaluationBase base;

      pi->staticValue =
         getValue(&variation->singlePosition, &base, variation->pawnHashtable,
                  variation->kingsafetyHashtable);
      pi->staticValueAvailable = TRUE;
   }

   return pi->staticValue;
}

static int searchBestQuiescence(Variation * variation, int alpha, int beta,
                                const int ply, const int restDepth,
                                const bool pvNode, PrincipalVariation * pv)
{
#ifdef HASHTABLE_IN_QS
   const int oldAlpha = alpha;
   Hashentry *tableHit = NULL;
   UINT8 hashentryFlag;
   UINT8 hashDepth = (restDepth >= 0 ? 2 : 1);
#endif
   Position *position = &variation->singlePosition;
   int best = VALUE_MATED, currentValue = VALUE_MATED, historyLimit, i;
   const int VALUE_MATE_HERE = -VALUE_MATED - ply + 1;
   const int VALUE_MATED_HERE = VALUE_MATED + ply;
   Movelist movelist;
   Move currentMove, bestMove = NO_MOVE, hashmove = NO_MOVE;
   const bool inCheck = variation->plyInfo[ply - 1].currentMoveIsCheck;
   PrincipalVariation newPv;
   EvaluationBase base;

   assert(alpha >= VALUE_MATED && alpha <= -VALUE_MATED);
   assert(beta >= VALUE_MATED && beta <= -VALUE_MATED);
   assert(alpha < beta);
   assert(ply > 0 && ply < MAX_DEPTH);
   assert(restDepth < DEPTH_RESOLUTION);
   assert(passiveKingIsSafe(position));
   assert((inCheck != FALSE) == (activeKingIsSafe(position) == FALSE));

   initPlyInfo(&variation->plyInfo[ply]);

   base.futilityMargin[WHITE] = base.futilityMargin[BLACK] = 0;
   newPv.length = pv->length = 0;
   variation->nodes++;

   if (variation->nodes == (variation->nodes & ~((UINT64) 0xFFFF)))
   {
      checkTime(variation);
   }

   if (variation->terminate &&
       movesAreEqual(variation->bestBaseMove, NO_MOVE) == FALSE)
   {
      variation->searchStatus = SEARCH_STATUS_TERMINATE;
   }

   if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
       variation->nominalDepth > 1)
   {
      return 0;
   }

   /* Check for a draw by repetition. */
   /* ------------------------------- */
   historyLimit = POSITION_HISTORY_OFFSET + variation->ply -
      position->halfMoveClock;

   assert(historyLimit >= 0);

   for (i = POSITION_HISTORY_OFFSET + variation->ply - 4;
        i >= historyLimit; i -= 2)
   {
      if (position->hashValue == variation->positionHistory[i])
      {
         return variation->drawScore[position->activeColor];
      }
   }

#ifdef HASHTABLE_IN_QS
   /* Probe the transposition table */
   /* ----------------------------- */
   tableHit = getDatedEntry(variation->hashtable,
                            variation->singlePosition.hashValue);

   if (tableHit != NULL)
   {                            /* 45% */
      const int importance =
         getHashentryImportance(tableHit) - HASH_DEPTH_OFFSET;
      const int value = getHashentryValue(tableHit);
      const int hashValue = calcEffectiveValue(value, ply);
      const int flag = getHashentryFlag(tableHit);

      hashmove = (Move) getHashentryMove(tableHit);

      if (hashmove != NO_MOVE)
      {                         /* 81% */
         assert(moveIsPseudoLegal(position, hashmove));

         if (moveIsPseudoLegal(position, hashmove))
         {
            assert(moveIsLegal(position, hashmove));

            if (variation->handleSearchEvent != 0)
            {
               appendMoveToPv(&newPv, pv, hashmove);
            }
         }
         else
         {
            hashmove = NO_MOVE;
         }
      }

      if (flag == HASHVALUE_EVAL)
      {
         PlyInfo *pi = &variation->plyInfo[variation->ply];

         pi->staticValue = hashValue;
         pi->staticValueAvailable = TRUE;
         base.futilityMargin[WHITE] = base.futilityMargin[BLACK] = 500;
      }
      else if (pvNode == FALSE && restDepth <= importance * DEPTH_RESOLUTION)
      {                         /* 99% */
         assert(flag == HASHVALUE_UPPER_LIMIT || flag == HASHVALUE_EXACT ||
                flag == HASHVALUE_LOWER_LIMIT);
         assert(hashValue >= VALUE_MATED && hashValue < -VALUE_MATED);

         switch (flag)
         {
         case HASHVALUE_UPPER_LIMIT:
            if (hashValue <= alpha)
            {
               return (hashValue <= VALUE_ALMOST_MATED ? alpha : hashValue);
            }
            break;

         case HASHVALUE_EXACT:
            return hashValue;

         case HASHVALUE_LOWER_LIMIT:
            if (hashValue >= beta)
            {
               return (hashValue >= -VALUE_ALMOST_MATED ? beta : hashValue);
            }
            break;

         default:;
         }
      }
   }
#endif

   if (inCheck == FALSE)
   {
      assert(flipTest(position,
                      variation->pawnHashtable,
                      variation->kingsafetyHashtable) != FALSE);

      if (variation->plyInfo[ply].staticValueAvailable == FALSE)
      {
         best = getValue(position,
                         &base,
                         variation->pawnHashtable,
                         variation->kingsafetyHashtable);
         variation->plyInfo[ply].staticValue = best;
         variation->plyInfo[ply].staticValueAvailable = TRUE;
      }
      else
      {
         best = variation->plyInfo[ply].staticValue;
      }

      if (best > alpha)
      {
         alpha = best;

         if (best >= beta)
         {
#ifdef HASHTABLE_IN_QS
            setDatedEntry(variation->hashtable, position->hashValue,
                          calcHashtableValue(best, ply),
                          0, packedMove(NO_MOVE), HASHVALUE_EVAL);
#endif

            return best;
         }
      }

      currentValue = best;
   }

   if (ply >= MAX_DEPTH)
   {
      assert(flipTest(position,
                      variation->pawnHashtable,
                      variation->kingsafetyHashtable) != FALSE);

      return getStaticValue(variation, ply);
   }

   if (alpha < VALUE_MATED_HERE && inCheck == FALSE)
   {
      alpha = VALUE_MATED_HERE;

      if (alpha >= beta)
      {
         return VALUE_MATED_HERE;
      }
   }

   if (beta > VALUE_MATE_HERE)
   {
      beta = VALUE_MATE_HERE;

      if (beta <= alpha)
      {
         return VALUE_MATE_HERE;
      }
   }

   initQuiescenceMovelist(&movelist, &variation->singlePosition,
                          &variation->plyInfo[ply],
                          &variation->historyValue[0],
                          (inCheck ? hashmove : NO_MOVE), restDepth, inCheck);
   initializePlyInfo(variation);

   while ((currentMove = getNextMove(&movelist)) != NO_MOVE)
   {
      int value, newDepth =
         (inCheck ? restDepth : restDepth - DEPTH_RESOLUTION);

#ifdef QUIESCENCE_FUTILITY_CUT
      const int delta = 50 + base.futilityMargin[position->activeColor];
      int optValue = currentValue + delta +
         basicValue[position->piece[getToSquare(currentMove)]];

      if (pvNode == FALSE && inCheck == FALSE && optValue < alpha &&
          pieceType(position->piece[getToSquare(currentMove)]) != QUEEN &&
          getNewPiece(currentMove) != QUEEN &&
          (pieceType(position->piece[getFromSquare(currentMove)]) != PAWN ||
           colorRank(position->activeColor,
                     getToSquare(currentMove)) != RANK_7))
      {
         if (getNewPiece(currentMove) != NO_PIECE)
         {
            optValue +=
               basicValue[getNewPiece(currentMove)] - basicValue[PAWN];
         }

         if (getToSquare(currentMove) == position->enPassantSquare &&
             pieceType(position->piece[getFromSquare(currentMove)]) == PAWN)
         {
            optValue += basicValue[PAWN];
         }

         if (optValue < alpha && moveIsCheck(currentMove, position) == FALSE)
         {
            best = max(best, optValue);

            continue;
         }
      }
#endif

      assert(moveIsPseudoLegal(position, currentMove));

      if (makeMoveFast(variation, currentMove) != 0 ||
          passiveKingIsSafe(&variation->singlePosition) == FALSE)
      {
         unmakeLastMove(variation);

         continue;
      }

      variation->plyInfo[ply].currentMoveIsCheck =
         activeKingIsSafe(&variation->singlePosition) == FALSE;

      assert(position->piece[getToSquare(currentMove)] != NO_PIECE ||
             (getToSquare(currentMove) == position->enPassantSquare &&
              position->piece[getFromSquare(currentMove)] ==
              (PAWN | position->activeColor)) ||
             getNewPiece(currentMove) != NO_PIECE ||
             inCheck || variation->plyInfo[ply].currentMoveIsCheck);

      assert(inCheck != FALSE ||
             basicValue[position->piece[getFromSquare(currentMove)]] <=
             basicValue[position->piece[getToSquare(currentMove)]] ||
             seeMove(position, currentMove) >= 0);

      value = -searchBestQuiescence(variation, -beta, -alpha, ply + 1,
                                    newDepth, pvNode, &newPv);

      unmakeLastMove(variation);

      if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
          variation->nominalDepth > 1)
      {
         return 0;
      }

      if (value > best)
      {
         best = value;
         bestMove = currentMove;

         if (variation->handleSearchEvent != 0)
         {
            appendMoveToPv(&newPv, pv, currentMove);
         }

         if (best > alpha)
         {
            alpha = best;

            if (best >= beta)
            {
               break;
            }
         }
      }
   }

   if (best == VALUE_MATED)
   {
      /* mate */

      assert(inCheck != FALSE);

      best = VALUE_MATED + ply;
   }

#ifdef HASHTABLE_IN_QS
   /* Store the value in the transposition table. */
   /* ------------------------------------------- */
   if (best > oldAlpha)
   {
      hashentryFlag =
         (best >= beta ? HASHVALUE_LOWER_LIMIT : HASHVALUE_EXACT);
   }
   else
   {
      hashentryFlag =
         (inCheck == FALSE && best == getStaticValue(variation, ply) ?
          HASHVALUE_EVAL : HASHVALUE_UPPER_LIMIT);
   }

   setDatedEntry(variation->hashtable, position->hashValue,
                 calcHashtableValue(best, ply),
                 hashDepth, packedMove(bestMove), hashentryFlag);
#endif

   return best;
}

INLINE static bool moveIsQuiet(const Move move, const Position * position)
{
   return (bool) (position->piece[getToSquare(move)] == NO_PIECE &&
                  getNewPiece(move) == NO_PIECE &&
                  (getToSquare(move) != position->enPassantSquare ||
                   pieceType(position->piece[getFromSquare(move)]) != PAWN));
}

static int searchBestWithoutNullmove(Variation * variation, int alpha,
                                     int beta, const int ply,
                                     const int restDepth, const bool pvNode,
                                     const Move hashmove,
                                     PrincipalVariation * pv)
{
   Position *position = &variation->singlePosition;
   int best = VALUE_MATED;
   Movelist movelist;
   int i, historyLimit;
   Move currentMove;
   const bool inCheck = variation->plyInfo[ply - 1].currentMoveIsCheck;
   PrincipalVariation newPv;

   assert(alpha >= VALUE_MATED && alpha <= -VALUE_MATED);
   assert(beta >= VALUE_MATED && beta <= -VALUE_MATED);
   assert(alpha < beta);
   assert(ply > 0 && ply < MAX_DEPTH);
   assert(restDepth > 0);
   assert(passiveKingIsSafe(position));
   assert(inCheck == FALSE);

   initPlyInfo(&variation->plyInfo[ply]);
   newPv.length = pv->length = 0;
   variation->nodes++;

   if (variation->nodes == (variation->nodes & ~((UINT64) 0xFFFF)))
   {
      checkTime(variation);
   }

   if (variation->terminate &&
       movesAreEqual(variation->bestBaseMove, NO_MOVE) == FALSE)
   {
      variation->searchStatus = SEARCH_STATUS_TERMINATE;
   }

   if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
       variation->nominalDepth > 1)
   {
      return 0;
   }

   if (ply >= MAX_DEPTH)
   {
      EvaluationBase base;

      assert(flipTest(position,
                      variation->pawnHashtable,
                      variation->kingsafetyHashtable) != FALSE);

      return getValue(position, &base,
                      variation->pawnHashtable,
                      variation->kingsafetyHashtable);
   }

   /* Check for a draw by repetition. */
   /* ------------------------------- */
   historyLimit = POSITION_HISTORY_OFFSET + variation->ply -
      position->halfMoveClock;

   assert(historyLimit >= 0);

   for (i = POSITION_HISTORY_OFFSET + variation->ply - 4;
        i >= historyLimit; i -= 2)
   {
      if (position->hashValue == variation->positionHistory[i])
      {
         return variation->drawScore[position->activeColor];
      }
   }

#ifdef TEST_FOREIGN_KILLERS
   if (ply >= 2)
   {
      variation->plyInfo[ply].killerMove3 =
         variation->plyInfo[ply - 2].killerMove1;
      variation->plyInfo[ply].killerMove4 =
         variation->plyInfo[ply - 2].killerMove2;
   }
   else
   {
#endif
      variation->plyInfo[ply].killerMove3 = NO_MOVE;
      variation->plyInfo[ply].killerMove4 = NO_MOVE;
#ifdef TEST_FOREIGN_KILLERS
   }
#endif

   initStandardMovelist(&movelist, &variation->singlePosition,
                        &variation->plyInfo[ply],
                        &variation->historyValue[0], hashmove, inCheck);
   initializePlyInfo(variation);

   /* Loop through all moves in this node. */
   /* ------------------------------------ */
   while ((currentMove = getNextMove(&movelist)) != NO_MOVE)
   {
      int value, newDepth;
      bool check;
      const int historyIdx = historyIndex(currentMove, position);
      const bool quietMove = moveIsQuiet(currentMove, position);

      variation->plyInfo[ply].indexCurrentMove = historyIdx;
      variation->plyInfo[ply].quietMove = quietMove;

      assert(moveIsPseudoLegal(position, currentMove));

      /* Execute the current move and check if it is legal. */
      /* -------------------------------------------------- */
      if (makeMoveFast(variation, currentMove) != 0 ||
          passiveKingIsSafe(&variation->singlePosition) == FALSE)
      {
         unmakeLastMove(variation);

         continue;
      }

      /* Check the conditions for search extensions and finally */
      /* calculate the rest depth for the next ply.             */
      /* ------------------------------------------------------ */
      variation->plyInfo[ply].currentMoveIsCheck = check =
         activeKingIsSafe(&variation->singlePosition) == FALSE;

      newDepth = (check ? restDepth : restDepth - DEPTH_RESOLUTION);

      /* Recursive search */
      /* ---------------- */
      value = -searchBest(variation, -beta, -alpha, ply + 1,
                          newDepth, pvNode, FALSE, &newPv, NO_MOVE);

      unmakeLastMove(variation);

      if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
          variation->nominalDepth > 1)
      {
         return 0;
      }

      /* Calculate the parameters controlling the search tree. */
      /* ----------------------------------------------------- */
      if (value > best)
      {
         best = value;

         if (variation->handleSearchEvent != 0)
         {
            appendMoveToPv(&newPv, pv, currentMove);
         }

         if (best > alpha)
         {
            alpha = best;

            if (best >= beta)
            {
               return best;
            }
         }
      }
   }

   /* No legal move was found. Check if it's mate or stalemate. */
   /* --------------------------------------------------------- */
   if (best == VALUE_MATED)
   {
      /* stalemate */

      assert(inCheck == FALSE);

      best = variation->drawScore[position->activeColor];
   }

   return best;
}

INLINE static bool moveIsCastling(const Move move, const Position * position)
{
   return (bool) (pieceType(position->piece[getFromSquare(move)]) == KING &&
                  distance(getFromSquare(move), getToSquare(move)) == 2);
}

INLINE static bool moveAttacksSquare(const Move move, const Square square,
                                     const Position * position)
{
   const Piece piece = position->piece[getFromSquare(move)];
   Bitboard orginalMoves =
      getCaptureMoves(getFromSquare(move), piece, position->allPieces);
   Bitboard movesFromTarget =
      getCaptureMoves(getToSquare(move), piece, position->allPieces);

   return testSquare(orginalMoves, square) == FALSE &&
      testSquare(movesFromTarget, square);
}

INLINE static bool movesAreConnected(const Move firstMove,
                                     const Move secondMove,
                                     Position * position)
{
   const Square firstFrom = getFromSquare(firstMove);
   const Square firstTo = getToSquare(firstMove);
   const Square secondFrom = getFromSquare(secondMove);
   const Square secondTo = getToSquare(secondMove);
   const Piece firstPiece = position->piece[firstTo];
   const Piece secondPiece = position->piece[secondFrom];
   const bool firstPieceIsSlider =
      (bool) (firstPiece & PP_SLIDING_PIECE) != 0;
   const bool secondPieceIsSlider =
      (bool) (secondPiece & PP_SLIDING_PIECE) != 0;

   if (firstTo == secondFrom)
   {
      assert(firstPieceIsSlider == secondPieceIsSlider);

      if (secondTo == firstFrom)
      {
         return FALSE;          /* avoid shuffling */
      }

      if (firstPieceIsSlider)
      {
         const Bitboard backyards = squaresBehind[firstTo][firstFrom] |
            squaresBehind[firstFrom][secondFrom];

         return (testSquare(backyards, secondTo) == FALSE);     /* slider changes direction */
      }
      else
      {
         return (bool) distance(firstFrom, secondTo) >= 2;
      }
   }

   if (secondPieceIsSlider)
   {
      const Bitboard corridor = squaresBetween[secondFrom][secondTo];

      if (testSquare(corridor, firstFrom))
      {
         return TRUE;           /* first move was clearing square */
      }
   }

   if (secondTo == firstFrom)
   {
      return TRUE;              /* first move was clearing square */
   }

   return FALSE;
}

INLINE static bool moveIsDefence(const Move move, const Move threatMove,
                                 Position * position)
{
   const Square threatTarget = getToSquare(threatMove);
   const Square threatFromSquare = getFromSquare(threatMove);
   const Square fromSquare = getFromSquare(move);

   if (fromSquare == threatTarget && seeMove(position, move) >= 0)
   {
      return TRUE;              /* threatened piece was moved */
   }
#ifdef PASSIVE_DEFENCE_MOVES
   else if (basicValue[position->piece[threatFromSquare]] >=
            basicValue[position->piece[fromSquare]] &&
            (position->piece[threatFromSquare] & PP_SLIDING_PIECE) != 0 &&
            distance(threatFromSquare, threatTarget) > 1)
   {
      Bitboard sliderCorridor =
         squaresBetween[threatFromSquare][threatTarget];

      assert(pieceType(position->piece[threatFromSquare]) == QUEEN ||
             pieceType(position->piece[threatFromSquare]) == ROOK ||
             pieceType(position->piece[threatFromSquare]) == BISHOP);
      assert(sliderCorridor != EMPTY_BITBOARD);

      if (testSquare(sliderCorridor, getToSquare(move)) &&
          seeMove(position, move) >= 0)
      {
         /*
            dumpMove(threatMove);
            dumpMove(move);
            dumpPosition(position);
            /* */

         return TRUE;           /* move blocks slider */
      }
   }

   if (basicValue[position->piece[threatTarget]] > 0 &&
       basicValue[position->piece[threatFromSquare]] >=
       basicValue[position->piece[threatTarget]])
   {
      if (moveAttacksSquare(move, threatTarget, position) &&
          seeMove(position, move) >= 0)
      {
         return TRUE;           /* move defends target */
      }
   }
#endif

   return FALSE;
}

static INLINE bool isPassedPawnMove(const Square pawnSquare,
                                    const Square targetSquare,
                                    const Position * position)
{
   const Piece piece = position->piece[pawnSquare];

   if (pieceType(piece) == PAWN)
   {
      return pawnIsPassed(position, targetSquare, pieceColor(piece));
   }
   else
   {
      return FALSE;
   }
}

static INLINE int getFutilityMargin(const int restDepth, const int moveCount)
{
   return (restDepth < 16 ? futilityMargin[restDepth][min(63, moveCount)] :
           -VALUE_MATED);
}

static int searchBest(Variation * variation, int alpha, int beta,
                      const int ply, const int restDepth, const bool pvNode,
                      const bool reduction, PrincipalVariation * pv,
                      Move excludeMove)
{
   const int oldAlpha = alpha;
   Position *position = &variation->singlePosition;
   int best = VALUE_MATED;
   const int VALUE_MATE_HERE = -VALUE_MATED - ply + 1;
   const int VALUE_MATED_HERE = VALUE_MATED + ply;
   Movelist movelist;
   Hashentry *tableHit = NULL;
   UINT8 hashentryFlag;
   int i, historyLimit, numMovesPlayed = 0, numQuietMoves = 0;
   Move hashmove = NO_MOVE, bestMove = NO_MOVE, threatMove = NO_MOVE;
   Move currentMove;
   Move lastMove = variation->plyInfo[ply - 1].currentMove;
   const bool inCheck = variation->plyInfo[ply - 1].currentMoveIsCheck;
   PrincipalVariation newPv;
   int historyIndexPlayed[MAX_MOVES_PER_POSITION];
   const bool dangerousPawns =
      hasDangerousPawns(position, position->activeColor);
   const int rdBasic = restDepth / DEPTH_RESOLUTION;
   bool singularExtensionNode = FALSE;

#ifndef RAZORING
   bool futilityCut = FALSE;
#endif

   initPlyInfo(&variation->plyInfo[ply]);

#ifndef FUTILITY_CUTOFF_HISTORY
   getStaticValue(variation, ply);

   if (variation->plyInfo[ply - 1].quietMove)
   {
      const int moveIndex = variation->plyInfo[ply - 1].indexCurrentMove;
      INT16 *storedGain = &variation->positionalGain[moveIndex];
      const INT16 currentDiff = (INT16)
         (getStaticValue(variation, ply) +
          variation->plyInfo[ply - 1].staticValue);

      if (currentDiff > *storedGain)
      {
         *storedGain = currentDiff;
      }
      else
      {
         *storedGain--;
      }
   }
#endif

#ifdef CRV_CHECK
   bool crvFail = FALSE;

   for (i = 0; crv[i] != NO_MOVE; i++)
   {
      if (movesAreEqual(variation->plyInfo[i].currentMove, crv[i]) == FALSE)
      {
         crvFail = TRUE;
         break;
      }
   }

   if (crvFail == FALSE && i != ply)
   {
      crvFail = TRUE;
   }
#endif /* CRV_CHECK */

   if (ply + 1 > variation->selDepth)
   {
      variation->selDepth = ply + 1;
   }

   assert(alpha >= VALUE_MATED && alpha <= -VALUE_MATED);
   assert(beta >= VALUE_MATED && beta <= -VALUE_MATED);
   assert(alpha < beta);
   assert(ply > 0 && ply < MAX_DEPTH);
   assert(restDepth >= -3 * DEPTH_RESOLUTION);
   assert(passiveKingIsSafe(position));
   assert((inCheck != FALSE) == (activeKingIsSafe(position) == FALSE));

   newPv.length = pv->length = 0;
   newPv.move[0] = (UINT16) NO_MOVE;

   if (restDepth < DEPTH_RESOLUTION)
   {                            /* 63% */
      assert(excludeMove == NO_MOVE);

      return searchBestQuiescence(variation, alpha, beta, ply, 0, pvNode, pv);
   }

   variation->nodes++;

   if (variation->nodes == (variation->nodes & ~((UINT64) 0xFFFF)))
   {
      checkTime(variation);
   }

   if (variation->terminate &&
       movesAreEqual(variation->bestBaseMove, NO_MOVE) == FALSE)
   {
      variation->searchStatus = SEARCH_STATUS_TERMINATE;
   }

   if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
       variation->nominalDepth > 1)
   {
      return 0;
   }

   /* Check for a draw according to the 50-move-rule */
   /* ---------------------------------------------- */
   if (position->halfMoveClock > 100)
   {
      return variation->drawScore[position->activeColor];
   }

   /* Check for a draw by repetition. */
   /* ------------------------------- */
   historyLimit = POSITION_HISTORY_OFFSET + variation->ply -
      position->halfMoveClock;

   assert(historyLimit >= 0);

   for (i = POSITION_HISTORY_OFFSET + variation->ply - 4;
        i >= historyLimit; i -= 2)
   {
      if (position->hashValue == variation->positionHistory[i])
      {
         return variation->drawScore[position->activeColor];
      }
   }

#ifdef INCLUDE_TABLEBASE_ACCESS
   /* Probe the tablebases in case of reduced material */
   /* ------------------------------------------------ */
   if (ply >= 2 && (pvNode || restDepth >= 6 * DEPTH_RESOLUTION) &&
       position->numberOfPieces[WHITE] +
       position->numberOfPieces[BLACK] <= 5 &&
       excludeMove == NO_MOVE && tbAvailable != FALSE)
   {
      int tbValue;

      tbValue = probeTablebase(position);

      if (tbValue != TABLEBASE_ERROR)
      {
         variation->tbHits++;

         if (tbValue == 0)
         {
            return variation->drawScore[position->activeColor];
         }

         return (tbValue > 0 ? tbValue - ply : tbValue + ply);
      }
   }
#endif

   if (ply >= 2 && rdBasic >= 4 && kpkpValueAvailable(position) &&
       getStaticValue(variation, ply) == 0 && excludeMove == NO_MOVE)
   {
      return 0;
   }

   /* Probe the transposition table */
   /* ----------------------------- */
   if (excludeMove == NO_MOVE || excludeMove == NULLMOVE)
   {
      tableHit = getDatedEntry(variation->hashtable,
                               variation->singlePosition.hashValue);
   }

   if (tableHit != NULL)
   {                            /* 45% */
      const int importance =
         getHashentryImportance(tableHit) - HASH_DEPTH_OFFSET;

#ifdef SINGLE_MOVE_EXTENSION
      const int singleMoveExtensionDepth =
         (pvNode ? 6 : 8) * DEPTH_RESOLUTION;
#endif

      hashmove = (Move) getHashentryMove(tableHit);

      if (hashmove != NO_MOVE)
      {                         /* 81% */
         assert(moveIsPseudoLegal(position, hashmove));

         if (moveIsPseudoLegal(position, hashmove))
         {
            assert(moveIsLegal(position, hashmove));

            if (variation->handleSearchEvent != 0)
            {
               appendMoveToPv(&newPv, pv, hashmove);
            }
         }
         else
         {
            hashmove = NO_MOVE;
         }
      }

      if (pvNode == FALSE && restDepth <= importance * DEPTH_RESOLUTION &&
          excludeMove != NULLMOVE)
      {                         /* 99% */
         const int value = getHashentryValue(tableHit);
         const int hashValue = calcEffectiveValue(value, ply);
         const int flag = getHashentryFlag(tableHit);

         assert(flag == HASHVALUE_UPPER_LIMIT || flag == HASHVALUE_EXACT ||
                flag == HASHVALUE_LOWER_LIMIT);
         assert(hashValue >= VALUE_MATED && hashValue < -VALUE_MATED);

         switch (flag)
         {
         case HASHVALUE_UPPER_LIMIT:
            if (hashValue <= alpha)
            {
               return (hashValue <= VALUE_ALMOST_MATED ? alpha : hashValue);
            }
            break;

         case HASHVALUE_EXACT:
            return hashValue;

         case HASHVALUE_LOWER_LIMIT:
            if (hashValue >= beta)
            {
               return (hashValue >= -VALUE_ALMOST_MATED ? beta : hashValue);
            }
            break;

         default:;
         }
      }
#ifdef SINGLE_MOVE_EXTENSION
      if (restDepth >= singleMoveExtensionDepth &&
          excludeMove == NO_MOVE && hashmove != NO_MOVE &&
          getHashentryFlag(tableHit) == HASHVALUE_LOWER_LIMIT &&
          /* abs(getHashentryValue(tableHit)) <=
             2 * DEFAULTVALUE_QUEEN_OPENING && */
          importance * DEPTH_RESOLUTION >= restDepth - 3 * DEPTH_RESOLUTION)
      {
         singularExtensionNode = TRUE;
      }
#endif
   }

   if (ply >= MAX_DEPTH)
   {
      return getStaticValue(variation, ply);
   }

   if (alpha < VALUE_MATED_HERE && inCheck == FALSE)
   {
      alpha = VALUE_MATED_HERE;

      if (alpha >= beta)
      {
         return VALUE_MATED_HERE;
      }
   }

   if (beta > VALUE_MATE_HERE)
   {
      beta = VALUE_MATE_HERE;

      if (beta <= alpha)
      {
         return VALUE_MATE_HERE;
      }
   }

   initializePlyInfo(variation);

#ifdef RAZORING
   if (pvNode == FALSE && rdBasic <= 3 && lastMove != NULLMOVE &&
       inCheck == FALSE && hashmove == NO_MOVE && excludeMove == NO_MOVE &&
       alpha > VALUE_ALMOST_MATED && beta < -(VALUE_ALMOST_MATED) &&
       dangerousPawns == FALSE)
   {
      const static int delta[4] = { 0, 200, 225, 250 };
      const int limit = alpha - delta[rdBasic];

      if (getStaticValue(variation, ply) < limit)
      {
         const int qsValue =
            searchBestQuiescence(variation, limit - 1, limit, ply, 0,
                                 pvNode, pv);

         if (qsValue < limit)
         {
            return qsValue;
         }
      }
   }
#endif

#ifdef STATIC_NULLMOVE_CUTS
   if (pvNode == FALSE && rdBasic <= 3 && inCheck == FALSE &&
       alpha > VALUE_ALMOST_MATED && beta < -(VALUE_ALMOST_MATED) &&
       excludeMove == NO_MOVE &&
       numberOfNonPawnPieces(position, position->activeColor) >= 2)
   {
      const int limit = beta + futilityMargin[restDepth][0];

      if (getStaticValue(variation, ply) > limit)
      {
         return getStaticValue(variation, ply) - futilityMargin[restDepth][0];
      }
   }
#endif

   if (restDepth >= 2 * DEPTH_RESOLUTION && inCheck == FALSE &&
       pvNode == FALSE && alpha > VALUE_MATED && excludeMove == NO_MOVE &&
       numberOfNonPawnPieces(position, position->activeColor) >= 2 &&
       getNumberOfPieceMoves(position, position->activeColor, 6) >= 6 &&
       getStaticValue(variation, ply) >= beta - THREAT_LIMIT)
   {                            /* 16-32% */
      const int verificationReduction = 5 * DEPTH_RESOLUTION;
      int newDepth = restDepth - 3 * DEPTH_RESOLUTION;
      int nullValue;

#ifdef NULLMOVE_EXTRA_REDUCTION
      if (newDepth >= 4 * DEPTH_RESOLUTION &&
          getStaticValue(variation, ply) > beta + PAWN_VALUE_ENDGAME)
      {
         newDepth -= DEPTH_RESOLUTION;
      }

      if (restDepth >= verificationReduction)
      {
         newDepth -= (restDepth / 8) * DEPTH_RESOLUTION;
      }
#endif

      assert(flipTest(position,
                      variation->pawnHashtable,
                      variation->kingsafetyHashtable) != FALSE);

      makeMoveFast(variation, NULLMOVE);
      variation->plyInfo[ply].currentMoveIsCheck = FALSE;
      nullValue = -searchBest(variation, -beta, -beta + 1, ply + 1,
                              newDepth, pvNode, FALSE, &newPv, NO_MOVE);
      unmakeLastMove(variation);
      threatMove = (Move) newPv.move[0];

      if (restDepth - verificationReduction >= DEPTH_RESOLUTION &&
          nullValue >= beta)
      {                         /* 2% */
#ifdef NONULL_SPECIALIZED
         nullValue =
            searchBestWithoutNullmove(variation, alpha, beta, ply,
                                      restDepth - verificationReduction,
                                      FALSE, hashmove, &newPv);
#else
         const int noNullValue = searchBest(variation, alpha, beta, ply,
                                            restDepth - verificationReduction,
                                            FALSE, FALSE, &newPv, NULLMOVE);
#endif

         if (noNullValue >= beta)
         {                      /* 99% */
            return nullValue;
         }
      }
      else if (nullValue >= beta)
      {                         /* 70% */
         return min(nullValue, VALUE_MATE_HERE);
      }
#ifdef EXTEND_CONNECTED_MOVES
      else if (reduction && restDepth >= 5 * DEPTH_RESOLUTION &&
               nullValue < beta - THREAT_LIMIT && threatMove != NO_MOVE &&
               movesAreConnected(lastMove, threatMove, position))
#else
      else if (reduction && restDepth >= 5 * DEPTH_RESOLUTION &&
               nullValue < beta - THREAT_LIMIT)
#endif
      {
         return beta - 1;
      }
   }

   /* Internal iterative deepening. */
   /* ----------------------------- */
   if (hashmove == NO_MOVE &&
       restDepth >= (pvNode ? 3 : 7) * DEPTH_RESOLUTION &&
       (pvNode ||
        (inCheck == FALSE && getStaticValue(variation, ply) >= beta - 100)))
   {
      const Move excludeHere =
         (excludeMove != NO_MOVE ? excludeMove : NULLMOVE);
      const int searchDepth =
         (pvNode ? restDepth - 2 * DEPTH_RESOLUTION : restDepth / 2);
      /* const int value = */

      searchBest(variation, alpha, beta, ply, searchDepth, pvNode,
                 FALSE, &newPv, excludeHere);

      /*
         if (value <= alpha)
         {
         searchBest(variation, VALUE_MATED, beta, ply, searchDepth, pvNode,
         FALSE, &newPv, excludeHere);

         }
       */

      if (newPv.length > 0 &&
          moveIsPseudoLegal(position, (Move) newPv.move[0]))
      {
         hashmove = (Move) newPv.move[0];
      }
   }

#ifndef RAZORING
   if (pvNode == FALSE && restDepth <= 6 * DEPTH_RESOLUTION &&
       inCheck == FALSE && lastMove != NULLMOVE && excludeMove != NULLMOVE &&
       dangerousPawns == FALSE)
   {
      const static int delta[6] = { 150, 150, 350, 350, 550, 550 };
      const int limit = alpha - delta[rdBasic - 1];

      if (getStaticValue(variation, ply) < limit)
      {
         best = getStaticValue(variation, ply);
         futilityCut = TRUE;
      }
   }
#endif

#ifdef TEST_FOREIGN_KILLERS
   if (ply >= 2)
   {
      variation->plyInfo[ply].killerMove3 =
         variation->plyInfo[ply - 2].killerMove1;
      variation->plyInfo[ply].killerMove4 =
         variation->plyInfo[ply - 2].killerMove2;
   }
   else
   {
#endif
      variation->plyInfo[ply].killerMove3 = NO_MOVE;
      variation->plyInfo[ply].killerMove4 = NO_MOVE;
#ifdef TEST_FOREIGN_KILLERS
   }
#endif

#ifndef RAZORING
   if (futilityCut)
   {
      assert(excludeMove != NULLMOVE);

      initPreQuiescenceMovelist(&movelist, &variation->singlePosition,
                                &variation->plyInfo[ply],
                                &variation->historyValue[0], hashmove,
                                inCheck);
   }
   else
   {
#endif
      initStandardMovelist(&movelist, &variation->singlePosition,
                           &variation->plyInfo[ply],
                           &variation->historyValue[0], hashmove, inCheck);
#ifndef RAZORING
   }
#endif

   /* Loop through all moves in this node. */
   /* ------------------------------------ */
   while ((currentMove = getNextMove(&movelist)) != NO_MOVE)
   {
      const int historyIdx = historyIndex(currentMove, position);
      const int stage = moveGenerationStage[movelist.currentStage];
      int value, newDepth;
      int extension = 0, reduction = 0;
      bool check;
      const bool quietMove = moveIsQuiet(currentMove, position);
      const Square toSquare = getToSquare(currentMove);
      const bool capture = (bool) (position->piece[toSquare] != NO_PIECE);
      const bool pieceCapture = (bool)
         (capture && pieceType(position->piece[toSquare]) != PAWN);
      const bool recapture = (bool)
         (toSquare == getToSquare(lastMove) && capture &&
          (variation->plyInfo[ply - 1].captured != NO_PIECE ||
           (pieceType(position->piece[toSquare]) == PAWN &&
            variation->plyInfo[ply - 1].enPassantSquare == toSquare)));
      const bool pieceRecapture = (bool)
         (recapture && pieceCapture &&
          pieceType(variation->plyInfo[ply - 1].captured) != PAWN &&
          variation->plyInfo[ply - 1].captured != NO_PIECE);
      const bool hashMoveNode = movesAreEqual(currentMove, hashmove);

      if (excludeMove != NO_MOVE && movesAreEqual(currentMove, excludeMove))
      {
         assert(excludeMove != NULLMOVE);

         continue;              /* exclude excludeMove */
      }

#ifdef EXTEND_ON_THREATS
      variation->plyInfo[ply].indexCurrentMove = historyIdx;
      variation->plyInfo[ply].quietMove = quietMove;
#endif

      assert(moveIsPseudoLegal(position, currentMove));
      assert(hashmove == NO_MOVE || numMovesPlayed > 0 ||
             movesAreEqual(currentMove, hashmove));

#ifdef FUTILITY_CUTOFF_HISTORY
      if (restDepth <= 6 * DEPTH_RESOLUTION && pvNode == FALSE &&
          inCheck == FALSE && numQuietMoves >= rdBasic + (rdBasic & 1) &&
          quietMove != FALSE && dangerousPawns == FALSE &&
          excludeMove != NULLMOVE && !moveIsCastling(currentMove, position) &&
          !isPassedPawnMove(getFromSquare(currentMove), toSquare, position) &&
          best > VALUE_ALMOST_MATED)
      {
         const int depthDivisor = (rdBasic + 1) / 2;
         const int limit = FUTILITY_CUT_LIMIT_PCT / depthDivisor;
         const int historyValue =
            (variation->historyHitCount[historyIdx] * 100) /
            variation->historyTotalCount[historyIdx];

         if (historyValue < limit &&
             simpleMoveIsCheck(position, currentMove) == FALSE &&
             (threatMove == NO_MOVE || quietMove == FALSE ||
              moveIsDefence(currentMove, threatMove, position) == FALSE))
         {
            continue;
         }
      }
#else
      if (pvNode == FALSE && inCheck == FALSE &&
          quietMove != FALSE && dangerousPawns == FALSE &&
          numMovesPlayed >= 3 && excludeMove != NULLMOVE &&
          !isPassedPawnMove(getFromSquare(currentMove), toSquare, position) &&
          !moveIsCastling(currentMove, position) && best > VALUE_ALMOST_MATED)
      {
         const int rdIndex = min(31, restDepth);

         if (numMovesPlayed > quietMoveCountLimit[rdIndex] &&
             simpleMoveIsCheck(position, currentMove) == FALSE &&
             (threatMove == NO_MOVE ||
              moveIsDefence(currentMove, threatMove, position) == FALSE))
         {
            continue;
         }
         else if (restDepth < 9 * DEPTH_RESOLUTION)
         {
            const int futLimit =
               getStaticValue(variation, ply) +
               variation->positionalGain[historyIdx] +
               getFutilityMargin(restDepth - DEPTH_RESOLUTION,
                                 numMovesPlayed);

            if (futLimit < beta &&
                simpleMoveIsCheck(position, currentMove) == FALSE &&
                (threatMove == NO_MOVE ||
                 moveIsDefence(currentMove, threatMove, position) == FALSE))
            {
               if (futLimit > best)
               {
                  best = futLimit;
               }

               continue;
            }
         }
      }
#endif

      /* Execute the current move and check if it is legal. */
      /* -------------------------------------------------- */
      if (makeMoveFast(variation, currentMove) != 0 ||
          passiveKingIsSafe(&variation->singlePosition) == FALSE)
      {
         unmakeLastMove(variation);

         continue;
      }

      /* Check the conditions for search extensions and finally */
      /* calculate the rest depth for the next ply.             */
      /* ------------------------------------------------------ */
      variation->plyInfo[ply].currentMoveIsCheck = check =
         activeKingIsSafe(&variation->singlePosition) == FALSE;

      if (inCheck && movelist.numberOfMoves <= 2)
      {
         if (movelist.numberOfMoves <= 1)
         {
            extension = DEPTH_RESOLUTION;
         }
         else if (pvNode)
         {
            extension = DEPTH_RESOLUTION / 2;
         }
      }

      if (extension == 0 && pieceRecapture)
      {
         extension = DEPTH_RESOLUTION / 2;
      }

      if (check)
      {
         extension = DEPTH_RESOLUTION;
      }
      else if (pvNode)
      {
         const Color passiveColor = opponent(position->activeColor);

         if (pieceType(position->piece[toSquare]) == PAWN &&
             (colorRank(passiveColor, toSquare) >= RANK_6 ||
              pawnIsPassed(position, toSquare, passiveColor)))
         {
            extension = DEPTH_RESOLUTION;
         }
         else if (recapture)
         {
            extension = DEPTH_RESOLUTION;
         }
         else if (pieceCapture)
         {
            if (restDepth >= 6 * DEPTH_RESOLUTION)
            {
               extension = DEPTH_RESOLUTION;
            }
            else if (numberOfNonPawnPieces(position, WHITE) <= 2 &&
                     numberOfNonPawnPieces(position, WHITE) ==
                     numberOfNonPawnPieces(position, BLACK))
            {
               extension = DEPTH_RESOLUTION;
            }
         }
      }

      if (singularExtensionNode != FALSE &&
          extension < DEPTH_RESOLUTION && hashMoveNode)
      {
         const int value = getHashentryValue(tableHit);
         const int hashValue = calcEffectiveValue(value, ply);
         const int limitValue = hashValue - 40;

         assert(tableHit != 0);
         assert(excludeMove == NO_MOVE);

         if (limitValue > VALUE_MATED && limitValue <= -VALUE_MATED)
         {
            int excludeValue;
            PrincipalVariation tmpPv;

            tmpPv.length = 0;
            tmpPv.move[0] = (UINT16) NO_MOVE;

            unmakeLastMove(variation);
            excludeValue =
               searchBest(variation, limitValue - 1, limitValue, ply,
                          restDepth / 2, FALSE, FALSE, &tmpPv, hashmove);
            makeMoveFast(variation, currentMove);
            variation->plyInfo[ply].indexCurrentMove = historyIdx;
            variation->plyInfo[ply].quietMove = quietMove;
            variation->plyInfo[ply].currentMoveIsCheck = check;

            if (excludeValue < limitValue)
            {
               extension = DEPTH_RESOLUTION;
            }
         }
      }

      newDepth = restDepth - DEPTH_RESOLUTION +
         (excludeMove != NULLMOVE || check ? extension : 0);

      /* History pruning */
      /* --------------- */
#ifdef STANDARD_HISTORY_PRUNING
      if (inCheck == FALSE && extend == FALSE && numMovesPlayed >= 3 &&
          restDepth >= 3 * DEPTH_RESOLUTION &&
          stage != MGS_GOOD_CAPTURES_AND_PROMOTIONS &&
          isPassedPawnMove(toSquare, toSquare, position) == FALSE &&
          pvNode == FALSE && oppThreat == FALSE)
      {
         newDepth -= DEPTH_RESOLUTION;
         historyCut = TRUE;
      }
#else
      if (inCheck == FALSE && extension == 0 &&
          restDepth >= 3 * DEPTH_RESOLUTION && quietMove != FALSE &&
          stage != MGS_GOOD_CAPTURES_AND_PROMOTIONS &&
          stage != MGS_KILLER_MOVES && excludeMove != NULLMOVE &&
          isPassedPawnMove(toSquare, toSquare, position) == FALSE)
      {
         const int moveIndex = min(63, numMovesPlayed);
         const int depthIndex = min(63, rdBasic);

         reduction =
            (pvNode ? quietPvMoveReduction[depthIndex][moveIndex] :
             quietMoveReduction[depthIndex][moveIndex]);
      }
#endif

      /* Recursive search */
      /* ---------------- */
      if (pvNode == FALSE || best == VALUE_MATED)
      {
         value = -searchBest(variation, -beta, -alpha, ply + 1,
                             newDepth - reduction, pvNode,
                             (bool) reduction > 0, &newPv, NO_MOVE);

         if (reduction > 0 && value > alpha)
         {
            assert(excludeMove != NULLMOVE);

            value = -searchBest(variation, -beta, -alpha, ply + 1,
                                newDepth, pvNode, FALSE, &newPv, NO_MOVE);
         }
      }
      else                      /* pvNode == TRUE && best > VALUE_MATED */
      {
         value = -searchBest(variation, -alpha - 1, -alpha, ply + 1,
                             newDepth - reduction, FALSE,
                             (bool) reduction > 0, &newPv, NO_MOVE);

         if (reduction > 0 && value > alpha)
         {
            value = -searchBest(variation, -alpha - 1, -alpha, ply + 1,
                                newDepth, FALSE, FALSE, &newPv, NO_MOVE);
         }

         if (value > alpha /*&& value < beta */ )
         {                      /* 2% */
            value = -searchBest(variation, -beta, -alpha, ply + 1,
                                newDepth, TRUE, FALSE, &newPv, NO_MOVE);
         }
      }

      assert(value >= VALUE_MATED && value <= -VALUE_MATED);

      unmakeLastMove(variation);
      numMovesPlayed++;

      if (quietMove)
      {
         assert(numQuietMoves < MAX_MOVES_PER_POSITION);

         historyIndexPlayed[numQuietMoves++] = historyIdx;

         if (value <= best)
         {
            variation->historyValue[historyIdx] = (UINT16)
               (variation->historyValue[historyIdx] -
                (variation->historyValue[historyIdx] * restDepth) / 256);
         }
      }

      if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
          variation->nominalDepth > 1)
      {
         return 0;
      }

      /* Calculate the parameters controlling the search tree. */
      /* ----------------------------------------------------- */
      if (value > best)
      {                         /* 32% */
         best = value;

         if (variation->handleSearchEvent != 0)
         {
            appendMoveToPv(&newPv, pv, currentMove);
         }

         if (best > alpha)
         {                      /* 63% */
            alpha = best;
            bestMove = currentMove;

            if (best >= beta)
            {                   /* 99% */
               goto finish;
            }
         }
      }
   }

   /* No legal move was found. Check if it's mate or stalemate. */
   /* --------------------------------------------------------- */
   if (best == VALUE_MATED)
   {
      if (excludeMove != NO_MOVE && excludeMove != NULLMOVE)
      {
         return beta - 1;
      }

      if (inCheck)
      {
         /* mate */

         best = VALUE_MATED + ply;
      }
      else
      {
         /* stalemate */

         best = variation->drawScore[position->activeColor];
      }
   }

 finish:

#ifdef CRV_CHECK
   if (crvFail == FALSE)
   {
      logDebug
         ("\nbestValue=%d beta=%d, rd=%d extensions=%d pvNode=%d\nbest move: ",
          best, beta, restDepth, (ply + restDepth) - variation->nominalDepth,
          pvNode);
      dumpMove(bestMove);
      dumpVariation(variation);
   }
#endif /* CRV_CHECK */

   /* Calculate the parameters controlling the move ordering. */
   /* ------------------------------------------------------- */
   if (bestMove != NO_MOVE && moveIsQuiet(bestMove, position) &&
       (excludeMove == NO_MOVE || excludeMove == NULLMOVE))
   {
      Move killerMove = bestMove;
      const Piece movingPiece = position->piece[getFromSquare(killerMove)];
      const int index = historyIndex(bestMove, position);

      assert(numQuietMoves >= 1);

      variation->historyValue[index] = (UINT16)
         (variation->historyValue[index] +
          ((HISTORY_MAX - variation->historyValue[index]) * restDepth) / 256);

#ifdef EXTRA_BONUS_BETA_CUT
      if (best > beta + 20)
      {
         variation->historyValue[index] += historyBonus;
      }
#endif

      setMoveValue(&killerMove, movingPiece);
      registerKillerMove(&variation->plyInfo[ply], killerMove);

#ifdef FUTILITY_CUTOFF_HISTORY
      if (best >= beta)
      {
         int i;

         for (i = 0; i < numQuietMoves; i++)
         {
            const int playedIndex = historyIndexPlayed[i];

            variation->historyTotalCount[playedIndex]++;

            if (variation->historyTotalCount[playedIndex] >= HISTORY_MAX)
            {
               shrinkHistoryHitValues(variation);
            }
         }

         variation->historyHitCount[index]++;
      }
#endif
   }

   /* Store the value in the transposition table. */
   /* ------------------------------------------- */
   if (excludeMove == NO_MOVE || excludeMove == NULLMOVE)
   {
      if (best > oldAlpha)
      {
         hashentryFlag =
            (best >= beta ? HASHVALUE_LOWER_LIMIT : HASHVALUE_EXACT);
      }
      else
      {
         hashentryFlag = HASHVALUE_UPPER_LIMIT;
      }

      setDatedEntry(variation->hashtable, position->hashValue,
                    calcHashtableValue(best, ply),
                    (INT8) (rdBasic + HASH_DEPTH_OFFSET),
                    packedMove(bestMove), hashentryFlag);
   }

   return best;
}

void registerNewBestMove(Variation * variation, Move * move,
                         const int value, PrincipalVariation * pv)
{
   if (variation->searchStatus == SEARCH_STATUS_RUNNING)
   {
      setMoveValue(move, value);
      variation->pv.score = value;

      if (variation->handleSearchEvent != 0)
      {
         appendMoveToPv(pv, &variation->pv, *move);
      }

      variation->bestBaseMove = *move;

      if (variation->nominalDepth > 4 &&
          variation->numberOfCurrentBaseMove > 1)
      {
         variation->bestMoveChange = TRUE;
         variation->easyMove = FALSE;
      }

      if (variation->handleSearchEvent != 0)
      {
         variation->handleSearchEvent(SEARCHEVENT_NEW_PV, variation);
      }
   }
}

static int getExactValue(Variation * variation, PrincipalVariation * pv,
                         Move * move, const int alpha,
                         const int depth, const int estimation)
{
   const int INITIAL_WINDOW_SIZE = 20;
   int window = INITIAL_WINDOW_SIZE, value;
   bool valueFound = FALSE;
   int tempAlpha = max(alpha, estimation - window);
   int tempBeta = min(-VALUE_MATED, estimation + window);
   int count = 0;

   do
   {
      value = -searchBest(variation, -tempBeta, -tempAlpha,
                          1, depth - DEPTH_RESOLUTION, TRUE, FALSE, pv,
                          NO_MOVE);

      if (value < variation->previousBest - FAIL_LOW_MARGIN)
      {
         variation->failingHighOrLow = TRUE;
      }

      if (value > alpha)
      {
         if (value >= tempBeta)
         {
            registerNewBestMove(variation, move, value, pv);
         }
      }
      else
      {
         return VALUE_MATED;
      }

      if (value > tempAlpha && value < tempBeta)
      {
         valueFound = TRUE;
      }
      else
      {
         if (++count == 5)
         {
            tempAlpha = VALUE_MATED;
            tempBeta = -VALUE_MATED;
         }
         else
         {
            window += 10;
            tempAlpha = max(alpha, value - window);
            tempBeta = min(-VALUE_MATED, value + window);
         }
      }
   }
   while (valueFound == FALSE);

   return value;
}

#ifdef ASPIRATION_SEARCH_ROOT
static void exploreBaseMoves(Variation * variation, Movelist * basemoves,
                             const int aspirationWindow)
{
   int best = VALUE_MATED;
   int alpha = variation->previousBest, beta = variation->previousBest;
   int restDepth = DEPTH_RESOLUTION * variation->nominalDepth;
   int ply = 0;
   Position *position = &variation->singlePosition;
   const bool fullWindow = (bool) (variation->nominalDepth <= 1);
   PrincipalVariation pv;
   int failLowCount = 0, failHighCount = 0;
   bool exactValueFound = FALSE;

   initializeMoveValues(basemoves);
   variation->failingHighOrLow = FALSE;
   variation->selDepth = variation->nominalDepth;
   initPlyInfo(&variation->plyInfo[ply]);
   getStaticValue(variation, ply);

   do
   {
      best = VALUE_MATED;
      alpha =
         max(VALUE_MATED, alpha - aspirationWindow * (1 << failLowCount));

      for (variation->numberOfCurrentBaseMove = 1;
           variation->numberOfCurrentBaseMove <= basemoves->numberOfMoves;
           variation->numberOfCurrentBaseMove++)
      {
         int value = best, newDepth = restDepth;
         const int icm = variation->numberOfCurrentBaseMove - 1;

         variation->currentBaseMove = basemoves->moves[icm];
         variation->plyInfo[ply].indexCurrentMove =
            historyIndex(variation->currentBaseMove, position);
         makeMoveFast(variation, basemoves->moves[icm]);

         if (variation->handleSearchEvent != 0)
         {
            variation->handleSearchEvent(SEARCHEVENT_NEW_BASEMOVE, variation);
         }

         if (activeKingIsSafe(&variation->singlePosition) == FALSE)
         {
            variation->plyInfo[ply].currentMoveIsCheck = TRUE;
            newDepth += DEPTH_RESOLUTION;
         }
         else
         {
            variation->plyInfo[ply].currentMoveIsCheck = FALSE;
         }

         unmakeLastMove(variation);

         do
         {
            beta =
               min(-VALUE_MATED,
                   beta + aspirationWindow * (1 << failHighCount));

            makeMoveFast(variation, basemoves->moves[icm]);

            if (icm == 0 || value > alpha || fullWindow)
            {
               value = -searchBest(variation, -beta, -alpha, ply + 1,
                                   newDepth - DEPTH_RESOLUTION, TRUE, FALSE,
                                   &pv, NO_MOVE);

               if (value < alpha)
               {
                  variation->failingHighOrLow = TRUE;
               }
            }
            else
            {
               value = -searchBest(variation, -alpha - 1, -alpha, ply + 1,
                                   newDepth - DEPTH_RESOLUTION, FALSE, FALSE,
                                   &pv, NO_MOVE);

               if (value > alpha)
               {
                  variation->failingHighOrLow = TRUE;

                  value = -searchBest(variation, -beta, -alpha, ply + 1,
                                      newDepth - DEPTH_RESOLUTION, TRUE,
                                      FALSE, &pv, NO_MOVE);
               }
            }

            unmakeLastMove(variation);

            if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
                variation->nominalDepth > 1)
            {
               break;
            }

            if (value > best || fullWindow)
            {
               setMoveValue(&basemoves->moves[icm], value);
            }

            if (icm == 0 && value > best && value <= alpha)
            {
               registerNewBestMove(variation, &basemoves->moves[icm], value,
                                   &pv);
            }

            if (value > best)
            {
               best = value;

               if (best > alpha)
               {
                  exactValueFound = TRUE;
                  variation->failingHighOrLow = FALSE;

                  if (fullWindow == FALSE)
                  {
                     alpha = best;
                  }

                  registerNewBestMove(variation, &basemoves->moves[icm], best,
                                      &pv);

                  if (best >= beta)
                  {
                     failHighCount++;
                  }
               }
            }
         }
         while (best >= beta);
      }

      if (exactValueFound == FALSE)
      {
         failLowCount++;
         variation->failingHighOrLow = TRUE;
         sortMoves(basemoves);
         initializeMoveValues(basemoves);
      }
   }
   while (variation->searchStatus == SEARCH_STATUS_RUNNING &&
          exactValueFound == FALSE);

   sortMoves(basemoves);

   if (fullWindow &&
       (basemoves->numberOfMoves < 2 ||
        getMoveValue(basemoves->moves[0]) >
        getMoveValue(basemoves->moves[1]) + 150))
   {
      variation->easyMove = TRUE;
   }
}
#else
static void exploreBaseMoves(Variation * variation, Movelist * basemoves)
{
   int best = VALUE_MATED, alpha = best, beta = -alpha;
   int restDepth = DEPTH_RESOLUTION * variation->nominalDepth;
   int ply = 0;
   Position *position = &variation->singlePosition;
   const bool fullWindow = (bool) (variation->nominalDepth <= 1);
   PrincipalVariation pv;

   initializeMoveValues(basemoves);
   variation->failingHighOrLow = FALSE;
   variation->selDepth = variation->nominalDepth;
   initPlyInfo(&variation->plyInfo[ply]);
   getStaticValue(variation, ply);

   for (variation->numberOfCurrentBaseMove = 1;
        variation->numberOfCurrentBaseMove <= basemoves->numberOfMoves;
        variation->numberOfCurrentBaseMove++)
   {
      int value, newDepth = restDepth;
      const int icm = variation->numberOfCurrentBaseMove - 1;

      variation->currentBaseMove = basemoves->moves[icm];
      variation->plyInfo[ply].indexCurrentMove =
         historyIndex(variation->currentBaseMove, position);
      makeMoveFast(variation, basemoves->moves[icm]);

      if (variation->handleSearchEvent != 0)
      {
         variation->handleSearchEvent(SEARCHEVENT_NEW_BASEMOVE, variation);
      }

      if (activeKingIsSafe(&variation->singlePosition) == FALSE)
      {
         variation->plyInfo[ply].currentMoveIsCheck = TRUE;
         newDepth += DEPTH_RESOLUTION;
      }
      else
      {
         variation->plyInfo[ply].currentMoveIsCheck = FALSE;
      }

      if (best == VALUE_MATED || fullWindow)
      {
         if (fullWindow == FALSE)
         {
            value = getExactValue(variation, &pv, &basemoves->moves[icm],
                                  alpha, newDepth, variation->previousBest);
         }
         else
         {
            value = -searchBest(variation, -beta, -alpha, ply + 1,
                                newDepth - DEPTH_RESOLUTION, TRUE, FALSE,
                                &pv, NO_MOVE);
         }

         if (value < variation->previousBest - FAIL_LOW_MARGIN)
         {
            variation->failingHighOrLow = TRUE;
         }
      }
      else
      {
         value = -searchBest(variation, -alpha - 1, -alpha, ply + 1,
                             newDepth - DEPTH_RESOLUTION, FALSE, FALSE, &pv,
                             NO_MOVE);

         if (value > alpha)
         {
            variation->failingHighOrLow = TRUE;

            value = -searchBest(variation, -alpha - 1, -alpha, ply + 1,
                                newDepth - DEPTH_RESOLUTION, TRUE, FALSE,
                                &pv, NO_MOVE);

            if (value > alpha)
            {
               value = getExactValue(variation, &pv, &basemoves->moves[icm],
                                     alpha, newDepth, value);
            }
         }
      }

      unmakeLastMove(variation);

      if (variation->searchStatus != SEARCH_STATUS_RUNNING &&
          variation->nominalDepth > 1)
      {
         break;
      }

      if (value > best || fullWindow)
      {
         setMoveValue(&basemoves->moves[icm], value);
      }

      if (value > best)
      {
         best = value;

         if (best > alpha)
         {
            if (fullWindow == FALSE)
            {
               alpha = best;
            }

            registerNewBestMove(variation, &basemoves->moves[icm], best, &pv);

            if (best >= beta && fullWindow == FALSE)
            {
               goto finish;
            }
         }
      }

      variation->failingHighOrLow =
         (bool) (best < variation->previousBest - FAIL_LOW_MARGIN);
   }

 finish:

   sortMoves(basemoves);

   if (fullWindow &&
       (basemoves->numberOfMoves < 2 ||
        getMoveValue(basemoves->moves[0]) >
        getMoveValue(basemoves->moves[1]) + 150))
   {
      variation->easyMove = TRUE;
   }
}
#endif

static void initializePawnHashtable(PawnHashInfo * pawnHashtable)
{
   int i;

   for (i = 0; i < PAWN_HASHTABLE_SIZE; i++)
   {
      pawnHashtable[i].hashValue = 0;
   }
}

static void initializeKingsafetyHashtable(KingSafetyHashInfo *
                                          kingsafetyHashtable)
{
   int i;

   for (i = 0; i < KINGSAFETY_HASHTABLE_SIZE; i++)
   {
      kingsafetyHashtable[i].hashValue = 0;
   }
}

Move search(Variation * variation, Movelist * acceptableSolutions)
{
   Movelist movelist;
   int iteration;
   long timeTarget;
   int stableIterationCount = 0;
   int stableBestMoveCount = 0;
   Move bestMove = NO_MOVE;
   UINT64 nodeCount = 0;
   bool failLow = FALSE;
   int loScore = -VALUE_MATED, hiScore = VALUE_MATED;
   int iv1 = 0, iv2 = 0, iv3 = 0;

   if (resetGlobalHashtable)
   {
      resetHashtable(variation->hashtable);
      initializePawnHashtable(variation->pawnHashtable);
      initializeKingsafetyHashtable(variation->kingsafetyHashtable);
      resetGlobalHashtable = FALSE;
   }

   resetHistoryValues(variation);
   resetHistoryHitValues(variation);
   resetGainValues(variation);

   variation->startTime = getTimestamp();
   variation->startTimeProcess = getProcessTimestamp();
   variation->timestamp = variation->startTime + 1;
   variation->tbHits = 0;
   variation->numPvUpdates = 0;
   variation->easyMove = FALSE;
   variation->terminatePondering = FALSE;
   variation->previousBest = VALUE_MATED;
   variation->bestBaseMove = NO_MOVE;
   getLegalMoves(variation, &movelist);
   initializePlyInfo(variation);

#ifdef TRACE_EVAL
   getValue(&variation->singlePosition, VALUE_MATED, -VALUE_MATED,
            variation->pawnHashtable, variation->kingsafetyHashtable);
#endif

#ifdef USE_BOOK
   if (globalBook.indexFile != NULL && globalBook.moveFile != NULL &&
       &variation->singlePosition->moveNumber <= 12)
   {
      Move bookMove = getBookmove(&globalBook,
                                  &variation->singlePosition->hashValue,
                                  &movelist);

      if (bookMove != NO_MOVE)
      {
         variation->bestBaseMove = bookMove;
         variation->searchStatus = SEARCH_STATUS_TERMINATE;
         variation->finishTime = getTimestamp();

         if (variation->handleSearchEvent != 0)
         {
            variation->handleSearchEvent(SEARCHEVENT_SEARCH_FINISHED,
                                         variation);
         }

         variation->nodes = 0;

         return variation->bestBaseMove;
      }
   }
#endif

   variation->numberOfBaseMoves = movelist.numberOfMoves;
   setMoveValue(&variation->bestBaseMove, VALUE_MATED);

   for (iteration = 1; iteration <= MAX_DEPTH; iteration++)
   {
      long calculationTime;
      int iterationValue, referenceValue, aspirationWindow;
      int scoreWidth;

      variation->bestMoveChange = FALSE;
      variation->nominalDepth = iteration;

      aspirationWindow = max(20, (abs(iv1 - iv2) + abs(iv2 - iv3)) / 2);
#ifdef ASPIRATION_SEARCH_ROOT
      exploreBaseMoves(variation, &movelist, aspirationWindow);
#else
      exploreBaseMoves(variation, &movelist);
#endif
      calculationTime =
         (unsigned long) (getTimestamp() - variation->startTime);

      if (movesAreEqual(variation->bestBaseMove, bestMove))
      {
         stableBestMoveCount++;
      }
      else
      {
         stableBestMoveCount = 0;
      }

      bestMove = variation->bestBaseMove;
      iv3 = iv2;
      iv2 = iv1;
      iv1 = iterationValue = getMoveValue(variation->bestBaseMove);

      loScore = (calculationTime >= variation->timeTarget / 32 ?
                 min(loScore, iterationValue) : loScore);
      hiScore = (calculationTime >= variation->timeTarget / 32 ?
                 max(hiScore, iterationValue) : hiScore);
      scoreWidth = max(hiScore, variation->expectedScore) -
         min(loScore, variation->expectedScore);
      failLow = (bool) (iterationValue <
                        variation->previousBest - FAIL_LOW_MARGIN);
      referenceValue = max(variation->previousBest, variation->expectedScore);
      variation->previousBest = iterationValue;

      assert(calculationTime >= 0);

      if (acceptableSolutions != 0 &&
          listContainsMove(acceptableSolutions, variation->bestBaseMove))
      {
         stableIterationCount++;

         if (stableIterationCount == 1)
         {
            nodeCount = variation->nodes;
         }
      }
      else
      {
         stableIterationCount = 0;
         nodeCount = variation->nodes;
      }

      /* Check for a fail low. */
      /* --------------------- */

      if (variation->numberOfBaseMoves == 1)
      {
         timeTarget = variation->timeTarget / 16;
      }
      else if (failLow)
      {
         timeTarget = 4 * variation->timeTarget;

         if (calculationTime >= variation->timeTarget / 32)
         {
            variation->easyMove = FALSE;
         }
      }
      else if (variation->bestMoveChange ||
               (iterationValue < referenceValue &&
                scoreWidth >= FAIL_LOW_MARGIN))
      {
         timeTarget = variation->timeTarget;
      }
      else if (variation->easyMove && iterationValue >= referenceValue)
      {
         timeTarget = variation->timeTarget / 4;
      }
      else
      {
         timeTarget = variation->timeTarget / 2;
      }

      if (variation->searchStatus == SEARCH_STATUS_RUNNING &&
          variation->timeLimit != 0 && calculationTime >= timeTarget)
      {
#ifdef DEBUG_THREAD_COORDINATION
         logDebug
            ("Time target reached (%lu/%lu ms, %lu%%)).\n",
             calculationTime, variation->timeTarget,
             (calculationTime * 100) / variation->timeTarget);
#endif

         if (variation->ponderMode)
         {
            variation->terminatePondering = TRUE;

#ifdef DEBUG_THREAD_COORDINATION
            logDebug("Setting ponder termination flag.\n");
#endif
         }
         else
         {
            variation->searchStatus = SEARCH_STATUS_TERMINATE;

#ifdef DEBUG_THREAD_COORDINATION
            logDebug("Terminating search.\n");
#endif
         }
      }

      if (variation->searchStatus == SEARCH_STATUS_RUNNING &&
          (getMoveValue(variation->bestBaseMove) <= VALUE_MATED + iteration ||
           getMoveValue(variation->bestBaseMove) >= -VALUE_MATED - iteration))
      {
#ifdef DEBUG_THREAD_COORDINATION
         logDebug("Best value out of bounds (%d). Terminating search.\n",
                  getMoveValue(variation->bestBaseMove));
#endif
         variation->searchStatus = SEARCH_STATUS_TERMINATE;
      }

      if (variation->searchStatus == SEARCH_STATUS_RUNNING &&
          iteration == MAX_DEPTH)
      {
#ifdef DEBUG_THREAD_COORDINATION
         logDebug("Max depth reached. Terminating search.\n");
#endif
         variation->searchStatus = SEARCH_STATUS_TERMINATE;
      }

      if (variation->searchStatus != SEARCH_STATUS_RUNNING)
      {
#ifdef DEBUG_THREAD_COORDINATION
         logReport
            ("search status != SEARCH_STATUS_RUNNING -> exiting search.\n",
             getMoveValue(variation->bestBaseMove));
#endif

         break;
      }

      if (acceptableSolutions != 0 && stableIterationCount >= 1 &&
          (getMoveValue(variation->bestBaseMove) > 20000 ||
           (stableIterationCount >= 2 &&
            (getMoveValue(variation->bestBaseMove) >= 25 ||
             (getTimestamp() - variation->startTime) >= 3000))))
      {
#ifdef DEBUG_THREAD_COORDINATION
         logDebug("Solution found (value=%d). Terminating search.\n",
                  getMoveValue(variation->bestBaseMove));
#endif
         variation->searchStatus = SEARCH_STATUS_TERMINATE;

         break;
      }
   }

   variation->terminatePondering = TRUE;
   variation->searchStatus = SEARCH_STATUS_TERMINATE;
   incrementDate(variation->hashtable);
   variation->finishTime = getTimestamp();
   variation->finishTimeProcess = getProcessTimestamp();

   if (variation->handleSearchEvent != 0)
   {
      variation->handleSearchEvent(SEARCHEVENT_SEARCH_FINISHED, variation);
   }

   variation->nodes = nodeCount;

   if (statCount1 != 0 || statCount2 != 0)
   {
      logDebug("statCount1=%lld statCount2=%lld (%lld%%) \n",
               statCount1, statCount2,
               (statCount2 * 100) / max(1, statCount1));
   }

   return variation->bestBaseMove;
}

void initializeArrays()
{
   int i, j;

   quietPvMoveReduction[0][0] = 0;
   quietMoveReduction[0][0] = 0;

   for (i = 1; i < 64; i++)
   {
      for (j = 1; j < 64; j++)
      {
         double pvReduction =
            0.5 + log((double) (i)) * log((double) (j)) / 6.0;
         double nonPvReduction =
            0.5 + log((double) (i)) * log((double) (j)) / 3.0;
         quietPvMoveReduction[i][j] =
            (int) (pvReduction >= 1.0 ?
                   floor(pvReduction * DEPTH_RESOLUTION) : 0);
         quietMoveReduction[i][j] =
            (int) (nonPvReduction >= 1.0 ?
                   floor(nonPvReduction * DEPTH_RESOLUTION) : 0);
      }
   }

   for (i = 0; i < 32; i++)
   {
      quietMoveCountLimit[i] = 3 + (1 << (3 * i / 8));
   }

   for (i = 0; i < 16; i++)
   {
      for (j = 0; j < 64; j++)
      {
         if (i >= 2)
         {
            const double factor = log((i * i) / 2.0) / log(2.0) + 1.001;

            futilityMargin[i][j] = 44 * (int) (factor) - 3 * j + 18;
         }
         else
         {
            futilityMargin[i][j] = 0;
         }
      }
   }
}

int initializeModuleSearch()
{
   const Move m1 = getPackedMove(F3, H5, NO_PIECE);
   const Move m2 = getPackedMove(G7, G6, NO_PIECE);
   Move ccrv[3];
   int i = 0;

   ccrv[0] = m1;
   ccrv[1] = m2;
   ccrv[2] = NO_MOVE;

   do
   {
      crv[i] = ccrv[i];
   }
   while (ccrv[i++] != NO_MOVE);

   initializeArrays();

   return 0;
}

int testModuleSearch()
{
   return 0;
}
