/**
 * MoveHelper.java
 *
 * Version 3.0   
 * 
 * Copyright (c) 2010 Eric Stock
 
Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:
 
The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * MoveHelper.java
 *
 * follows singleton design pattern
 * this class initializes and stores in memory arrays of attacks for non sliding pieces
 * the pieces are king
 * There are also functions to assist in rook, bishop and queen moves  
 * 
 * To-do - The king move and castle move board should be placed into the Board.java class
 *       - All functions which rotate the bitboards, and collect the moves for sliding pieces to be placed in Magic move arrays
 *       should be placed in this class
 *
 *       So basically, the Board class needs to have only the data structures containing the pre-computed moves and functions
 *          to retrieve the moves
 *       This class should only have data and functions to fill the Board class's data structures
 *
 * @version 	3.00 25 Oct 2010
 * @author 	Eric Stock
 */


public final class MoveHelper{	
	
    /** king moves for each square */
	private static final long[] KingMoveBoard = new long[64];
   private static final long[] kingCastleBoard = new long[256];
	
   /** only instance of class due to singleton pattern */
   private static final MoveHelper INSTANCE = new MoveHelper();	
    
	/*
     * MoveHelper Constructor
     *
     * calls methods to fill arrays with needed values for move generation
     */
    private MoveHelper() {
		initKingBoard(KingMoveBoard);
        for(int i=0;i<256;i++) {
			setKingCastle(i);
        }
	};
	/*
     * Method getInstance()
     *
     * returns initialized instance of class
     * 
     * @return MoveHelper - the object
     * 
     */
	public static MoveHelper getInstance() {
        return INSTANCE;
    }    
	
	
	
	/***********************************************************************
		Name:		getKingPosition
		Parameters:	int
		Returns:	BitSet
		Description:This method returns a BitSet from KingMoveBoard
					specified via the parameter
	***********************************************************************/
	public final long getKingPosition(int square){
		return KingMoveBoard[square];
	}
	
	
	
	/***********************************************************************
		Name:		initKingBoard
		Parameters:	BitSet[]
		Returns:	None
		Description:This method initializes the BitSet representing the 
					king move board
	***********************************************************************/
	private final void initKingBoard(long[] board){
		int square;
		
		for(square=0;square<64;square++){
			int rank = square/8;
			int file = square%8;
			board[square]=0;
			if(rank>0) {
				board[square] |= (long)1<<(square-8);
				if(file>0)
					board[square] |= (long)1<<(square-9);
				if(file<7)
					board[square] |= (long)1<<(square-7);
			}
			if(rank<7) {
				board[square] |= (long)1<<(square+8);
				if(file>0) 
					board[square] |= (long)1<<(square+7);
				if(file<7)
					board[square] |= (long)1<<(square+9);
			}
			if(file>0)
				board[square] |= (long)1<<(square-1);
			if(file<7)
				board[square] |= (long)1<<(square+1);
		}
	}
	
	/*
     * Method getKingCastle
     *
     * returns the potential castle moves 
     * 
     * @param int rank - the occupancy of the rank the king is on
     * 
     */
	public final long getKingCastle(int rank) {
		return kingCastleBoard[rank];
	}

    /*
     * Method setKingCastle
     *
     * initializes array of all potential castle moves
     * 
     * @param int rank - the occupancy of the rank the king is on
     * ex// if the file is full (255) there are no castle moves
     * 
     */
	private final void setKingCastle(int file) {
		if(file>=144 && file < 160) {
			kingCastleBoard[file] = (long)1<<6;
		}
		long temp = file&31;
		if(temp==17)
			kingCastleBoard[file] |= (long)1<<2;
	}
	
	/*
     * Method getDiag1Attack
     *
     * calculates all sliding moves along diagonal pointing NW
     * 
     * @param int index1 - the index the diagonal sliding piece is on
     * @param int index2 - the occupancy of the diagonal
     * 
     * @return long - a bitset representing the moves along this diagonal
     * 
     */
	public final long getDiag1Attack(int index1, int index2) {
		
		int DistToLeft = 0;						//used to find spot in index2
		int temp = index1;	
		index2 &= ((1<<Global.Diag1Length[index1])-1);
		long diag = 0;
		while(temp%8!=0 && temp<56) {			//while not at most left square
			temp+=7;
			DistToLeft++;
		}
		
		if(index1%8!=0) {
			int ind2 = DistToLeft;
			for(int i=index1+7;i<64&&i%8!=7;i+=7) {			//move up and left along diagonal
				ind2--;
				diag |= (long)1<<i;
				if((index2&(long)1<<ind2)!=0) break; 
			}
		}
		
		if(index1%8!=7) {
			int ind2=DistToLeft;
			for(int i=index1-7;i>=0&&i%8!=0;i-=7) {
				ind2++;
				diag |= (long)1<<i;
				if((index2&(long)1<<(ind2))!=0) break;
		
			}
		}
		return diag;
	}
	
    /*
     * Method getDiag1Attack
     *
     * calculates all sliding moves along diagonal pointing NE
     * 
     * @param int index1 - the index the diagonal sliding piece is on
     * @param int index2 - the occupancy of the diagonal
     * 
     * @return long - a bitset representing the moves along this diagonal
     * 
     */
	public final long getDiag2Attack(int index1, int index2) {
		
		int DistToLeft = 0;						//used to find spot in index2
		int temp = index1;	
		index2 &= ((1<<Global.Diag2Length[index1])-1);
		long diag = 0;
		
		while(temp%8!=0 && temp>7) {			//while not at most left square
			temp-=9;
			DistToLeft++;
		}
		
		if(index1%8!=0) {
			int ind2 = DistToLeft;
			for(int i=index1-9;i>=0&&i%8!=7;i-=9) {			//move down and left along diagonal
				ind2--;
				diag |= (long)1<<i;
				if((index2&(long)1<<ind2)!=0) break; 
			}
		}
		
		if(index1%8!=7) {
			int ind2=DistToLeft;
			for(int i=index1+9;i<64&&i%8!=0;i+=9) {
				ind2++;
				diag |= (long)1<<i;
				if((index2&(long)1<<(ind2))!=0) break;
		
			}
		}
		return diag;
	}

	/*
     * Method getRooksRank2
     *
     * calculates sliding moves along a rank
     * 
     * @param int index1 - the index the rank sliding piece is on
     * @param int index2 - the occupancy of the diagonal
     * 
     * @return long - a bitset representing the moves along the rank
     * 
     */
	public final long getRooksRank2(int index1,int index2) {
	
		long rank = 0; 
		for(int i=index1+1;i<index1/8*8+8;i++)   {		//move up file and look for pieces in the way
			rank |= (long)1<<(i);
			if((index2&(long)1<<i%8)!=0) break;
		}
		for(int i=index1-1;i>=index1/8*8;i--) {				//move down file and look for pieces in the way
			rank |= (long)1<<(i);
			if((index2&(long)1<<i%8)!=0) break;
		}
		return rank;
    }
		
	/*
     * Method getRooksFile2
     *
     * calculates sliding moves along a file
     * 
     * @param int index1 - the index the rank sliding piece is on
     * @param int index2 - the occupancy of the diagonal
     * 
     * @return long - a bitset representing the moves along the file
     * 
     */
	public final long getRooksFile2(int index1,int index2) {
	
		long file = 0;
		 
		for(int i=index1+8;i<64;i+=8)   {		//move up file and look for pieces in the way
			file |= (long)1<<(i);
			if((index2&(long)1<<i/8)!=0) break;
		}
		for(int i=index1-8;i>=0;i-=8) {				//move down file and look for pieces in the way
			file |= (long)1<<(i);
			if((index2&(long)1<<i/8)!=0) break;
		}
		return file;
    }
	
}	
		