#ifdef _MSC_VER
#pragma warning(disable: 4996)
#endif

#include "book.h"
#include "commands.h"
#include "eval.h"
#include "moves.h"
#include "notation.h"
#include "search.h"
#include "utils.h"

#define  max(x, y) (((x) < (y))? (y) : (x))

const EVAL ROOT_WINDOW = VAL_P;

extern Position   g_pos;
extern PROTOCOL_T g_protocol;

inline bool IsUCI() { return g_protocol == UCI; }

static Position pos;
static MoveList moves[MAX_PLY];
static int g_iter = 0;

vector<Move> g_PV[MAX_PLY + 2];
vector<Move> g_rootPV;

bool g_limitStrength = false;
int g_npsLimit = 0; // 0 = unlimited

int g_multipv_size = 1;
MultiPVEntry g_multipv_storage[MAX_BRANCH];

HashEntry* g_hash = NULL;
long g_hash_size = 0;
U8 g_hash_age = 0;

int g_history[14][64];
void UpdateHistory(Move mv, int depth);
Move g_killers[MAX_PLY + 2];
Move g_matekillers[MAX_PLY + 2];

NODES   g_nodes = 0;
clock_t g_start_time = 0;
inline int GetSearchTime() { return 1000 * (clock() - g_start_time) / CLOCKS_PER_SEC; } // milliseconds

Limits g_limits = {0, 0, 0, 0, 2000, 2000};

#define TERMINATE_SEARCH (1)
#define FINALIZE_SEARCH  (2)
int g_flag = 0;

MODE_T g_mode = IDLE;
extern list<string> g_commandQueue;

EVAL AlphaBetaRoot(Position& pos0, EVAL alpha, EVAL beta, const int depth);
EVAL AlphaBeta(EVAL alpha, EVAL beta, const int depth, int ply, bool isNull);
EVAL AlphaBetaQ(EVAL alpha, EVAL beta, int ply, int qply);

void CheckInput();
void CheckLimits();

void AdjustKNPS()
{
  if (g_limitStrength)
  {
    int timeToSleep = 1000 * int(g_nodes) / g_npsLimit - GetSearchTime();
    if (g_limits.m_restMillisec)
    {
      if (GetSearchTime() + timeToSleep > g_limits.m_stHard)
        timeToSleep = g_limits.m_stHard - GetSearchTime();
    }
    if (timeToSleep > 0)
      SleepMilliseconds(timeToSleep);
  }
}
////////////////////////////////////////////////////////////////////////////////

EVAL AlphaBetaRoot(Position& pos0, EVAL alpha, EVAL beta, const int depth)
{
  pos = pos0;

  int ply = 0;
  bool inCheck = pos.InCheck();

  g_hash_age++;
  g_PV[ply].clear();
  g_nodes++;

  int J = 0;
  for (J = 0; J < MAX_BRANCH; J++)
  {
    g_multipv_storage[J].m_score = - INFINITY_SCORE;
    g_multipv_storage[J].m_pv.clear();
  }

  Move hash_mv = 0;
  U8 hash_type = HASH_ALPHA;
  HashEntry* pentry = ProbeHash(&pos);
  if (pentry)
  {
    hash_mv = pentry->mv;
  }

  MoveList& mvlist = moves[ply];
  if (inCheck)
    mvlist.GenCheckEvasions(pos);
  else
    mvlist.GenAllMoves(pos);
  mvlist.UpdateScores(pos, hash_mv, g_killers[ply], g_matekillers[ply]);

  EVAL e = 0;
  int legalMoves = 0;
  Move best_mv = 0;

  for (int i = 0; i < mvlist.Size(); ++i)
  {
    AdjustKNPS();

    Move mv = mvlist.GetNthBest(i);
    if (pos.MakeMove(mv))
    {
      legalMoves++;
      if (IsUCI() && GetSearchTime() > 1000)
      {
        char mvbuf[16];
        out("info currmove %s", MoveToStrLong(mv, mvbuf));
        out(" currmovenumber %d\n", legalMoves);
      }

      int newDepth = depth - 1;
      bool givesCheck = pos.InCheck();
      
      //
      //   EXTENSIONS
      //

      if (givesCheck)
        ++newDepth;
      else if (mv.Piece() == PW && (mv.To() / 8) == 1)
        ++newDepth;
      else if (mv.Piece() == PB && (mv.To() / 8) == 6)
        ++newDepth;

      if (g_multipv_size > 1)
        e = -AlphaBeta(-beta - VAL_Q, -alpha + VAL_Q, newDepth, ply + 1, false);
      else
      {
        if (legalMoves == 1)
          e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
        else
        {
          e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
          if (e > alpha && e < beta)
            e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
        }
      }
      pos.UnmakeMove();

      if (g_flag)
        return alpha;

      if (legalMoves == 1)
      {
        best_mv = mv;
        g_PV[ply].clear();
        g_PV[ply].push_back(mv);
        g_PV[ply].insert(g_PV[ply].end(), g_PV[ply + 1].begin(), g_PV[ply + 1].end());
      }

      //
      //   Update multipv
      //

      if (legalMoves < MAX_BRANCH)
      {
        MultiPVEntry *mpv = &(g_multipv_storage[legalMoves - 1]);
        
        mpv->m_pv.clear();
        mpv->m_score = e;
        mpv->m_pv.push_back(mv);
        mpv->m_pv.insert(mpv->m_pv.end(), g_PV[ply + 1].begin(), g_PV[ply + 1].end());
      }

      if (e > alpha)
      {
        best_mv = mv;
        UpdateHistory(mv, depth);
        hash_type = HASH_EXACT;

        g_PV[ply].clear();
        g_PV[ply].push_back(mv);
        g_PV[ply].insert(g_PV[ply].end(), g_PV[ply + 1].begin(), g_PV[ply + 1].end());

        alpha = e;

        if (alpha >= beta)
        {
          hash_type = HASH_BETA;
          if (!mv.Captured() && !mv.Promotion())
          {
            if (e > CHECKMATE_SCORE - MAX_PLY && e <= CHECKMATE_SCORE)
              g_matekillers[ply] = mv;
            else
              g_killers[ply] = mv;
          }

          break;
        }
      }
    }
  }

  if (legalMoves == 0)
  {
    if (inCheck)
      alpha = - CHECKMATE_SCORE + ply;
    else
      alpha = DRAW_SCORE;
  }

  RecordHash(&pos, best_mv, depth, alpha, hash_type, ply);

  return alpha;
}
////////////////////////////////////////////////////////////////////////////////

EVAL AlphaBeta(EVAL alpha, EVAL beta, const int depth, int ply, bool isNull)
{
  g_PV[ply].clear();
  ++g_nodes;

  COLOR side = pos.Side();
  bool onPV = (beta - alpha > 1);
  bool inCheck = pos.InCheck();
  bool lateEndgame = pos.MatIndex(side) < 5;

  AdjustKNPS();
  CheckLimits();
  if (g_nodes % 8192 == 0)
    CheckInput();

  //
  //   DRAW DETECTION
  //

  if (ply >= MAX_PLY)
    return DRAW_SCORE;

  if (pos.IsDraw())
    return DRAW_SCORE;

  if (!isNull)
  {
    int rep_total = pos.GetRepetitions();
    if (rep_total >= 2)
      return DRAW_SCORE;
  }

  //
  //   PROBING HASH
  //

  Move hash_mv = 0;
  U8 hash_type = HASH_ALPHA;
  HashEntry* pentry = ProbeHash(&pos);

  if (pentry)
  {
    hash_mv = pentry->mv;
    if (pentry->depth >= depth)
    {
      EVAL hash_eval = pentry->eval;
      if (hash_eval > CHECKMATE_SCORE - 50 && hash_eval <= CHECKMATE_SCORE)
        hash_eval -= ply;
      else if (hash_eval < - CHECKMATE_SCORE + 50 && hash_eval >= -CHECKMATE_SCORE)
        hash_eval += ply;

      if (pentry->type == HASH_EXACT)
        return hash_eval;
      
      if (pentry->type == HASH_ALPHA && hash_eval <= alpha)
        return alpha;
      else if (pentry->type == HASH_BETA && hash_eval >= beta)
        return beta;
    }
  }

  //
  //   QSEARCH
  //

  if (depth <= 0 && !inCheck)
  {
    --g_nodes;
    return AlphaBetaQ(alpha, beta, ply, 0);
  }

  //
  //   PRUNING
  //

  EVAL MARGIN[4] = { 20000, 100, 400, 600 };
  if (!inCheck && !onPV && (depth > 0) && (depth < 4))
  {
    EVAL staticScore = Evaluate(pos, -INFINITY_SCORE, INFINITY_SCORE);
    if (staticScore < beta - MARGIN[depth])
      return AlphaBetaQ(alpha, beta, ply, 0);
    if (staticScore > beta + MARGIN[depth])
      return beta;
  }

  //
  //   NULLMOVE
  //

  const int R = 3;
  do
  {
    if (onPV) break;
    if (depth < 2) break;
    if (isNull || inCheck) break;
    if (lateEndgame) break;

    pos.MakeNullMove();
    EVAL nullEval;
    if (depth - 1 - R > 0)
      nullEval = -AlphaBeta(-beta, -beta + 1, depth - 1 - R, ply + 1, true);
    else
      nullEval = -AlphaBetaQ(-beta, -beta + 1, ply + 1, 0);
    pos.UnmakeNullMove();

    if (nullEval >= beta)
      return beta;
  }
  while (0);

  MoveList& mvlist = moves[ply];
  if (inCheck)
    mvlist.GenCheckEvasions(pos);
  else
    mvlist.GenAllMoves(pos);
  mvlist.UpdateScores(pos, hash_mv, g_killers[ply], g_matekillers[ply]);

  int legalMoves = 0;
  EVAL e = 0;
  Move best_mv = 0;

  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist.GetNthBest(i);
    if (pos.MakeMove(mv))
    {
      legalMoves++;
      int newDepth = depth - 1;
      bool givesCheck = pos.InCheck();

      //
      //   EXTENSIONS
      //

      if (givesCheck)
        ++newDepth;
      else if (mv.Piece() == PW && (mv.To() / 8) == 1)
        ++newDepth;
      else if (mv.Piece() == PB && (mv.To() / 8) == 6)
        ++newDepth;

      //
      //   LMR
      //

      bool reduced = false;
      if (!onPV && !inCheck && !mv.Captured() && !mv.Promotion())
      {
        if (depth > 4 && legalMoves > 4)
        {
          --newDepth;
          reduced = true;
        }
      }
      
      if (legalMoves == 1)
        e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
      else
      {
        e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
        if (reduced && e > alpha)
        {
          ++newDepth;
          e = -AlphaBeta(-alpha - 1, -alpha, newDepth, ply + 1, false);
        }
        if (e > alpha && e < beta)
          e = -AlphaBeta(-beta, -alpha, newDepth, ply + 1, false);
      }
      pos.UnmakeMove();

      if (g_flag)
        return alpha;

      if (e > alpha)
      {
        best_mv = mv;
        UpdateHistory(mv, depth);
        hash_type = HASH_EXACT;

        alpha = e;
        if (alpha >= beta)
        {
          hash_type = HASH_BETA;
          if (!mv.Captured() && !mv.Promotion())
          {
            if (e > CHECKMATE_SCORE - MAX_PLY && e <= CHECKMATE_SCORE)
              g_matekillers[ply] = mv;
            else
              g_killers[ply] = mv;
          }

          break;
        }

        g_PV[ply].clear();
        g_PV[ply].push_back(mv);
        g_PV[ply].insert(g_PV[ply].end(), g_PV[ply + 1].begin(), g_PV[ply + 1].end());
      }
    }
  }

  if (legalMoves == 0)
  {
    if (inCheck)
      alpha = - CHECKMATE_SCORE + ply;
    else
      alpha = DRAW_SCORE;
  }
  else if (pos.Fifty() >= 100)
    alpha = DRAW_SCORE;

  RecordHash(&pos, best_mv, depth, alpha, hash_type, ply);
  return alpha;
}
////////////////////////////////////////////////////////////////////////////////

EVAL AlphaBetaQ(EVAL alpha, EVAL beta, int ply, int qply)
{
  g_PV[ply].clear();
  ++g_nodes;

  AdjustKNPS();
  CheckLimits();
  if (g_nodes % 8192 == 0)
    CheckInput();

  if (ply >= MAX_PLY)
    return DRAW_SCORE;

  bool inCheck = pos.InCheck();
  if (!inCheck)
  {
    EVAL staticEval = Evaluate(pos, alpha, beta);
    if (staticEval > alpha)
      alpha = staticEval;
  }

  if (alpha >= beta)
    return beta;

  MoveList& mvlist = moves[ply];

  if (inCheck)
    mvlist.GenCheckEvasions(pos);
  else
  {
    mvlist.GenCapturesAndPromotions(pos);
    if (qply < 2)
      mvlist.AddSimpleChecks(pos);
  }
  mvlist.UpdateScores(pos, 0, g_killers[ply], g_matekillers[ply]);

  EVAL e = 0;
  int legalMoves = 0;
  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist.GetNthBest(i);
    
    if (!inCheck && SEE(pos, mv) < 0)
      continue;

    if (pos.MakeMove(mv))
    {
      legalMoves++;
      e = -AlphaBetaQ(-beta, -alpha, ply + 1, qply + 1);
      pos.UnmakeMove();

      if (g_flag)
        return alpha;

      if (e > alpha)
      {
        alpha = e;
        if (alpha >= beta)
          break;

        g_PV[ply].clear();
        g_PV[ply].push_back(mv);
        g_PV[ply].insert(g_PV[ply].end(), g_PV[ply + 1].begin(), g_PV[ply + 1].end());
      }
    }
  }

  if (inCheck && legalMoves == 0)
    alpha = - CHECKMATE_SCORE + ply;

  return alpha;
}
////////////////////////////////////////////////////////////////////////////////

void CheckInput()
{
  if (g_rootPV.empty())
    return;

  if (InputAvailable())
  {
    char s[4096];
    ReadInput(s, sizeof(s));

    if (g_mode == ANALYZE)
    {
      Move mv = StrToMove(s, g_pos);
      if (mv)
      {
        g_flag = TERMINATE_SEARCH;
        g_commandQueue.push_back("force");
        g_commandQueue.push_back(s);
        g_commandQueue.push_back("analyze");
      }
      else if (Is(s, "board", 1))
        g_pos.Print();
      else if (Is(s, "quit", 1))
        exit(0);
      else if (Is(s, "isready", 7))
        out("readyok\n");
      else if (Is(s, "exit", 2))
        g_flag = TERMINATE_SEARCH;
      else if (Is(s, "stop", 2))
        g_flag = TERMINATE_SEARCH;
      else if (Is(s, "new", 3))
        g_flag = TERMINATE_SEARCH;
      else if (Is(s, "setoption", 8))
      {
        char *token = strtok(s, " ");
        token = strtok(NULL, " ");
        token = strtok(NULL, " ");
        if (token && !strcmp(token, "MultiPV"))
        {
          token = strtok(NULL, " ");
          token = strtok(NULL, " ");
          g_multipv_size = atoi(token);
        }
      }
      else if (Is(s, "undo", 4))
      {
        g_flag = TERMINATE_SEARCH;
        g_commandQueue.push_back("undo");
        g_commandQueue.push_back("analyze");
      }
      else if (Is(s, "remove", 6))
      {
        g_flag = TERMINATE_SEARCH;
        g_commandQueue.push_back("undo");
        g_commandQueue.push_back("analyze");
      }
    }

    else if (g_mode == THINKING)
    {
      if (Is(s, "quit", 1))
        exit(0);
      else if (Is(s, "isready", 7))
        out("readyok\n");
      else if (Is(s, "?", 1))
        g_flag = FINALIZE_SEARCH;
      else if (Is(s, "stop", 4))
        g_flag = FINALIZE_SEARCH;
      else if (Is(s, "new", 3))
        g_flag = TERMINATE_SEARCH;
      else if (Is(s, "result", 6))
        g_flag = TERMINATE_SEARCH;
      else if (Is(s, "setoption", 8))
      {
        char *token = strtok(s, " ");
        token = strtok(NULL, " ");
        token = strtok(NULL, " ");
        if (token && !strcmp(token, "MultiPV"))
        {
          token = strtok(NULL, " ");
          token = strtok(NULL, " ");
          g_multipv_size = atoi(token);
        }
      }
    }

    else if (g_mode == EPDTEST)
    {
      if (Is(s, "board", 1))
        g_pos.Print();
      else if (Is(s, "quit", 1))
        exit(0);
      else if (Is(s, "exit", 2))
        g_flag = TERMINATE_SEARCH;
      else if (Is(s, "new", 3))
        g_flag = TERMINATE_SEARCH;
    }
  }
}
////////////////////////////////////////////////////////////////////////////////

void CheckLimits()
{
  if (g_rootPV.empty())
    return;

  if (g_limits.m_stHard)
  {
    if (GetSearchTime() >= g_limits.m_stHard)
    {
      if (g_mode == THINKING || g_mode == EPDTEST)
        g_flag = FINALIZE_SEARCH;
    }
  }

  if (g_limits.m_sn && g_nodes >= g_limits.m_sn)
  {
    if (g_mode == THINKING || g_mode == EPDTEST)
      g_flag = FINALIZE_SEARCH;
  }
}
////////////////////////////////////////////////////////////////////////////////

void ClearHistory()
{
  memset(g_killers, 0, MAX_PLY * sizeof(Move));
  memset(g_matekillers, 0, MAX_PLY * sizeof(Move));
  memset(g_history, 0, 64 * 14 * sizeof(int));
}
////////////////////////////////////////////////////////////////////////////////

void ClearHashAndHistory()
{
  for (long i = 0; i < g_hash_size; ++i)
    g_hash[i].hashLock = 0;

  ClearHistory();
}
////////////////////////////////////////////////////////////////////////////////

void Epdtest(FILE* psrc, double time_in_seconds, int reps)
{
  char fen[256];
  Position tmp = g_pos;

  int dt = int(1000 * time_in_seconds);
  SetLimits(0, 0, 0, dt, dt);

  int total = 0, solved = 0;
  double sqtime = 0.0, sec = 0.0;

  while (fgets(fen, 256, psrc))
  {
    int res = g_pos.SetFEN(fen);
    out(fen);
    out("\n");

    if (!res)
    {
      g_pos = tmp;
      return;
    }

    if (!StartEpd(g_pos, fen, reps, &total, &solved, &sec))
    {
      g_pos = tmp;
      return;
    }

    sqtime += sec * sec;
    out("\nScore: %d / ", solved);
    out("%d\n", total);
    out("<Sq. time> = %lf\n\n", sqtime / total);

    if (g_flag == TERMINATE_SEARCH)
      break;
  }

  g_pos = tmp;
}
////////////////////////////////////////////////////////////////////////////////

NODES Perft(Position& pos, int depth)
{
  if (depth == 0)
    return 1;

  MoveList& mvlist = moves[depth];
  mvlist.GenAllMoves(pos);
  NODES sum = 0, delta = 0;

  for (int i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist[i];
    if (pos.MakeMove(mv))
    {
      delta = Perft(pos, depth - 1);
      sum += delta;
      pos.UnmakeMove();
    }
  }

  return sum;
}
////////////////////////////////////////////////////////////////////////////////

void PrintPV(const Position* pos, int iter, EVAL e, const char* comment)
{
  if (g_limitStrength)
  {
    int timeToSleep = 1000 * int(g_nodes) / g_npsLimit - GetSearchTime();
    bool canSleep = timeToSleep > 0;
    if (g_limits.m_restMillisec)
    {
      if (GetSearchTime() + timeToSleep < g_limits.m_restMillisec / 2)
        canSleep = false;
    }
    if (canSleep)
      SleepMilliseconds(timeToSleep);
  }

  char buf[16];
  Move mv = 0;

  if (IsUCI())
  {
    int J = 0;
    for (J = 0; J < MAX_BRANCH; J++)
      g_multipv_storage[J].m_seen = false;

    for (J = 1; J <= g_multipv_size; J++)
    {
      EVAL best_score = - INFINITY_SCORE;
      MultiPVEntry *best_mpv = NULL;  
      MultiPVEntry *mpv = NULL;

      for (int K = 0; K < MAX_BRANCH; K++)
      {
        mpv = &(g_multipv_storage[K]);
        if (mpv->m_seen)
          continue;

        if (mpv->m_pv.empty())
          break;

        if (mpv->m_score > best_score)
        {
          best_score = mpv->m_score;
          best_mpv = mpv;
        }
      }

      if (best_mpv)
      {
        best_mpv->m_seen = true;

        mpv = best_mpv;
        out("info multipv %d ", J);
        out("depth %d ", iter);
        out("score");
        if (mpv->m_score > CHECKMATE_SCORE - 50)
        {
          out(" mate %d", 1 + (CHECKMATE_SCORE - mpv->m_score) / 2);
        }
        else if (mpv->m_score < - CHECKMATE_SCORE + 50)
        {
          out(" mate -%d", (mpv->m_score + CHECKMATE_SCORE) / 2);
        }
        else
          out(" cp %d", mpv->m_score);

        out(" time %ld", GetSearchTime());
        out(" nodes %ld", g_nodes);
        if (GetSearchTime() > 0)
        {
          long knps = long(g_nodes / GetSearchTime());
          printf (" nps %ld", (1000 * knps));
        }
        out(" pv ");
        for (size_t m = 0; m < mpv->m_pv.size(); ++m)
        {
          mv = mpv->m_pv[m];
          out("%s ", MoveToStrLong(mv, buf));
        }
        out("\n");
      }
    }
    if (GetSearchTime() > 0)
    {
      out("info time %ld", GetSearchTime());
      out(" nodes %ld", g_nodes);
      long knps = long(g_nodes / GetSearchTime());
      out (" nps %ld\n", (1000 * knps));
    }
    return;
  }
  else
  {
    out(" %2d", iter);
    out(" %9d ", e);
    out(" %7d ", GetSearchTime() / 10);
    out(" %12d ", g_nodes);
    out(" ");
  }

  Position tmp = *pos;

  int movenum = tmp.Ply() / 2 + 1;
  for (size_t m = 0; m < g_rootPV.size(); ++m)
  {
    mv = g_rootPV[m];
    if (tmp.Side() == WHITE)
    {
      out("%d. ", movenum++);
    }
    else if (m == 0)
    {
      out("%d. ... ", movenum++);
    }

    MoveList mvlist;
    mvlist.GenAllMoves(tmp);
    out("%s", MoveToStrShort(mv, tmp, buf));
    tmp.MakeMove(mv);

    if (tmp.InCheck())
    {
      if (int(e + m + 1) == CHECKMATE_SCORE || int(e - m - 1) == - CHECKMATE_SCORE)
      {
        out("#");

        if (e > 0)
          out(" {+");
        else
          out(" {-");

        out("Mate in %d}", m / 2 + 1);
      }
      else
        out("+");
    }

    if (m == 0)
      out(comment);

    out(" ");
  }

  out("\n");
}
////////////////////////////////////////////////////////////////////////////////

HashEntry* ProbeHash(const Position* pos)
{
  long index = long(pos->Hash() % g_hash_size);
  if (g_hash[index].hashLock == U32(pos->Hash() & LL(0xffffffff)))
    return g_hash + index;
  else
    return NULL;
}
////////////////////////////////////////////////////////////////////////////////

void RecordHash(const Position* pos, Move best_mv,
                 I8 depth, EVAL eval, U8 type,
                 int ply)
{
  long index = long(pos->Hash() % g_hash_size);
  HashEntry* pentry = g_hash + index;

  if (pentry->age == g_hash_age && pentry->depth > depth)
    return;

  if (eval > CHECKMATE_SCORE - 50 && eval <= CHECKMATE_SCORE)
    eval += ply;

  else if (eval < - CHECKMATE_SCORE + 50 && eval >= -CHECKMATE_SCORE)
    eval -= ply;

  pentry->age = g_hash_age;
  pentry->depth = depth;
  pentry->eval = eval;
  pentry->hashLock = U32(pos->Hash() & LL(0xffffffff));
  pentry->mv = best_mv;
  pentry->type = type;
}
////////////////////////////////////////////////////////////////////////////////

void SetElo(int rating)
{
  int R0 = 2100;
  g_npsLimit = static_cast<int> (200000 * pow(10, (rating - R0) / 400.));
}
////////////////////////////////////////////////////////////////////////////////

void SetHashMB(double mb)
{
  if (g_hash)
    delete[] g_hash;

  g_hash_size = long (1024 * 1024 * mb / sizeof(HashEntry));
  if (g_hash_size <= 0)
    g_hash_size = 1;

  g_hash = new HashEntry[g_hash_size];

  if (!IsUCI())
  {
    out("main hash: %8ld nodes = ", g_hash_size);
    out("%lf MB\n", g_hash_size * sizeof(HashEntry) / (1024. * 1024.));
  }
}
////////////////////////////////////////////////////////////////////////////////

void SetLimits(int restMillisec, int sd, NODES sn, int stHard, int stSoft)
{
  g_limits.m_restMillisec = restMillisec;
  g_limits.m_sd = sd;
  g_limits.m_sn = sn;
  g_limits.m_stHard = stHard;
  g_limits.m_stSoft = stSoft;
}
////////////////////////////////////////////////////////////////////////////////

void StartAnalyze(const Position& pos0)
{
  SetLimits(0, 0, 0, 0, 0);
  g_mode = ANALYZE;
  g_flag = 0;

  g_nodes = 0;
  g_start_time = clock();

  Position pos = pos0;
  ClearHashAndHistory();

  if (!IsUCI())
  {
    char buf[4096];
    out("\n%s\n\n", pos0.FEN(buf));
  }

  EVAL alpha = - INFINITY_SCORE;
  EVAL beta = INFINITY_SCORE;

  for (g_iter = 1; g_iter < MAX_PLY; g_iter++)
  {
    int print_iter = g_iter;

    EVAL e = AlphaBetaRoot(pos, alpha, beta, g_iter);

    if (g_flag)
      break;

    if (e > alpha)
    {
      g_rootPV = g_PV[0];
    }

    if (e > alpha && e < beta)
    {
      PrintPV(&pos, print_iter, e, "");

      alpha = e - ROOT_WINDOW / 2;
      beta = e + ROOT_WINDOW / 2;
    }
    else
    {
      PrintPV(&pos, print_iter, e, (e <= alpha)? "?" : "!");

      alpha = - INFINITY_SCORE;
      beta = INFINITY_SCORE;
      g_iter--;
    }
  }
}
////////////////////////////////////////////////////////////////////////////////

int StartEpd(const Position& pos0, const char* fen,
              int reps, int* ptotal, int* psolved, double* psec)
{
  g_mode = EPDTEST;
  g_flag = 0;

  g_nodes = 0;
  g_start_time = clock();

  Position pos = pos0;
  ClearHashAndHistory();

  int good_iters = 0;
  int sufficient_iters = reps;
  int prev_iter = 0;

  EVAL alpha = - INFINITY_SCORE;
  EVAL beta = INFINITY_SCORE;

  *psec = g_limits.m_stHard / 1000.;
  for (g_iter = 1; g_iter < MAX_PLY; g_iter++)
  {
    int print_iter = g_iter;

    int move_found = 0;
    EVAL score = AlphaBetaRoot(pos, alpha, beta, g_iter);

    if (g_flag)
      break;

    if (score > alpha)
    {
      g_rootPV = g_PV[0];
    }

    if (!g_rootPV.empty())
    {
      Move mv = g_rootPV[0];
      char mvstr[16];
      MoveToStrShort(mv, pos, mvstr);

      if (strstr(fen, mvstr) != NULL)
        move_found = 1;
      else
      {
        MoveToStrLong(mv, mvstr);
        if (strstr(fen, mvstr) != NULL)
          move_found = 1;
      }

      if (strstr(fen, "am") && !strstr(fen, "bm"))
        move_found = 1 - move_found;
    }

    char comment[3] = "";

    if (score > alpha && score < beta)
    {
      strcpy(comment, "");
      alpha = score - ROOT_WINDOW / 2;
      beta = score + ROOT_WINDOW / 2;
    }
    else
    {
      strcpy(comment, (score <= alpha)? "?" : "!");
      alpha = - INFINITY_SCORE;
      beta = INFINITY_SCORE;
      g_iter--;
    }

    if (!move_found)
    {
      good_iters = 0;
    }
    else if (g_iter > prev_iter)
    {
      good_iters++;
      prev_iter = g_iter;
    }

    if (move_found)
    {
      Highlight(1);
      out(" yes ");
      Highlight(0);

      if (good_iters == 1)
        *psec = GetSearchTime() / 1000.;
    }
    else
    {
      out("  no ");
      *psec = g_limits.m_stHard / 1000.;
    }

    int complete_iter = (score > alpha && score < beta);

    if (good_iters >= sufficient_iters && complete_iter)
      Highlight(1);

    PrintPV(&pos, print_iter, score, comment);
    Highlight(0);

    if (good_iters >= sufficient_iters && complete_iter)
      break;
  }

  (*ptotal)++;
  if (good_iters > 0)
    (*psolved)++;

  return 1;
}
////////////////////////////////////////////////////////////////////////////////

void StartThinking(Position& pos0)
{
  g_rootPV.clear();

  EVAL e = 0;
  int print_iter = 0;
  char comment[32];

  if (pos0.GameOver(comment))
  {
    out(comment);
    return;
  }

  g_mode = THINKING;
  g_flag = 0;
  g_nodes = 0;
  g_start_time = clock();

  Position pos = pos0;
  ClearHistory();

  if (!IsUCI())
  {
    char buf[4096];
    out("\n%s\n\n", pos0.FEN(buf));
  }

  EVAL alpha = - INFINITY_SCORE;
  EVAL beta = INFINITY_SCORE;
  Move best_move = 0;

  // Book

  char buf[256];
  Move book_move = g_book.GetMove(pos, buf);
  if (book_move)
  {
    best_move = book_move;
    if (!IsUCI())
    {
      out(" 0 0 0 0      (");
      out(buf);
      out(")\n");
    }
    g_flag = FINALIZE_SEARCH;
    goto MAKE_MOVE;
  }

  for (g_iter = 1; g_iter < MAX_PLY; ++g_iter)
  {
    e = AlphaBetaRoot(pos, alpha, beta, g_iter);
    if (g_flag)
      break;

    if (e > alpha)
    {
      g_rootPV = g_PV[0];
    }

    if (e > alpha && e < beta)
    {
      print_iter = g_iter;
      strcpy(comment, "");
      
      alpha = e - ROOT_WINDOW / 2;
      beta = e + ROOT_WINDOW / 2;

      if (g_limits.m_stSoft)
      {
        if (GetSearchTime() >= g_limits.m_stSoft)
          g_flag = FINALIZE_SEARCH;
      }
      if (g_limits.m_sd)
      {
        if (g_iter >= g_limits.m_sd)
          g_flag = FINALIZE_SEARCH;
      }
    }
    else
    {
      strcpy(comment, (e <= alpha)? "?" : "!");
      
      // Additional time
      if (g_limits.m_restMillisec)
      {
        if (g_limits.m_restMillisec > 10 * g_limits.m_stHard)
        {
          int delta = g_limits.m_stHard / 2;
          g_limits.m_stHard += delta;
          g_limits.m_stSoft += delta;
        }
      }

      // Opening window
      if (e <= alpha)
        alpha = - INFINITY_SCORE;
      else
        beta = INFINITY_SCORE;
      --g_iter;
    }

    PrintPV(&pos, print_iter, e, comment);
    best_move = g_rootPV[0];

    if (best_move && (g_iter + e >= CHECKMATE_SCORE))
      g_flag = FINALIZE_SEARCH;
  }

  if (!g_flag && g_iter == MAX_PLY && best_move)
    g_flag = FINALIZE_SEARCH;

MAKE_MOVE:

  if (g_flag == FINALIZE_SEARCH)
  {
    char buf[16];
    if (IsUCI())
    {
      out("\nbestmove %s\n", MoveToStrLong(best_move, buf));
    }
    else
    {
      Highlight(1);
      out("\nmove %s\n\n", MoveToStrLong(best_move, buf));
      Highlight(0);
    }

    pos0.MakeMove(best_move);
  }
}
////////////////////////////////////////////////////////////////////////////////

void StartPerft(const Position& pos0, int depth)
{
  Position pos = pos0;
  clock_t t0 = clock();
  NODES sum = 0, delta = 0;

  MoveList& mvlist = moves[depth];
  mvlist.GenAllMoves(pos);

  out("\n");
  int i = 0;
  for (i = 0; i < mvlist.Size(); ++i)
  {
    Move mv = mvlist[i];

    if (pos.MakeMove(mv))
    {
      delta = Perft(pos, depth - 1);
      sum += delta;
      pos.UnmakeMove();

      char buf[16];
      out(" %s - ", MoveToStrLong(mv, buf));
      out("%d\n", delta);
      if (InputAvailable())
      {
        char s[4096];
        ReadInput(s, sizeof(s));

        if (Is(s, "exit", 2))
          break;
      }
    }
  }

  if (i == mvlist.Size())
  {
    clock_t t1 = clock();

    out("\n Nodes: %d\n", sum);
    out(" Time:  %lf\n", GetSearchTime() / 1000.);
    out(" Knps:  %lf\n\n", 0.001 * (double) sum * CLOCKS_PER_SEC / (t1 - t0));
  }
}
////////////////////////////////////////////////////////////////////////////////

void UpdateHistory(Move mv, int depth)
{
  if (mv.Captured())
    return;

  PIECE piece = mv.Piece();
  FLD to = mv.To();
  g_history[piece][to] += depth * depth;

  if (g_history[piece][to] >= MAX_HISTORY)
  {
    for (piece = 0; piece < 14; piece++)
      for (to = 0; to < 64; to++)
        g_history[piece][to] /= 2;
  }
}
////////////////////////////////////////////////////////////////////////////////

const EVAL SEE_VALUE[14] =
{ 0, 0, 100, 100, 400, 400, 400, 400, 600, 600, 1200, 1200, 20000, 20000 };

EVAL SEE_Exchange(const Position& pos, FLD to, COLOR side, EVAL currScore, EVAL target, U64 occupied)
{
  U64 att = pos.GetAttacks(to, side, occupied) & occupied;
  if (att == 0)
    return currScore;

  FLD from = NF;
  PIECE piece;
  EVAL newTarget = SEE_VALUE[KW] + 1;

  while (att)
  {
    FLD f = PopLSB(att);
    piece = pos[f];
    if (SEE_VALUE[piece] < newTarget) 
    {
      from = f;
      newTarget = SEE_VALUE[piece];
    }
  }

  occupied ^= BB_SINGLE[from];
  EVAL score = - SEE_Exchange(pos, to, side ^ 1, -(currScore + target), newTarget, occupied);
  return max(score, currScore);
}
////////////////////////////////////////////////////////////////////////////////

EVAL SEE(const Position& pos, Move mv)
{
  FLD from = mv.From();
  FLD to = mv.To();
  PIECE piece = mv.Piece();
  PIECE captured = mv.Captured();
  PIECE promotion = mv.Promotion();
  COLOR side = GetColor(piece);

  EVAL score0 = SEE_VALUE[captured];
  if (promotion)
  {
    score0 += SEE_VALUE[promotion] - SEE_VALUE[PW];
    piece = promotion;
  }

  U64 occupied = pos.BitsAll() ^ BB_SINGLE[from];
  EVAL score = - SEE_Exchange(pos, to, side ^ 1, -score0, SEE_VALUE[piece], occupied);

  return score;
}
////////////////////////////////////////////////////////////////////////////////

