// bitboard.h

#ifndef BITBOARD_H
#define BITBOARD_H

// includes

#include "GCint.h"
#include "util.h"



// macros

#define NULLBITBOARD  ( ULL(0x0000000000000000))

/*  Some bit macros  */

/*
 * gcc 2.95.4 completely screws up the macros with lookup tables 
 * with -O2 on PPC, maybe this check has to be refined. (I don't know
 * whether other architectures also suffer from this gcc bug.) However,
 * with gcc 3.0, the lookup tables are _much_ faster than this direct
 * calculation.
 */
#if defined(__GNUC__) && defined(__PPC__) && __GNUC__ < 3
#  define SETBIT(b,i)   ((b) |=  ((ULL(1)<<63)>>(i)))
#  define CLEARBIT(b,i) ((b) &= ~((ULL(1)<<63)>>(i)))
#else
#  define SETBIT(b,i)   ((b) |= bitboards.BitPosArray[i])
#  define CLEARBIT(b,i) ((b) &= bitboards.NotBitPosArray[i])
#endif


// types

typedef uint64_t BitBoard;

typedef struct {

/***************************************************************************
 *
 * DistMap[sq][d] returns a bitboard of all squares at king
 * distance <= d from square sq.
 *
 ***************************************************************************/
  BitBoard DistMap[64][8];


/***************************************************************************
 *
 *  BitPosArray[i] returns the bitboard whose ith bit is set to 1 
 *  and every other bits 0.  This ought to be faster than doing
 *  shifting all the time (I think).  
 *  Also compute the NotBitPosArray = ~BitPosArray.
 *
 ***************************************************************************/
  BitBoard BitPosArray[64];


/***************************************************************************
 *
 *  NotBitPosArray[i] = ~BitPosArray[i]
 *
 ***************************************************************************/
  BitBoard NotBitPosArray[64];


/***************************************************************************
 * 
 *  The bitboard for all
 *  the moves of a knight on F3 is given by MoveArray[knight][F3].
 *
 ***************************************************************************/
  BitBoard MoveArray[8][64];


/**************************************************************************
 *
 *  For each square, there are 8 rays.  The first 4 rays are diagonals 
 *  for the bishops and the next 4  are file/rank for the rooks.  
 *  The queen uses all 8 rays.
 *  These rays are used for move generation rather than MoveArray[].
 *  Also initialize the directions[][] array.  directions[f][t] returns
 *  the index into Ray[f] array allow us to find the ray in that direction.
 *
 **************************************************************************/
  BitBoard Ray[64][8];


/***************************************************************************
 *
 *  The FromToRay[B2][F6] gives the diagonal ray from C3 to F6 (note that 
 *  B2 is not included);
 *  It also produces horizontal/vertical rays as well.   If no
 *  ray is possible, then a 0 is returned.
 *
 ***************************************************************************/
  BitBoard FromToRay[64][64];


/***************************************************************************
 *
 *  RankBit[2] has all the bits on the 3rd rank 1 and others 0.
 *
 ***************************************************************************/
  BitBoard RankBit[8];


/***************************************************************************
 *
 *  FileBit[2] has all the bits on the 3rd file 1 and others 0.
 *
 ***************************************************************************/
  BitBoard FileBit[8];


/**************************************************************************
 *
 *  The PassedPawnMask variable is used to determine if a pawn is passed.
 *  This mask is basically all 1's from the square in front of the pawn to
 *  the promotion square, also duplicated on both files besides the pawn
 *  file.  Other bits will be set to zero.
 *  E.g. PassedPawnMask[white][B3] = 1's in A4-C4-C8-A8 rect, 0 otherwise.
 *
 **************************************************************************/
  BitBoard PassedPawnMask[2][64];


/**************************************************************************
 *
 *  The IsolaniMask variable is used to determine if a pawn is an isolani.
 *  This mask is basically all 1's on files beside the file the pawn is on.
 *  Other bits will be set to zero.
 *  E.g. IsolaniMask[D_FILE] = 1's in C-file & E-file, 0 otherwise.
 *
 **************************************************************************/
  BitBoard IsolaniMask[8];


/**************************************************************************
 *
 *  The SquarePawnMask is used to determine if a king is in the square of
 *  the passed pawn and is able to prevent it from queening.  
 *  Caveat:  Pawns on 2nd rank have the same mask as pawns on the 3rd rank
 *  as they can advance 2 squares.
 *
 **************************************************************************/
  BitBoard SquarePawnMask[2][64];


/*************************************************************************
 *
 * boardhalf[white] is the lower half of the board. I.e. the lowest 4 ranks.
 * boardhalf[black] is the upper half of the board.
 *
 **************************************************************************/
  BitBoard boardhalf[2];


/*************************************************************************
 *
 * boardhalf[ks] is the right half of the board. I.e. the files E-H.
 * boardhalf[qs] is the left hafl of the board. I.e. the files A-D.
 *
 **************************************************************************/
  BitBoard boardside[2];


/**************************************************************************
 *
 * rings[0-3] represent successively growing rings around the center.
 * E.g. rings[0] is the square D4-E4-D5-E5. rings[3] is the board's border.
 *
 **************************************************************************/ 
  BitBoard rings[4];


/**************************************************************************
 *
 * boxes[0,1] are filled versions of rings[1,2].
 *
 **************************************************************************/
  BitBoard boxes[2];


/**************************************************************************
 *
 * stonewall[white] is the configuration D4,E3,F4.
 * stonewall[black] is the configuration D5,E6,F5
 *
 **************************************************************************/
  BitBoard stonewall[2];


/**************************************************************************
 *
 * WhiteSquares are the white squares on the board.
 * BlackSquares are the black squares on the board.
 *
 **************************************************************************/
  BitBoard WhiteSquares;
  BitBoard BlackSquares;


/***************************************************************************
 *
 * The arrays mask_{k,q}r_trapped_{b,w} are used in eval to check if a rook
 * is trapped. I don't understand the code and I think it is meaningless.
 *
 **************************************************************************/
  BitBoard mask_kr_trapped_w[3];
  BitBoard mask_kr_trapped_b[3];
  BitBoard mask_qr_trapped_w[3];
  BitBoard mask_qr_trapped_b[3];
} bitboard_t;


// variables

extern bitboard_t bitboards;

// These are not really bitboards but they are computed here. Move later.

extern short distance[64][64];
extern short taxicab[64][64];
extern short directions[64][64];
extern const short raybeg[7];
extern const short rayend[7];

// functions

void BitBoardInit();
static inline unsigned char leadz (BitBoard b);
static inline unsigned char trailz(BitBoard b);
static inline unsigned char nbits (BitBoard b);
BitBoard BitBoardFlipVertical(BitBoard x);
BitBoard BitBoardFlipHorizontal (BitBoard x);
void BitBoardShow (BitBoard b);

// static inlined functions for efficiency

extern unsigned char _lzArray[65536];
extern unsigned char _BitCount[65536];

static inline unsigned char leadz (BitBoard b)
/**************************************************************************
 *
 *  Returns the leading bit in a bitboard.  Leftmost bit is 0 (i.e. square a1) and
 *  rightmost bit is 63 (i.e. square h8).  Thanks to Robert Hyatt for this algorithm.
 *
 ***************************************************************************/
{
  if (b >> 48) return _lzArray[b >> 48];
  if (b >> 32) return _lzArray[b >> 32] + 16;
  if (b >> 16) return _lzArray[b >> 16] + 32;
  return _lzArray[b] + 48;
}


static inline unsigned char trailz(BitBoard b){
    return (leadz ((b) & ((~b) + 1)));
}

static inline unsigned char nbits (BitBoard b)
/***************************************************************************
 *
 *  Count the number of bits in b.
 *
 ***************************************************************************/
{
  return _BitCount[b>>48] + _BitCount[(b>>32) & 0xffff]
    + _BitCount[(b>>16) & 0xffff] + _BitCount[b & 0xffff];
}

#endif
