// Search.cpp: implementation of the Search class.
//
//////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "Search.h"
#include <sys/timeb.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Search::Search(EvaluationParameters* evaluationParameters, int hashSize)
{
	evaluation = new Evaluation(evaluationParameters);
	hashTable = new HashTable(hashSize);
}

Search::~Search()
{
	delete evaluation;
	delete hashTable;
}

int whiteHistoryHeuristic[64][64] = 
{
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
};
int blackHistoryHeuristic[64][64] = 
{
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0,
};

//Implements a negamax search whit alfa beta cuttof and some modifications for speed-up checkmate.
void Search::MainSearch(Board* board, int depth, int alpha, int beta, int startTime, int stopTime)
{
	ply=0;
	maxPly=0;
	nodeCount=0;

	initialDepth = depth;
	
	searchStoped = false;
	setjmp(env);
	if (searchStoped) 
	{
		/* make sure to take back the line we were searching */
		while (ply)
		{
			board->TakeBackMove();
			ply--;
		}
		return;
	}
	else
	{
		searchStartTime = startTime;
		searchStopTime = stopTime;

		int score = AlphaBetaNegamax(board, depth, alpha, beta);
		
		//Update the lastLine to recover the information of bestMove and score.
		memcpy((void*)lastLine, pv[0], sizeof(Move)*MAX_PLY);
		lastLineCount = pv_length[0];
		lastLine[0].Score = score;
	}
}

int Search::AlphaBetaNegamax(Board* board, int depth, int alpha, int beta)
{
	int moveScore = 0;
	int svc = -1;
	int hashFlag = HASH_FLAG_UPPER;
	bool hasMoves = false;
	Move bestMove;
	bool mateThreat = false;
	bool foundPV = false;

	if ((nodeCount & 1023) == 0)
		CheckTime();
	
	pv_length[ply] = ply;
	__int64 boardKey = board->GetKey();

	if (ply >= MAX_PLY - 1)
		return beta;

	if (ply && CountPositionReps(board, boardKey))
		return 0;

	if (depth == 0)
		return Quiescent(board, alpha, beta);
		
	Hash* hpos = hashTable->GetPosition(boardKey);
	if (hpos)
	{	
		if (hpos->depth >= depth) 
		{
			switch (hpos->flags)
			{		
				case HASH_FLAG_EXACT:
				{
					alpha = hpos->value;
					if (hpos->value < beta)
					{
						//update the PV
						UpdatePV(hashTable->GetBestMove(hpos));
					}
					return hpos->value;
				}
				case HASH_FLAG_LOWER:
				{
					if(hpos->value>=beta) 
					{ 
						return beta;	
					}
					break;
				}
				case HASH_FLAG_UPPER:
				{
					if(hpos->value<=alpha) 
					{ 
						return alpha;
					}
				}
			}
		}
	}

	//! End game testing (moverlo)
	pieceMat[WHITE] = 0;
	pieceMat[BLACK] = 0;
	for (square=0; square<64; square++)
	{
		sideInSquare = board->squareSide[square];
		pieceInSquare = board->squarePiece[square];
	
		if (sideInSquare == EMPTY) continue;

		if (pieceInSquare != PAWN)
		{
			pieceMat[sideInSquare] += evaluation->pieceValue[pieceInSquare];
		}
	}

	//!Esto no deberia hacerlo devuelta
    bool inCheck = board->InCheck(board->sideToMove);

	// Null move prunning
	if (ply>1 && !inCheck &&

		//El move no fue null move
		(board->moveCount == 0 || !board->history[board->moveCount-1].nullMove) &&
		pieceMat[board->sideToMove]>ENDING_NO_PAWN_SCORE)
	{
		board->MakeNullMove();
		ply++;
		int nullMoveDepth = (depth>5)?depth-4:depth-3;
		int nullMoveValue;
		if (nullMoveDepth > 0)
			nullMoveValue = -AlphaBetaNegamax(board, nullMoveDepth, -beta, -beta+1);
		else
			nullMoveValue = -Quiescent(board, -beta, -beta+1);
		board->TakeBackMove();
		ply--;

		if (nullMoveValue >= beta)
			return beta;
		
		if (nullMoveValue <= -(MATE_SCORE/2))
			mateThreat = true;
	}

	//Return pseudo legal moves or check evasion moves
	MoveList* moves = board->GetAllMoves();
	Sort(board, moves, ply, boardKey, false);

	for (int i=0; i<moves->MovesCount; i++)
    {		
		hasMoves = true;
		int extensions = 0;
		double mydepth = 2;
		Move movei = moves->AllLegalMoves[i];
		
		board->MakeMove(movei);
		
		ply++;
		nodeCount++;
		if (ply > maxPly)
			maxPly = ply;
			
		

		if (ply <= pow(mydepth, initialDepth))
		{
			//Search extension: In Check
			if (inCheck)
				extensions++;
			
			//!Search extension: Promote threat (in test)
			else if (movei.IsPromoteThreat())
				extensions++;

			//Search extension: Mate threat
			else if (mateThreat)
				extensions ++;

			//Search extension: Recapture extension
			else if (	board->moveCount>1 &&
						board->history[board->moveCount-2].move.Capture &&
						board->history[board->moveCount-2].move.To == movei.To &&
						board->history[board->moveCount-2].move.DestinationPiece != PAWN &&
						evaluation->pieceValue[board->history[board->moveCount-2].move.DestinationPiece] == evaluation->pieceValue[board->history[board->moveCount-2].move.MovingPiece])
			{
				extensions++;
			}
			
			//Search extension: Single move (let this extension add to another extensions
			if (moves->MovesCount == 1)
				extensions++;
		}
		
		if (foundPV)
		{
			moveScore = -AlphaBetaNegamax(board, depth-1 + extensions, -alpha - 1, -alpha);
			if (moveScore > alpha && moveScore < beta) // Check for failure.
				moveScore = -AlphaBetaNegamax(board, depth-1 + extensions, -beta, -alpha);
		}
		else
			moveScore = -AlphaBetaNegamax(board, depth-1 + extensions, -beta, -alpha);

	
		board->TakeBackMove();
		ply--;

		if (moveScore > alpha)
		{
			foundPV = true;
			alpha = moveScore;
			bestMove = movei;
			hashFlag = HASH_FLAG_EXACT;
				
			if (movei.MovingSide == WHITE)
				whiteHistoryHeuristic[movei.From][movei.To] += depth;
			else
				blackHistoryHeuristic[movei.From][movei.To] += depth;

			// update the PV
			UpdatePV(bestMove);

			if (alpha >= beta)
			{	
				delete moves;			
				bestMove = movei;
				hashTable->AddPosition(boardKey, depth, alpha, bestMove, HASH_FLAG_LOWER);
				return alpha;
			}
		}	
    }

	if (!hasMoves)
	{	
		delete moves;
		int retVal = inCheck? -MATE_SCORE + ply : 0; 
		hashTable->AddPosition(boardKey, MAX_DEPTH, retVal, HASH_FLAG_EXACT);
		return retVal;
	}
	
	delete moves;
	if (hashFlag == HASH_FLAG_EXACT)
	{					
		hashTable->AddPosition(boardKey, depth, alpha, bestMove, hashFlag);
	}
	else
		hashTable->AddPosition(boardKey, depth, alpha, hashFlag);

    return alpha;
}

void Search::Sort(Board* board, MoveList* moves, int ply, __int64 boardKey, bool onlyCaptures)
{
	//http://groups.google.com/group/rec.games.chess.computer/browse_thread/thread/36f725b2ec3bcf2e/4a10f349d3b3163a?lnk=gst&q=move+ordering&rnum=11#4a10f349d3b3163a
	board->SortMoveList(moves);

	/*
	PLY	0	1	2	3

	1)	1

	2)	1	2

	3)	1	2	3			lastPv

	4)	1	2	3	4		pv
	*/
	


	followPv = true;
	if (initialDepth>1)
	{
		for (int i=0; i<ply; i++)
		{		
			if (pv[0][i].From != lastLine[i].From || pv[0][i].To != lastLine[i].To || (lastLine[i].From == 0 && lastLine[i].To == 0))
			{
				followPv = false;
				break;
			}
		}
		
		if (followPv)
		{
			for (int i=0; i<moves->MovesCount; i++)
			{
				if (moves->AllLegalMoves[i].From == lastLine[ply].From && moves->AllLegalMoves[i].To == lastLine[ply].To)
				{
					Move pvMove = moves->AllLegalMoves[i];
					for (int j=i; j>0; j--)
					{
						moves->AllLegalMoves[j] = moves->AllLegalMoves[j-1];
					}
					moves->AllLegalMoves[0] = pvMove; 
					break;
				}
			}
		}
	}

	if (!followPv)
	{
		Hash* hpos = hashTable->GetPosition(boardKey);
		if (hpos)
		{
			Move hashMove = hashTable->GetBestMove(hpos);
			if (hashMove.From != hashMove.To)
			{
				for (int i=0; i<moves->MovesCount; i++)
				{
					if (hashMove.IsEqual(moves->AllLegalMoves[i]))
					{
						for (int j=i; j>0; j--)
						{
							moves->AllLegalMoves[j] = moves->AllLegalMoves[j-1];
						}
						moves->AllLegalMoves[0] = hashMove;
						break;
					}
				}
			}	
		}
	}
}


int Search::Quiescent(Board* board, int alpha, int beta)
{
	bool hasMoves = false;
	int value = evaluation->Evaluate(board);
	Move capture;
    
	pv_length[ply] = ply;
	
    if (value > alpha)
	{
		alpha = value;
		if (alpha >= beta)
			return alpha;
	}

	//!Esto no deberia hacerlo devuelta
    MoveList* moves;
	bool inCheck = board->InCheck(board->sideToMove);

	if (inCheck)
		moves = board->GetAllMoves();
	else
		moves = board->GetAllCaptures();

	__int64 boardKey = board->GetKey();
	Sort(board, moves, ply, boardKey, true);

	for (int i=0; i<moves->MovesCount; i++)
    {
		hasMoves = true;
		capture = moves->AllLegalMoves[i];
		board->MakeMove(capture);

		ply++;
		nodeCount++;

		if (ply > maxPly)
			maxPly = ply;

		value = -Quiescent(board, -beta, -alpha);
		board->TakeBackMove();
		ply--;
				
		if (value > alpha)
		{
			alpha = value;
			UpdatePV(capture);		
		
			if (alpha >= beta)
			{
				delete moves;
				return alpha;
			}
		}
	}

	delete moves;
	if (!hasMoves && inCheck)
	{	
		hashTable->AddPosition(boardKey, MAX_DEPTH, -MATE_SCORE, HASH_FLAG_EXACT);
		return -MATE_SCORE;
	}
	else
		return alpha;	
}

void Search::UpdatePV(Move move)
{
	pv[ply][ply] = move;
	for (int j = ply + 1; j < pv_length[ply + 1]; ++j)
		pv[ply][j] = pv[ply + 1][j];
	pv_length[ply] = pv_length[ply + 1];
}


int Search::CountPositionReps(Board* board, __int64 hash)
{
	int r = 0;
	for (int j=(board->moveCount-1); j>=0; j--)
	{			
		if (board->history[j].irreversibleMove)
			return r;
		
		if (board->history[j].key == hash)
			++r;	
	}
	return r;
}

void Search::CheckTime()
{
	/* is the engine's time up? if so, longjmp back to the
	   beginning of think() */
	if (GetMs() >= searchStopTime) 
	{
		searchStoped = true;
		longjmp(env, 0);
	}
}


bool ftime_ok = false;  /* does ftime return milliseconds? */
int Search::GetMs()
{
	struct timeb timebuffer;
	ftime(&timebuffer);
	if (timebuffer.millitm != 0)
		ftime_ok = true;
	return (timebuffer.time * 1000) + timebuffer.millitm;
}
