#include "stdafx.h"
#include "Evaluation.h"

Evaluation::Evaluation(EvaluationParameters* evaluationParameters)
{
	ep = *evaluationParameters;
	for(int square=0; square<64; square++)
	{		
		blankAttacks[square] = 1;
	}

	//Initialize pieceVal array
	pieceValue[PAWN] = ep.pawnValue;
	pieceValue[KNIGHT] = ep.knightValue;
	pieceValue[BISHOP] = ep.bishopValue;
	pieceValue[ROCK] = ep.rockValue;
	pieceValue[QUEEN] = ep.queenValue;
	pieceValue[KING] = ep.kingValue;

	//!Esto tambien deberia ser parametro.
	isolatedPawnPenalty[0] = ep.isolatedPawnAPenalty;
	isolatedPawnPenalty[1] = ep.isolatedPawnBPenalty;
	isolatedPawnPenalty[2] = ep.isolatedPawnCPenalty;
	isolatedPawnPenalty[3] = ep.isolatedPawnDPenalty;
	isolatedPawnPenalty[4] = ep.isolatedPawnEPenalty;
	isolatedPawnPenalty[5] = ep.isolatedPawnFPenalty;
	isolatedPawnPenalty[6] = ep.isolatedPawnGPenalty;
	isolatedPawnPenalty[7] = ep.isolatedPawnHPenalty;
}

Evaluation::~Evaluation()
{
}

int Evaluation::Evaluate(Board* board)
{
    int score[2] = {0};
	int square, s;
	int sideInSquare, pieceInSquare;
	int column, rank;
	int offsetColumn;
	int moveSquare, directionSquare;
	bool oppening = board->moveCount<OPENING_MOVES;

	memcpy(whiteAttacks, blankAttacks, sizeof(int)*64);
	memcpy(blackAttacks, blankAttacks, sizeof(int)*64);

	for (int j = 0; j < 10; j++)
    {
        pawnRank[WHITE][j] = 7;
        pawnRank[BLACK][j] = 0;
    }

	pieceMat[WHITE] = 0;
    pieceMat[BLACK] = 0;
    pawnMat[WHITE] = 0;
    pawnMat[BLACK] = 0;
	
	for (square=0; square<64; square++)
    {
		sideInSquare = board->squareSide[square];
		pieceInSquare = board->squarePiece[square];
		column = COL(square);
		rank = ROW(square);
		offsetColumn = column + 1;

        if (sideInSquare == EMPTY) continue;

        if (pieceInSquare == PAWN)
        {
            pawnMat[sideInSquare] += pieceValue[PAWN];
           
            if (sideInSquare == WHITE)
            {
                if (pawnRank[WHITE][offsetColumn] > rank)
                    pawnRank[WHITE][offsetColumn] = rank;
            }
            else
            {
                if (pawnRank[BLACK][offsetColumn] < rank)
                    pawnRank[BLACK][offsetColumn] = rank;
            }
        }
        else
        {
            pieceMat[sideInSquare] += pieceValue[pieceInSquare];
        }
    }

	score[WHITE] = pieceMat[WHITE] + pawnMat[WHITE];
    score[BLACK] = pieceMat[BLACK] + pawnMat[BLACK];

    for (square=0; square<64; square++)
    {
		sideInSquare = board->squareSide[square];
		pieceInSquare = board->squarePiece[square];
		column = COL(square);
		rank = ROW(square);
		offsetColumn = column + 1;
		
		bool endingForWhite = pieceMat[WHITE]<=ENDING_NO_PAWN_SCORE;
		bool endingForBlack = pieceMat[BLACK]<=ENDING_NO_PAWN_SCORE;

        if (sideInSquare == EMPTY) continue;

		ComputeAttacks(board, square);
        if (sideInSquare == WHITE)
        {
			switch (pieceInSquare)
            {
                case PAWN:
                    score[WHITE] += EvalWhitePawn(board, square);

                    break;

                case KNIGHT:

					if (blackAttacks[square] > 1)
					{
						//Not defended
						if (whiteAttacks[square] == 1)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if ((blackAttacks[square] % 3) == 0)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[WHITE] += endingForBlack? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
	
                    score[WHITE] += knightPcSq[square];

					for(s=0; s<64; s++)
					{
						if (board->squareSide[s] == BLACK && board->GetDistance(s, square)<=2)
							score[WHITE] += oppening? ep.knightNearOpponentPieceBonusOpe:ep.knightNearOpponentPieceBonusEnd;	
						
					}
					score[WHITE] += board->GetDistance(board->whiteKingSquare, square) * ep.knightFarFromKingPenalty;
				
					if (endingForBlack)
					{
						//There are no pawn than can capture this knight
						if(pawnRank[BLACK][offsetColumn-1]<=rank && pawnRank[BLACK][offsetColumn+1]<=rank)
							score[WHITE] += ep.knightSecureFromEnemyPawnsBonus;
					}

                    break;
                case BISHOP:

					if (blackAttacks[square] > 1)
					{
						//Not defended
						if (whiteAttacks[square] == 1)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if ((blackAttacks[square] % 3) == 0)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[WHITE] += endingForBlack? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
				

                    score[WHITE] += bishopPcSq[square];

					score[WHITE] += endingForBlack? ep.bishopExtraMaterialInEnding : 0;

					board->GenerateNoPawnDirections(KING, board->blackKingSquare);
					
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = board->oneDirection[directionSquare];
						if (moveSquare==INVALID_SQUARE)
							break;

						if (BishopAttackSquare(board, moveSquare, square))
							score[WHITE] += ep.bishopAttackAdyacentEnemyKingBonus;
					}

					score[WHITE] += GetXRayScore(board, square, BISHOP, WHITE);

                    break;
                case ROCK:

					if (blackAttacks[square] > 1)
					{
						//Not defended
						if (whiteAttacks[square] == 1)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if ((blackAttacks[square] % 3 == 0) || (blackAttacks[square] % 5 == 0) || (blackAttacks[square] % 7) == 0)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[WHITE] += endingForBlack? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
					

                    if (pawnRank[WHITE][offsetColumn] == 7)
                        score[WHITE] += (pawnRank[BLACK][offsetColumn]==0)? ep.rockOpenFileBonus : ep.rockSemiOpenFileBonus;

                    if (rank == 6)
                        score[WHITE] += ep.rockSeventhBonus;

					score[WHITE] += GetXRayScore(board, square, ROCK, WHITE);

                    break;
				case QUEEN:

					if (blackAttacks[square] > 1)
					{
						//Not defended
						if (whiteAttacks[square] == 1)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if ((blackAttacks[square] % 3 == 0) || (blackAttacks[square] % 5 == 0) || (blackAttacks[square] % 7) == 0 || (blackAttacks[square] % 9) == 0)
							score[WHITE] += endingForBlack? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[WHITE] += endingForBlack? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
				

					break;
                case KING:

					score[WHITE] += ep.checkPenalty * GetAttackCount(blackAttacks[board->whiteKingSquare]);
                    score[WHITE] += endingForBlack? kingEndgamePcSq[square] : EvalWhiteKing(board, square);

                    break;
            }
        }
        else
        {
            switch (pieceInSquare)
            {
                case PAWN:
                    score[BLACK] += EvalBlackPawn(board, square);

                    break;
                case KNIGHT:

					if (whiteAttacks[square] > 1)
					{
						//Not defended
						if (blackAttacks[square] == 1)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if (whiteAttacks[square] % 3 == 0)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[BLACK] += endingForWhite? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
					

                    score[BLACK] += knightPcSq[flip[square]];

					for(s=0; s<64; s++)
					{
						if (board->squareSide[s] == WHITE && board->GetDistance(s, square)<=2)
							score[BLACK] += oppening? ep.knightNearOpponentPieceBonusOpe:ep.knightNearOpponentPieceBonusEnd;	
						
					}
					score[BLACK] += board->GetDistance(board->blackKingSquare, square) * ep.knightFarFromKingPenalty;
				
					if (endingForWhite)
					{
						//There are no pawn than can capture this knight
						if(pawnRank[WHITE][offsetColumn-1]>=rank &&  pawnRank[WHITE][offsetColumn+1]>=rank)
							score[BLACK] += ep.knightSecureFromEnemyPawnsBonus;
					}

                    break;
                case BISHOP:

					if (whiteAttacks[square] > 1)
					{
						//Not defended
						if (blackAttacks[square] == 1)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if (whiteAttacks[square] % 3 == 0)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[BLACK] += endingForWhite? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
				
					score[BLACK] += bishopPcSq[flip[square]];

					score[BLACK] += endingForWhite? ep.bishopExtraMaterialInEnding:0;

					board->GenerateNoPawnDirections(KING, board->whiteKingSquare);
					
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = board->oneDirection[directionSquare];
						if (moveSquare==INVALID_SQUARE)
							break;

						if (BishopAttackSquare(board, moveSquare, square))
							score[BLACK] += ep.bishopAttackAdyacentEnemyKingBonus;
					}

					score[BLACK] += GetXRayScore(board, square, BISHOP, BLACK);

                    break;
                case ROCK:

					if (whiteAttacks[square] > 1)
					{
						//Not defended
						if (blackAttacks[square] == 1)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if ((whiteAttacks[square] % 3 == 0) || (whiteAttacks[square] % 5 == 0) || (whiteAttacks[square] % 7) == 0)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[BLACK] += endingForWhite? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
					

					if (pawnRank[BLACK, offsetColumn] == 0)
                        score[BLACK] += pawnRank[WHITE, offsetColumn] == 0 ? ep.rockOpenFileBonus : ep.rockSemiOpenFileBonus;

                    if (rank == 1)
                        score[BLACK] += ep.rockSeventhBonus;

					score[BLACK] += GetXRayScore(board, square, ROCK, BLACK);

                    break;
				case QUEEN:

					if (whiteAttacks[square] > 1)
					{
						//Not defended
						if (blackAttacks[square] == 1)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
						//Atatcked by a pawn or less value piece
						else if ((whiteAttacks[square] % 3 == 0) || (whiteAttacks[square] % 5 == 0) || (whiteAttacks[square] % 7) == 0 || (whiteAttacks[square] % 9) == 0)
							score[BLACK] += endingForWhite? ep.hungPiecePenaltyEnd:ep.hungPiecePenalty;
					}
					if (board->IsBlocked(square))
						score[BLACK] += endingForWhite? ep.blockedPiecePenaltyEnd:ep.blockedPiecePenalty;
					
					break;
                case KING:
					
					score[BLACK] += ep.checkPenalty * GetAttackCount(whiteAttacks[board->blackKingSquare]);
                    score[BLACK] += endingForWhite? kingEndgamePcSq[flip[square]] : EvalBlackKing(board, square);
                    break;
            }
        }
    }

	return board->sideToMove == WHITE ? score[WHITE] - score[BLACK] : score[BLACK] - score[WHITE];

}

int Evaluation::EvalWhitePawn(Board* board, int square)
{
	int retVal = pawnPcSq[square];
    
	int column = COL(square);
	int rank = ROW(square);
	int offsetColumn = column + 1;

    if (pawnRank[WHITE][offsetColumn] < rank)
        retVal += ep.doubledPawnPenalty;

    if (pawnRank[WHITE][offsetColumn - 1] == 7 && pawnRank[WHITE][offsetColumn + 1] == 7)
        retVal += isolatedPawnPenalty[column];
    
	else if (pawnRank[WHITE][offsetColumn - 1] > rank && pawnRank[WHITE][offsetColumn + 1] > rank)
        retVal += ep.backwardsPawnPenalty;

    if (pawnRank[BLACK][offsetColumn - 1] <= rank && pawnRank[BLACK][offsetColumn] <= rank && pawnRank[BLACK][offsetColumn + 1] <= rank)
	{
		retVal += rank * ep.passedPawnBonus;
		
		bool emptyColumn = true;
		for (int r=rank+1; r<=7; r++)
		{
			if (board->squareSide[r*8 + column] != EMPTY)
			{
				emptyColumn = false;
				break;
			}
		}
		if (emptyColumn)
			retVal += ep.passedPawnEmptyColumnBonus;
	}

    return retVal;
}

int Evaluation::EvalBlackPawn(Board* board, int square)
{
    int retVal = pawnPcSq[flip[square]];
    int column = COL(square);
	int rank = ROW(square);
	int offsetColumn = column + 1;

    
    if (pawnRank[BLACK][offsetColumn] > rank)
        retVal += ep.doubledPawnPenalty;

    if (pawnRank[BLACK, offsetColumn - 1] == 0 && pawnRank[BLACK][offsetColumn + 1] == 0)
        retVal += isolatedPawnPenalty[column];
    
	else if (pawnRank[BLACK][offsetColumn - 1] < rank && pawnRank[BLACK][offsetColumn + 1] < rank)
        retVal += ep.backwardsPawnPenalty;

    // add a bonus if the pawn is passed
    if (pawnRank[WHITE][offsetColumn - 1] >= rank && pawnRank[WHITE][offsetColumn] >= rank && pawnRank[WHITE][offsetColumn + 1] >= rank)
    {
		retVal += (7-rank) * ep.passedPawnBonus;
			
		bool emptyColumn = true;
		for (int r=rank-1; r>=0; r--)
		{
			if (board->squareSide[r*8 + column] != EMPTY)
			{
				emptyColumn = false;
				break;
			}
		}
		if (emptyColumn)
			retVal += ep.passedPawnEmptyColumnBonus;
	}

    return retVal;
}

int Evaluation::EvalWhiteKing(Board* board, int square)
{
	//int retVal = (board->whiteKingCastle==0)? kingPcSqNoCastle[square]:kingPcSq[square];
	int retVal = kingPcSq[square];
    int column = COL(square);
	int rank = ROW(square);
	int offsetColumn = column + 1;

    // if the king is castled, use a special function to evaluate the pawns on the appropriate side
    if (column < 3)
    {
        retVal += EvalWhiteKingPawn(1);
        retVal += EvalWhiteKingPawn(2);
        retVal += EvalWhiteKingPawn(3) / 2;// problems with pawns on the 'c' file are not as severe
    }
    else if (column > 4)
    {
        retVal += EvalWhiteKingPawn(8);
        retVal += EvalWhiteKingPawn(7);
        retVal += EvalWhiteKingPawn(6) / 2;// problems with pawns on the 'f' file are not as severe
    }
    // otherwise, just assess a penalty if there are open files near the king
    else
    {
        for (int i=column; i <= column + 2; i++)
        {
            if (pawnRank[WHITE][i] == 7 && pawnRank[BLACK][i] == 0)
                retVal += ep.kingNearOpenFilePenalty;
        }
    }

    retVal *= pieceMat[BLACK];
    retVal /= 3100;

    return retVal;
}

int Evaluation::EvalWhiteKingPawn(int offsetColumn)
{
    int retVal = 0;// return value

    if (pawnRank[WHITE][offsetColumn] == 1)
        ;// pawn hasn't moved
    else if (pawnRank[WHITE][offsetColumn] == 2)
        retVal += ep.kingPawnMoved1Penalty;			// pawn moved one square
    else if (pawnRank[WHITE][offsetColumn] != 7)
        retVal += ep.kingPawnMovedMoreThan1Penalty;	// pawn moved more than one square
    else
        retVal += ep.noKingPawnPenalty;				// no pawn on this file

    if (pawnRank[BLACK][offsetColumn] == 0)
        retVal += ep.kingNoEnemyPawnPenalty;		// no enemy pawn
    else if (pawnRank[BLACK][offsetColumn] == 2)
        retVal += ep.kingEnemyPawn1RankClosePenalty;// enemy pawn on the 3rd rank
    else if (pawnRank[BLACK][offsetColumn] == 3)
        retVal += ep.kingEnemyPawn2RankClosePenalty;// enemy pawn on the 4th rank

    return retVal;
}

int Evaluation::EvalBlackKing(Board* board, int square)
{	
    //int retVal = (board->blackKingCastle==0)? kingPcSqNoCastle[flip[square]]:kingPcSq[flip[square]];
    int retVal = kingPcSq[flip[square]];
    
	int column = COL(square);
	int rank = ROW(square);
	int offsetColumn = column + 1;

    if (column < 3)
    {
        retVal += EvalBlackKingPawn(1);
        retVal += EvalBlackKingPawn(2);
        retVal += EvalBlackKingPawn(3) / 2;
    }
    else if (column > 4)
    {
        retVal += EvalBlackKingPawn(8);
        retVal += EvalBlackKingPawn(7);
        retVal += EvalBlackKingPawn(6) / 2;
    }
    // otherwise, just assess a penalty if there are open files near the king short castle side
    else
    {
        for (int i=column; i <= column + 2; i++)
        {
            if (pawnRank[WHITE][i] == 7 && pawnRank[BLACK][i] == 0)
                retVal += ep.kingNearOpenFilePenalty;
        }
    }

    retVal *= pieceMat[WHITE];
    retVal /= 3100;

    return retVal;
}

int Evaluation::EvalBlackKingPawn(int offsetColumn)
{
    int retVal = 0;// return value

    if (pawnRank[BLACK][offsetColumn] == 6)
        ;// pawn hasn't moved
    else if (pawnRank[BLACK][offsetColumn] == 5)
        retVal += ep.kingPawnMoved1Penalty;				// pawn moved one square
    else if (pawnRank[BLACK][offsetColumn] != 0)
        retVal += ep.kingPawnMovedMoreThan1Penalty;		// pawn moved more than one square
    else
        retVal += ep.noKingPawnPenalty;					// no pawn on this file

    if (pawnRank[WHITE][offsetColumn] == 7)
        retVal += ep.kingNoEnemyPawnPenalty;			// no enemy pawn
    else if (pawnRank[WHITE][offsetColumn] == 5)
        retVal += ep.kingEnemyPawn1RankClosePenalty;	// enemy pawn on the 6th rank
    else if (pawnRank[WHITE][offsetColumn] == 4)
        retVal += ep.kingEnemyPawn2RankClosePenalty;	// enemy pawn on the 5th rank

    return retVal;
}

int Evaluation::GetXRayScore(Board* board, int square, int piece, int sideOfPiece)
{
	//Bishop or rock compute xray score.
	int xRayScore = (piece==BISHOP)? -4:0;

	board->GenerateNoPawnDirections(piece, square);
   	int directionCounter = 0; 
	int pieceInSquare, sideInSquare, direction, directionSquare, moveSquare=0;

	for (direction=0; direction<4; direction++)
	{
		directionCounter++;
		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moveSquare = board->fourDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE )
				break;

			sideInSquare = board->squareSide[moveSquare];
			pieceInSquare = board->squarePiece[moveSquare];
			
			if (sideInSquare == EMPTY)
				xRayScore += 1;
			else if (sideInSquare != sideOfPiece)
            {
                if (pieceInSquare == PAWN )
				{
					break;
				}
                else if (piece==BISHOP && (pieceInSquare == ROCK || pieceInSquare == QUEEN || pieceInSquare == KING))
                    xRayScore += 1 + ep.bishopAttackMostValuedBonus;
				else if (piece==ROCK && (pieceInSquare == QUEEN || pieceInSquare == KING)) 
                    xRayScore += 1 + ep.rockAttackMostValuedBonus;
				else
					xRayScore += 1;
            }
			else
				break;
		}
	}

	return xRayScore;
}

bool Evaluation::BishopAttackSquare(Board* board, int square, int attackerSquare)
{
	board->GenerateNoPawnDirections(BISHOP, square);
	int sideInSquare, direction, directionSquare, moveSquare=0;

	for (direction=0; direction<4; direction++)
	{
		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moveSquare = board->fourDirections[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = board->squareSide[moveSquare];

			if (moveSquare==attackerSquare)
				return true;
			else if (sideInSquare != EMPTY )
				break;
		}
	}

	return false;
}

void Evaluation::ComputeAttacks(Board* board, int square)
{
	board->GenerateSquareDirections(square);
   	int directionCounter = 0; //1 : 0, 2:45, 3:90, 4:135, 5:180, 6:225, 7:270, 8:315
	int moves, pieceInSquare, sideInSquare, direction, directionSquare, moveSquare;

	int originalSquareSide = board->squareSide[square];
	
	for (direction=0; direction<8; direction++)
	{
		moves = 0;
        directionCounter++;

		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moves++;
			moveSquare = board->squareSlidingDirection[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = board->squareSide[moveSquare];
			pieceInSquare = board->squarePiece[moveSquare];
			
			if (sideInSquare != EMPTY)
            {
				//PAWN=3, KNIGHT=5, BISHOP=7, ROCK=9, QUEEN=11, KING=13
                if (pieceInSquare == KING && moves == 1) 
				{
					if (sideInSquare == WHITE)
						whiteAttacks[square] *= 13;
					else
						blackAttacks[square] *= 13;
				}
                else if (pieceInSquare == BISHOP && (directionCounter == 2 || directionCounter == 4 || directionCounter == 6 || directionCounter == 8))
				{
					if (sideInSquare == WHITE)
						whiteAttacks[square] *= 7;
					else
						blackAttacks[square] *= 7;
				}
				else if (pieceInSquare == ROCK && (directionCounter == 1 || directionCounter == 3 || directionCounter == 5 || directionCounter == 7))
                {
					if (sideInSquare == WHITE)
						whiteAttacks[square] *= 9;
					else
						blackAttacks[square] *= 9;
				}
				else if (pieceInSquare == QUEEN)
				{
					if (sideInSquare == WHITE)
						whiteAttacks[square] *= 11;
					else
						blackAttacks[square] *= 11;
                }
				else if (pieceInSquare == PAWN && moves == 1)
                {
                    if (sideInSquare == WHITE)
                    {
						if ((square == moveSquare + 7 && COL(square) != 0) ||
                            (square == moveSquare + 9) && COL(square) != 7)
						{
							whiteAttacks[square] *= 3;
						}
                    }
                    else
                    {
						if ((square == moveSquare - 7 && COL(square) != 7) ||
                           (square == moveSquare - 9) && COL(square) != 0)
						{
							blackAttacks[square] *= 3;
						}
                    }
                }
                break;
            }
            else if (sideInSquare != EMPTY)
                break;
		}
	}
    
    //Pruebo el rey como si fuera un caballo y busco otros caballos
    for (directionSquare=0; directionSquare<8; directionSquare++)
	{
		moveSquare = board->squareKnightDirection[directionSquare];
		if (moveSquare == INVALID_SQUARE)
			break;

		sideInSquare = board->squareSide[moveSquare];
		pieceInSquare = board->squarePiece[moveSquare];
		
		if (sideInSquare != EMPTY && pieceInSquare == KNIGHT) //testeo caballo
		{
			if (sideInSquare == WHITE)
				whiteAttacks[square] *= 5;
			else
				blackAttacks[square] *= 5;
		}
	}
}

int Evaluation::GetAttackCount(int attacksCode)
{
	int retVal = 0;
	int aCode = attacksCode;
	if ((aCode % 11)==0)
	{
		//Attacked By Queen
		retVal++;
		aCode = aCode/11;
	}
	if ((aCode % 9)==0)
	{
		//Attacked By ROCK
		retVal++;
		aCode = aCode/9;

		if ((aCode % 9)==0)
		{
			//Attacked By another ROCK
			retVal++;
			aCode = aCode/9;
 
		}
	}
	if ((aCode % 7)==0)
	{
		//Attacked By BISHOP
		retVal++;
		aCode = aCode/7;

		if ((aCode % 7)==0)
		{
			//Attacked By another BISHOP
			retVal++;
			aCode = aCode/7;
 
		}
	}
	if ((aCode % 5)==0)
	{
		//Attacked By KNIGHT
		retVal++;
		aCode = aCode/5;

		if ((aCode % 5)==0)
		{
			//Attacked By another KNIGHT
			retVal++;
			aCode = aCode/5;
 
		}
	}

	//Count pwan attacks
	retVal = retVal + aCode/3;

	return retVal;
}
