// Board.cpp: implementation of the Board class.
//
//////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "Board.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Board::Board()
{
	pcmg = new PrecalculatedMoveGeneration();
	
	for (int square=0; square<64; square++)
	{
		for (int piece=0; piece<6; piece++)
		{
			zobristTable[piece][WHITE][square] = Rand64();
			zobristTable[piece][BLACK][square] = Rand64();
		}
	}

	for (int emPassIndex=0; emPassIndex<9; emPassIndex++)
	{
		zobristEmPassantTable[emPassIndex] = Rand64();
	}

	zobristCastleTable[SHORT_CASTLE | LONG_CASTLE][WHITE] = Rand64();
	zobristCastleTable[SHORT_CASTLE | LONG_CASTLE][BLACK] = Rand64();
	zobristCastleTable[SHORT_CASTLE][WHITE] = Rand64();
	zobristCastleTable[SHORT_CASTLE][BLACK] = Rand64();
	zobristCastleTable[LONG_CASTLE][WHITE] = Rand64();
	zobristCastleTable[LONG_CASTLE][BLACK] = Rand64();
	zobristCastleTable[0][WHITE] = Rand64();
	zobristCastleTable[0][BLACK] = Rand64();

	white_to_move_code = Rand64();

	InitializeBoard();
}

Board::~Board()
{
	delete pcmg;
}

void Board::GenerateNoPawnDirections(int slidingPiece, int square)
{	
	if (slidingPiece == KNIGHT)
		oneDirection = pcmg->knightMoves[square];
	else if (slidingPiece == BISHOP)
	{
		fourDirections[0] = pcmg->diagonal45Moves[square];
		fourDirections[1] = pcmg->diagonal135Moves[square];
		fourDirections[2] = pcmg->diagonal225Moves[square];
		fourDirections[3] = pcmg->diagonal315Moves[square];
	}
	else if (slidingPiece == ROCK)
	{
		fourDirections[0] = pcmg->columnUpMoves[square];
		fourDirections[1] = pcmg->columnDownMoves[square];
		fourDirections[2] = pcmg->rankRightMoves[square];
		fourDirections[3] = pcmg->rankLeftMoves[square];
	}
	else if (slidingPiece == QUEEN)
	{
		eightDirections[0] = pcmg->columnUpMoves[square];
		eightDirections[1] = pcmg->diagonal45Moves[square];
		eightDirections[2] = pcmg->rankRightMoves[square];
		eightDirections[3] = pcmg->diagonal135Moves[square];
		eightDirections[4] = pcmg->columnDownMoves[square];
		eightDirections[5] = pcmg->diagonal225Moves[square];
		eightDirections[6] = pcmg->rankLeftMoves[square];
		eightDirections[7] = pcmg->diagonal315Moves[square];
	}
	else if (slidingPiece == KING)
		oneDirection = pcmg->kingMoves[square];
}

void Board::GenerateSquareDirections(int square)
{		
	squareKnightDirection = pcmg->knightMoves[square];
	squareSlidingDirection[0] = pcmg->columnUpMoves[square];
	squareSlidingDirection[1] = pcmg->diagonal45Moves[square];
	squareSlidingDirection[2] = pcmg->rankRightMoves[square];
	squareSlidingDirection[3] = pcmg->diagonal135Moves[square];
	squareSlidingDirection[4] = pcmg->columnDownMoves[square];
	squareSlidingDirection[5] = pcmg->diagonal225Moves[square];
	squareSlidingDirection[6] = pcmg->rankLeftMoves[square];
	squareSlidingDirection[7] = pcmg->diagonal315Moves[square];
}

//////////////////////////////////////////////////////////////////////
// Board - Move generation
//////////////////////////////////////////////////////////////////////

void Board::InitializeBoard()
{	
	srand(unsigned(time(NULL)));
	moveCount = 0;
	emPassCol = -10;	

	//Square A1 is 0 position and square H8 is 63 position
	int i;
	
	for (i=0; i<=15; i++)
        squareSide[i] = WHITE;
    for (i=16; i<=47; i++)
        squareSide[i] = EMPTY;
    for (i = 48; i <= 63; i++)
        squareSide[i] = BLACK;

    squarePiece[0] = ROCK;
    squarePiece[1] = KNIGHT;
    squarePiece[2] = BISHOP;
    squarePiece[3] = QUEEN;
    squarePiece[4] = KING;
    squarePiece[5] = BISHOP;
    squarePiece[6] = KNIGHT;
    squarePiece[7] = ROCK;

	//Add pawns in ALL nopawns squares only fot clean garbage;
    for (i = 8; i <= 55; i++)
        squarePiece[i] = PAWN;
	
    squarePiece[56] = ROCK;
    squarePiece[57] = KNIGHT;
    squarePiece[58] = BISHOP;
    squarePiece[59] = QUEEN;
    squarePiece[60] = KING;
    squarePiece[61] = BISHOP;
    squarePiece[62] = KNIGHT;
    squarePiece[63] = ROCK;

    sideToMove = WHITE;
	
	whiteKingSquare = 4;
    blackKingSquare = 60;
	
	whiteKingCastle = SHORT_CASTLE | LONG_CASTLE;
	blackKingCastle = SHORT_CASTLE | LONG_CASTLE;
}

MoveList* Board::GetAllMoves()
{
	MoveList* moves = new MoveList();
	int column, rank;

	int movingPiece;
	int pawnFrontSquare;
	int pawnFirstMoveSquare;
	int pawnCaptureLeftSquare;
	int pawnCaptureRightSquare;

	int direction;
	int directionSquare;
	int moveSquare;
	int moveSquareSide;

	for (int square=0; square<64; square++)
    {
		if (sideToMove == squareSide[square])
        {
			movingPiece = squarePiece[square];
			column = COL(square);
			rank = ROW(square);
		
			if (movingPiece == PAWN)
            {
				if (sideToMove == WHITE)
                {
					pawnFrontSquare = square + 8;
					if (squareSide[pawnFrontSquare] == EMPTY)
					{
						TestAndAddMove(moves, square, pawnFrontSquare, PAWN, false, 0);
						if (square <= 15)
						{
							pawnFirstMoveSquare = square + 16;
							if (squareSide[pawnFirstMoveSquare] == EMPTY)
								TestAndAddMove(moves, square, pawnFirstMoveSquare, PAWN, false, 0);
						}
					}

					//Camtures
                    pawnCaptureLeftSquare = square + 7;
                    if (column != 0 && squareSide[pawnCaptureLeftSquare] == BLACK)
                        TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);

                    pawnCaptureRightSquare = square + 9;
                    if (column != 7 && squareSide[pawnCaptureRightSquare] == BLACK)
                        TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);
					
					//em passant
					if (emPassCol == (column-1) && rank == 4)
						TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);
					else if (emPassCol == (column+1) && rank == 4)
						TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);

				}
				else
				{
					pawnFrontSquare = square - 8;
					if (squareSide[pawnFrontSquare] == EMPTY)
					{
						TestAndAddMove(moves, square, pawnFrontSquare, PAWN, false, 0);
						if (square >= 48)
						{
							pawnFirstMoveSquare = square - 16;
							if (squareSide[pawnFirstMoveSquare] == EMPTY)
								TestAndAddMove(moves, square, pawnFirstMoveSquare, PAWN, false, 0);
						}
					}

					//Camtures
                    pawnCaptureLeftSquare = square - 7;
                    if (column != 7 && squareSide[pawnCaptureLeftSquare] == WHITE)
                        TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);

                    pawnCaptureRightSquare = square - 9;
                    if (column != 0 && squareSide[pawnCaptureRightSquare] == WHITE)
                        TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);

					//em passant
					if (emPassCol == (column+1) && rank == 3)
						TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);
					else if (emPassCol == (column-1) && rank == 3)
						TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);
				}
			}
			else
			{				
				GenerateNoPawnDirections(movingPiece, square);
				if (movingPiece == ROCK || movingPiece == BISHOP)
				{
					for (direction=0; direction<4; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = fourDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];

							//se deslizo a una pieza amiga
							if (moveSquareSide == sideToMove) 
								break;
							//se deslizo a un casillero vacio
							else if (moveSquareSide == EMPTY) 
								TestAndAddMove(moves, square, moveSquare, movingPiece, false, 0);
							//capturo una pieza
							else 
							{
								TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
								break;
							}
						}
					}
				}
				else if (movingPiece == KING)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						
						//Los casilleros de piezas amigas son ignorados completamente
						//pero el testeo sigue en el resto de los casilleros
						if (moveSquareSide == EMPTY) 
							TestAndAddMove(moves, square, moveSquare, movingPiece, false, 0);
						//capturo una pieza
						else if (moveSquareSide != sideToMove)
							TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
					}

					//Castle
					if (sideToMove == WHITE)
					{
						if (whiteKingCastle & SHORT_CASTLE &&
							square == E1 &&	squarePiece[H1] == ROCK && squareSide[H1] == WHITE &&
							squareSide[F1] == EMPTY && squareSide[G1] == EMPTY &&
							!IsInAttack(E1, WHITE) && !IsInAttack(F1, WHITE) && !IsInAttack(G1, WHITE))
							TestAndAddMove(moves, square, G1, movingPiece, false, SHORT_CASTLE);

						if (whiteKingCastle & LONG_CASTLE &&
							square == E1 && squarePiece[A1] == ROCK && squareSide[A1] == WHITE &&
							squareSide[D1] == EMPTY && squareSide[C1] == EMPTY && squareSide[B1] == EMPTY &&
							!IsInAttack(E1, WHITE) && !IsInAttack(D1, WHITE) && !IsInAttack(C1, WHITE)) //PASS3 is not tested for check
							TestAndAddMove(moves, square, C1, movingPiece, false, LONG_CASTLE);
					}	
					else
					{
						if (blackKingCastle & SHORT_CASTLE &&
							square == E8 && squarePiece[H8] == ROCK && squareSide[H8] == BLACK &&
							squareSide[F8] == EMPTY && squareSide[G8] == EMPTY &&
							!IsInAttack(E8, BLACK) && !IsInAttack(F8, BLACK) && !IsInAttack(G8, BLACK))
							TestAndAddMove(moves, square, G8, movingPiece, false, SHORT_CASTLE);
						
						if (blackKingCastle & LONG_CASTLE &&
							square == E8 && squarePiece[A8] == ROCK && squareSide[A8] == BLACK &&
							squareSide[D8] == EMPTY && squareSide[C8] == EMPTY && squareSide[B8] == EMPTY &&
							!IsInAttack(E8, BLACK) && !IsInAttack(D8, BLACK) && !IsInAttack(C8, BLACK)) //PASS3 is not tested for check
							TestAndAddMove(moves, square, C8, movingPiece, false, LONG_CASTLE);
					
					}
				
				}
				else if (movingPiece == KNIGHT)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						
						//Los casilleros de piezas amigas son ignorados completamente
						//pero el testeo sigue en el resto de los casilleros
						if (moveSquareSide == EMPTY) 
							TestAndAddMove(moves, square, moveSquare, movingPiece, false, 0);
						//capturo una pieza
						else if (moveSquareSide != sideToMove)
							TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
					}
				}
				else //(QUEEN)
				{
					for (direction=0; direction<8; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = eightDirections[direction][directionSquare];
							int test = moveSquare & 0x88;
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];

							//se deslizo a una pieza amiga
							if (moveSquareSide == sideToMove) 
								break;
							//se deslizo a un casillero vacio
							else if (moveSquareSide == EMPTY) 
								TestAndAddMove(moves, square, moveSquare, movingPiece, false, 0);
							//capturo una pieza
							else 
							{
								TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
								break;
							}
						}
					}
				}
			}
		}
	}

	return moves;
}

MoveList* Board::GetAllCaptures()
{
	MoveList* moves = new MoveList();
	int column, rank;

	int movingPiece;
	int pawnCaptureLeftSquare;
	int pawnCaptureRightSquare;

	int direction;
	int directionSquare;
	int moveSquare;
	int moveSquareSide;

	for (int square=0; square<64; square++)
    {
		if (sideToMove == squareSide[square])
        {
			movingPiece = squarePiece[square];
			column = COL(square);
			rank = ROW(square);
		
			if (movingPiece == PAWN)
            {
				if (sideToMove == WHITE)
                {
					//Camtures
                    pawnCaptureLeftSquare = square + 7;
                    if (column != 0 && squareSide[pawnCaptureLeftSquare] == BLACK)
                        TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);

                    pawnCaptureRightSquare = square + 9;
                    if (column != 7 && squareSide[pawnCaptureRightSquare] == BLACK)
                        TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);

					//em passant
					if (emPassCol == (column-1) && rank == 4)
						TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);
					else if (emPassCol == (column+1) && rank == 4)
						TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);
				}
				else
				{					
					//Camtures
                    pawnCaptureLeftSquare = square - 7;
                    if (column != 7 && squareSide[pawnCaptureLeftSquare] == WHITE)
                        TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);

                    pawnCaptureRightSquare = square - 9;
                    if (column != 0 && squareSide[pawnCaptureRightSquare] == WHITE)
                        TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);

					//em passant
					if (emPassCol == (column+1) && rank == 3)
						TestAndAddMove(moves, square, pawnCaptureLeftSquare, PAWN, true, 0);
					else if (emPassCol == (column-1) && rank == 3)
						TestAndAddMove(moves, square, pawnCaptureRightSquare, PAWN, true, 0);
				}
			}
			else
			{
				
				GenerateNoPawnDirections(movingPiece, square);
				if (movingPiece == ROCK || movingPiece == BISHOP)
				{
					for (direction=0; direction<4; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = fourDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];

							//se deslizo a una pieza amiga
							if (moveSquareSide == sideToMove) 
								break;
							else if (moveSquareSide != EMPTY)
							{
								TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
								break;
							}
						}
					}
				}
				else if (movingPiece == KING)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						if (moveSquareSide != sideToMove && moveSquareSide != EMPTY)
							TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
					}			
				}
				else if (movingPiece == KNIGHT)
				{
					for (directionSquare=0; directionSquare<8; directionSquare++)
					{
						moveSquare = oneDirection[directionSquare];
						if (moveSquare == INVALID_SQUARE)
							break;

						moveSquareSide = squareSide[moveSquare];
						if (moveSquareSide != sideToMove && moveSquareSide != EMPTY)
							TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
					}
				}
				else //(QUEEN)
				{
					for (direction=0; direction<8; direction++)
					{
						for (directionSquare=0; directionSquare<8; directionSquare++)
						{
							moveSquare = eightDirections[direction][directionSquare];
							if (moveSquare == INVALID_SQUARE)
								break;

							moveSquareSide = squareSide[moveSquare];
							
							if (moveSquareSide == sideToMove) 
								break;
							else if (moveSquareSide != EMPTY)
							{
								TestAndAddMove(moves, square, moveSquare, movingPiece, true, 0);
								break;
							}
						}
					}
				}
			
			}
		}
	}

	return moves;
}

void Board::TestAndAddMove(MoveList* moves, char from, char to, char movingPiece, bool capture, char castleMoveCode)
{
	int rank = ROW(to);
	bool promote = ((rank == 0) || (rank == 7));

	if (movingPiece == PAWN)
    {
		if (capture && promote)
		{
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, true, true, QUEEN);
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, true, true, ROCK);
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, true, true, BISHOP);
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, true, true, KNIGHT);
		}
		else if (promote)
		{
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, false, true, QUEEN);
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, false, true, ROCK);
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, false, true, BISHOP);
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, false, true, KNIGHT);
		}
		else if (capture)
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, true, false, 0);
		else
			AddToMoveList(moves, from, to, PAWN, castleMoveCode, false, false, 0);
	}
	else
	{
		if (capture)
			AddToMoveList(moves, from, to, movingPiece, castleMoveCode, true, false, 0);	
        else 
			AddToMoveList(moves, from, to, movingPiece, castleMoveCode, false, false, 0);	
	}
}

void Board::AddToMoveList(MoveList* moves, char from, char to, char movingPiece, char castleMoveCode, bool capture, bool promote, char pieceToPromote)
{
	moves->AllLegalMoves[moves->MovesCount].From = from;
	moves->AllLegalMoves[moves->MovesCount].To = to;
	moves->AllLegalMoves[moves->MovesCount].MovingSide = sideToMove;
	moves->AllLegalMoves[moves->MovesCount].MovingPiece = movingPiece;
	moves->AllLegalMoves[moves->MovesCount].DestinationPiece = capture? squarePiece[to]:0;
	moves->AllLegalMoves[moves->MovesCount].DestinationSide = squareSide[to];
	moves->AllLegalMoves[moves->MovesCount].Capture = capture;
	moves->AllLegalMoves[moves->MovesCount].Promote = promote;
	moves->AllLegalMoves[moves->MovesCount].PieceToPromote = pieceToPromote;
	moves->AllLegalMoves[moves->MovesCount].CastleMoveCode = castleMoveCode;
	
	if (capture && !IsInAttack(to, (sideToMove==WHITE)? BLACK:WHITE))
	{
		//! Use the score==1 to mark the move as capture to undefended piece
		moves->AllLegalMoves[moves->MovesCount].Score = 1;
	}
	
	MakeMove(moves->AllLegalMoves[moves->MovesCount]);
	if(!InCheck(moves->AllLegalMoves[moves->MovesCount].MovingSide))
	{
		moves->AllLegalMoves[moves->MovesCount].Check = InCheck(sideToMove);
		moves->MovesCount++;	
	}
	TakeBackMove();
}

void Board::SortMoveList(MoveList* moves)
{
	qsort(moves->AllLegalMoves, moves->MovesCount, sizeof(Move), MoveComparer);
}

//Implements MVV/LVA for captures and promotes-captures
int Board::MoveComparer (const void* a, const void* b)
{
	/*

0	move in the last principal continuation (I pass this through
	the hash table, by "stuffing" the PV moves before the next
	iteration starts.  This simplifies the code and eliminates
	a special test that is *very* rarely true.  Therefore, this
	case falls into the next case quite nicely.
0	move suggested by hash table.
1	captures of greater value
2	captures of equal value
3	killer moves (they can be tried before generating legal
	moves because you normally don't let a capture become a
	killer since it is tried before the killers anyway.
4	history moves... sort of a global killer move idea reported
	by Jonathan Schaeffer.  I can give details if you don't know
	about this idea.
5	rest of moves...  can be ordered anyway you like.  I try
	castling first, then all knight moves, then bishop moves,
	then rook moves, then queen moves, then king moves, and
	finally pawn moves.  

	*/

	Move movea = *(const Move *)a;
	Move moveb = *(const Move *)b;
	int moveAScore=0, moveBScore=0;

	//Promote moves has score between 100 and 140
	if (movea.Promote)
		moveAScore += 100 + movea.PieceToPromote*10;
	if (moveb.Promote)
		moveBScore += 100 + moveb.PieceToPromote*10;

	//Captures has scored beetween -4 and 4
	if (movea.Capture)
	{
		if (movea.Score == 1)
			moveAScore += movea.DestinationPiece;
		else
			moveAScore += (movea.DestinationPiece - movea.MovingPiece);
	}
	if (moveb.Capture)
	{
		if (moveb.Score == 1)
			moveBScore += moveb.DestinationPiece;
		else
			moveBScore += (moveb.DestinationPiece - moveb.MovingPiece);
	}

	if (movea.Check)
	{
		if ((movea.Capture && moveAScore<0) || !movea.Capture)
			moveAScore = 1;
	}
	if (moveb.Check)
	{
		if ((moveb.Capture && moveBScore<0) || !moveb.Capture)
			moveBScore = 1;
	}

	//Sort promotes, checks and good capures first
	if (moveAScore > moveBScore)
		return -1;
	else if (moveAScore < moveBScore)
		return 1;

	//Same score moves
	else 
	{
		//Both moves are good captures, checks or promotes or combination, sort promotes first
		if (moveAScore > 0)
		{
			if (movea.Promote && !moveb.Promote)
				return -1;
			else if (moveb.Promote && !movea.Promote)
				return 1;
			else
			{
				//Same positive score and both promote or both no-promote, 
				//then, same capture score, both check or not check, sort ramdomly
				if ((rand() % 2) == 0)
					return 1;
				else 
					return -1;	
			}
		}
		//No captures or captures of same value.
		else if (moveAScore == 0)
		{
			if (movea.Capture && !moveb.Capture)
				return -1;
			else if (moveb.Capture && !movea.Capture)
				return 1;
			else
			{
				bool aPromoteThreat = movea.IsPromoteThreat();
				bool bPromoteThreat = moveb.IsPromoteThreat();

				if (aPromoteThreat && !bPromoteThreat)
					return -1;
				else if (!aPromoteThreat && bPromoteThreat)
					return 1;
				
				//same promote threat
				moveAScore += (movea.MovingSide==WHITE)? whiteHistoryHeuristic[movea.From][movea.To]:blackHistoryHeuristic[movea.From][movea.To];
				moveBScore += (moveb.MovingSide==WHITE)? whiteHistoryHeuristic[moveb.From][moveb.To]:blackHistoryHeuristic[moveb.From][moveb.To];
				
				if (moveAScore > moveBScore)
					return -1;
				else if (moveAScore < moveBScore)
					return 1;
				else
				{
					((Move *)a)->Score = -1;
					((Move *)b)->Score = -1;

					//Both moves has same history score, sort by move type
					moveAScore = 0;
					moveBScore = 0;

					if (movea.CastleMoveCode != 0)
						moveAScore += 10;
					if (moveb.CastleMoveCode != 0)
						moveBScore += 10;

					moveAScore += KING - movea.MovingPiece;
					if (movea.MovingPiece == PAWN)
						moveAScore = -1;

					moveBScore += KING - moveb.MovingPiece;
					if (moveb.MovingPiece == PAWN)
						moveBScore = -1;
					
					if (moveAScore > moveBScore)
						return -1;
					else if (moveAScore < moveBScore)
						return 1;
					else
					{
						//Same move type, sort randomly
						if ((rand() % 2) == 0)
							return 1;
						else 
							return -1;	
					}
				}
			}

		}
		//Both moves are bad captures captures
		else
		{
			((Move *)a)->Score = -1;
			((Move *)b)->Score = -1;

			//Both captures are bad and equal, sort ramdomly
			if ((rand() % 2) == 0)
				return 1;
			else 
				return -1;	
		}
	}
}

//////////////////////////////////////////////////////////////////////
// Board - Moke move / take back move
//////////////////////////////////////////////////////////////////////

void Board::MakeMove(Move move)
{
	history[moveCount].move = move;
	history[moveCount].KingCastle = (move.MovingSide==WHITE)? whiteKingCastle: blackKingCastle;
	history[moveCount].key = GetKey();
	history[moveCount].emPassCol = emPassCol;
	history[moveCount].nullMove = false;
	history[moveCount].irreversibleMove = false;

	emPassCol = -10;		
    squareSide[move.From] = EMPTY;
    squareSide[move.To] = move.MovingSide;

    if (move.Promote)
        squarePiece[move.To] = move.PieceToPromote;
    else
        squarePiece[move.To] = move.MovingPiece;

    //Update kings position
    if (move.MovingPiece == KING)
    {
        if (sideToMove == WHITE)
            whiteKingSquare = move.To;
        else
            blackKingSquare = move.To;
    }
	else if (move.MovingPiece == PAWN)
	{
		history[moveCount].irreversibleMove = true;

		//If one pawn moves 2 squares mark the col as em passant
		if (GetDistance(move.From, move.To) == 2)
		{
			emPassCol = COL(move.From);
		}

		//Em passant, delete the captured pawn
		if (move.Capture && move.DestinationSide == EMPTY)
		{
			squareSide[8*ROW(move.From)+COL(move.To)] = EMPTY;
		}
	}

	if (move.CastleMoveCode != 0)
	{
		history[moveCount].irreversibleMove = true;

		if (move.MovingSide == WHITE)
		{
			if (move.CastleMoveCode == SHORT_CASTLE)
			{
				squareSide[H1] = EMPTY;
				squareSide[F1] = WHITE;
				squarePiece[F1] = ROCK;	
			}
			else
			{
				squareSide[A1] = EMPTY;
				squareSide[D1] = WHITE;
				squarePiece[D1] = ROCK;
			}
			whiteKingCastle = 0;
		}
		else
		{
			if (move.CastleMoveCode == SHORT_CASTLE)
			{
				squareSide[H8] = EMPTY;
				squareSide[F8] = BLACK;
				squarePiece[F8] = ROCK;
			}
			else 
			{
				squareSide[A8] = EMPTY;
				squareSide[D8] = BLACK;
				squarePiece[D8] = ROCK;
			} 
			blackKingCastle = 0;
		}
	}
	else 
	{
		if (move.MovingSide == WHITE)
		{
			if (move.From == E1 && move.MovingPiece == KING && whiteKingCastle!=0)
				whiteKingCastle = 0;
			else if (move.From == H1 && move.MovingPiece == ROCK && (whiteKingCastle & SHORT_CASTLE)!=0)
				whiteKingCastle = whiteKingCastle & LONG_CASTLE;
			else if (move.From == A1 && move.MovingPiece == ROCK && (whiteKingCastle & LONG_CASTLE)!=0)
				whiteKingCastle = whiteKingCastle & SHORT_CASTLE;
		}
		else
		{
			if (move.From == E8 && move.MovingPiece == KING && blackKingCastle!=0)
				blackKingCastle = 0;
			else if (move.From == H8 && move.MovingPiece == ROCK && (blackKingCastle & SHORT_CASTLE)!=0)
				blackKingCastle = blackKingCastle & LONG_CASTLE;
			else if (move.From == A8 && move.MovingPiece == ROCK && (blackKingCastle & LONG_CASTLE)!=0)
				blackKingCastle = blackKingCastle & SHORT_CASTLE;
		}
	}
		
	sideToMove = (sideToMove == WHITE) ? BLACK : WHITE;
	moveCount++;
}

void Board::TakeBackMove()
{
	moveCount--;
	MoveRecord mr = history[moveCount];
	Move move = mr.move;

	//If the move is a null move the take back the null move
	if (mr.nullMove)
	{
		emPassCol = mr.emPassCol;
		(move.MovingSide==WHITE)? whiteKingCastle=mr.KingCastle: blackKingCastle=mr.KingCastle;
		sideToMove = move.MovingSide;
		return;
	}

    squareSide[move.From] = move.MovingSide;
    squarePiece[move.From] = move.MovingPiece;

    squareSide[move.To] = move.DestinationSide;
    squarePiece[move.To] = move.DestinationPiece;

	emPassCol = mr.emPassCol;

    //Update kings position
    if (move.MovingPiece == KING)
    {
        if (move.MovingSide == WHITE)
            whiteKingSquare = move.From;
        else
            blackKingSquare = move.From;
    }
	
	else if (move.MovingPiece == PAWN)
	{	 
		//Em passant, put a pawn on the side to the captor.
		if (move.Capture && move.DestinationSide == EMPTY)
		{
			squareSide[8*ROW(move.From)+COL(move.To)] = (move.MovingSide==WHITE)?BLACK:WHITE;
			squarePiece[8*ROW(move.From)+COL(move.To)] = PAWN;
		}			
	}

	(move.MovingSide==WHITE)? whiteKingCastle=mr.KingCastle: blackKingCastle=mr.KingCastle;
	if (move.CastleMoveCode != 0)
	{
		if (move.MovingSide == WHITE)
		{
			if (move.CastleMoveCode == SHORT_CASTLE)
			{
				squareSide[H1] = WHITE;
				squareSide[F1] = EMPTY;
				squarePiece[H1] = ROCK;
			}
			else
			{
				squareSide[A1] = WHITE;
				squareSide[D1] = EMPTY;
				squarePiece[A1] = ROCK;
			}	
		}
		else
		{
			if (move.CastleMoveCode == SHORT_CASTLE)
			{
				squareSide[H8] = BLACK;
				squareSide[F8] = EMPTY;
				squarePiece[H8] = ROCK;
			}
			else
			{
				squareSide[A8] = BLACK;
				squareSide[D8] = EMPTY;
				squarePiece[A8] = ROCK;
			}   
		}
	}

	sideToMove = move.MovingSide;
}

void Board::MakeNullMove()
{
	Move nullMove;
	nullMove.MovingSide = sideToMove;
	history[moveCount].move = nullMove;
	history[moveCount].KingCastle = (nullMove.MovingSide==WHITE)? whiteKingCastle: blackKingCastle;
	history[moveCount].key = GetKey();
	history[moveCount].emPassCol = emPassCol;
	history[moveCount].nullMove = true;
	history[moveCount].irreversibleMove = true;

	emPassCol = -10;		
   		
	sideToMove = (sideToMove == WHITE) ? BLACK : WHITE;
	moveCount++;
}

/*void Board::TakeBackNullMove()
{
	moveCount--;
	MoveRecord mr = history[moveCount];
	Move nullMove = mr.move;

    emPassCol = mr.emPassCol;

	(nullMove.MovingSide==WHITE)? whiteKingCastle=mr.KingCastle: blackKingCastle=mr.KingCastle;

	sideToMove = nullMove.MovingSide;
}*/

bool Board::InCheck(int side)
{
	bool retVal = false;
	if (side == BLACK)
	{
		retVal = IsInAttack(blackKingSquare, BLACK);	
	}
	else if (side == WHITE)
	{
		retVal = IsInAttack(whiteKingSquare, WHITE);
	}
	return retVal;
}

bool Board::IsInAttack(int square, int sideOfSquare)
{
	GenerateSquareDirections(square);
   	int directionCounter = 0; 
	int moves, pieceInSquare, sideInSquare, direction, directionSquare, moveSquare=0;

	for (direction=0; direction<8; direction++)
	{
		moves = 0;
        directionCounter++;

		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moves++;
			moveSquare = squareSlidingDirection[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;

			sideInSquare = squareSide[moveSquare];
			pieceInSquare = squarePiece[moveSquare];
			
			if (sideInSquare != EMPTY  && sideInSquare != sideOfSquare)
            {
                if (pieceInSquare == KING && moves == 1) //testeo rey
                    return true;
                //testeo bishop solo en diagonales
                else if (pieceInSquare == BISHOP && (directionCounter == 2 || directionCounter == 4 || directionCounter == 6 || directionCounter == 8))
                    return true;
                //testeo rock solo en filas y columnas
                else if (pieceInSquare == ROCK && (directionCounter == 1 || directionCounter == 3 || directionCounter == 5 || directionCounter == 7))
                    return true;
                //testeo queen en todas las direcciones
                else if (pieceInSquare == QUEEN)
                    return true;
                else if (pieceInSquare == PAWN && moves == 1)
                {
                    if (sideInSquare == WHITE)
                    {
                        if (((square == moveSquare + 7) && COL(moveSquare) != 0) ||
                            ((square == moveSquare + 9) && COL(moveSquare) != 7))
                            return true;
                    }
                    else
                    {
                        if (((square == moveSquare - 7) && COL(moveSquare) != 7) ||
                           ((square == moveSquare - 9) && COL(moveSquare) != 0))
                            return true;
                    }
                }
                break;
            }
            else if (sideInSquare != EMPTY)
                break;
		}
	}
    
    //Pruebo el rey como si fuera un caballo y busco otros caballos
    for (directionSquare=0; directionSquare<8; directionSquare++)
	{
		moveSquare = squareKnightDirection[directionSquare];
		if (moveSquare == INVALID_SQUARE)
			break;

		sideInSquare = squareSide[moveSquare];
		pieceInSquare = squarePiece[moveSquare];
		
		if (sideInSquare != EMPTY && sideInSquare != sideOfSquare && pieceInSquare == KNIGHT) //testeo caballo
			return true;
	}

	return false;
}

bool Board::IsBlocked(int square)
{
	int sideOfSquare = squareSide[square];
	int kingSquare = sideOfSquare==WHITE? whiteKingSquare:blackKingSquare;

	GenerateSquareDirections(kingSquare);
   	int directionCounter = 0; 
	int pieceInSquare, sideInSquare, direction, directionSquare, moveSquare=0;
	bool pieceFound = false;

	for (direction=0; direction<8; direction++)
	{
        directionCounter++;

		for (directionSquare=0; directionSquare<8; directionSquare++)
		{
			moveSquare = squareSlidingDirection[direction][directionSquare];
			if (moveSquare == INVALID_SQUARE)
				break;
		
			sideInSquare = squareSide[moveSquare];
			pieceInSquare = squarePiece[moveSquare];

			//Friendly piece was found but not the piece to test. Break direction
			if (sideInSquare == sideOfSquare && moveSquare != square)
				break;

			//The piece was found. continue testing direction
			if (moveSquare == square)
				pieceFound = true;

			//Enemy piece was found in direction and piece was previously found. return true if enemy piece
			//can block other piece
			if (sideInSquare != EMPTY  && sideInSquare != sideOfSquare && pieceFound)
            {
				if (pieceInSquare == BISHOP && (directionCounter == 2 || directionCounter == 4 || directionCounter == 6 || directionCounter == 8))
                    return true;
                //testeo rock solo en filas y columnas
                else if (pieceInSquare == ROCK && (directionCounter == 1 || directionCounter == 3 || directionCounter == 5 || directionCounter == 7))
                    return true;
                //testeo queen en todas las direcciones
                else if (pieceInSquare == QUEEN)
                    return true;
                else 
					return false;
            }
			//Enemy piece was found in direction but piece not found. The side is in check but the piece is not blocked
			//in this direction. break.
			else if (sideInSquare != EMPTY  && sideInSquare != sideOfSquare)
				break;
		}
	}

	return false;
}

int Board::GetDistance(int square1, int square2)
{
	int d1 = abs(COL(square1)-COL(square2));
	int d2 = abs(ROW(square1)-ROW(square2));

	if (d1>=d2)
		return d1;
	else
		return d2;
}

__int64 Board::Rand64()
{
    return rand() ^ ((__int64)rand() << 15) ^ ((__int64)rand() << 30) ^
        ((__int64)rand() << 45) ^ ((__int64)rand() << 60);

}

__int64 Board::GetKey()
{
	__int64 zobristKey = 0;
	
	for (int square=0; square<64; square++)
	{
		if (squareSide[square] != EMPTY)
			zobristKey = zobristKey ^ zobristTable[squarePiece[square]][squareSide[square]][square];
	}

	if (sideToMove == WHITE)
		zobristKey = zobristKey ^ white_to_move_code;

	zobristKey = zobristKey ^ zobristCastleTable[whiteKingCastle][WHITE];
	zobristKey = zobristKey ^ zobristCastleTable[blackKingCastle][BLACK];
	
	int emPassIndex = emPassCol;
	if (emPassIndex == -10)
		emPassIndex = 8;

	zobristKey = zobristKey ^ zobristEmPassantTable[emPassIndex];
	
	return zobristKey;
}
