// EXchess source code, (c) Daniel C. Homan  1997-2011
// Released under the GNU public license, see file license.txt

/* Piece captures only */

#include "define.h"
#include "chess.h"
#include "const.h"
#include "funct.h"

/*------------------------- Semi-legal capture generator ---------------------*/

int position::captures(move_list *list, int delta_score)
{
  register int i;
  list->count = 0;

  // function calls to individual piece routines will return 1
  //  immediately if a move is found that captures a KING, 
  //  otherwise the accumluation of captures into the list
  //  continues.  Note it is safe to stop with capturing
  //  a KING as this will terminate the current qsearch node.

  for(i=1;i<=plist[wtm][PAWN][0];i++) 
    if(pawn_capts(list, plist[wtm][PAWN][i], delta_score)) return 1;
  for(i=1;i<=plist[wtm][KNIGHT][0];i++) 
    if(knight_capts(list, plist[wtm][KNIGHT][i], delta_score)) return 1;
  for(i=1;i<=plist[wtm][BISHOP][0];i++) 
    if(bishop_capts(list, plist[wtm][BISHOP][i], delta_score)) return 1;
  for(i=1;i<=plist[wtm][ROOK][0];i++) 
    if(rook_capts(list, plist[wtm][ROOK][i], delta_score)) return 1;
  for(i=1;i<=plist[wtm][QUEEN][0];i++) {
    if(bishop_capts(list, plist[wtm][QUEEN][i], delta_score)) return 1;
    if(rook_capts(list, plist[wtm][QUEEN][i], delta_score)) return 1;
  }
  for(i=1;i<=plist[wtm][KING][0];i++) 
    if(king_capts(list, plist[wtm][KING][i], delta_score)) return 1;

  return 0;
}

/* Add capture function */
// Function to add a capture to the capture list
// The move is also scored for the alpha-beta search
// fsq = from square,  tsq = to square
int position::add_capt(int fsq, int tsq, move_list *list, char type, int delta_score)
{
  int i = list->count;      // index of move list

  if(sq[tsq].type == KING) return 1;

  // add move to list...
  list->mv[i].m.b.from = fsq;
  list->mv[i].m.b.to = tsq;
  list->mv[i].m.b.type = type;
  list->mv[i].m.b.promote = 0;

  // Give an initial score to the move below delta_score
  // so that by default it will not be put in the list 
  // unless a step below changes the score
  list->mv[i].score = delta_score-100;  

  // give a bonus for a promotion
  if((tsq > 55 || tsq < 8) && sq[fsq].type == PAWN)
  {
    list->mv[i].m.b.type |= PROMOTE; 
    list->mv[i].m.b.promote = QUEEN;
    if(swap(tsq,(*this),wtm,fsq) >= 0) list->mv[i].score += 1000;
  }

 
  // If the move is a pawn capture 
  //  give a bonus by the rank of the pawn
  int pawn_bonus = 0;
  if(sq[tsq].type == PAWN) {
    if(wtm && RANK(tsq) < 5) pawn_bonus = 15*(5-RANK(tsq)); 
    if(!wtm && RANK(tsq) > 2) pawn_bonus = 15*(RANK(tsq)-2);
  }

  // score move with the expected gain if it already hasn't made the list
  if((type&CAPTURE) && list->mv[i].score < delta_score) 
    if(value[sq[tsq].type]+pawn_bonus >= delta_score) {
      list->mv[i].score = swap(tsq,(*this),wtm,fsq)+pawn_bonus;
      // looking for a revealed check if score isn't high enough
      if(list->mv[i].score < delta_score && slide_check_table[fsq][plist[wtm^1][KING][1]])  {
	position temp_pos = (*this);
	temp_pos.wtm ^= 1;
	temp_pos.sq[tsq] = temp_pos.sq[fsq];
	temp_pos.sq[fsq] = empty;
	if(temp_pos.simple_check(fsq)) list->mv[i].score = value[sq[tsq].type]+pawn_bonus;
      }
    } 
  
 
  //  list->mv[i].score = value[sq[tsq].type] + pawn_bonus + promote_score;

  if(list->mv[i].score >= delta_score) {
    list->count++;           // increase list count
  }

  return 0;

}

/*---------------------------- Bishop Captures --------------------------*/
int position::bishop_capts(move_list *list, int sqr, int ds)
{
  int mm, nn, ii;
  int tsq;                            // to square

  mm = FILE(sqr); nn = RANK(sqr);     // set rank and file of bishop

  ii = 1;
  while (mm + ii <= 7 && nn + ii <= 7)
  {
   tsq = SQR((mm+ii),(nn+ii));             // set to square
   if (!sq[tsq].type) { ii++; }            // if empty, move on
   else if (sq[tsq].side != wtm)          // else if other side,
     { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; } // add to capture list
   else break;                             // break if our piece
  }

  ii = 1;
  while (mm - ii >= 0 && nn + ii <= 7)
  {
   tsq = SQR((mm-ii),(nn+ii));
   if (!sq[tsq].type) { ii++; }
   else if (sq[tsq].side != wtm)
     { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; }
   else break;
  }

  ii = 1;
  while(mm + ii <= 7 && nn - ii >= 0)
  {
   tsq = SQR((mm+ii),(nn-ii));
   if (!sq[tsq].type) { ii++; }
   else if (sq[tsq].side != wtm)
     { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; }
   else break;
  }

  ii = 1;
  while (mm - ii >= 0 && nn - ii >= 0)
  {
   tsq = SQR((mm-ii),(nn-ii));
   if (!sq[tsq].type) { ii++; }
   else if (sq[tsq].side != wtm)
     { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; }
   else break;
  }

  return 0;

}

/*--------------------------- Rook Captures ---------------------------*/
int position::rook_capts(move_list *list, int sqr, int ds)
{
  int mm, nn, ii;
  int tsq;                              // to square

  mm = FILE(sqr); nn = RANK(sqr);       // set file and rank of rook

  ii = 1;
  while (mm + ii <= 7)
  {
   tsq = SQR((mm+ii),nn);                  // set to square
   if (!sq[tsq].type) { ii++; }            // if empty, move on
   else if (sq[tsq].side != wtm)          // else if other side,
   { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; } // add to capture list
   else break;                             // else if our side, break
  }

  ii = 1;
  while (mm - ii >= 0)
  {
   tsq = SQR((mm-ii),nn);
   if (!sq[tsq].type) { ii++; }
   else if (sq[tsq].side != wtm)
   { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; }
   else break;
  }

  ii = 1;
  while(nn - ii >= 0)
  {
   tsq = SQR(mm,(nn-ii));
   if (!sq[tsq].type) { ii++; }
   else if (sq[tsq].side != wtm)
   { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; }
   else break;
  }

  ii = 1;
  while (nn + ii <= 7)
  {
   tsq = SQR(mm,(nn+ii));
   if (!sq[tsq].type) { ii++; }
   else if (sq[tsq].side != wtm)
   { if(add_capt(sqr, tsq, list, 1, ds)) return 1; break; }
   else break;
  }

  return 0;

}

/*--------------------------- Knight Captures ----------------------------*/
int position::knight_capts(move_list *list, int sqr, int ds)
{
  int tsq;                                // to square

  if(FILE(sqr) < 6 && RANK(sqr) < 7) {
   tsq = sqr + 10;                                   // set to square
   if(sq[tsq].side != wtm) {                        // if occupied by
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;   // other side, add
    }                                                // to capture list
  }
  if(FILE(sqr) < 6 && RANK(sqr)) {
   tsq = sqr - 6;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) > 1 && RANK(sqr) < 7) {
   tsq = sqr + 6;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) > 1 && RANK(sqr)) {
   tsq = sqr - 10;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr) < 6) {
   tsq = sqr + 17;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) && RANK(sqr) < 6) {
   tsq = sqr + 15;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr) > 1) {
   tsq = sqr - 15;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) && RANK(sqr) > 1) {
   tsq = sqr - 17;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }

  return 0;

}

/*--------------------------- King Captures ----------------------------*/
int position::king_capts(move_list *list, int sqr, int ds)
{
  int tsq;                            // to square

  if(FILE(sqr) && RANK(sqr) < 7) {
   tsq = sqr + 7;                                   // set to square
   if(sq[tsq].side != wtm) {                       // if occupied by
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;  // other side, add
    }                                               // to capture list
  }
  if(RANK(sqr) < 7) {
   tsq = sqr + 8;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr) < 7) {
   tsq = sqr + 9;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr)) {
   tsq = sqr - 1;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) < 7) {
   tsq = sqr + 1;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr)) {
   tsq = sqr - 7;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(RANK(sqr)) {
   tsq = sqr - 8;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }
  if(FILE(sqr) && RANK(sqr)) {
   tsq = sqr - 9;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) if(add_capt(sqr, tsq, list, 1, ds)) return 1;
    }
  }

  return 0;
}

/*------------------------ Pawn Captures ------------------------------*/
// Not including en passant!
// Do include promotions!
int position::pawn_capts(move_list *list, int sqr, int ds)
{

  if(wtm) {                        // if it is white's pawn
   if(FILE(sqr)) {
    if (sq[sqr+7].type && sq[sqr+7].side != wtm)
      { if(add_capt(sqr, sqr+7, list, (PAWN_PUSH|1), ds)) return 1; }  
   }
   if(FILE(sqr) < 7) {
    if (sq[sqr+9].type && sq[sqr+9].side != wtm)
      { if(add_capt(sqr, sqr+9, list, (PAWN_PUSH|1), ds)) return 1; }
   }
   if(RANK(sqr) == 6) {
    if (!sq[sqr+8].type)
      { if(add_capt(sqr, sqr+8, list, (PAWN_PUSH|PROMOTE), ds)) return 1; }
   }
  } else {                           // or if it is black's pawn
   if(FILE(sqr)) {
    if (sq[sqr-9].type && sq[sqr-9].side != wtm)
      { if(add_capt(sqr, sqr-9, list, (PAWN_PUSH|1), ds)) return 1; }
   }
   if(FILE(sqr) < 7) {
    if (sq[sqr-7].type && sq[sqr-7].side != wtm)
      { if(add_capt(sqr, sqr-7, list, (PAWN_PUSH|1), ds)) return 1; }
   }
   if(RANK(sqr) == 1) {
    if (!sq[sqr-8].type)
      { if(add_capt(sqr, sqr-8, list, (PAWN_PUSH|PROMOTE), ds)) return 1; }
   }
  }

  return 0;
}
























