// EXchess source code, (c) Daniel C. Homan  1997-2011
// Released under the GNU public license, see file license.txt

/* Piece captures and checks */

#include "define.h"
#include "chess.h"
#include "const.h"
#include "funct.h"
#include "extern.h"

/*------------------------- Semi-legal capture/check generator ---------------------*/

void position::captchecks(move_list *list, int delta_score)
{
  register int i;
  list->count = 0;

  for(i=1;i<=plist[wtm][PAWN][0];i++) 
    pawn_cc(list, plist[wtm][PAWN][i], delta_score);
  for(i=1;i<=plist[wtm][KNIGHT][0];i++) 
    knight_cc(list, plist[wtm][KNIGHT][i], delta_score);
  for(i=1;i<=plist[wtm][BISHOP][0];i++) 
    bishop_cc(list, plist[wtm][BISHOP][i], delta_score);
  for(i=1;i<=plist[wtm][ROOK][0];i++) 
    rook_cc(list, plist[wtm][ROOK][i], delta_score);
  for(i=1;i<=plist[wtm][QUEEN][0];i++) {
    bishop_cc(list, plist[wtm][QUEEN][i], delta_score);
    rook_cc(list, plist[wtm][QUEEN][i], delta_score);
  }
  for(i=1;i<=plist[wtm][KING][0];i++) 
    king_cc(list, plist[wtm][KING][i], delta_score);

}

/* Add move to move list */
// Function to add move to move list
// Move is scored for alpha-beta algorithm
void position::add_cc(int fsq, int tsq, move_list *list, char type, int delta_score)
{

  int i = list->count;                 // move list index
  int pawn_bonus = 0;

  // add move to list
  list->mv[i].m.b.from = fsq;
  list->mv[i].m.b.to = tsq;
  list->mv[i].m.b.type = type;
  list->mv[i].m.b.promote = 0;

  // Give an initial score to the move below delta_score
  // so that by default it will not be put in the list 
  // unless a step below changes the score
  list->mv[i].score = delta_score-100;  
  
  // give a bonus for a promotion
  if((tsq > 55 || tsq < 8) && sq[fsq].type == PAWN)
  {
    list->mv[i].m.b.type |= PROMOTE; 
    list->mv[i].m.b.promote = QUEEN;
    //if(swap(tsq,(*this),wtm,fsq) >= 0) list->mv[i].score += 1000;
    list->mv[i].score += 1000;
  }

  // if the move is a capture or a promotion we
  // will score it to keep it in the movelist...
  if((type&CAPTURE) && list->mv[i].score < delta_score) {
    // If the move is a pawn capture
    //  give a bonus by the rank of the pawn
    if(sq[tsq].type == PAWN) {
      if(wtm && RANK(tsq) < 5) pawn_bonus = 15*(5-RANK(tsq)); 
      if(!wtm && RANK(tsq) > 2) pawn_bonus = 15*(RANK(tsq)-2);
    }
    // score move with the expected gain ....
    if(value[sq[tsq].type]+pawn_bonus >= delta_score) {
      list->mv[i].score = swap(tsq,(*this),wtm,fsq)+pawn_bonus;
      // looking for a revealed check if score isn't high enough
      if(list->mv[i].score < delta_score && slide_check_table[fsq][plist[wtm^1][KING][1]])  {
	position temp_pos = (*this);
	temp_pos.wtm ^= 1;
	temp_pos.sq[tsq] = temp_pos.sq[fsq];
	temp_pos.sq[fsq] = empty;
	if(temp_pos.simple_check(fsq)) list->mv[i].score = value[sq[tsq].type]+pawn_bonus;
      }
    }
  } 

  // if the score won't keep the move in the list, see if it might be a check
  if(list->mv[i].score < delta_score && !(type&CAPTURE) && !(type&PROMOTE)) {
    int ptype = sq[fsq].type; 
    int ksq = plist[wtm^1][KING][1];
    int ocheck = 0;
    int csq;

    if((ptype == BISHOP || ptype == QUEEN)
       && bishop_check_table[tsq][ksq]) {
      ocheck = dia_slide_attack(tsq,ksq);
    } else if((ptype == ROOK || ptype == QUEEN)
         && rook_check_table[tsq][ksq])
      ocheck = hor_slide_attack(tsq,ksq);
    else if(ptype == PAWN && bishop_check_table[tsq][ksq]) {
      // note the bishop_check_table guarantees that the pawn/king
      //   relationships below are OK... otherwise we would have to
      //   restrict the files for these comparisons
      if(!wtm && (tsq == ksq+9 || tsq == ksq+7))
	{ ocheck = 1; }
      else if(wtm && (tsq == ksq-9 || tsq == ksq-7))
	{ ocheck = 1; }
    } 
    else if(ptype == KNIGHT && knight_check_table[tsq][ksq]) ocheck = 1;    

    if(ocheck) { 
      //list->mv[i].score += delta_score + swap(tsq,(*this),wtm,fsq);
      list->mv[i].score = delta_score + swap(tsq,(*this),wtm,fsq);
    } else {
      // looking for a revealed check
      if(slide_check_table[fsq][ksq])  {
	position temp_pos = (*this);
	temp_pos.wtm ^= 1;
	temp_pos.sq[tsq] = temp_pos.sq[fsq];
	temp_pos.sq[fsq] = empty;
	ocheck = temp_pos.simple_check(fsq);
	if(ocheck) list->mv[i].score = delta_score; // + 100;
      }
    }
   
  }   

  if(list->mv[i].score >= delta_score) {
    list->count++;           // increase list count
  }

}

/*------------------------------- Bishop Moves --------------------------*/
void position::bishop_cc(move_list *list, int sqr, int ds)
{
  int mm, nn, ii, tsq;

  mm = FILE(sqr); nn = RANK(sqr);

  ii = 1;
  while (mm + ii <= 7 && nn + ii <= 7)
  {
   tsq = SQR((mm+ii),(nn+ii));
   if (!sq[tsq].type)
     { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

  ii = 1;
  while (mm - ii >= 0 && nn + ii <= 7)
  {
   tsq = SQR((mm-ii),(nn+ii));
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

  ii = 1;
  while(mm + ii <= 7 && nn - ii >= 0)
  {
   tsq = SQR((mm+ii),(nn-ii));
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

  ii = 1;
  while (mm - ii >= 0 && nn - ii >= 0)
  {
   tsq = SQR((mm-ii),(nn-ii));
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

}

/*--------------------------- Rook Moves ---------------------------*/

void position::rook_cc(move_list *list, int sqr, int ds)
{
  int mm, nn, ii, tsq;

  mm = FILE(sqr); nn = RANK(sqr);

  ii = 1;
  while (mm + ii <= 7)
  {
   tsq = SQR((mm+ii),nn);
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

  ii = 1;
  while (mm - ii >= 0)
  {
   tsq = SQR((mm-ii),nn);
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

  ii = 1;
  while(nn - ii >= 0)
  {
   tsq = SQR(mm,(nn-ii));
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

  ii = 1;
  while (nn + ii <= 7)
  {
   tsq = SQR(mm,(nn+ii));
   if (!sq[tsq].type)
   { add_cc(sqr, tsq, list, 0, ds); }
   else if (sq[tsq].side != wtm)
   { add_cc(sqr, tsq, list, 1, ds); break; }
   else break;
   ii++;
  }

}

/*--------------------------- Knight Moves ----------------------------*/
void position::knight_cc(move_list *list, int sqr, int ds)
{
  int tsq;

  if(FILE(sqr) < 6 && RANK(sqr) < 7) {
   tsq = sqr + 10;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) < 6 && RANK(sqr)) {
   tsq = sqr - 6;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) > 1 && RANK(sqr) < 7) {
   tsq = sqr + 6;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) > 1 && RANK(sqr)) {
   tsq = sqr - 10;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr) < 6) {
   tsq = sqr + 17;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) && RANK(sqr) < 6) {
   tsq = sqr + 15;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr) > 1) {
   tsq = sqr - 15;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) && RANK(sqr) > 1) {
   tsq = sqr - 17;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }

}

/*--------------------------- King Moves ----------------------------*/
void position::king_cc(move_list *list, int sqr, int ds)
{
  int tsq;

  if(FILE(sqr) && RANK(sqr) < 7) {
   tsq = sqr + 7;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(RANK(sqr) < 7) {
   tsq = sqr + 8;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr) < 7) {
   tsq = sqr + 9;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr)) {
   tsq = sqr - 1;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) < 7) {
   tsq = sqr + 1;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) < 7 && RANK(sqr)) {
   tsq = sqr - 7;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(RANK(sqr)) {
   tsq = sqr - 8;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }
  if(FILE(sqr) && RANK(sqr)) {
   tsq = sqr - 9;
   if(sq[tsq].side != wtm) {
     if(sq[tsq].type) add_cc(sqr, tsq, list, 1, ds);
     else add_cc(sqr, tsq, list, 0, ds);
    }
  }

}

/*------------------------ Pawn Moves ------------------------------*/
void position::pawn_cc(move_list *list, int sqr, int ds)
{

  if(wtm) {                        // if it is white's pawn
   if(!sq[sqr+8].type) {
     add_cc(sqr, sqr+8, list, 16, ds);
     if(RANK(sqr) == 1) {
       if(!sq[sqr+16].type) { add_cc(sqr, sqr+16, list, 8|16, ds); }
      }
   }
   if(FILE(sqr)) {
    if (sq[sqr+7].type && sq[sqr+7].side != wtm)
      { add_cc(sqr, sqr+7, list, (1|PAWN_PUSH), ds); }
    else if((sqr+7) == ep && ep)
      { add_cc(sqr, sqr+7, list, (4|1|PAWN_PUSH), ds); }
   }
   if(FILE(sqr) < 7) {
    if (sq[sqr+9].type && sq[sqr+9].side != wtm)
      { add_cc(sqr, sqr+9, list, (1|PAWN_PUSH), ds); }
    else if((sqr+9) == ep && ep)
      { add_cc(sqr, sqr+9, list, (4|1|PAWN_PUSH), ds); }
   }
  } else {                           // or if it is black's pawn
   if(!sq[sqr-8].type) {
     add_cc(sqr, sqr-8, list, 16, ds);
     if(RANK(sqr) == 6) {
       if(!sq[sqr-16].type) { add_cc(sqr, sqr-16, list, 8|16, ds); }
      }
   }
   if(FILE(sqr)) {
    if (sq[sqr-9].type && sq[sqr-9].side != wtm)
      { add_cc(sqr, sqr-9, list, (1|PAWN_PUSH), ds); }
    else if((sqr-9) == ep && ep)
      { add_cc(sqr, sqr-9, list, (4|1|PAWN_PUSH), ds); }
   }
   if(FILE(sqr) < 7) {
    if (sq[sqr-7].type && sq[sqr-7].side != wtm)
      { add_cc(sqr, sqr-7, list, (1|PAWN_PUSH), ds); }
    else if((sqr-7) == ep && ep)
      { add_cc(sqr, sqr-7, list, (4|1|PAWN_PUSH), ds); }
   }
  }
}
























