/* This file is part of Cassandre.
   Copyright (c) 2003 Romang Jean-Franois, Adoplh Thomas, Grundrich Raphael

   Cassandre is based on the DessChess program, a student project relised at
   University Louis Pasteur in Strasbourg, France ; under the direction of
   professor J.Korczak.

   Cassandre is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Cassandre is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Cassandre; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info: 
     jeff@proxone.net
*/

#ifndef CASSANDRE_POSITION_H
#define CASSANDRE_POSITION_H 1

#include "bitboardtoolkit.h"
#include "move.h"
#include <cstdio>
#include <cstring>
#include <iostream>
#include <queue>
#include <stack>

/* masks for postion 'flags' attribute */
#define whiteLongCastleMask 0x100
#define whiteShortCastleMask 0x80
#define blackLongCastleMask 0x40
#define blackShortCastleMask 0x20
#define playerMask 0x10 /* 0:black 1:white */
#define enpassantMask 0x0F

#define movementMask 63

#define EMPTY		0
#define W_PAWN		1
#define W_KNIGHT	2
#define W_BISHOP	3
#define W_ROOK		4
#define W_QUEEN		5
#define W_KING		6
#define B_PAWN		7
#define B_KNIGHT	8
#define B_BISHOP	9
#define B_ROOK		10
#define B_QUEEN		11
#define B_KING		12

/**
 * This class represents a position during the game.
 * The main information stored here is the location of pieces,
 * but a position contains also casteling states, the side to play, etc...
 * The pieces are stored with a special structure called bitboards.
 */
class Position
{
    public:
        Position(char *input)
        {
            bitMask=BitboardToolkit::instance()->bitMask;
            pawnAdvances.push(0);
            set(input);
        }
    
        bitboard       w_occupied;
        bitboard       b_occupied;
        bitboard       occupied_rl90;
        bitboard       occupied_a8h1;
        bitboard       occupied_a1h8;
        bitboard       w_pawn;
        bitboard       w_knight;
        bitboard       w_bishop;
        bitboard       w_rook;
        bitboard       w_queen;
        bitboard       b_pawn;
        bitboard       b_knight;
        bitboard       b_bishop;
        bitboard       b_rook;
        bitboard       b_queen;
        unsigned int 	 flags; 
        
        int		hash_key;
        bitboard	hash_checksum;
        
        //bitboard       hash_key;
        //unsigned int   pawn_hash_key;
        
        
        

        const static unsigned char rl90Index[64];
        
        void doMove(Move m);
        void undoMove(Move m);
        void set(char *input); //notation FEN
        bool empty(unsigned char location);
        unsigned char pieceAt(unsigned char location);
        void displayChessboard();
        void displayBitboards();
        void insertPieceInRotatedBitboards(unsigned char location);
        void removePieceFromRotatedBitboards(unsigned char location);
        bool operator==(Position& p) const;
        unsigned char getEnPassantStatus();
        
    private:
        //bool lastMoveWasPawnAdvance;  //usefull for the movegenerator
        std::stack<unsigned char> pawnAdvances; //push 0 if no pawn advance
    
        bitboard *bitMask;
        const static unsigned char a8h1_map[64];
        const static unsigned char a1h8_map[64];
    
        void addPiece(unsigned char piece,unsigned char location);
        void deletePiece(unsigned char location); //(!) not used
        void deleteBlackPiece(unsigned char location);
        void deleteWhitePiece(unsigned char location);
        void set(unsigned char *array);
};

inline void Position::insertPieceInRotatedBitboards(unsigned char location)
{
    occupied_rl90=occupied_rl90|bitMask[rl90Index[location]];  
    occupied_a8h1=occupied_a8h1|bitMask[a8h1_map[location]];
    occupied_a1h8=occupied_a1h8|bitMask[a1h8_map[location]];
}

inline void Position::removePieceFromRotatedBitboards(unsigned char location)
{
    occupied_rl90=occupied_rl90^bitMask[rl90Index[location]];   
    occupied_a8h1=occupied_a8h1^bitMask[a8h1_map[location]];
    occupied_a1h8=occupied_a1h8^bitMask[a1h8_map[location]];
}

inline bool Position::operator==(Position& p) const
{
    return(	this->w_occupied==p.w_occupied &&
                this->b_occupied==p.b_occupied &&
                this->occupied_rl90==p.occupied_rl90 &&
                this->occupied_a8h1==p.occupied_a8h1 &&
                this->occupied_a1h8==p.occupied_a1h8 &&
                this->w_pawn==p.w_pawn &&
                this->w_knight==p.w_knight &&
                this->w_bishop==p.w_bishop &&
                this->w_rook==p.w_rook &&
                this->w_queen==p.w_queen &&
                this->b_pawn==p.b_pawn &&
                this->b_knight==p.b_knight &&
                this->b_bishop==p.b_bishop &&
                this->b_rook==p.b_rook &&
                this->b_queen==p.b_queen &&
                this->flags==p.flags );
}

inline void Position::deletePiece(unsigned char location)
{ 
    removePieceFromRotatedBitboards(location);    
    w_occupied=w_occupied&(~bitMask[location]);
    b_occupied=b_occupied&(~bitMask[location]);
    w_pawn=w_pawn&(~bitMask[location]);
    w_knight=w_knight&(~bitMask[location]);
    w_bishop=w_bishop&(~bitMask[location]);
    w_rook=w_rook&(~bitMask[location]);
    w_queen=w_queen&(~bitMask[location]);
    b_pawn=b_pawn&(~bitMask[location]);
    b_knight=b_knight&(~bitMask[location]);
    b_bishop=b_bishop&(~bitMask[location]);
    b_rook=b_rook&(~bitMask[location]);
    b_queen=b_queen&(~bitMask[location]);
}

inline void Position::deleteBlackPiece(unsigned char location)
{
    b_occupied=b_occupied^bitMask[location];
    b_pawn=b_pawn&(~bitMask[location]);
    b_knight=b_knight&(~bitMask[location]);
    b_bishop=b_bishop&(~bitMask[location]);
    b_rook=b_rook&(~bitMask[location]);
    b_queen=b_queen&(~bitMask[location]);
}

inline void Position::deleteWhitePiece(unsigned char location)
{
    w_occupied=w_occupied^bitMask[location];
    w_pawn=w_pawn&(~bitMask[location]);
    w_knight=w_knight&(~bitMask[location]);
    w_bishop=w_bishop&(~bitMask[location]);
    w_rook=w_rook&(~bitMask[location]);
    w_queen=w_queen&(~bitMask[location]);
}

inline bool Position::empty(unsigned char location)
{
    return(((w_occupied|b_occupied)&bitMask[location])==0);
}

inline unsigned char Position::getEnPassantStatus()
{
    return pawnAdvances.top();
}

#endif
