/* This file is part of Cassandre.
   Copyright (c) 2003 Romang Jean-Franois, Adoplh Thomas, Grundrich Raphael

   Cassandre is based on the DessChess program, a student project relised at
   University Louis Pasteur in Strasbourg, France ; under the direction of
   professor J.Korczak.

   Cassandre is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Cassandre is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Cassandre; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
     jeff@proxone.net
*/

#include "position.h"
#include <assert.h>

const unsigned char Position::rl90Index[64] = {
    56, 48, 40, 32, 24, 16, 8, 0,
    57, 49, 41, 33, 25, 17, 9, 1,
    58, 50, 42, 34, 26, 18, 10, 2,
    59, 51, 43, 35, 27, 19, 11, 3,
    60, 52, 44, 36, 28, 20, 12, 4,
    61, 53, 45, 37, 29, 21, 13, 5,
    62, 54, 46, 38, 30, 22, 14, 6,
    63, 55, 47, 39, 31, 23, 15, 7
};

const unsigned char Position::a8h1_map[64] = {
    15, 22, 29, 36, 43, 50, 57, 0,
    23, 30, 37, 44, 51, 58, 1, 8,
    31, 38, 45, 52, 59, 2, 9, 16,
    39, 46, 53, 60, 3, 10, 17, 24,
    47, 54, 61, 4, 11, 18, 25, 32,
    55, 62, 5, 12, 19, 26, 33, 40,
    63, 6, 13, 20, 27, 34, 41, 48,
    7, 14, 21, 28, 35, 42, 49, 56
};

const unsigned char Position::a1h8_map[64] = {
    56, 49, 42, 35, 28, 21, 14, 7,
    0, 57, 50, 43, 36, 29, 22, 15,
    8, 1, 58, 51, 44, 37, 30, 23,
    16, 9, 2, 59, 52, 45, 38, 31,
    24, 17, 10, 3, 60, 53, 46, 39,
    32, 25, 18, 11, 4, 61, 54, 47,
    40, 33, 26, 19, 12, 5, 62, 55,
    48, 41, 34, 27, 20, 13, 6, 63
};

void            Position::set(char *input)
{
    int             i,
                    c,
                    l;
    unsigned char   board[64];
    l = strlen(input);

    /* board */
    for (i = 0; i < 64; i++)
        board[i] = 0;
    i = c = 0;
    while (c < 64) {
        switch (input[i]) {
        case 'P':
            board[c++] = W_PAWN;
            break;
        case 'N':
            board[c++] = W_KNIGHT;
            break;
        case 'B':
            board[c++] = W_BISHOP;
            break;
        case 'R':
            board[c++] = W_ROOK;
            break;
        case 'Q':
            board[c++] = W_QUEEN;
            break;
        case 'K':
            board[c++] = W_KING;
            break;
        case 'p':
            board[c++] = B_PAWN;
            break;
        case 'n':
            board[c++] = B_KNIGHT;
            break;
        case 'b':
            board[c++] = B_BISHOP;
            break;
        case 'r':
            board[c++] = B_ROOK;
            break;
        case 'q':
            board[c++] = B_QUEEN;
            break;
        case 'k':
            board[c++] = B_KING;
            break;
        case '1':
            c++;
            break;
        case '2':
            c += 2;
            break;
        case '3':
            c += 3;
            break;
        case '4':
            c += 4;
            break;
        case '5':
            c += 5;
            break;
        case '6':
            c += 6;
            break;
        case '7':
            c += 7;
            break;
        case '8':
            c += 8;
            break;
        case '/':
            if (c % 8)
                c += (8 - (c % 8));
            break;
        default:
            printf("ERROR : Unknown value in setPositionWithString");
        }
        i++;
    }
    set(board);

    /* side on move */
    flags = 0;
    while (input[i] != 'b' && input[i] != 'w')
        i++;
    if (input[i] == 'w')
        flags = flags | playerMask;
    i++;

    /* castle and enpassant */
    while (i < l) {
        switch (input[i]) {
        case 'K':
            flags = flags | whiteShortCastleMask;
            break;
        case 'Q':
            flags = flags | whiteLongCastleMask;
            break;
        case 'k':
            flags = flags | blackShortCastleMask;
            break;
        case 'q':
            flags = flags | blackLongCastleMask;
            break;
        case 'a':
            flags = flags | 1;
            break;
        case 'b':
            flags = flags | 2;
            break;
        case 'c':
            flags = flags | 3;
            break;
        case 'd':
            flags = flags | 4;
            break;
        case 'e':
            flags = flags | 5;
            break;
        case 'f':
            flags = flags | 6;
            break;
        case 'g':
            flags = flags | 7;
            break;
        case 'h':
            flags = flags | 8;
            break;
        default:
            break;
        }
        i++;
    }
}

void            Position::set(unsigned char *array)
{
    int             i,
                    bitNumber;

    w_occupied = b_occupied = occupied_rl90
        = w_pawn = occupied_a1h8 = occupied_a8h1
        = w_knight = w_bishop = w_rook = w_queen = b_pawn
        = b_knight = b_bishop = b_rook = b_queen = 0;

    for (i = 0; i < 64; i++) {
        bitNumber = 7 - (i % 8) + (i / 8) * 8;
        addPiece(array[i], bitNumber);
    }
}

unsigned char   Position::pieceAt(unsigned char location)
{
	if (!(bitMask[location] & (b_occupied | w_occupied))) return (EMPTY);
    if (w_occupied & bitMask[location]) {
        if (w_pawn & bitMask[location])
            return (W_PAWN);
        else if (w_rook & bitMask[location])
            return (W_ROOK);
        else if (w_bishop & bitMask[location])
            return (W_BISHOP);
        else if (w_knight & bitMask[location])
            return (W_KNIGHT);
        else if (w_queen & bitMask[location])
            return (W_QUEEN);
        else
            return (W_KING);
    } else {
        if (b_pawn & bitMask[location])
            return (B_PAWN);
        else if (b_rook & bitMask[location])
            return (B_ROOK);
        else if (b_bishop & bitMask[location])
            return (B_BISHOP);
        else if (b_knight & bitMask[location])
            return (B_KNIGHT);
        else if (b_queen & bitMask[location])
            return (B_QUEEN);
        else
            return (B_KING);
    } 
    assert(false);
}

void            Position::displayChessboard()
{
    int             l,
                    c;
    for (l = 0; l < 8; l++) {
        printf("\n   +---+---+---+---+---+---+---+---+\n");
        printf(" %i |", 8 - l);
        for (c = 0; c < 8; c++) {
            switch (pieceAt(l * 8 + (7 - c))) {
            case W_PAWN:
                printf(" P |");
                break;
            case W_KNIGHT:
                printf(" N |");
                break;
            case W_BISHOP:
                printf(" B |");
                break;
            case W_ROOK:
                printf(" R |");
                break;
            case W_QUEEN:
                printf(" Q |");
                break;
            case W_KING:
                printf(" K |");
                break;
            case B_PAWN:
                printf(" *P|");
                break;
            case B_KNIGHT:
                printf(" *N|");
                break;
            case B_BISHOP:
                printf(" *B|");
                break;
            case B_ROOK:
                printf(" *R|");
                break;
            case B_QUEEN:
                printf(" *Q|");
                break;
            case B_KING:
                printf(" *K|");
                break;
            case EMPTY:
                printf("   |");
                break;
            default:
                break;
            }
        }
    }
    printf("\n   +---+---+---+---+---+---+---+---+\n");
    printf(".    a   b   c   d   e   f   g   h\n");

    std::cout << "Roques:";
    if (flags & whiteShortCastleMask)
        std::cout << "K";
    if (flags & whiteLongCastleMask)
        std::cout << "Q";
    if (flags & blackShortCastleMask)
        std::cout << "k";
    if (flags & blackLongCastleMask)
        std::cout << "q";
    std::cout << std::endl;
}

void            Position::displayBitboards()
{
    printf("POSITION:\nw_occupied\n");
    BitboardToolkit::displayBitboard(w_occupied);
    printf("b_occupied\n");
    BitboardToolkit::displayBitboard(b_occupied);
    printf("occupied_rl90\n");
    BitboardToolkit::displayBitboard(occupied_rl90);
    printf("occupied_a8h1\n");
    BitboardToolkit::displayBitboard(occupied_a8h1);
    printf("occupied_a1h8\n");
    BitboardToolkit::displayBitboard(occupied_a1h8);
    printf("w_pawn\n");
    BitboardToolkit::displayBitboard(w_pawn);
    printf("w_knight\n");
    BitboardToolkit::displayBitboard(w_knight);
    printf("w_bishop\n");
    BitboardToolkit::displayBitboard(w_bishop);
    printf("w_rook\n");
    BitboardToolkit::displayBitboard(w_rook);
    printf("w_queen\n");
    BitboardToolkit::displayBitboard(w_queen);
    printf("b_pawn\n");
    BitboardToolkit::displayBitboard(b_pawn);
    printf("b_knight\n");
    BitboardToolkit::displayBitboard(b_knight);
    printf("b_bishop\n");
    BitboardToolkit::displayBitboard(b_bishop);
    printf("b_rook\n");
    BitboardToolkit::displayBitboard(b_rook);
    printf("b_queen\n");
    BitboardToolkit::displayBitboard(b_queen);
    printf("flags %c\nEND POSITION\n", flags);
}

void            Position::doMove(Move m)
{
    unsigned char   to = m.to();
    unsigned char   from = m.from();

    //lastMoveWasPawnAdvance=false;
    pawnAdvances.push(0);

    /* mise a jour des flags des roques */

    if (m.breaksWhiteKingCastle())
        flags = flags & (~whiteShortCastleMask);
    if (m.breaksWhiteQueenCastle())
        flags = flags & (~whiteLongCastleMask);
    if (m.breaksBlackKingCastle())
        flags = flags & (~blackShortCastleMask);
    if (m.breaksBlackQueenCastle())
        flags = flags & (~blackLongCastleMask);


    /* mouvements */
    switch (m.getMovingPiece()) {
    case W_PAWN:
        if (m.isPromotion()) {
            if (m.isCapture()) {
                w_occupied = w_occupied ^ bitMask[from];
                w_occupied = w_occupied | bitMask[to];
                w_pawn = w_pawn ^ bitMask[from];
                switch (m.promotionPiece()) {
                case W_KNIGHT:
                    w_knight = w_knight | bitMask[to];
                    break;
                case W_BISHOP:
                    w_bishop = w_bishop | bitMask[to];
                    break;
                case W_ROOK:
                    w_rook = w_rook | bitMask[to];
                    break;
                case W_QUEEN:
                    w_queen = w_queen | bitMask[to];
                    break;
                default:
                    std::cout << "Erreur promotionpiece";
                    break;
                }
                deleteBlackPiece(to);
                removePieceFromRotatedBitboards(from);
            } else {
                w_occupied = w_occupied ^ bitMask[from];
                w_occupied = w_occupied | bitMask[to];
                w_pawn = w_pawn ^ bitMask[from];
                switch (m.promotionPiece()) {
                case W_KNIGHT:
                    w_knight = w_knight | bitMask[to];
                    break;
                case W_BISHOP:
                    w_bishop = w_bishop | bitMask[to];
                    break;
                case W_ROOK:
                    w_rook = w_rook | bitMask[to];
                    break;
                case W_QUEEN:
                    w_queen = w_queen | bitMask[to];
                    break;
                default:
                    std::cout << "Erreur promotionpiece";
                    break;
                }
                removePieceFromRotatedBitboards(from);
                insertPieceInRotatedBitboards(to);
            }
        } else {
            /* mvt pion avec prise -> TODO : analyser capture en passant */
            if (m.isCapture()) {
                /* m.setCapturedPiece(pieceAt(to));  (!) se mefier en cas de
                 * prise en passant, ne pas utiliser to */
                w_occupied = w_occupied ^ bitMask[from];
                w_occupied = w_occupied | bitMask[to];
                w_pawn = w_pawn ^ bitMask[from];
                w_pawn = w_pawn | bitMask[to];
                if(!m.isEnPassantCapture())
                {
                    deleteBlackPiece(to);
                    removePieceFromRotatedBitboards(from);
                }
                else //en passant (necessite insert et autre remove)
                {
                    deleteBlackPiece(to+8);
                    removePieceFromRotatedBitboards(to+8);
                    removePieceFromRotatedBitboards(from);
                    insertPieceInRotatedBitboards(to);
                }
                                                         
            } else {            /* mvt pion normal sans prise */
                w_occupied = w_occupied ^ bitMask[from];
                w_occupied = w_occupied | bitMask[to];
                w_pawn = w_pawn ^ bitMask[from];
                w_pawn = w_pawn | bitMask[to];
                removePieceFromRotatedBitboards(from);
                insertPieceInRotatedBitboards(to);
                if ((from-to)==16) //2 cases, memoriser pour E.P.
                {
                    //lastMoveWasPawnAdvance=true;
                    pawnAdvances.pop();
                    pawnAdvances.push(from);
                }
            }
        }
        flags = flags ^ playerMask;
        break;

    case B_PAWN:
        if (m.isPromotion()) {
            if (m.isCapture()) {

                b_occupied = b_occupied ^ bitMask[from];
                b_occupied = b_occupied | bitMask[to];
                b_pawn = b_pawn ^ bitMask[from];
                switch (m.promotionPiece()) {
                case B_KNIGHT:
                    b_knight = b_knight | bitMask[to];
                    break;
                case B_BISHOP:
                    b_bishop = b_bishop | bitMask[to];
                    break;
                case B_ROOK:
                    b_rook = b_rook | bitMask[to];
                    break;
                case B_QUEEN:
                    b_queen = b_queen | bitMask[to];
                    break;
                default:
                    std::cout << "Erreur promotionpiece";
                    break;
                }
                deleteWhitePiece(to);
                removePieceFromRotatedBitboards(from);
            } else {
                b_occupied = b_occupied ^ bitMask[from];
                b_occupied = b_occupied | bitMask[to];
                b_pawn = b_pawn ^ bitMask[from];
                switch (m.promotionPiece()) {
                case B_KNIGHT:
                    b_knight = b_knight | bitMask[to];
                    break;
                case B_BISHOP:
                    b_bishop = b_bishop | bitMask[to];
                    break;
                case B_ROOK:
                    b_rook = b_rook | bitMask[to];
                    break;
                case B_QUEEN:
                    b_queen = b_queen | bitMask[to];
                    break;
                default:
                    std::cout << "Erreur promotionpiece";
                    break;
                }
                removePieceFromRotatedBitboards(from);
                insertPieceInRotatedBitboards(to);
            }
        } else {
            if (m.isCapture()) {/* mvt avec prise */
                /* m.setCapturedPiece(pieceAt(to)); (!) se mefier en cas de
                 * prise en passant, ne pas utiliser to */

                b_occupied = b_occupied ^ bitMask[from];
                b_occupied = b_occupied | bitMask[to];
                b_pawn = b_pawn ^ bitMask[from];
                b_pawn = b_pawn | bitMask[to];
                if(!m.isEnPassantCapture())
                {
                    deleteWhitePiece(to);
                    removePieceFromRotatedBitboards(from);
                }
                else //en passant (necessite insert et autre remove)
                {
                    deleteWhitePiece(to-8);
                    removePieceFromRotatedBitboards(to-8);
                    removePieceFromRotatedBitboards(from);
                    insertPieceInRotatedBitboards(to);
                }
            } else {            /* mvt pion normal sans prise */
                b_occupied = b_occupied ^ bitMask[from];
                b_occupied = b_occupied | bitMask[to];
                b_pawn = b_pawn ^ bitMask[from];
                b_pawn = b_pawn | bitMask[to];
                removePieceFromRotatedBitboards(from);
                insertPieceInRotatedBitboards(to);
                if((to-from)==16)//2 cases, memoriser pour E.P.
                {
                    //lastMoveWasPawnAdvance=true;
                    pawnAdvances.pop();
                    pawnAdvances.push(from);
                }
            }
        }
        flags = flags ^ playerMask;
        break;

    case W_KNIGHT:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_knight = w_knight ^ bitMask[from];
            w_knight = w_knight | bitMask[to];
            deleteBlackPiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_knight = w_knight ^ bitMask[from];
            w_knight = w_knight | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case B_KNIGHT:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_knight = b_knight ^ bitMask[from];
            b_knight = b_knight | bitMask[to];
            deleteWhitePiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_knight = b_knight ^ bitMask[from];
            b_knight = b_knight | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case W_BISHOP:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_bishop = w_bishop ^ bitMask[from];
            w_bishop = w_bishop | bitMask[to];
            deleteBlackPiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_bishop = w_bishop ^ bitMask[from];
            w_bishop = w_bishop | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case B_BISHOP:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_bishop = b_bishop ^ bitMask[from];
            b_bishop = b_bishop | bitMask[to];
            deleteWhitePiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_bishop = b_bishop ^ bitMask[from];
            b_bishop = b_bishop | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case W_ROOK:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_rook = w_rook ^ bitMask[from];
            w_rook = w_rook | bitMask[to];
            deleteBlackPiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_rook = w_rook ^ bitMask[from];
            w_rook = w_rook | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case B_ROOK:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_rook = b_rook ^ bitMask[from];
            b_rook = b_rook | bitMask[to];
            deleteWhitePiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_rook = b_rook ^ bitMask[from];
            b_rook = b_rook | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case W_QUEEN:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_queen = w_queen ^ bitMask[from];
            w_queen = w_queen | bitMask[to];
            deleteBlackPiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            w_queen = w_queen ^ bitMask[from];
            w_queen = w_queen | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case B_QUEEN:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_queen = b_queen ^ bitMask[from];
            b_queen = b_queen | bitMask[to];
            deleteWhitePiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            b_queen = b_queen ^ bitMask[from];
            b_queen = b_queen | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case W_KING:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            deleteBlackPiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt sans prise */
            if (from == 59) {   /* roque */
                if (to == 57) { /* petit */
                    w_occupied = w_occupied ^ bitMask[56];
                    w_occupied = w_occupied | bitMask[58];
                    w_rook = w_rook ^ bitMask[56];
                    w_rook = w_rook | bitMask[58];
                    removePieceFromRotatedBitboards(56);
                    insertPieceInRotatedBitboards(58);
                }
                if (to == 61) { /* grand */
                    w_occupied = w_occupied ^ bitMask[63];
                    w_occupied = w_occupied | bitMask[60];
                    w_rook = w_rook ^ bitMask[63];
                    w_rook = w_rook | bitMask[60];
                    removePieceFromRotatedBitboards(63);
                    insertPieceInRotatedBitboards(60);
                }
            }
            w_occupied = w_occupied ^ bitMask[from];
            w_occupied = w_occupied | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    case B_KING:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            deleteWhitePiece(to);
            removePieceFromRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            if (from == 3) {    /* roque */
                if (to == 1) {  /* petit */
                    b_occupied = b_occupied ^ bitMask[0];
                    b_occupied = b_occupied | bitMask[2];
                    b_rook = b_rook ^ bitMask[0];
                    b_rook = b_rook | bitMask[2];
                    removePieceFromRotatedBitboards(0);
                    insertPieceInRotatedBitboards(2);
                }
                if (to == 5) {  /* grand */
                    b_occupied = b_occupied ^ bitMask[7];
                    b_occupied = b_occupied | bitMask[4];
                    b_rook = b_rook ^ bitMask[7];
                    b_rook = b_rook | bitMask[4];
                    removePieceFromRotatedBitboards(7);
                    insertPieceInRotatedBitboards(4);
                }
            }
            b_occupied = b_occupied ^ bitMask[from];
            b_occupied = b_occupied | bitMask[to];
            removePieceFromRotatedBitboards(from);
            insertPieceInRotatedBitboards(to);
        }
        flags = flags ^ playerMask;
        break;

    default:
        printf("ERROR : Unknown value in doMove");
    }
}

void            Position::addPiece(unsigned char piece, unsigned char location)
{
    switch (piece) {
 case EMPTY:
        break;
    case W_PAWN:
        w_pawn = w_pawn | bitMask[location];
        w_occupied = w_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case W_KNIGHT:
        w_knight = w_knight | bitMask[location];
        w_occupied = w_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case W_BISHOP:
        w_bishop = w_bishop | bitMask[location];
        w_occupied = w_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case W_ROOK:
        w_rook = w_rook | bitMask[location];
        w_occupied = w_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case W_QUEEN:
        w_queen = w_queen | bitMask[location];
        w_occupied = w_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case W_KING:
        w_occupied = w_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case B_PAWN:
        b_pawn = b_pawn | bitMask[location];
        b_occupied = b_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case B_KNIGHT:
        b_knight = b_knight | bitMask[location];
        b_occupied = b_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case B_BISHOP:
        b_bishop = b_bishop | bitMask[location];
        b_occupied = b_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case B_ROOK:
        b_rook = b_rook | bitMask[location];
        b_occupied = b_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case B_QUEEN:
        b_queen = b_queen | bitMask[location];
        b_occupied = b_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    case B_KING:
        b_occupied = b_occupied | bitMask[location];
        insertPieceInRotatedBitboards(location);
        break;
    default:
        printf("ERROR : Unknown value in addPiece");
    }
}


void            Position::undoMove(Move m)
{
    unsigned char   to = m.to();
    unsigned char   from = m.from();
    
    pawnAdvances.pop();

    /* mise a jour des flags des roques */
    if (m.breaksWhiteKingCastle())
        flags = flags | whiteShortCastleMask;
    if (m.breaksWhiteQueenCastle())
        flags = flags | whiteLongCastleMask;
    if (m.breaksBlackKingCastle())
        flags = flags | blackShortCastleMask;
    if (m.breaksBlackQueenCastle())
        flags = flags | blackLongCastleMask;


    flags = flags ^ playerMask;


    switch (m.getMovingPiece()) {
    case W_PAWN:
        if (m.isPromotion()) {
            if (m.isCapture()) {
                w_occupied = w_occupied | bitMask[from];
                deleteWhitePiece(to);
                w_pawn = w_pawn | bitMask[from];
                addPiece(m.capturedPiece(), to);
                insertPieceInRotatedBitboards(from);
            } else {
                w_occupied = w_occupied | bitMask[from];
                deleteWhitePiece(to);
                w_pawn = w_pawn | bitMask[from];
                removePieceFromRotatedBitboards(to);
                insertPieceInRotatedBitboards(from);
            }
        } else {
            if (m.isCapture()) {/* mvt pion avec prise -> TODO : analyser
                                 * capture en passant */
                w_occupied = w_occupied ^ bitMask[to];
                w_occupied = w_occupied | bitMask[from];
                w_pawn = w_pawn ^ bitMask[to];
                w_pawn = w_pawn | bitMask[from];
                if(!m.isEnPassantCapture())
                {
                    addPiece(m.capturedPiece(), to);
                    insertPieceInRotatedBitboards(from);    
                }
                else  // prise en passant
                { 
                    addPiece(m.capturedPiece(), to+8);
                    insertPieceInRotatedBitboards(to+8);
                    insertPieceInRotatedBitboards(from);
                    removePieceFromRotatedBitboards(to);
                }
            } else {            /* mvt pion normal sans prise */
                w_occupied = w_occupied ^ bitMask[to];
                w_occupied = w_occupied | bitMask[from];
                w_pawn = w_pawn ^ bitMask[to];
                w_pawn = w_pawn | bitMask[from];
                removePieceFromRotatedBitboards(to);
                insertPieceInRotatedBitboards(from);
            }
        }
        break;

    case B_PAWN:
        if (m.isPromotion()) {
            if (m.isCapture()) {
                b_occupied = b_occupied | bitMask[from];
                deleteBlackPiece(to);
                b_pawn = b_pawn | bitMask[from];
                addPiece(m.capturedPiece(), to);
                insertPieceInRotatedBitboards(from);
            } else {
                b_occupied = b_occupied | bitMask[from];
                deleteBlackPiece(to);
                b_pawn = b_pawn | bitMask[from];
                removePieceFromRotatedBitboards(to);
                insertPieceInRotatedBitboards(from);
            }
        } else {
            if (m.isCapture()) {/* mvt pion avec prise -> TODO : analyser
                                 * capture en passant */
                b_occupied = b_occupied ^ bitMask[to];
                b_occupied = b_occupied | bitMask[from];
                b_pawn = b_pawn ^ bitMask[to];
                b_pawn = b_pawn | bitMask[from];
                if(!m.isEnPassantCapture())
                {
                    addPiece(m.capturedPiece(), to);
                    insertPieceInRotatedBitboards(from);
                }
                else  // prise en passant
                { 
                    addPiece(m.capturedPiece(), to-8);
                    insertPieceInRotatedBitboards(to-8);
                    insertPieceInRotatedBitboards(from);
                    removePieceFromRotatedBitboards(to);
                }
                  
            } else {            /* mvt pion normal sans prise */
                b_occupied = b_occupied ^ bitMask[to];
                b_occupied = b_occupied | bitMask[from];
                b_pawn = b_pawn ^ bitMask[to];
                b_pawn = b_pawn | bitMask[from];
                removePieceFromRotatedBitboards(to);
                insertPieceInRotatedBitboards(from);
            }
        }
        break;

    case W_KNIGHT:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_knight = w_knight ^ bitMask[to];
            w_knight = w_knight | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_knight = w_knight ^ bitMask[to];
            w_knight = w_knight | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case B_KNIGHT:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_knight = b_knight ^ bitMask[to];
            b_knight = b_knight | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_knight = b_knight ^ bitMask[to];
            b_knight = b_knight | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case W_BISHOP:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_bishop = w_bishop ^ bitMask[to];
            w_bishop = w_bishop | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_bishop = w_bishop ^ bitMask[to];
            w_bishop = w_bishop | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case B_BISHOP:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_bishop = b_bishop ^ bitMask[to];
            b_bishop = b_bishop | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_bishop = b_bishop ^ bitMask[to];
            b_bishop = b_bishop | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;


    case W_ROOK:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_rook = w_rook ^ bitMask[to];
            w_rook = w_rook | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_rook = w_rook ^ bitMask[to];
            w_rook = w_rook | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case B_ROOK:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_rook = b_rook ^ bitMask[to];
            b_rook = b_rook | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_rook = b_rook ^ bitMask[to];
            b_rook = b_rook | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case W_QUEEN:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_queen = w_queen ^ bitMask[to];
            w_queen = w_queen | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            w_queen = w_queen ^ bitMask[to];
            w_queen = w_queen | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case B_QUEEN:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_queen = b_queen ^ bitMask[to];
            b_queen = b_queen | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            b_queen = b_queen ^ bitMask[to];
            b_queen = b_queen | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case W_KING:
        if (m.isCapture()) {    /* mvt avec prise */
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            if (from == 59) {   /* roque */
                if (to == 57) { /* petit */
                    w_occupied = w_occupied ^ bitMask[58];
                    w_occupied = w_occupied | bitMask[56];
                    w_rook = w_rook ^ bitMask[58];
                    w_rook = w_rook | bitMask[56];
                    removePieceFromRotatedBitboards(58);
                    insertPieceInRotatedBitboards(56);
                }
                if (to == 61) { /* grand */
                    w_occupied = w_occupied ^ bitMask[60];
                    w_occupied = w_occupied | bitMask[63];
                    w_rook = w_rook ^ bitMask[60];
                    w_rook = w_rook | bitMask[63];
                    removePieceFromRotatedBitboards(60);
                    insertPieceInRotatedBitboards(63);
                }
            }
            w_occupied = w_occupied ^ bitMask[to];
            w_occupied = w_occupied | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    case B_KING:
        if (m.isCapture()) {    /* mvt avec prise */
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            addPiece(m.capturedPiece(), to);
            insertPieceInRotatedBitboards(from);
        } else {                /* mvt normal sans prise */
            if (from == 3) {    /* roque */
                if (to == 1) {  /* petit */
                    b_occupied = b_occupied ^ bitMask[2];
                    b_occupied = b_occupied | bitMask[0];
                    b_rook = b_rook ^ bitMask[2];
                    b_rook = b_rook | bitMask[0];
                    removePieceFromRotatedBitboards(2);
                    insertPieceInRotatedBitboards(0);
                }
                if (to == 5) {  /* grand */
                    b_occupied = b_occupied ^ bitMask[4];
                    b_occupied = b_occupied | bitMask[7];
                    b_rook = b_rook ^ bitMask[4];
                    b_rook = b_rook | bitMask[7];
                    removePieceFromRotatedBitboards(4);
                    insertPieceInRotatedBitboards(7);
                }
            }
            b_occupied = b_occupied ^ bitMask[to];
            b_occupied = b_occupied | bitMask[from];
            removePieceFromRotatedBitboards(to);
            insertPieceInRotatedBitboards(from);
        }
        break;

    default:
        printf("ERROR : Unknown value in undoMove");
    }
}
