/* This file is part of Cassandre.
   Copyright (c) 2003 Romang Jean-Franois, Adoplh Thomas, Grundrich Raphael

   Cassandre is based on the DessChess program, a student project relised at
   University Louis Pasteur in Strasbourg, France ; under the direction of
   professor J.Korczak.

   Cassandre is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Cassandre is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Cassandre; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info: 
     jeff@proxone.net
*/

#ifndef CASSANDRE_MOVEGENERATOR_H
#define CASSANDRE_MOVEGENERATOR_H 1

#include <vector>
#include "bitboardtoolkit.h"
#include "move.h"
#include "position.h"

/**
 * This class is used to compute all the moves according to chess rules
 * given a Position. 
 */
class MoveGenerator
{
    public:
        MoveGenerator()
        {
            bbtk=BitboardToolkit::instance();
            initBitMask();
            initializeAttackBoards();
        }
        
        std::vector<Move>* generateMoves(Position *p);
        std::vector<Move>* generateNonCaptureMoves(Position *p,std::vector<Move> *moveList=0);
        std::vector<Move>* generateCaptureMoves(Position *,std::vector<Move> *moveList=0);
        
        bitboard getAttacks(Position *p, bool color);
        
        bitboard knight_moves(unsigned char c);
        bitboard bishop_moves(unsigned char c, Position *p);
        bitboard rook_moves(unsigned char c, Position *p);
        bitboard queen_moves(unsigned char c, Position *p);
        bitboard king_moves(unsigned char c);
        
        bool isLegal(Position *p); //teste la couleur qui n'a pas le trait a son roi en echec
        
    private:
        static BitboardToolkit *bbtk;
        const static unsigned char A8H1_start[64];
        const static unsigned char H8A1_start[64];
        const static unsigned char length_H8A1_diag[64];
        const static unsigned char length_A8H1_diag[64];
        const static unsigned char A8H1_bitsToShift[64];
        const static unsigned char H8A1_bitsToShift[64];
        const static bitboard diagMask[9];
        bitboard bitMask[64]; //masques 1 bit
        bitboard	knight_attacks[64],
			rank_attacks[64][256],
			file_attacks[64][256],
			diag_A8H1_attacks[64][256],
			diag_H8A1_attacks[64][256], 
			king_attacks[64];
        
        void initBitMask();
        void initializeAttackBoards();
};

inline bitboard MoveGenerator::knight_moves(unsigned char c)
{
    return knight_attacks[c];
}

inline bitboard MoveGenerator::bishop_moves(unsigned char c, Position *p)
{
    int A8H1diag, H8A1diag;
    
    H8A1diag = (p->occupied_a1h8>>H8A1_bitsToShift[c])&diagMask[length_H8A1_diag[c]];
    A8H1diag = (p->occupied_a8h1>>A8H1_bitsToShift[c])&diagMask[length_A8H1_diag[c]];
    return(diag_A8H1_attacks[c][A8H1diag]|diag_H8A1_attacks[c][H8A1diag]);
}

inline bitboard MoveGenerator::rook_moves(unsigned char c, Position *p)
{
    int rank,file;
    rank=((p->w_occupied|p->b_occupied)>>(8*(c/8)))&BitboardToolkit::FFMask;
    file=(p->occupied_rl90>>(8*(Position::rl90Index[c]/8)))&BitboardToolkit::FFMask;
    return(rank_attacks[c][rank]|file_attacks[c][file]);
}

inline bitboard MoveGenerator::queen_moves(unsigned char c, Position *p)
{
    return(bishop_moves(c,p)|rook_moves(c,p));
}

inline bitboard MoveGenerator::king_moves(unsigned char c)
{
    return king_attacks[c];
}

#endif
