/* This file is part of Cassandre.
   Copyright (c) 2003 Romang Jean-Franois, Adoplh Thomas, Grundrich Raphael

   Cassandre is based on the DessChess program, a student project relised at
   University Louis Pasteur in Strasbourg, France ; under the direction of
   professor J.Korczak.

   Cassandre is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Cassandre is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Cassandre; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
     jeff@proxone.net
*/

#include "movegenerator.h"
#include <iostream>
#include <cstring>
#include <cmath>


BitboardToolkit *MoveGenerator::bbtk = 0;

const bitboard  MoveGenerator::diagMask[9] = {0, 1, 3, 7, 15, 31, 63, 127, 255};

const unsigned char MoveGenerator::A8H1_start[64] = {
    0, 1, 2, 3, 4, 5, 6, 7,
    1, 2, 3, 4, 5, 6, 7, 15,
    2, 3, 4, 5, 6, 7, 15, 23,
    3, 4, 5, 6, 7, 15, 23, 31,
    4, 5, 6, 7, 15, 23, 31, 39,
    5, 6, 7, 15, 23, 31, 39, 47,
    6, 7, 15, 23, 31, 39, 47, 55,
    7, 15, 23, 31, 39, 47, 55, 63
};

const unsigned char MoveGenerator::H8A1_start[64] = {
    0, 1, 2, 3, 4, 5, 6, 7,
    8, 0, 1, 2, 3, 4, 5, 6,
    16, 8, 0, 1, 2, 3, 4, 5,
    24, 16, 8, 0, 1, 2, 3, 4,
    32, 24, 16, 8, 0, 1, 2, 3,
    40, 32, 24, 16, 8, 0, 1, 2,
    48, 40, 32, 24, 16, 8, 0, 1,
    56, 48, 40, 32, 24, 16, 8, 0
};

const unsigned char MoveGenerator::length_H8A1_diag[64] = {
    8, 7, 6, 5, 4, 3, 2, 1,
    7, 8, 7, 6, 5, 4, 3, 2,
    6, 7, 8, 7, 6, 5, 4, 3,
    5, 6, 7, 8, 7, 6, 5, 4,
    4, 5, 6, 7, 8, 7, 6, 5,
    3, 4, 5, 6, 7, 8, 7, 6,
    2, 3, 4, 5, 6, 7, 8, 7,
    1, 2, 3, 4, 5, 6, 7, 8
};

const unsigned char MoveGenerator::length_A8H1_diag[64] = {
    1, 2, 3, 4, 5, 6, 7, 8,
    2, 3, 4, 5, 6, 7, 8, 7,
    3, 4, 5, 6, 7, 8, 7, 6,
    4, 5, 6, 7, 8, 7, 6, 5,
    5, 6, 7, 8, 7, 6, 5, 4,
    6, 7, 8, 7, 6, 5, 4, 3,
    7, 8, 7, 6, 5, 4, 3, 2,
    8, 7, 6, 5, 4, 3, 2, 1
};

const unsigned char MoveGenerator::A8H1_bitsToShift[64] = {
    15, 22, 29, 36, 43, 50, 57, 0,
    22, 29, 36, 43, 50, 57, 0, 8,
    29, 36, 43, 50, 57, 0, 8, 16,
    36, 43, 50, 57, 0, 8, 16, 24,
    43, 50, 57, 0, 8, 16, 24, 32,
    50, 57, 0, 8, 16, 24, 32, 40,
    57, 0, 8, 16, 24, 32, 40, 48,
    0, 8, 16, 24, 32, 40, 48, 56
};

const unsigned char MoveGenerator::H8A1_bitsToShift[64] = {
    56, 49, 42, 35, 28, 21, 14, 7,
    0, 56, 49, 42, 35, 28, 21, 14,
    8, 0, 56, 49, 42, 35, 28, 21,
    16, 8, 0, 56, 49, 42, 35, 28,
    24, 16, 8, 0, 56, 49, 42, 35,
    32, 24, 16, 8, 0, 56, 49, 42,
    40, 32, 24, 16, 8, 0, 56, 49,
    48, 40, 32, 24, 16, 8, 0, 56
};

/*
 *	Attaques
 */

bool MoveGenerator::isLegal(Position * p)
{
    bitboard        attacks;
    if (p->flags & playerMask)
    {/* white plays */
        attacks = getAttacks(p, true);
        if ((p->b_occupied ^ p->b_pawn ^ p->b_knight ^ p->b_bishop ^ p->b_rook ^ p->b_queen) & attacks)
            return false;
        else
            return true;
    } else
    {                    /* black plays */
        attacks = getAttacks(p, false);
        if ((p->w_occupied ^ p->w_pawn ^ p->w_knight ^ p->w_bishop ^ p->w_rook ^ p->w_queen) & attacks)
            return false;
        else
            return true;
    }
}



bitboard        MoveGenerator::getAttacks(Position * p, bool color)
{
    bitboard        attacks = 0;
    bitboard my_chessmen, one;
    one = 1;
	unsigned char   c;
    BitboardToolkit *t=BitboardToolkit::instance();

    if (color) {// white plays
		my_chessmen = p->w_occupied;
		while (c = t->firstOne(my_chessmen), c < 64) {
            my_chessmen &= ~(one << c);
            switch (p->pieceAt(c)) {
            case W_PAWN:
                // a droite
                if ((c % 8) < 7 )
                    attacks = attacks | bitMask[c - 7];
                // a gauche 
                if ((c % 8) > 0 )
                    attacks = attacks | bitMask[c - 9];
                break;
            case W_KNIGHT:
                attacks = attacks | (knight_moves(c) );
                break;
            case W_BISHOP:
                attacks = attacks | (bishop_moves(c, p) );
                break;
            case W_ROOK:
                attacks = attacks | (rook_moves(c, p) );
                break;
            case W_QUEEN:
                attacks = attacks | (queen_moves(c, p));
                break;
            case W_KING:
                attacks = attacks | (king_moves(c));
                break;
            default:
                break;
            }
        }
    } else {                    // black attacks 
		my_chessmen = p->b_occupied;
		while (c = t->firstOne(my_chessmen), c < 64) {
            my_chessmen &= ~(one << c);
            switch (p->pieceAt(c)) {
            case B_PAWN:
                // a droite
                if ((c % 8) < 7 )
                    attacks = attacks | bitMask[c + 9];
                // a gauche
                if ((c % 8) > 0 )
                    attacks = attacks | bitMask[c + 7];
                break;
            case B_KNIGHT:
                attacks = attacks | (knight_moves(c) );
                break;
            case B_BISHOP:
                attacks = attacks | (bishop_moves(c, p) );
                break;
            case B_ROOK:
                attacks = attacks | (rook_moves(c, p) );
                break;
            case B_QUEEN:
                attacks = attacks | (queen_moves(c, p) );
                break;
            case B_KING:
                attacks = attacks | (king_moves(c) );
                break;
            default:
                break;
            }
        }
    }
    return (attacks);
}


/*
 *	Generation de mouvements
 */


void            MoveGenerator::initializeAttackBoards()
{

    int             i,
                    c,
                    col,
                    lin,
                    avance,
                    s,
                    t;
    bitboard        o,
                    occupied;
    /* initialize knight attack board */
    /* initialize king attack board  */
    for (i = 0; i < 64; i++) {
        knight_attacks[i] = 0;
         king_attacks[i] = 0;
        c = i % 8;
        if (i > 15 && c > 0)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i - 17]);
        if (i > 15 && c < 7)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i - 15]);
        if (i > 7 && c < 6)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i - 6]);
        if (i > 7 && c > 1)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i - 10]);
        if (i < 56 && c < 6)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i + 10]);
        if (i < 56 && c > 1)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i + 6]);
        if (i < 48 && c < 7)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i + 17]);
        if (i < 48 && c > 0)
            knight_attacks[i] = knight_attacks[i] | (bitMask[i + 15]);
  
        if (i > 7 && c > 0)
            king_attacks[i] = king_attacks[i] | (bitMask[i - 9]);
        if (i > 7)
            king_attacks[i] = king_attacks[i] | (bitMask[i - 8]);
        if (i > 7 && c < 7)
            king_attacks[i] = king_attacks[i] | (bitMask[i - 7]);
        if (c < 7)
            king_attacks[i] = king_attacks[i] | (bitMask[i + 1]);
        if (i < 56 && c < 7)
            king_attacks[i] = king_attacks[i] | (bitMask[i + 9]);
        if (i < 56)
            king_attacks[i] = king_attacks[i] | (bitMask[i + 8]);
        if (i < 56 && c > 0)
            king_attacks[i] = king_attacks[i] | (bitMask[i + 7]);
        if (c > 0)
            king_attacks[i] = king_attacks[i] | (bitMask[i - 1]);
  
  

        
    }

    /* initialize rank_attacks */
    for (i = 0; i < 64; i++)
        for (o = 0; o < 256; o++) {
            rank_attacks[i][o] = 0;
            /* a gauche */
            col = i % 8;
            avance = 1;
            while (avance && col < 7) {
                col++;
                if (o & bitMask[col])
                    avance = 0; /* on regarde s'il y a une piece */
                rank_attacks[i][o] = rank_attacks[i][o] | bitMask[(i / 8) * 8 + col];
            }
            /* a droite */
            col = i % 8;
            avance = 1;
            while (avance && col > 0) {
                col--;
                if (o & bitMask[col])
                    avance = 0; /* on regarde s'il y a une piece */
                rank_attacks[i][o] = rank_attacks[i][o] | bitMask[(i / 8) * 8 + col];
            }
        }

    /* initialize file_attacks */
    for (i = 0; i < 64; i++)
        for (o = 0; o < 256; o++) {
            file_attacks[i][o] = 0;
            /* en haut */
            lin = i / 8;
            avance = 1;
            while (avance && lin > 0) {
                lin--;
                /* on regarde s'il y a une piece */
                if (o & bitMask[lin])
                    avance = 0;
                file_attacks[i][o] = file_attacks[i][o] | bitMask[i % 8 + lin * 8];
            }

            /* en bas */
            lin = i / 8;
            avance = 1;
            while (avance && lin < 7) {
                lin++;
                /* on regarde s'il y a une piece */
                if (o & bitMask[lin])
                    avance = 0;
                file_attacks[i][o] = file_attacks[i][o] | bitMask[i % 8 + lin * 8];
            }
        }

    /* initialize diag_A8H1_attacks */
    for (i = 0; i < 64; i++) {  /* every square (0-63)  */
        /* number of diagonal "states" will vary with length of diagonal */
        for (o = 0; o < powf(2, length_A8H1_diag[i]); o++) {     /* every diagonal state
                                                                 * (0-(2^diag length)-1)  */
            /* decide which squares are occupied */
            /* note that not all squares are used in some diagonals */
            occupied = 0;
            c = A8H1_start[i];
            for (s = 0; s <= length_A8H1_diag[i] - 1; s++) {
                if (o & bitMask[s])
                    occupied = occupied | bitMask[c];
                c += 7;
            }

            diag_A8H1_attacks[i][o] = 0;
            /* compute all moves on diagonal for this state up and left */
            t = i;
            avance = 1;
            while (avance && t > 7 && (t % 8) < 7) {
                t -= 7;
                /* on regarde s'il y a une piece */
                if (occupied & bitMask[t])
                    avance = 0;
                diag_A8H1_attacks[i][o] = diag_A8H1_attacks[i][o] | bitMask[t];
            }

            /* compute all moves on diagonal for this state down and right */

            t = i;
            avance = 1;
            while (avance && t < 56 && (t % 8) > 0) {
                t += 7;
                /* on regarde s'il y a une piece */
                if (occupied & bitMask[t])
                    avance = 0;
                diag_A8H1_attacks[i][o] = diag_A8H1_attacks[i][o] | bitMask[t];
            }
        }
    }

    /* initialize diag_H8A1_attacks */
    for (i = 0; i < 64; i++) {  /* every square (0-63)  */
        /* number of diagonal "states" will vary with length of diagonal */
        for (o = 0; o < powf(2, length_H8A1_diag[i]); o++) {     /* every diagonal state
                                                                 * (0-(2^diag length)-1)  */
            /* decide which squares are occupied */
            /* note that not all squares are used in some diagonals */
            occupied = 0;
            c = H8A1_start[i];
            for (s = 0; s <= length_H8A1_diag[i] - 1; s++) {
                if (o & bitMask[s])
                    occupied = occupied | bitMask[c];
                c += 9;
            }

            diag_H8A1_attacks[i][o] = 0;
            /* compute all moves on diagonal for this state up and right */
            t = i;
            avance = 1;
            while (avance && t > 7 && (t % 8) > 0) {
                t -= 9;
                /* on regarde s'il y a une piece */
                if (occupied & bitMask[t])
                    avance = 0;
                diag_H8A1_attacks[i][o] = diag_H8A1_attacks[i][o] | bitMask[t];
            }

            /* compute all moves on diagonal for this state down and left */
            t = i;
            avance = 1;
            while (avance && t < 56 && (t % 8) < 7) {
                t += 9;
                /* on regarde s'il y a une piece */
                if (occupied & bitMask[t])
                    avance = 0;
                diag_H8A1_attacks[i][o] = diag_H8A1_attacks[i][o] | bitMask[t];
            }
        }
    }
}

void            MoveGenerator::initBitMask()
{
    int             i;
    bitboard        x;
    /* bitMask initialisation */
    x = 1;
    for (i = 0; i < 64; i++) {
        bitMask[i] = x;
        x = x << 1;
    }
}


std::vector < Move > *MoveGenerator::generateMoves(Position * p)
{
    return (MoveGenerator::generateNonCaptureMoves(p, MoveGenerator::generateCaptureMoves(p)));
}

std::vector < Move > *MoveGenerator::generateNonCaptureMoves(Position * p, std::vector < Move > *moveList)
{
    unsigned char   c,
                    i;
    bitboard        possibleMoves,
                    attacks;
    bitboard my_chessmen, one;
    one = 1;
    BitboardToolkit *t=BitboardToolkit::instance();

    if (!moveList)
        moveList = new std::vector < Move >;

    if (p->flags & playerMask) {/* white plays */
		my_chessmen = p->w_occupied;
		while (c = t->firstOne(my_chessmen), c < 64) {
            my_chessmen &= ~(one << c);
            possibleMoves = 0;
            switch (p->pieceAt(c)) {
            case W_PAWN:
                /* avance normale */
                if (c > 15 && !(bitMask[c - 8] & (p->w_occupied | p->b_occupied))) {
                    moveList->insert(moveList->end(), ((W_PAWN << 28) | (c << 6) | (c - 8)));
                }
                /* deux cases */
                if ((c > 47)
                    && !(bitMask[c - 8] & (p->w_occupied | p->b_occupied))
                    && !(bitMask[c - 16] & (p->w_occupied | p->b_occupied))
                    ) {
                    moveList->insert(moveList->end(), ((W_PAWN << 28) | (c << 6) | (c - 16)));  /* avec enpassant  */
                }
                /* promotion sans capture */
                if ((c < 16) && !(bitMask[c - 8] & (p->w_occupied | p->b_occupied))) {
                    Move            m = (W_PAWN << 28) | (c << 6) | (c - 8);
                    m.setPromotion();
                    m.setPromotionPiece(W_QUEEN);
                    moveList->insert(moveList->end(), m);
                    m.setPromotionPiece(W_ROOK);
                    moveList->insert(moveList->end(), m);
                    m.setPromotionPiece(W_BISHOP);
                    moveList->insert(moveList->end(), m);
                    m.setPromotionPiece(W_KNIGHT);
                    moveList->insert(moveList->end(), m);
                }
                break;
            case W_KNIGHT:
                possibleMoves = knight_moves(c);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    moveList->insert(moveList->end(), ((c << 6) | i | (W_KNIGHT << 28)));
                }
                break;
            case W_BISHOP:
                possibleMoves = bishop_moves(c, p);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    moveList->insert(moveList->end(), ((c << 6) | i | (W_BISHOP << 28)));
                }
                break;
            case W_ROOK:
                possibleMoves = rook_moves(c, p);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    Move            m = (c << 6) | i | (W_ROOK << 28);
                    if (c == 56 && (p->flags & whiteShortCastleMask))
                        m.setBreaksWhiteKingCastle();
                    if (c == 63 && (p->flags & whiteLongCastleMask))
                        m.setBreaksWhiteQueenCastle();
                    moveList->insert(moveList->end(), m);
                }
                break;
            case W_QUEEN:
                possibleMoves = queen_moves(c, p);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    moveList->insert(moveList->end(), ((c << 6) | i | (W_QUEEN << 28)));
                }
                break;
            case W_KING:
                possibleMoves = king_moves(c);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                attacks = 0;

                if (c == 59 && (p->flags & whiteShortCastleMask) && p->empty(57) && p->empty(58)) {
                    attacks = getAttacks(p, false);
                    if (!((attacks & bitMask[57]) | (attacks & bitMask[58]) | (attacks & bitMask[59])))
                        possibleMoves = possibleMoves | bitMask[57];
                }
                if (c == 59 && (p->flags & whiteLongCastleMask) && p->empty(60) && p->empty(61) && p->empty(62)) {
                    if (!attacks)
                        attacks = getAttacks(p, false);
                    if (!((attacks & bitMask[59]) | (attacks & bitMask[60]) | (attacks & bitMask[61])))
                        possibleMoves = possibleMoves | bitMask[61];
                }
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    Move            m = (c << 6) | i | (W_KING << 28);
                    if (p->flags & whiteShortCastleMask)
                        m.setBreaksWhiteKingCastle();
                    if (p->flags & whiteLongCastleMask)
                        m.setBreaksWhiteQueenCastle();
                    moveList->insert(moveList->end(), m);
                }
                break;
            default:
                break;
            }
        }
    } else {                    /* black plays */
		my_chessmen = p->b_occupied;
		while (c = t->firstOne(my_chessmen), c < 64) {
            my_chessmen &= ~(one << c);
            possibleMoves = 0;
            switch (p->pieceAt(c)) {
            case B_PAWN:
                /* avance normale */
                if (c < 48 && !(bitMask[c + 8] & (p->w_occupied | p->b_occupied))) {
                    moveList->insert(moveList->end(), ((B_PAWN << 28) | (c << 6) | (c + 8)));
                }
                /* deux cases */
                if ((c < 16)
                    && !(bitMask[c + 8] & (p->w_occupied | p->b_occupied))
                    && !(bitMask[c + 16] & (p->w_occupied | p->b_occupied))
                    ) {
                    moveList->insert(moveList->end(), ((B_PAWN << 28) | (c << 6) | (c + 16)));  /* avec enpassant */
                }
                /* promotion sans capture */
                if ((c > 47) && !(bitMask[c + 8] & (p->w_occupied | p->b_occupied))) {
                    Move            m = (B_PAWN << 28) | (c << 6) | (c + 8);
                    m.setPromotion();
                    m.setPromotionPiece(B_QUEEN);
                    moveList->insert(moveList->end(), m);
                    m.setPromotionPiece(B_ROOK);
                    moveList->insert(moveList->end(), m);
                    m.setPromotionPiece(B_BISHOP);
                    moveList->insert(moveList->end(), m);
                    m.setPromotionPiece(B_KNIGHT);
                    moveList->insert(moveList->end(), m);

                }
                break;
            case B_KNIGHT:
                possibleMoves = knight_moves(c);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    moveList->insert(moveList->end(), ((c << 6) | i | (B_KNIGHT << 28)));
                }
                break;
            case B_BISHOP:
                possibleMoves = bishop_moves(c, p);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    moveList->insert(moveList->end(), ((c << 6) | i | (B_BISHOP << 28)));
                }
                break;
            case B_ROOK:
                possibleMoves = rook_moves(c, p);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    Move            m = (c << 6) | i | (B_ROOK << 28);
                    if (c == 0 && (p->flags & blackShortCastleMask))
                        m.setBreaksBlackKingCastle();
                    if (c == 7 && (p->flags & blackLongCastleMask))
                        m.setBreaksBlackQueenCastle();
                    moveList->insert(moveList->end(), m);
                }
                break;
            case B_QUEEN:
                possibleMoves = queen_moves(c, p);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    moveList->insert(moveList->end(), ((c << 6) | i | (B_QUEEN << 28)));
                }
                break;
            case B_KING:
                possibleMoves = king_moves(c);
                possibleMoves = possibleMoves & (possibleMoves ^ (p->w_occupied | p->b_occupied));
                attacks = 0;
                
                if (c == 3 && (p->flags & blackShortCastleMask) && p->empty(1) && p->empty(2)) {
                    attacks = getAttacks(p, true);
                    if (!((attacks & bitMask[1]) | (attacks & bitMask[2]) | (attacks & bitMask[3])))
                        possibleMoves = possibleMoves | bitMask[1];
                }
                if (c == 3 && (p->flags & blackLongCastleMask) && p->empty(4) && p->empty(5) && p->empty(6)) {
                    if (!attacks)
                        attacks = getAttacks(p, true);
                    if (!((attacks & bitMask[3]) | (attacks & bitMask[4]) | (attacks & bitMask[5])))
                        possibleMoves = possibleMoves | bitMask[5];
                    //else std::cout<<"bloque2"<<std::endl;
                }
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    Move            m = (c << 6) | i | (B_KING << 28);
                    if (p->flags & blackShortCastleMask)
                        m.setBreaksBlackKingCastle();
                    if (p->flags & blackLongCastleMask)
                        m.setBreaksBlackQueenCastle();
                    moveList->insert(moveList->end(), m);
                }
                break;
            default:
                break;
            }
        }
    }
    return moveList;
}

#define CLEARBIT(db, bn) db &= ~(1 << bn) 

std::vector < Move > *MoveGenerator::generateCaptureMoves(Position * p, std::vector < Move > *moveList)
{
    unsigned char   c,
                    i;
    bitboard        possibleMoves;
    if (!moveList)
        moveList = new std::vector < Move >;
    bitboard my_chessmen, one;
    one = 1;
    BitboardToolkit *t=BitboardToolkit::instance();

    if (p->flags & playerMask) {/* white plays */
		my_chessmen = p->w_occupied;
		while (c = t->firstOne(my_chessmen), c < 64) {
            my_chessmen &= ~(one << c);
            possibleMoves = 0;
            Move            m = 0;
            m.setCapture();

            switch (p->pieceAt(c)) {
            case W_PAWN:
                /* a droite */
                if (c > 15 && (c % 8) < 7 && (bitMask[c - 7] & p->b_occupied)) {
                    m.setMovingPiece(W_PAWN);
                    m.setCapturedPiece(p->pieceAt(c - 7));
                    moveList->insert(moveList->end(), (m | (c << 6) | (c - 7)));
                }
                /* a gauche */
                if (c > 15 && (c % 8) > 0 && (bitMask[c - 9] & p->b_occupied)) {
                    m.setMovingPiece(W_PAWN);
                    m.setCapturedPiece(p->pieceAt(c - 9));
                    moveList->insert(moveList->end(), (m | (c << 6) | (c - 9)));
                }
                /* promotion a droite */
                if (c < 16 && (c % 8) < 7 && (bitMask[c - 7] & p->b_occupied)) {
                    Move            mp = (m | (c << 6) | (c - 7));
                    mp.setMovingPiece(W_PAWN);
                    mp.setCapturedPiece(p->pieceAt(c - 7));
                    mp.setPromotion();
                    mp.setPromotionPiece(W_QUEEN);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(W_ROOK);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(W_BISHOP);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(W_KNIGHT);
                    moveList->insert(moveList->end(), mp);
                }
                /* promotion a gauche */
                if (c < 16 && (c % 8) > 0 && (bitMask[c - 9] & p->b_occupied)) {
                    Move            mp = (m | (c << 6) | (c - 9));
                    mp.setMovingPiece(W_PAWN);
                    mp.setCapturedPiece(p->pieceAt(c - 9));
                    mp.setPromotion();
                    mp.setPromotionPiece(W_QUEEN);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(W_ROOK);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(W_BISHOP);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(W_KNIGHT);
                    moveList->insert(moveList->end(), mp);
                }
                /* en passant */
                if(c<32 && c>16 && (p->getEnPassantStatus()!=0))
                {
                    unsigned char epPawn=p->getEnPassantStatus();
                    /* a droite */
                    if((c % 8)<7 && epPawn==(c-15))
                    {
                        m.setMovingPiece(W_PAWN);
                        m.setCapturedPiece(B_PAWN);
                        m.setEnPassantCapture();
                        moveList->insert(moveList->end(), (m | (c << 6) | (c - 7)));
                    }
                    /* a gauche */
                    if((c % 8)>0 && epPawn==(c-17))
                    {
                        m.setMovingPiece(W_PAWN);
                        m.setCapturedPiece(B_PAWN);
                        m.setEnPassantCapture();
                        moveList->insert(moveList->end(), (m | (c << 6) | (c - 9)));
                    }
                }
                break;
            case W_KNIGHT:
                possibleMoves = knight_moves(c) & p->b_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(W_KNIGHT);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case W_BISHOP:
                possibleMoves = bishop_moves(c, p) & p->b_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(W_BISHOP);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case W_ROOK:
                possibleMoves = rook_moves(c, p) & p->b_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    if (c == 56 && (p->flags & whiteShortCastleMask))
                        m.setBreaksWhiteKingCastle();
                    if (c == 63 && (p->flags & whiteLongCastleMask))
                        m.setBreaksWhiteQueenCastle();
                    m.setMovingPiece(W_ROOK);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case W_QUEEN:
                possibleMoves = queen_moves(c, p) & p->b_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(W_QUEEN);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }

                break;
            case W_KING:
                possibleMoves = king_moves(c) & p->b_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(W_KING);
                    m.setCapturedPiece(p->pieceAt(i));
                    if (p->flags & whiteShortCastleMask)
                        m.setBreaksWhiteKingCastle();
                    if (p->flags & whiteLongCastleMask)
                        m.setBreaksWhiteQueenCastle();
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            default:
                break;
            }
        }

        /* maj des roques noirs si on capture une des tours noires */
        if ((p->flags && blackShortCastleMask) || (p->flags && blackLongCastleMask)) {
            std::vector < Move > ::iterator iter;
            for (iter = moveList->begin(); iter < moveList->end(); iter++) {
                if ((*iter).to() == 0 && (p->flags & blackShortCastleMask))
                    (*iter).setBreaksBlackKingCastle();
                if ((*iter).to() == 7 && (p->flags & blackLongCastleMask))
                    (*iter).setBreaksBlackQueenCastle();
            }
        }
    } else {                    /* black plays */
		my_chessmen = p->b_occupied;
		while (c = t->firstOne(my_chessmen), c < 64){
            my_chessmen &= ~(one << c);
            Move            m = 0;
            m.setCapture();
            possibleMoves = 0;
            switch (p->pieceAt(c)) {
            case B_PAWN:
                /* a droite */
                if (c < 48 && (c % 8) < 7 && (bitMask[c + 9] & p->w_occupied)) {
                    m.setMovingPiece(B_PAWN);
                    m.setCapturedPiece(p->pieceAt(c + 9));
                    moveList->insert(moveList->end(), (m | (c << 6) | (c + 9)));
                }
                /* a gauche */
                if (c < 48 && (c % 8) > 0 && (bitMask[c + 7] & p->w_occupied)) {
                    m.setMovingPiece(B_PAWN);
                    m.setCapturedPiece(p->pieceAt(c + 7));
                    moveList->insert(moveList->end(), (m | (c << 6) | (c + 7)));
                }
                /* promotion a droite */
                if (c > 47 && (c % 8) < 7 && (bitMask[c + 9] & p->w_occupied)) {
                    Move            mp = (m | (c << 6) | (c + 9));
                    mp.setMovingPiece(B_PAWN);
                    mp.setCapturedPiece(p->pieceAt(c + 9));
                    mp.setPromotion();
                    mp.setPromotionPiece(B_QUEEN);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(B_ROOK);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(B_BISHOP);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(B_KNIGHT);
                    moveList->insert(moveList->end(), mp);
                }
                /* promotion a gauche */
                if (c > 47 && (c % 8) > 0 && (bitMask[c + 7] & p->w_occupied)) {
                    Move            mp = (m | (c << 6) | (c + 7));
                    mp.setMovingPiece(B_PAWN);
                    mp.setCapturedPiece(p->pieceAt(c + 7));
                    mp.setPromotion();
                    mp.setPromotionPiece(B_QUEEN);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(B_ROOK);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(B_BISHOP);
                    moveList->insert(moveList->end(), mp);
                    mp.setPromotionPiece(B_KNIGHT);
                    moveList->insert(moveList->end(), mp);
                }
                /* en passant */
                if(c>24 && c<40 && (p->getEnPassantStatus()!=0))
                {
                    unsigned char epPawn=p->getEnPassantStatus();
                    /* a droite */
                    if((c % 8)<7 && epPawn==(c+17))
                    {
                        m.setMovingPiece(B_PAWN);
                        m.setCapturedPiece(W_PAWN);
                        m.setEnPassantCapture();
                        moveList->insert(moveList->end(), (m | (c << 6) | (c + 9)));
                    }
                    /* a gauche */
                    if((c % 8)>0 && epPawn==(c+15))
                    {
                        m.setMovingPiece(B_PAWN);
                        m.setCapturedPiece(W_PAWN);
                        m.setEnPassantCapture();
                        moveList->insert(moveList->end(), (m | (c << 6) | (c + 7)));
                    }
                }
                break;
            case B_KNIGHT:
                possibleMoves = knight_moves(c) & p->w_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(B_KNIGHT);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case B_BISHOP:
                possibleMoves = bishop_moves(c, p) & p->w_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(B_BISHOP);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case B_ROOK:
                possibleMoves = rook_moves(c, p) & p->w_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    if (c == 0 && (p->flags & blackShortCastleMask))
                        m.setBreaksBlackKingCastle();
                    if (c == 7 && (p->flags & blackLongCastleMask))
                        m.setBreaksBlackQueenCastle();
                    m.setMovingPiece(B_ROOK);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case B_QUEEN:
                possibleMoves = queen_moves(c, p) & p->w_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(B_QUEEN);
                    m.setCapturedPiece(p->pieceAt(i));
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            case B_KING:
                possibleMoves = king_moves(c) & p->w_occupied;
                while (possibleMoves) {
                    i = bbtk->firstOne(possibleMoves);
                    possibleMoves = possibleMoves ^ bitMask[i];
                    m.setMovingPiece(B_KING);
                    m.setCapturedPiece(p->pieceAt(i));
                    if (p->flags & blackShortCastleMask)
                        m.setBreaksBlackKingCastle();
                    if (p->flags & blackLongCastleMask)
                        m.setBreaksBlackQueenCastle();
                    moveList->insert(moveList->end(), (m | (c << 6) | i));
                }
                break;
            default:
                break;
            }
        }

        /* maj des roques blancs si on capture une des tours blanches */
        if ((p->flags && whiteShortCastleMask) || (p->flags && whiteLongCastleMask)) {
            std::vector < Move > ::iterator iter;
            for (iter = moveList->begin(); iter < moveList->end(); iter++) {
                if ((*iter).to() == 56 && (p->flags & whiteShortCastleMask))
                    (*iter).setBreaksWhiteKingCastle();
                if ((*iter).to() == 63 && (p->flags & whiteLongCastleMask))
                    (*iter).setBreaksWhiteQueenCastle();
            }
        }
    }
    return moveList;
}
