/* This file is part of Cassandre.
   Copyright (c) 2003 Romang Jean-Franois, Adolph Thomas, Grundrich Raphael

   Cassandre is based on the DessChess program, a student project realised at
   University Louis Pasteur in Strasbourg, France ; under the direction of
   professor J.Korczak.

   Cassandre is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Cassandre is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Cassandre; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info: 
     jeff@proxone.net
*/

#ifndef CASSANDRE_MOVE_H
#define CASSANDRE_MOVE_H 1

#include <iostream>

/**
 * This class represents a move during the game. All informations about
 * the move are stored in the <I>move</I> attribute of this class.
 * You can access and set various informations of a move using
 * the methods provided by this class (<I>from()</I>, <I>to()</I>,...)
 * or by using predefinied masks and binary operations (&, |)
 */
class Move
{
    public:
        Move()
        {}
    
        Move(const unsigned int n)
        {
            move=n;
        }
        
        unsigned int move;

        unsigned int operator|(const unsigned int n);
        unsigned int operator&(const unsigned int n);
        unsigned int operator>>(const unsigned char n);
        bool operator==(const Move& m);
        Move& operator=(const unsigned int n);
    
        unsigned char from();
        unsigned char to();
        unsigned char capturedPiece();
        void setCapturedPiece(unsigned char piece);
        unsigned char promotionPiece();
        void setPromotionPiece(unsigned char piece);
        bool breaksBlackKingCastle();
        bool breaksBlackQueenCastle();
        bool breaksWhiteKingCastle();
        bool breaksWhiteQueenCastle();
        //bool breaksPawnAdvance();
        bool isCapture();
        bool isPromotion();
        bool isEnPassantCapture();
        void setCapture();
        void setPromotion();
        void setEnPassantCapture();
        void setBreaksBlackKingCastle();
        void setBreaksBlackQueenCastle();
        void setBreaksWhiteKingCastle();
        void setBreaksWhiteQueenCastle();
        //void setBreaksPawnAdvance();
        
        void setMovingPiece(unsigned char piece);
        unsigned char getMovingPiece();
        
    private:
        const static unsigned int moveMask;
        const static unsigned int captureMask;
        const static unsigned int promotionMask;
        const static unsigned int pieceMask;
        const static unsigned int wQCastleMask;
        const static unsigned int wKCastleMask;
        const static unsigned int bQCastleMask;
        const static unsigned int bKCastleMask;
        const static unsigned int isEnPassantCaptureMask;
        //const static unsigned int breaksPawnAdvanceMask;
};

inline std::ostream& operator<<(std::ostream& os, const Move& m)
{
    //os<<std::hex<<(m.move>>12)<<'|'<<(m.move&0x0FFF);
    os<<std::hex<<m.move;
    return os;
}

inline unsigned int Move::operator|(const unsigned int n)
{
    return(move|n);
}

inline unsigned int Move::operator&(const unsigned int n)
{
    return(move&n);
}

inline unsigned int Move::operator>>(const unsigned char n)
{
    return(move>>n);
}

inline bool Move::operator==(const Move& m)
{
    return(m.move==this->move);
}

inline Move& Move::operator=(const unsigned int n)
{
    move=n;
    return *this;
}

inline unsigned char Move::from()
{
    return((move>>6)&moveMask);
}

inline unsigned char Move::to()
{
    return(move&moveMask);
}

inline unsigned char Move::capturedPiece()
{
    return((move>>12)&pieceMask);
}

inline void Move::setCapturedPiece(unsigned char piece)
{
    move=(move&~(pieceMask<<12))|(piece<<12);
}

inline unsigned char Move::promotionPiece()
{
    return((move>>16)&pieceMask);
}

inline void Move::setPromotionPiece(unsigned char piece)
{
    move=(move&~(pieceMask<<16))|(piece<<16);
}

inline bool Move::breaksBlackKingCastle()
{
    return((move&bKCastleMask)!=0);
}

inline bool Move::breaksBlackQueenCastle()
{
    return((move&bQCastleMask)!=0);
}

inline bool Move::breaksWhiteKingCastle()
{
    return((move&wKCastleMask)!=0);
}

inline bool Move::breaksWhiteQueenCastle()
{
    return((move&wQCastleMask)!=0);
}

/*
inline bool Move::breaksPawnAdvance()
{
    return((move&breaksPawnAdvanceMask)!=0);
}*/

inline bool Move::isCapture()
{
    return((move&captureMask)!=0);
}

inline bool Move::isPromotion()
{
    return((move&promotionMask)!=0);
}

inline bool Move::isEnPassantCapture()
{
    return((move&isEnPassantCaptureMask)!=0);
}

inline void Move::setCapture()
{
    move=move|captureMask;
}

inline void Move::setPromotion()
{
    move=move|promotionMask;
}

inline void Move::setEnPassantCapture()
{
    move=move|isEnPassantCaptureMask;
}

inline void Move::setBreaksBlackKingCastle()
{
    move=move|bKCastleMask;
}

inline void Move::setBreaksBlackQueenCastle()
{
    move=move|bQCastleMask;
}

inline void Move::setBreaksWhiteKingCastle()
{
    move=move|wKCastleMask;
}

inline void Move::setBreaksWhiteQueenCastle()
{
    move=move|wQCastleMask;
}

/*
inline void Move::setBreaksPawnAdvance()
{
    move=move|breaksPawnAdvanceMask;
}*/

inline void Move::setMovingPiece(unsigned char piece)
{
    move=(move&~(pieceMask<<28))|(piece<<28);
}

inline unsigned char Move::getMovingPiece()
{
    return((move>>28)&pieceMask);
}

#endif
