// Copyright 1993-2007 by Jon Dart.  All Rights Reserved.

#ifndef _TYPES_H
#define _TYPES_H

// Types and macros for OS, machine, and compiler portability

#include <stdlib.h>
#ifndef _WIN32
#include <sys/types.h>
#include <sys/param.h>
#endif
#include <iostream>
using namespace std;

// Define this here since Bitmap class depends on it

typedef unsigned int Square;

enum {InvalidSquare = 127};

#if defined(SMP) && !defined(_WIN32)
extern "C" {
#include <pthread.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
  /* union semun is defined by including <sys/sem.h> */
#else
  /* according to X/OPEN we have to define it ourselves */
  union semun {
    int val;                  /* value for SETVAL */
    struct semid_ds *buf;     /* buffer for IPC_STAT, IPC_SET */
    unsigned short *array;    /* array for GETALL, SETALL */
    /* Linux specific part: */
    struct seminfo *__buf;    /* buffer for IPC_INFO */
  };
#endif
};
#endif

#if defined(_WIN64) || defined(__LP64__)
#define _64BIT 1
#else
#undef _64BIT 
#endif

#ifndef _GNUC_PREREQ
#define _GNUC_PREREQ(major,minor) (__GNUC__ >= major || ( __GNUC__ == major && __GNUC_MINOR__ >= minor))
#endif

typedef unsigned char byte;
typedef int int32;
typedef unsigned int uint32;
typedef short int16;
typedef unsigned short uint16;
#ifdef _WIN32
#ifdef _MSC_VER
#include <windows.h>
  typedef unsigned __int64 hash_t;
  typedef _int64 int64;
  typedef unsigned __int64 uint64;
  #include <strstream>
#if _MSC_VER <= 1300
// workaround for lack of << operator on uint64
// Microsoft support article 168440
extern std::ostream& operator<<(std::ostream& os, uint64 i );
#endif
#else //GCC
  typedef unsigned long long hash_t;
  typedef long long int64;
  typedef unsigned long long uint64;
  #include <Windows32/Defines.h>
#if _GNUC_PREREQ(3,2)
  #include <sstream>
#else
  #include <strstream.h>
#endif
  #include <sys/time.h>
  #include <sys/types.h>
  #include <sys/times.h>
#endif
#else
typedef unsigned long long hash_t;
typedef long long int64;
typedef unsigned long long uint64;
  #include <limits.h>
  #define MAX_PATH PATH_MAX
  #include <sys/time.h>
  #include <sys/types.h>
  #include <sys/times.h>
#if _GNUC_PREREQ (3,2)
  #include <sstream>
#else
  #include <strstream.h>
#endif
  #include <unistd.h>
#endif
inline int getTime()
{
#ifdef _WIN32
#ifdef _MSC_VER
  return GetTickCount()/10;
#else
  struct tms t;
  (void) times(&t);
  return (t.tms_utime+t.tms_stime+t.tms_cutime+t.tms_cstime)/10;
#endif
#else
  struct tms t;
  (void) times(&t);
  return t.tms_utime+t.tms_stime+t.tms_cutime+t.tms_cstime;
#endif
}
#define MAKELONGLONG(hi,lo) ((hash_t)hi<<32 | (hash_t)lo)

#ifdef _WIN32
// force _cdecl even if compiler uses fastcall
#undef CDECL
#define CDECL _cdecl
#else
#define CDECL
#endif

#ifdef _WIN32
#if defined(_MSC_VER) && _MSC_VER >= 1300
#define CACHE_ALIGN __declspec(align(128))
#else
#define CACHE_ALIGN
#endif
#define FORCEINLINE __forceinline
#else // Linux/GCC
#define CACHE_ALIGN __attribute__ ((aligned (128)))
#define FORCEINLINE __inline
#endif

#ifdef _WIN64
         #define ALIGN_POINTER(ptr,x) ((void*)((DWORD_PTR)ptr+x & (DWORD_PTR)~(x-1)))
#elif defined(_WIN32)
         #define ALIGN_POINTER(ptr,x) ((void*)((int)ptr+x & (int)~(x-1)))
#elif defined(__LP64__)
         #define ALIGN_POINTER(ptr,x) ((void*)((uintptr_t)ptr+x & (uintptr_t)~(x-1)))
#else
         #define ALIGN_POINTER(ptr,x) ((void*)((int)ptr+x & (int)~(x-1)))
#endif
#ifdef _WIN32
#define ALIGN_VAR(n) __declspec(align(n))
#else
#define ALIGN_VAR(n)
#endif
         
#ifdef SMP
// multithreading support.
#ifdef _WIN32
#define LockDefine(x) CRITICAL_SECTION x
#define lock_t CRITICAL_SECTION
#define LockInit(x) InitializeCriticalSection(&x)
#define Lock(x) EnterCriticalSection(&x);
#define Unlock(x) LeaveCriticalSection(&x);
#define LockDestroy(x) DeleteCriticalSection(&x)
#define LockFree(x) DeleteCriticalSection(&x)
#define THREAD HANDLE
#else
// Linux
#define LockDefine(x) volatile int x
#define lock_t volatile int
#define LockInit(x) x = 0
static void inline LockAsm(lock_t *x) {
  // use atomic instruction (xchg) to test & set
  register lock_t dummy = 1;
  asm volatile ("1:\n"
                " xchgl %0,(%1)\n"  /* set lock=1 & get value */
                " testl %0, %0\n"   /* test old value */
                " jz  3f\n"         /* if =0, was unlocked, now locked */
                "2:\n"             
                " pause\n"          /* yield thread */
                " cmpb $0, (%1)\n"  /* try a non-locking test */
                " jne 2b\n"         /* if non-zero, wait again */
                " jmp 1b\n"         /* now confirm w. bus lock */
                "3:\n"
                : "=&q" (dummy)
                : "q" (x)
                : "memory","cc");
}
#define Lock(x) LockAsm(&x)
#define Unlock(x) x=0
#define LockDestroy(x) 
#define LockFree(x) 
#define THREAD pthread_t
#endif
#else // no SMP
#define LockDefine(x)
#define LockInit(x)
#define Lock(x)
#define Unlock(x)
#define LockDestroy(x)
#endif

#if !defined(_WIN32) && (__BYTE_ORDER == __BIG_ENDIAN)
// we assume Windows is little endian
typedef union endian_convert64
{
    uint64 data;
    unsigned byte bytes[8];
};

typedef union endian_convert32
{
    uint32 data;
    unsigned byte bytes[4];
};

typedef union endian_convert16
{
    uint16 data;
    unsigned byte bytes[2];
};

uint32 swapEndian64(const byte *input) {
    endian_convert64 ret;
    ret.bytes[0] = input[7];
    ret.bytes[1] = input[6];
    ret.bytes[2] = input[5];
    ret.bytes[3] = input[4];
    ret.bytes[4] = input[3];
    ret.bytes[5] = input[2];
    ret.bytes[6] = input[1];
    ret.bytes[7] = input[0];
    return ret.data;
}

uint32 swapEndian32(const byte *input) {
    endian_convert32 ret;
    ret.bytes[0] = input[3];
    ret.bytes[1] = input[2];
    ret.bytes[2] = input[1];
    ret.bytes[3] = input[0];
    return ret.data;
}

uint16 swapEndian16(const byte *input) {
    endian_convert16 ret;
    ret.bytes[0] = input[1];
    ret.bytes[1] = input[0];
    return ret.data;
}
#else
// no need to convert, just de-reference
#define swapEndian64(x) *((uint64*)(x))
#define swapEndian32(x) *((uint32*)(x))
#define swapEndian16(x) *((uint16*)(x))
#endif
#endif
