// Copyright 2005-2007 by Jon Dart. All Rights Reserved.

#include "threadp.h"
#include "search.h"
#ifdef _THREAD_TRACE
#include "globals.h"
#endif
#include <iostream>
#ifndef _WIN32
#include <errno.h>
#include <fcntl.h>
#endif
using namespace std;

#ifdef _THREAD_TRACE
static lock_t io_lock;

void log(const string &s) {
  Lock(io_lock);
  puts(s.c_str());
  Unlock(io_lock);
}
void log(const string &s,int param) {
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
  std::ostringstream out;
#else
  strstream out;
#endif
  out << s;
  out << ": ";
  out << param << '\0';
  log(out.str());
}
#endif

void idle_loop(ThreadInfo *ti, split_t *split) {

   while (ti->state != ThreadInfo::Terminating) {
#ifdef _THREAD_TRACE
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
     std::ostringstream s;
#else
     strstream s;
#endif
     s << "thread " << ti->index << " in idle loop:";
     s << "master=" << ti->master << " hasSplit=" << (split && split->slaves.size()) << '\0';
     log(s.str());
#endif
     if (ti->master && !(split && split->slaves.size())) {
#ifdef _THREAD_TRACE
       log("exiting",ti->index);
#endif
       ti->signal(); // ensure master will not wait
       break;
      }
      ti->state = ThreadInfo::Idle; // mark thread available again
      if (ti->wait()) break;
#ifdef _THREAD_TRACE
      log("unblocked",ti->index);
#endif
      // We've been woken up. There are three possible reasons:
      // 1. This thread is terminating.
      // 2. We are a master at a split point and all our slave threads
      // are done.
      // 3. This thread (a slave) has been assigned some work.
      //
      if (ti->state == ThreadInfo::Terminating) break;
      if (ti->master && !(split && split->slaves.size())) {
	  ti->signal();
	  break;
      }
#ifdef _THREAD_TRACE
      log("wakeup",ti->index);
#endif
      ASSERT(ti->work);
      ASSERT(ti->state == ThreadInfo::Working);
      ti->work->searchSMP(ti);  
      ti->pool->checkIn(ti);
   }
}

#ifdef _WIN32
static DWORD WINAPI parkingLot(void *x)
#else
static void * CDECL parkingLot(void *x)
#endif
{
  idle_loop((ThreadInfo*)x,NULL);
  return 0;
}

ThreadInfo::~ThreadInfo() {
#ifdef _WIN32
  CloseHandle(hEvent1);
#else
  if (semctl(semid,IPC_RMID,0)) perror("semctl");
#endif
  //  if (index != 0) delete work;
}

void ThreadInfo::start() {
    ASSERT(index>=0);
#ifdef _THREAD_TRACE
    log("start",index);
#endif
    signal();
}

void ThreadInfo::signal() {
    // unblock the thread so it can execute
#ifdef _WIN32
    if (!SetEvent(hEvent1)) {
        cerr << "error: SetEvent" << endl;
    }
#else
    struct sembuf sb = { 0, -1, IPC_NOWAIT };
    // decrement semaphore value:
    if (semop(semid,&sb,1)&& errno!=EAGAIN) 
       perror("semop: signal");
#endif
}

void ThreadInfo::reset() {
#ifdef _THREAD_TRACE
  log("reset",index);
#endif
#ifdef _WIN32
    if (!ResetEvent(hEvent1))
        cerr << "ResetEvent failed" << endl;
#else
    // use semctl, not semop, to mimic Windows behavior where
    // multiple resets have no extra effect. 
    union semun val;
    // set semaphore to 1
    val.val = 1; 
    if (semctl(semid, 0, SETVAL, val)) {
       perror("semctl");
       return;
    }
#endif
}

int ThreadInfo::wait() {
#ifdef _THREAD_TRACE
  log("in wait",index);
#endif
#ifdef _WIN32
  return (WaitForSingleObject(hEvent1,INFINITE) != WAIT_OBJECT_0);
#else
#ifdef _THREAD_TRACE
  union semun stuff;
  int val  = semctl(semid,0,GETVAL,stuff);
  char buf[20];
  sprintf(buf,"value=%d",val); 
  log(buf);
#endif
      struct sembuf sb = { 0, 0, 0 };
      // wait for semaphore value 0  
      if (semop(semid,&sb,1)) {
        if (errno == EINTR || errno == EIDRM) return 1;
        else {
          perror("semop: wait");
	}
      }
  return 0;
#endif
}

int ThreadInfo::wait(unsigned millisec) {
#ifdef _WIN32
  return (WaitForSingleObject(hEvent1,millisec) != WAIT_OBJECT_0);
#else
  struct timespec ts = { millisec/1000, (millisec%1000)*1000 };
      struct sembuf sb = { 0, 0, 0 };
      // wait for semaphore value 0  
      if (semtimedop(semid,&sb,1,&ts)) {
        if (errno == EINTR || errno == EIDRM) return 1;
        else if (errno != EAGAIN) {
          perror("semop: wait (timed)");
	}
      }
     return 0;
#endif
}

ThreadInfo::ThreadInfo(int i) 
 : state(Idle),
#ifdef _WIN32
   thread_id(NULL),
   work(NULL),
#else
   work(NULL),
#endif
   parentNode(NULL),
   master(0),
#ifndef _WIN32
   count(0),
#endif
   index(i)
{
#ifdef _THREAD_TRACE
  log("starting",i);
#endif
#ifdef _WIN32
      // initial state is FALSE (non-signaled), so the
      // the controller blocks the thread
      hEvent1 = CreateEvent(NULL,FALSE,FALSE,NULL);
      DWORD id;
      if (index == 0) {
	thread_id = GetCurrentThread();
      } else {
         thread_id = CreateThread(NULL,0,
            parkingLot,this,
            0,
            &id);
      }
#else
      semid = semget(IPC_PRIVATE,1,IPC_CREAT | 0777);
      if (semid == -1) {
	perror("semaphore creation failed");
        return;
      }
      union semun val;
      // initialize semaphore to 1
      val.val = 1; 
      if (semctl(semid, 0, SETVAL, val)) {
        perror("semctl");
        return;
      }
      if (index == 0) {
	thread_id = pthread_self();
      }
      else {
        if (pthread_create(&thread_id, NULL, parkingLot, this)) {
  	   perror("thread creation failed");
	}
      }
#endif
}

ThreadPool::ThreadPool(SearchController *controller,int n) 
: nthreads(n) {
#ifdef _THREAD_TRACE
  LockInit(io_lock);
#endif
   data = new ThreadInfo *[MAX_CPUS];
   memset(data,'\0',sizeof(ThreadInfo*)*MAX_CPUS);
   LockInit(poolLock);
   for (int i = 0; i < n; i++) {
      data[i] = new ThreadInfo(i);
      data[i]->pool = this;
      if (i==0) {
	     data[i]->work = new RootSearch(controller,data[i]);
      }
      else {
	     data[i]->work = new Search(controller,data[i]);
      }
      data[i]->work->ti = data[i];
      data[i]->pool = this;
   }
}

ThreadPool::~ThreadPool() {
    shutDown();
#ifdef _THREAD_TRACE
  LockDestroy(io_lock);
#endif
}

void ThreadPool::shutDown() {
    // do not shut down thread 0 (main thread)
    for (int i = 1; i < nthreads; i++) {
       // All threads should be idle when this function is called.
       if (data[i]->state == ThreadInfo::Idle) {
          // Set the thread to the terminating state that will force thread
	  // procedure exit
          data[i]->state = ThreadInfo::Terminating;
          // unblock the thread
	  data[i]->signal();
       }
       // wait for the thread to terminate
#ifdef _WIN32
       WaitForSingleObject(data[i]->thread_id,INFINITE);
#else
       void *value_ptr;
       pthread_join(data[i]->thread_id,&value_ptr);
#endif
       // Free thread data 
       delete data[i];
    }
    for (int i = nthreads; i < MAX_CPUS; i++) {
      // Free thread data - may be non-zero if we have shrunk the thread pool
      delete data[i];
    }
    delete data[0]; // main thread structure
    //delete [] data; 
}

ThreadInfo * ThreadPool::checkOut(Search *parent, NodeInfo *forNode,
  int ply, int depth) {
    Lock(poolLock);
    for (int i = 0; i < nthreads; i++) {
#ifdef _WIN32
       if (i!=parent->ti->index && data[i]->state == ThreadInfo::Idle) {
#else
       // verify the thread is available and is actually
       // at the synchronization point
       if (i!=parent->ti->index && data[i]->state == ThreadInfo::Idle &&
           semctl(data[i]->semid,0,GETZCNT)>0) {
#endif
	 // if this is a "master" thread it is not sufficient to just be
	 // idle - assign it only to one of its slave threads
	 if (!data[i]->master || 
	     (data[i]->parentNode->split &&
             data[i]->parentNode->split->slaves.exists(parent->ti))) {
  	   split_t *split = parent->split(forNode,data[i],ply,depth);
           if (!split) {
              Unlock(poolLock);
#ifdef _THREAD_TRACE
	      log("alloc failed, split limit");
#endif
              return NULL; // can't split this search instance any more
           }
           Unlock(poolLock);
           return data[i]; 
	 }
       }
    }
    // no luck, no free threads
    Unlock(poolLock);
    return NULL;
}

void ThreadPool::resize(int n, SearchController *controller) {
  if (n >= 1 && n < MAX_CPUS && n != nthreads) {
    Lock(poolLock);
    if (n>nthreads) {
      data[nthreads] = new ThreadInfo(nthreads);
      data[nthreads]->work = new Search(controller,data[nthreads]);
      data[nthreads]->pool = this;
      nthreads++;
    }
    else {
      // shrinking
      --nthreads;
      data[nthreads]->state = ThreadInfo::Terminating;
      if (data[nthreads-1]->state == ThreadInfo::Idle) {
        data[nthreads]->signal(); // unblock thread & exit thread proc
      }
    }
    Unlock(poolLock);
  }
}

void ThreadPool::checkIn(ThreadInfo *ti) {
#ifdef _THREAD_TRACE
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
  std::ostringstream s;
#else
  strstream s;
#endif
  s << "checkIn " << ti->index << " master=" << ti->master << '\0';
  log(s.str());
#endif
    Lock(poolLock);
    if (!ti->master) {
      Search *parent = ti->work->parent;
      if (!parent->join(ti->parentNode,ti)) {
	// all slave threads are completed, so signal parent that it
	// does not need to wait any more
#ifdef _THREAD_TRACE
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
	std::ostringstream s;
#else
	strstream s;
#endif
        s << "signaling parent (" << parent->ti->index << ")" << 
	  " parent state=" << parent->ti->state << '\0';
        log(s.str());
#endif
         parent->ti->signal();
      }
#ifndef _WIN32
      // semaphores are not auto-reset, so increment the value
      // here so we will wait when we re-enter the idle loop.
      ti->reset();
#endif
   }
   Unlock(poolLock);
}

int ThreadPool::activeCount() {
    int count = 0;
    for (int i = 0; i < nthreads; i++) {
       if (data[i]->state == ThreadInfo::Working) ++count;
    }
    return count;
}

void ThreadPool::clearHashTables(SearchController *c) {
  for (int i=0;i< nthreads;i++) {
    data[i]->work->clearHashTables();
  }
}
