// Copyright 1994-2008 by Jon Dart.  All Rights Reserved.

// Stand-alone console executable to test the search engine

#include "board.h"
#include "movegen.h"
#include "search.h"
#include "movearr.h"
#include "notation.h"
#include "globals.h"
#include "chessio.h"
#include "epdrec.h"
#include "scoring.h"
#include "bearing.h"
#include "calctime.h"
#include "config.h"
#include "arasvec.h"
#include "types.h"
#include <string>

extern "C"
{
#include <string.h>
#include <ctype.h>
};
#include <fstream>
#include <iostream>
#include <ctype.h>
#include <time.h>
#include <sys/timeb.h>
#include "tbprobe.h"

int verbose = 0;
uint64 total_nodes = (uint64)0;
long total_correct = 0L;
long total_tests = 0L;
long total_time = 0L;
int early_exit_plies = Constants::MaxPly;
int early_exit = 0;
int solution_move_count = 0;
time_t solution_time = 0;
int last_iteration_depth = -1;
int iterations_correct = 0;
int moves_to_search = 1;
Move solution_moves[10];
bool avoid = false;
ArasanVector<int> solution_times;

static void post(const Statistics &stats)
{
    if (avoid) {
      // note: doesn't handle multiple "am" moves
      int ok = !MovesEqual(solution_moves[0],stats.best_line[0]);
      if (ok) {
          if (solution_time == -1) solution_time = stats.elapsed_time;
      }
      else solution_time = -1;
      return;
   }
   for (int i = 0; i < solution_move_count; ++i)
   {
      if (MovesEqual(solution_moves[i],stats.best_line[0]))
      {
         if ((int)stats.depth > last_iteration_depth)
         {
            ++iterations_correct;
            last_iteration_depth = stats.depth;
         }
         // don't terminate in under 2 seconds unless we found
         // a mate
         if ((stats.elapsed_time >= 200 ||
              stats.value > Constants::BIG-100)
             && iterations_correct >= early_exit_plies)
	   early_exit = 1;
         if (solution_time == -1) solution_time = stats.elapsed_time;
         return;
      }
   }
   solution_time = -1;
}

static int terminate(const Statistics &stats)
{
   post(stats);
   return early_exit;
}

static Move search(Board &board, int ply_limit,
                   int time_limit, Statistics &stats, 
                   Move excludes [], int num_excludes)
{
   Move move = NullMove;
   SearchController searcher;

   searcher.register_post_function(post);
   searcher.register_terminate_function(terminate);
   solution_time = -1;

   move = searcher.find_best_move(board, 
               ply_limit >= 99 ? Fixed_Time : Fixed_Depth, 
               time_limit, 0, ply_limit,
               0, 0, stats,
               verbose ? Debug : Silent,
               excludes, num_excludes);

   char move_text[20];
   Notation::Image(board, move, move_text);
   if (num_excludes)
      cout << "result(" << num_excludes+1 << "):";
   else
      cout << "result:";
   char score_buf[20];
   Scoring::print_score(stats.display_value,score_buf);
   cout << '\t' << move_text << "\t" << stats.elapsed_time/100 <<
   " seconds.\tscore:" << score_buf << "\t"  <<
   stats.num_nodes << " nodes.";
   total_time += (long)stats.elapsed_time;
   total_nodes += stats.num_nodes;
   // call only after SearchController cleanup:
   game_moves->removeAll();
   Scoring::cleanup();
   clearHash(); 
   return move;
}

static void show_usage()
{
   cerr << "Usage:"  << endl;
   cerr << "  testsrc <options> <position file>" << endl;
   cerr << "Options are:" << endl;
#ifdef SMP
   cerr << "  -c [cpus]    - use specified # of cpus" << endl;    
#endif
   cerr << "  -e           - show root evaluation only" << endl;    
   cerr << "  -t [seconds] - seconds to search per position" << endl;
   cerr << "  -p [plies]   - ply limit for search" << endl;
   cerr << "  -n [0|1|2]   - null depth reduction (default 2)" << endl;
   cerr << "  -H [size]    - hash table size (kilobytes)" << endl;
   cerr  << "  -tb[+|-]     - use/don't use tablebases (default on)" << endl;
   cerr << "  -N [n]       - find <n> best moves" << endl;
   cerr << "  -x [plies]   - terminate if correct move holds for <plies>" << endl;
   cerr << "  -v           - be verbose\n" << endl;
}

static void do_eval(ifstream &pos_file, Board &board)
{
   if (!pos_file.good())
   {
      cout << "File not found, or bad format." << endl;
      return;
   }
   while (!pos_file.eof()) {
      pos_file >> board;
      if (!pos_file.good()) break;
      int tbscore;
      char score_buf[20];
      if (options.search.use_tablebases && probe_tb(board,tbscore,0)) {
        Scoring::print_score(tbscore,score_buf);
        static char line[128];
        sprintf(line,"score = %s (from tablebases)\n",score_buf);
        cout << line << endl;
        int score;
        if ((score = Scoring::tryBitbase(board))!= Scoring::INVALID_SCORE) 
           cout << "bitbase score=" << score << endl;
      }
      Scoring::init();
      SearchController searcher;
      Search s(&searcher,NULL);
      if (Scoring::is_draw(board))
        cout << "position evaluates to draw (statically)" << endl;
      else {
         cout << board << endl;
         Scoring::print_score(Scoring::evalu8(board,&s),score_buf);
         cout << "score = " << score_buf << endl;
         Board board2 = board;
         board.flip();
         cout << board << endl;
         Scoring::print_score(Scoring::evalu8(board,&s),score_buf);
         cout << "score(flip1) = " << score_buf << endl;
         board2.flip2();
         cout << board2 << endl;
         Scoring::print_score(Scoring::evalu8(board2,&s),score_buf);
         cout << "score(flip2) = " << score_buf << endl;
         cout << endl;
      }
      pos_file.ignore(128,'\n');
   }
}

static const char *get_move(const char *buf,const Board &board, Move &m)
{
   const char *p = buf;
   m = NullMove;
   if (*p)
   {
      while (isspace(*p) && *p != '\0') ++p;
      if (*p == '\0')
         return NULL;
      char tmp[10];
      int i = 0;
      char *q = tmp;
      while (!isspace(*p) && *p != '+' && *p != '\0' && i < 10)
      {
         *q++ = *p++;
         ++i;
      }
      *q = '\0'; 
      m = Notation::Value(board,board.Side(),tmp);
      if (*p == '+') ++p;
   }
   return p;
}

static void do_search(Board &board,int ply_limit,int time_limit)
{
   Statistics stats;
   Move excludes[Constants::MaxMoves];
   for (int index = 0; index < moves_to_search; index++) {
      Move result = search(board,ply_limit,time_limit,stats,
         excludes,index);
      if (IsNull(result)) break;
      excludes[index] = result;
      int correct = avoid ? 1 : 0;
      for (int i = 0; i < solution_move_count; ++i)
      {
         if (MovesEqual(solution_moves[i],result))
         {
            correct = (avoid) ? 0 : 1;
            break;
         }
      }
      solution_times.append((int)solution_time);
      total_tests++;
      if (correct)
      {
         total_correct++;
         cout << "\t++ correct" << endl;
      }
      else
         cout << "\t** error" << endl;
      cout << stats.best_line_image << endl;
   }
}

static void do_search(ifstream &pos_file, Board &board, int ply_limit,
                      int time_limit) 
{
   char buf[512];
   while (!pos_file.eof())
   {
      pos_file.getline(buf,511);
      if (!pos_file)
      {
         cout << "File not found, or bad format." << endl;
         return;
      }
      // Try to parse this line as an EPD command.
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
      string istr(buf);
      istringstream stream(istr);
#else
      istrstream stream(buf,512);
#endif
      string id, comment;
      EPDRecord *epd_rec =
        ChessIO::readEPDRecord(stream,board);
      if (epd_rec != NULL)
      {
         for (size_t i = 0; i < (size_t)epd_rec->getSize(); i++)
         {
            string key, val;
            epd_rec->getData(i,key,val);
            if (key == "bm" || key == "am")
            {
               Move m;
               solution_move_count = 0;
               const char *p = val.c_str();
               while (*p) {
                  p = get_move(p,board,m);
                  if (!IsNull(m))
                  {
                     assert(solution_move_count < 10);
                     solution_moves[solution_move_count++] = m;
                  }
                  avoid = (key == "am");
               }
	    }
            else if (key == "id")
            {
               id = val;
            }
	    else if (key == "c0") {
	       comment = val;
	    }
         }
         last_iteration_depth = -1;
         iterations_correct = 0;
         early_exit = 0;
         cout << id << ' ';
         if (comment.length()) cout << comment << ' ';
         char move_string[100];
         if (avoid) {
             Notation::Image(board,solution_moves[0],move_string);
             cout << "am " << move_string << endl;
         }
         else {
             cout << "bm";
             for (int i = 0; i < solution_move_count; i++) {
                Notation::Image(board,solution_moves[i],move_string);
                cout << ' ' << move_string;
             }
             cout << endl;
         }
         do_search(board,ply_limit,time_limit);
      }

      char c;
      while (!pos_file.eof())
      {
         c = pos_file.get();
         if (!isspace(c) && c != '\n')
         {
            if (!pos_file.eof())
               pos_file.putback(c);
            break;
         }
      }
   }
   pos_file.close();
   cout << endl << "solution times:" << endl;
   cout << "         ";
   int i = 0;
   for (i = 0; i < 10; i++)
    cout << i << "      ";
   cout << endl;   
   for (i = 0; i < solution_times.length(); i++) {
      char digits[15];
      if (i == 0) {
         sprintf(digits,"% 4d |       ",i);
         cout << endl << digits;
      }
      else if ((i+1) % 10 == 0) {
         sprintf(digits,"% 4d |",(i+1)/10);
         cout << endl << digits;
      }
      if (solution_times[i] == -1) {
         cout << "  ***  ";
      }
      else {
        sprintf(digits,"%6.2f ",solution_times[i]/100.0);
        cout << digits;
     }
   }
   cout << endl << endl << "correct : " << total_correct << '/' <<
       total_tests << endl;
   cout << endl;
}

int CDECL main(int argc, char **argv)
{
   init_options(argv[0]);
   options.book.book_enabled = options.log_enabled = 0;
   int ply_limit = 2;
   int time_limit = 99999;
   int arg = 1;
   int show_eval = 0;

   if (argc == 1)
   {
      show_usage();
      return -1;
   }
   else
   {
      while (arg < argc && (*(argv[arg]) == '-'))
      {
         char c = *(argv[arg]+1);
         switch (c)
         {
#ifdef SMP
         case 'c':
            ++arg;
            options.search.ncpus = Util::Min(32,atol(argv[arg]));
            break; 
#endif
         case 'e':
            ++show_eval;
            break;
         case 'p':
            ++arg;
            ply_limit = atoi(argv[arg]);
            break;
         case 't': 
            c = (*(argv[arg]+3));
            if (c == 'b') {
               c = (*(argv[arg]+4));
               if (c == '\0' || c == '+') {
 		          options.search.use_tablebases = 1;
               } else {
		          options.search.use_tablebases = 0;
               }
            }
            else {
               ++arg;
               time_limit = 100*atol(argv[arg]);
               ply_limit = 99;
            }
            break;
         case 'v':
            verbose++; break;
         case 'n':
            ++arg;
            if (arg < argc)
            {
               if (argv[arg][0] == '0')
                  options.search.null_depth_reduction = 0;
               else if (argv[arg][0] == '1')
                  options.search.null_depth_reduction = 1;
               else if (argv[arg][0] == '2')
                  options.search.null_depth_reduction = 2;
            }
            break;
         case 'H':
            {
               ++arg;
               options.search.hash_table_size = atol(argv[arg]);
            }
            break;
         case 'N': {
               ++arg;
               if (arg < argc)
               {
                  moves_to_search = atoi(argv[arg]);
               }
               break;
           }
         case 'x':
            {
               ++arg;
               if (arg < argc)
               {
                  early_exit_plies = atoi(argv[arg]);
               }
               break;
            }
         default:
            show_usage();
            return -1;
         }
         ++arg;
      }

      if (arg >= argc)
      {
         show_usage();
         return -1;
      }
   }
   if (!init_globals(argv[0],0,1)) {
      cleanup_globals();
      exit(-1);
   }
   atexit(cleanup_globals);
   delayed_init();

   Board board;
#if defined (_WIN32) || (defined (__GNUC__) && (__GNUC__ >=3))
      ifstream pos_file( argv[arg], ios::in);
#else
      ifstream pos_file( argv[arg], ios::in | ios::nocreate);
#endif
   if (pos_file.good())
   {
      if (show_eval)
         do_eval(pos_file,board);
      else
         do_search(pos_file,board,ply_limit,time_limit);         
   }
   else
   {
      cout << "file not found: " << argv[arg] << endl;
      return -1;
   }
   return 0;
}
