// Copyright 1994-2008 by Jon Dart.  All Rights Reserved.

#ifndef _SEARCH_H
#define _SEARCH_H

#include "board.h"
#include "hash.h"
#include "stats.h"
#include "pawnentr.h"
#include "arasset.h"
#include "searchc.h"
#include "scoring.h"
#ifdef SMP
#include "threadp.h"
#endif
#include <memory.h>
#include <time.h>
#include <set>
using namespace std;

class Root_Move_Generator;

class Move_Generator;

#ifdef SMP
struct split_t {
   ArasanSet<ThreadInfo *,Constants::MaxCPUs> slaves;
   Move moves[Constants::MaxMoves];
   int ply;
   int depth;
   lock_t nodeLock;
   split_t() {
     LockInit(nodeLock);
   }
   ~split_t() {
     LockDestroy(nodeLock);
   }
};
#endif

// Per-node info, part of search history stack
  struct NodeInfo {
	  NodeInfo() {
         init();
       }
  void init() {
    score = alpha = beta = hibound = cutoff = num_try = flags = threat = 0;
    best = last_move = NullMove;
    extensions = 0;
#ifdef MOVE_ORDER_STATS
    best_count = 0;
#endif
    fpruned_moves = 0;
    pos_score = Scoring::INVALID_SCORE;
#ifdef SMP
    parentNode = NULL;
    splits = 0;
    split = NULL;
    mg = NULL;
#endif

  }
    volatile int score;
    volatile int alpha, beta, hibound;
    volatile int cutoff;
    int num_try;
    int flags; 
    int threat;
    volatile Move best;
    Move last_move;
    int extensions; // mask of extensions
    volatile Move pv[Constants::MaxPly];
    volatile int pv_length;
    volatile Move done[Constants::MaxMoves];
#ifdef MOVE_ORDER_STATS
    int best_count;
#endif
    int pos_score;
    volatile int fpruned_moves;
    int ply, depth;
#ifdef SMP
    void lock() {
      Lock(split->nodeLock);
    }
    void unlock() {
      Unlock(split->nodeLock);
    }
    Move_Generator * mg;
    // parent of this node if it has an owning split point:
    NodeInfo *parentNode;
    // set of other threads executing searches that are slaves to this one.
    // (NULL if none)
    split_t *split;
    volatile int splits;
    int hasSplit() const {
      return (split && split->slaves.size());
    }
    int wasSplit() const {
      return splits;
    }
#endif
};

struct QSearchHashEntry {
   hash_t hash_code;
   Move best;
   int score;
};

class RootSearch;
class SearchController;

// There are 4 levels of verbosity.  Silent mode does no output to
// the console - it is used by the Windows GUI version.  Debug
// level is used to output debug info.  Whisper is used to "whisper"
// comments on a chess server.  "Trace" is used in trace mode
// (-t) of the chess server client.
enum Talk_Level { Silent, Debug, Whisper, Trace };
    
enum Search_Type { Fixed_Depth, Time_Limit, Fixed_Time };

typedef void (*PostFunction)(const Statistics &);
typedef int (*TerminateFunction)(const Statistics &);

class Search {
#ifdef SMP
    friend struct ThreadInfo;
    friend class ThreadPool;
#endif
    friend class Move_Ordering;

    // Encapsulates alpha-beta search routine. 

protected:
    enum Extension_Type { RECAPTURE=1, CHECK=2, PAWN_PUSH=4, CAPTURE=8,
       FORCED=16, HPRUNE=64, THREAT=128 };

    enum SearchFlags { IID=1, VERIFY=2, EXACT=4 };

public:

#ifdef SMP
    Search(SearchController *c, ThreadInfo *ti);
#else
    Search(SearchController *c);
#endif
        
    void init(const Board &board, Search *parent);

#ifdef SMP
    void init(const Board &board, Search *parent, NodeInfo *parentNode, ThreadInfo *ti, int ply, int depth);
#endif

    virtual ~Search();
    
    int is_draw(const Board &board, int &rep_count, int ply );
    
    int draw_score(const Board & board);
    
    int search(int alpha, int beta,
               int ply, int depth, int flags = 0) {
      PUSH(alpha,beta,flags,ply,depth);
      return POP(search());
    }

    // search based on current board & NodeInfo
    int search();

#ifdef SMP
    int master() const { return ti->index == 0; }

    // perform a subsidiary search in a separate thread
    void searchSMP(ThreadInfo *);
    int maybe_split(const Board &board, NodeInfo *node,int ply,int depth);
    void maybe_wait(NodeInfo *);
#else
    int master() const { return 1; }
#endif

    int quiesce(int alpha, int beta,
                int ply, int depth) {
       PUSHQ(alpha,beta);
       return POP(quiesce(ply,depth));
    }

    int quiesce(int ply,int depth);

    int evalu8(const Board & board, int alpha, int beta, int ply);

    Move getBest() const;

    void terminateSearch() {
       terminate = 1;
    }

    //    NodeInfo *getParentNode() const {
    //  return parentNode;
    //}
    
    int update_move(const Board &,
		    NodeInfo *parentNode, NodeInfo* myNode, Move move, 
                    int score, int ply, int depth);

    NodeInfo * getNode() const {
      return node;
    }

    static const int PAWN_HASH_SIZE = 8192;
    static const int KING_PAWN_HASH_SIZE = 65536;
    static const int KING_COVER_HASH_SIZE = 8192;
    static const int QSEARCH_CACHE_SIZE = 2*65536;

    struct King_Pawn_Entry {
        // defines one entry in the king/pawn hash table.
        hash_t hc;                                    // hashcode
        signed char white_king_position, black_king_position;
        signed char white_endgame_pawn_proximity;
        signed char black_endgame_pawn_proximity;
        byte w_uncatchable, b_uncatchable;
        byte pawns_both_sides;
        int score;
    };

    struct King_Cover_Entry {
       hash_t hc;
       int cover;
       int pawn_proximity;
    };

    King_Pawn_Entry *king_pawn_hash_table;
    Pawn_Entry *pawn_hash_table;
    King_Cover_Entry *king_cover_hash_table;
    QSearchHashEntry *qsearch_cache;

    virtual void clearHashTables();

protected:
    void *king_pawn_hash_table_allocated;
    void *pawn_hash_table_allocated;
    void *king_cover_hash_table_allocated;
    void *qsearch_cache_allocated;
    Search *parent;
    SearchController *controller;
    RootSearch *root;
    Board board;
#ifdef SMP
    ThreadInfo *ti;
#endif
    int terminate;
    NodeInfo *node; // pointer into node stack, below
    NodeInfo stack[Constants::MaxPly];

    int calc_extensions(const Board &board,
		         NodeInfo *node, NodeInfo *parentNode, 
 			 CheckStatusType check_status_after_move,
			 int ply, int depth,
 		         const Move &move);

    void update_pv(const Board &, Move m, int ply);

    void update_pv(const Board &,NodeInfo *node,NodeInfo *fromNode,Move move,int ply);
    void stopAllThreads();
#ifdef SMP
    split_t *split(NodeInfo *nodeToSplit,ThreadInfo *ti,int ply,int depth);
    int join(NodeInfo *splitNode,ThreadInfo *ti);
    void stopAllThreads(NodeInfo *);
    void waitForThreadCompletion(NodeInfo *);
    void idle_loop(NodeInfo *);

    LockDefine(split_lock);
    split_t split_stack[SPLIT_STACK_MAX_DEPTH];
    split_t *split_stack_end;

    struct Search_State {
      ThreadInfo *ti;
      Search *parent;
      Board board;
      NodeInfo nodeContents; 
      NodeInfo *node; // pointer into node stack
      split_t *split_stack_end;
      void init(const Search *);
    } saved;
    void restore(Search_State &);
#endif

private:
    // make this class uncopyable:
    Search( const Search & );
    Search &operator = ( const Search & );

    int qsearch_check(int ply,int depth);
    int qsearch_no_check(int ply,int depth);

    FORCEINLINE void PUSH(int alpha,int beta,int flags,
			  int ply, int depth) {
       ++node; 
       node->alpha = node->score = alpha; 
       node->beta = node->hibound = beta; 
       node->flags = flags; 
       node->best = NullMove; 
       node->num_try = 0;
       node->ply = ply;
       node->depth = depth;
#ifdef SMP
       node->split = NULL;
#endif
       node->cutoff = 0; 
	}

    FORCEINLINE void PUSHQ(int alpha,int beta) {
       ++node; 
       node->flags = 0;
       node->alpha = node->score = alpha; 
       node->beta = node->hibound = beta; 
       node->best = NullMove; 
	}

    FORCEINLINE int POP(int value)  {
       int val = value; 
       --node; 
       return val;
    }

};

class RootSearch : public Search {
    friend class Search;
    friend class SearchController;
public: 
    RootSearch(SearchController *c
#ifdef SMP
	       , ThreadInfo *ti
#endif
    );

    virtual ~RootSearch();

    Move ply0_search(Search_Type srcType,
                int32 timelimit,
                int32 xtratime,
                int plylimit,
                const int background,
                Statistics &stat_buf,
                Talk_Level t,
                Move *exclude, int num_exclude);

    time_t getTimeAdded() {
        return time_added;
    }

    // Get the starting time for the search in centiseconds
    time_t get_start_time() const
    {
        return start_time;
    }
    
    Search_Type get_search_type() const
    {
        return type_of_search;
    }
    
    void set_time_limit(time_t limit,time_t xtra_time_limit)
    {
       type_of_search = Time_Limit;
       time_target = limit;
       xtra_time = xtra_time_limit;
    }
    
    void set_ply_limit(int ply)
    {
       type_of_search = Fixed_Depth;
       ply_limit = ply;
    }
    
    time_t get_time_limit() const
    { 
        return time_target + (time_added ? xtra_time : 0);
    }
    
    int is_background_search() const
    {
        return bkgrnd_search;
    }
    
    int update_root_move(const Board &,NodeInfo *parentNode, NodeInfo *node, Move move, int score);

    void set_background(int b);

    void terminate_now();

    int was_terminated() const {
      return terminated;
    }

    virtual void clearHashTables();

private:
    void init(const Board &);

    int check_time(const Board &,int ply, int rank, int interation_depth);
        
    // output something to the console
    void show_status(const Board &board,const Move bestMove,int faillow,int failhigh,int complete);

    void iterate();

    int ply0_search(int alpha, int beta,
            int depth, Move exclude [], int num_exclude, int multipv_count);

    void update_stats(int score, int alpha, int beta,int multipv_count);

    void update_history(NodeInfo *parentNode, Move move, int cutoff, 
       int depth, ColorType side);

    Move excludes[MAX_PV];
    int num_excludes;
    Search_Type type_of_search;
    int easy;
    Move easy_move;
    Move ply0first; // first move to search
    ColorType computerSide;
    Statistics *stats;
    int ply0_move_count;
    Move ply0_moves[Constants::MaxMoves];
    time_t time_target, start_time, xtra_time, time_added;
    int time_check_counter;
    time_t last_time;
    int ply_limit;
    int bkgrnd_search;
    int root_total_mat;
    int null_depth;
    int terminated, time_up;
    Board initial_board;
    StateType state;
    int iteration_depth;
    SearchContext context;
    Root_Move_Generator *mg;
    LockDefine(history_lock);
};

class SearchController {
    // Manages shared state that is not particular to a search
    // thread
    friend class Search;
    friend class RootSearch;
    friend struct ThreadInfo;
public:

    SearchController();

    virtual ~SearchController();

    Move find_best_move( 
            const Board &board, 
            Search_Type srcType,
            int32 time_limit,
            int32 xtratime,
            int ply_limit,
            int background,
            int uci,
            Statistics &stats,
            Talk_Level t);
    
    Move find_best_move( 
            const Board &board, 
            Search_Type srcType,
            int32 time_limit,
            int32 xtratime,
            int ply_limit,
            int background,
            int uci,
            Statistics &stats,
            Talk_Level t,
            Move *exclude,
            int num_excludes);
    
    void register_command_function(void (*cmd)(char *,int &)) {
       command_function = cmd;
    }
     
    void register_pending_function(void (*cmd)(const char *)) {
       pending_function = cmd;
    }
     
    PostFunction register_post_function(PostFunction post) {
       PostFunction tmp = post_function;
       post_function = post;
       return tmp;
    }
     
    TerminateFunction register_terminate_function(TerminateFunction term) {
       TerminateFunction tmp = terminate_function;
       terminate_function = term;
       return tmp;
    }
     
    void set_background(int b);
    
    void init();

    void reinit();

    void setRatingDiff(int rdiff);

    // check console input
    int check_input(const Board &);

    // return # of nodes visited in current search
    uint64 get_numnodes() const;
            
    Talk_Level get_talk_level() const {
       return talkLevel;
    }

    void set_talk_level(Talk_Level t)
    {
       talkLevel = t;
    }
    
    int draw_score(const Board & board, ColorType computerSide,
       int root_total_mat);
    
    void set_time_limit(time_t limit,time_t xtra_time_limit) {
       rootSearch->set_time_limit(limit,xtra_time_limit);
    }

    RootSearch *getRootSearch() const {
      return rootSearch;
    }

    void clearHashTables();

#ifdef SMP
    void setThreadCount(int n) {
      pool->resize(n,this);
    }
#endif

private:
    void cleanup(int clearHash);

    // pointer to function, called to process a command
    // received during the search (console mode only).
    void (*command_function)(char *,int &);

    // pointer to function, called to output status during
    // a search (console mode only).
    void (*post_function)(const Statistics &);

    int (*terminate_function)(const Statistics &);

    // Function to add a command to the deferred execution
    // stack (UNIX_ONLY)
    void (*pending_function)(const char*);

    // Calculate the recommended hash table size (expressed
    // in number of entries).  
    static long calc_hash_size();
    
    RootSearch *rootSearch;
    int age;
    int explicit_excludes;
    int use_hash_table;
    int hash_table_size;
    int ratingDiff;
    int ratingFactor; 
    int pipe;
    int terminated;
    int uci;
    Talk_Level talkLevel;

    LockDefine(input_lock);

#ifdef SMP
    ThreadPool *pool;
#endif
};

#endif
