// Copyright 1987-2008 by Jon Dart.  All Rights Reserved.

#include "bearing.h"
#include "search.h"
#include "scoring.h"
#include "constant.h"
#include "movegen.h"
#include "moveord.h"
#include "util.h"
#include "pinfo.h"
#include "attacke.h"
#include "log.h"
#include "globals.h"
#include "notation.h"
#include "debug.h"
#include "config.h"
#include "tbprobe.h"
#include "legal.h"

#ifdef _WIN32
static const UINT SEARCH_TIMER = 2;
#include <process.h>
#include <wincon.h>
#endif
#include <iostream>
#include <iomanip>
#include <limits.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

void add_pending(char *buf);

static const int Illegal = -Constants::BIG - 1000;
static int Time_Check_Interval = 10000;

#if defined(EXTENDED_FUTILITY_PRUNE)
const int FUTILITY_RANGE = EXTENDED_FUTILITY_DEPTH;
#else
const int FUTILITY_RANGE = FUTILITY_DEPTH;
#endif
#define PRUNE -Constants::BIG

static const int limits[64] =
{
    0, 2,   4,  6,  8, 10, 12, 14, 16, 18,
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
    30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
    40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
    50, 51, 52, 53, 54, 55, 56, 57, 58, 59,
    60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
    70, 71, 72, 73
};

static int SPLIT_DEPTH[32];

void SearchController::init()
{
    // Allocate the hash table.  Size is in kilobytes.  Guesstimate
    // a conversion to the number of hash table entries.
    int hash_entries;
    use_hash_table = options.search.hash_table_size > 0;
    hash_entries = (1000*options.search.hash_table_size)/sizeof(Hash_Entry);
    if (hash_table_size != options.search.hash_table_size) {
        // hash table was resized (GUI can do this)
        reinit();
    }
    if (use_hash_table) {
        initHash(hash_entries);
        hash_table_size = options.search.hash_table_size;
    }
    for (int i = 0; i < 32; i++ ) {
        if (i<=6) SPLIT_DEPTH[i] = (DEPTH_INCREMENT*THREAD_SPLIT_MIN_DEPTH*7)/4;
        else if (i<=8) SPLIT_DEPTH[i] = (DEPTH_INCREMENT*THREAD_SPLIT_MIN_DEPTH*5)/4;
        else SPLIT_DEPTH[i] = DEPTH_INCREMENT*THREAD_SPLIT_MIN_DEPTH;
    }
}


void SearchController::cleanup(int clearHash)
{
    delete rootSearch;
#ifdef SMP
    delete pool;
#endif
    LockDestroy(input_lock);
}


void SearchController::clearHashTables()
{
    age = 0;
#ifdef SMP
    pool->clearHashTables(this);
#else
    rootSearch->clearHashTables();
#endif
    clearHash();
}


long SearchController::calc_hash_size()
{
    // Size the hash table according to how much memory we have.
    static int sizes[12]= {
        997, 1997, 3001, 4003, 4999, 6007, 6997, 8003,
        10007, 12007, 15013, 18103
    };

    int i = 0;
#ifdef _WIN32
    SIZE_T memSize = GlobalCompact(0);
    for (i = 0; i < 11; i++)
        if (sizeof(Position_Info)*((long)sizes[i]) > memSize/16)
            break;
    return 32L*sizes[i];
#else
    return 32L*sizes[i];
#endif
}


SearchController::SearchController()
:
command_function(NULL),post_function(NULL),terminate_function(NULL),
pending_function(NULL),
rootSearch(NULL),
age(0),
ratingDiff(0),
ratingFactor(0)
{
    LockInit(input_lock);
    hash_table_size = options.search.hash_table_size;
#ifdef SMP
    pool = new ThreadPool(this,options.search.ncpus);
    ThreadInfo *ti = pool->mainThread();
    ti->state = ThreadInfo::Working;
    rootSearch = (RootSearch*)ti->work;
    ti->parentNode = ti->work->getNode();
#else
    rootSearch = new RootSearch(this);
#endif

    init();
#ifdef _WIN32
    HANDLE inh = GetStdHandle(STD_INPUT_HANDLE);
    DWORD dw;
    pipe = !GetConsoleMode(inh, &dw);
    if (!pipe) {
        SetConsoleMode(inh, dw & ~(ENABLE_LINE_INPUT |
            ENABLE_MOUSE_INPUT | ENABLE_WINDOW_INPUT));
    }
#endif
}


SearchController::~SearchController()
{
    cleanup(1);
}


Move SearchController::find_best_move(
const Board &board,
Search_Type srcType,
int32 timelimit,
int32 xtratime,
int plylimit,
int background,
int isUCI,
Statistics &stat_buf,
Talk_Level t)
{
    explicit_excludes = 0;
    Move excludes[1];
    return find_best_move(board,srcType,timelimit,xtratime,plylimit,background,
        isUCI, stat_buf, t, excludes, 0);
}


Move SearchController::find_best_move(
const Board &board,
Search_Type srcType,
int32 timelimit,
int32 xtratime,
int plylimit,
int background,
int isUCI,
Statistics &stat_buf,
Talk_Level t,
Move *exclude, int num_exclude)
{
    uci = isUCI;
    if (num_exclude) explicit_excludes = 1;
    // Positions are stored in the hashtable with an "age" to identify
    // which search they came from.  "Newer" positions can replace
    // "older" ones. Update the age here since we are starting a
    // new search.
    age = (age + 1) % 3;
    rootSearch->init(board);
    return rootSearch->ply0_search(srcType,timelimit,xtratime,plylimit,
        background,stat_buf,t,exclude,num_exclude);
}


void SearchController::setRatingDiff(int rdiff)
{
    ratingDiff = rdiff;
    ratingFactor = 0;
    if (ratingDiff < -100) {
        ratingFactor = Util::Max((PAWN_VALUE*(ratingDiff+100))/100,-PAWN_VALUE);
    }
    else if (ratingDiff > 100) {
        ratingFactor = Util::Min((PAWN_VALUE*(ratingDiff-100))/100,PAWN_VALUE);
    }
}


int SearchController::draw_score(const Board & board, ColorType computerSide,
int root_total_mat)
{
    int score = 0;

    // if we know the opponent's rating (which will be the case if playing
    // on ICC), factor that into the draw score - a draw against a high-rated
    // opponent is good; a draw against a lower-rated one is bad.
    if (ratingDiff != 0) {
        if (board.Side() == computerSide)
            score += ratingFactor;
        else
            score -= ratingFactor;
    }
    /**
    // Penalize the side to move for allowing a draw if it is ahead in
    // material
    int mat = Scoring::material_score(board);
    if (mat > 2*PAWN_VALUE) {
        score -= (mat-2*PAWN_VALUE)/8;
    }
    // A draw is worse if there is a lot of material on the board
    if (root_total_mat > 6000) {
        score -= 16;
    }
    else if (root_total_mat > 4800) {
        score -= 16*(root_total_mat-4800)/1200;
    }
    **/
    return score;
}


void SearchController::set_background(int b)
{
    rootSearch->set_background(b);
}


void SearchController::reinit()
{
    // After the search options change, we may need to reallocate
    // the hash table.  Free it here so that next time through init()
    // the reallocation will be done.
    freeHash();
}


#ifdef _WIN32
static int is_move(const Board &b,char *buf)
{
    if (isalpha(*buf) && isdigit(buf[1]) &&
        isalpha(buf[2]) && isdigit(buf[3]))
        return 1;
    else if (strncmp(buf,"o-o",3)==0 || strncmp(buf,"O-O",3)==0 ||
        strncmp(buf,"o-o-o",5)==0 || strncmp(buf,"O-O-O",5) ==0)
        return 1;
    else
        return Notation::Value(b,b.Side(),buf) != NullMove;
}
#endif

int SearchController::check_input(const Board &board)
{
    // two threads should not be checking input at once, at least not
    // under Windows
    Lock(input_lock);
#ifdef _WIN32
    DWORD nchar;
    DWORD dw;
    if (pipe) {
        // Use the GnuChess trick of calling PeekNamedPipe to check
        // for user input.  We need to do this because Winboard does
        // not send ^C to interrupt pondering.
        if (PeekNamedPipe(GetStdHandle(STD_INPUT_HANDLE), NULL, 0,
        NULL, &nchar, NULL)) {
            if (nchar && talkLevel == Trace) cout << "(debug) got console input nchar=" << nchar << endl;
            int nextc;
            BOOL terminate;
            for (unsigned i = 0; i < nchar; i++) {
                nextc=getc(stdin);
                //                if (talkLevel == Trace) cout << "(debug) got: " << (char)nextc << endl;
                if (nextc == '\n') {
                    cmd_buf[cmd_buf_count]  = '\0';
                    cmd_buf_count = 0;
                    // call a function (supplied by the calling code) that
                    // sets terminate = 1 if we should terminate.
                    command_function(cmd_buf,terminate);
                    if (terminate) {
                        if (talkLevel == Trace) cout << "(debug) terminating" << endl;
                        Unlock(input_lock);
                        return 1;
                    }
                }
                else
                    cmd_buf[cmd_buf_count++] = nextc;
            }
            // Hack alert. Sometimes we get a move with no trailing
            // carriage return.
            if (cmd_buf_count > 0) {
                cmd_buf[cmd_buf_count] = '\0';
                if (is_move(board,cmd_buf) || (strncmp(cmd_buf,"usermove",8) == 0 && is_move(board,cmd_buf+9))) {
                    if (talkLevel == Trace) cout << "(debug) got move w/o <CR>: " << cmd_buf << endl;
                    cmd_buf_count = 0;
                    command_function(cmd_buf,terminate);
                    if (terminate) {
                        Unlock(input_lock);
                        return 1;
                    }
                }
            }
        }
    }
    else {
        // We are reading from the console, not connected to Winboard.
        // Not something we usually do, except for testing.
        int retVal = WaitForSingleObject(GetStdHandle(STD_INPUT_HANDLE), 0);
        if (retVal == WAIT_OBJECT_0) {
            INPUT_RECORD recBuf[128];
            ReadConsoleInput(GetStdHandle(STD_INPUT_HANDLE),recBuf,
                128,&dw);
            for (unsigned i = 0; i < dw; i++) {
                if (!recBuf[i].Event.KeyEvent.bKeyDown) continue;
                char nextc=recBuf[i].Event.KeyEvent.uChar.AsciiChar;
                if (nextc == (char)13 /* carriage return */) {
                   putchar('\n');
                   cmd_buf[cmd_buf_count]  = '\0';
                   cmd_buf_count = 0;
                   BOOL terminate;
                   // call a function (supplied by the calling code) that
                   // sets terminate = 1 if we should terminate.
                   command_function(cmd_buf,terminate);
                   if (terminate) {
                       Unlock(input_lock);
                       return 1;
                   }
                   break;
               }
               else {
                   putchar(nextc);
                   cmd_buf[cmd_buf_count++] = nextc;
               }
           }
        }
    }
    Unlock(input_lock);
    return 0;
#else
    // UNIX version, uses select like Crafty does
    fd_set readfds;
    struct timeval tv;
    int data;

    FD_ZERO(&readfds);
    FD_SET(fileno(stdin), &readfds);
    tv.tv_sec=0;
    tv.tv_usec=0;
    select(16, &readfds, 0, 0, &tv);
    data=FD_ISSET(fileno(stdin), &readfds);
    if (data == -1)
        cerr << "warning: select returned error. errno=" << errno << endl;
    else if (data) {
        // we have something to read
        //     cout << "(debug) got input, data = " << data << endl;
        *cmd_buf = '\0'; cmd_buf_count = 0;
        char buf[256];
        memset(buf,'\0',256);
        int bytes = read(fileno(stdin),buf,255);
        if (bytes <= 0) {
            Unlock(input_lock);
            return 0;
        }
        char *p = buf;
        int stop = 0;
        while (*p) {
            char *q = cmd_buf;
            while (*p != '\n' && *p != '\0' && cmd_buf_count < 255) {
                *q++ = *p++;
                ++cmd_buf_count;
            }
            *q = '\0';
            p++;                                      // skip over null or CR
            //cout << "(debug) cmd_buf = " << cmd_buf << endl;
            // call a function (supplied by the calling code) that
            // sets terminate = 1 if we should terminate.
            command_function(cmd_buf,stop);
            cmd_buf_count = 0;
            if (stop) {
                rootSearch->terminate_now();
            }
        }
        Unlock(input_lock);
        return rootSearch->was_terminated();
    }
    Unlock(input_lock);
    return(0);
#endif
}

#ifdef SMP
RootSearch::RootSearch(SearchController *c,ThreadInfo *ti)
: Search(c,ti)
#else
RootSearch::RootSearch(SearchController *c)
   :Search(c)
#endif
{
    root = c->rootSearch = this;
    parent = this;
#ifdef SMP
    //   ti = controller->pool->mainThread();
    LockInit(history_lock);
#endif
}


RootSearch::~RootSearch()
{
    LockDestroy(history_lock);
}


Move RootSearch::ply0_search(Search_Type srcType,
int32 timelimit,
int32 xtratime,
int plylimit,
const int background,
Statistics &stat_buf,
Talk_Level t,
Move *exclude, int num_exclude)
{

    controller->set_talk_level(t);
    time_target = timelimit;
    xtra_time = xtratime;
    ply_limit = plylimit;
    if (ply_limit > 0)
        ply_limit = Util::Min(ply_limit-1,Constants::MaxPly-1);
    type_of_search = srcType;
    num_excludes = Util::Min(MAX_PV,num_exclude);
    if (num_excludes)
        memcpy(excludes,exclude,sizeof(Move)*num_excludes);
    state = NormalState;
    stats = &stat_buf;
    stats->clear();
    node = stack;                                 // top of stack
#ifdef EVAL_STATS
    Scoring::clearStats();
#endif
    null_depth = options.search.null_depth_reduction*DEPTH_INCREMENT;

    bkgrnd_search = background;
    initial_board = board;

    node->best = NullMove;
    if (Scoring::is_legal_draw(board)) {
        // If it's a legal draw situation before we even move, then
        // just return a draw score and don't search.
        stats->state = Draw;
        stats->value = draw_score(board);
    }
    else {
        // Generate the ply 0 moves here:
        Root_Move_Generator rmg(board,&root->context,ply0_moves,
            NullMove,1);
        mg = &rmg;
        ply0_move_count = stats->mvtot = rmg.move_count();
        root->stats->num_moves += ply0_move_count;
        // See if there is an "easy" move (a recapture that appears to
        // gain more material than any other alternative).  If so we
        // may be able to terminate the search early.
        easy = (ply0_move_count > 0 &&
            board[DestSquare(rmg.first())] != EmptyPiece &&
            game_moves->num_moves() > 0 &&
            (DestSquare(game_moves->move(game_moves->num_moves()-1)) ==
            DestSquare(rmg.first())) &&
            (Capture(game_moves->move(game_moves->num_moves()-1)) != EmptyPiece &&
            Capture(rmg.first()) != EmptyPiece)
            );
        easy_move = rmg.first();
        // Do the actual search
        iterate();
    }
    static const int end_of_game[] = {0, 0, 0, 1, 1, 1, 1};
    stats->end_of_game = end_of_game[(int)stats->state];
    if (!controller->uci && !stats->end_of_game && options.search.can_resign) {
        if (stats->depth >= 2) {
            // Use the display value - i.e. do not use score if
            // we are failing low
            if (stats->display_value <= 8-(int)Constants::BIG) {
                // TBD we can sometimes get -10000 as a score!
                // this is bogus, we should not resign.
                if (stats->display_value != -(int)Constants::BIG)
                    stats->state = Resigns;
            }
            else if (stats->display_value > 200-Constants::BIG &&
                ((100*stats->display_value)/PAWN_VALUE <= options.search.resign_threshold))
                stats->state = Resigns;
            stats->end_of_game = end_of_game[(int)stats->state];
        }
    }

    if (controller->talkLevel == Debug) {
        cout.setf(ios::fixed);
        cout << setprecision(2);
        if (stats->elapsed_time > 0) {
            char buf[128];
            stats->print_nps(buf);
            cout << buf << " nodes/second." << endl;
        }
        cout << (stats->num_nodes-stats->num_qnodes) << " regular nodes, " <<
            stats->num_qnodes << " quiescence nodes." << endl;
        cout << stats->hash_searches << " searches of hash table, " <<
            stats->hash_hits << " successful";
        if (stats->hash_searches != 0)
            cout << " (" <<
                (int)((100.0*(float)stats->hash_hits)/((float)stats->hash_searches)) <<
                " percent).";
        cout << endl;
        cout << stats->hash_inserts << " hash entries inserted, " <<
            stats->hash_replaces << " entries replaced, " <<
            stats->hash_inserts_failed << " inserts failed." << endl;
#ifdef MOVE_ORDER_STATS
        cout << "move ordering: ";
        static const char *labels[] = {"1st","2nd","3rd","4th"};
        for (int i = 0; i < 4; i++) {
            cout << setprecision(2) << labels[i] << " " <<
                (100.0*stats->move_order[i])/(float)stats->move_order_count << "% " ;
        }
        cout << endl;
#endif
        cout <<  "null cutoffs: " << stats->num_null_cuts << " (" <<
            (int)((100.0*(float)stats->num_null_cuts)/((float)stats->num_nodes)) <<
            "%)" << endl;
        cout << "pruning: " <<
#ifdef FUTILITY_PRUNE
        stats->futility_pruning << " moves (futility)";
        cout << ", ";
#endif
        cout << stats->num_razored << " nodes razored";
        cout << endl;
        cout << "extensions: " <<
             stats->check_extensions << " check, " <<
             stats->recap_extensions << " recapture, " <<
             stats->pawn_extensions << " pawn, " << endl;
        cout << "            " <<
           stats->threat_extensions << " threat, " <<
	   stats->forced_extensions << " forced move." << endl;
        cout << stats->tb_probes << " tablebase probes, " <<
           stats->tb_hits << " tablebase hits" << endl;
#if defined(SMP) && defined(SMP_STATS)
        cout << stats->splits << " splits," <<
            " average thread usage=" << (float)(stats->threads)/(float)stats->samples << endl;
#endif
#ifdef EVAL_STATS
        extern long position_hash_probes, position_hash_hits,
            pawn_hash_probes, pawn_hash_hits,
            king_pawn_hash_probes, king_pawn_hash_hits,
            king_cover_hash_probes, king_cover_hash_hits,
            lazy_cut_chances, lazy_cuts;
        cout << "position hash hit rate = " << 100.0*(float)position_hash_hits/(float)position_hash_probes << "%" << endl;
        cout << "pawn hash hit rate     = " << 100.0*(float)pawn_hash_hits/(float)pawn_hash_probes << "%" << endl;
        cout << "king/pawn hash hit rate= " << 100.0*(float)king_pawn_hash_hits/(float)king_pawn_hash_probes << "%" << endl;
        cout << "king cover hash hit rate= " << 100.0*(float)king_cover_hash_hits/(float)king_cover_hash_probes << "%" << endl;
        if (lazy_cuts > 0L)
            cout <<"lazy cutoff in eval    = " << 100.0*(float)lazy_cuts/(float)lazy_cut_chances << "%" << endl;
#endif
        fflush(stdout);
    }
    return node->best;
}


void RootSearch::set_background(int b)
{
    bkgrnd_search = b;
    if (!b) {
        // changing from background to foreground search,
        // update display
        show_status(initial_board,getBest(),0,0,0);
    }
}


void RootSearch::show_status(const Board &board,const Move best,int faillow,
int failhigh,int complete)
{
    if (was_terminated())
        return;
    int ply = stats->depth;
    static char move_text[20];
    Notation::Image(board, best, move_text);
    char val[20];
    stats->complete = complete;
    Scoring::print_score(stats->display_value,val);
    if (controller->talkLevel == Debug) {
        cout.setf(ios::fixed);
        cout << setprecision(2);
        cout << ply << '\t';
        cout << stats->elapsed_time/100.0 << '\t';
        if (faillow) {
            cout << " --";
        }
        else if (best != NullMove) {
            cout << move_text;
            if (failhigh) cout << '!';
        }
        cout << '\t' << val <<
            '\t' << stats->num_nodes << endl;
    }
    // Post during ponder if UCI
    if (!is_background_search() || controller->uci) {
        if (controller->post_function) {
            controller->post_function(*stats);
        }
    }
}


void RootSearch::init(const Board &aboard)
{
    board = aboard;
    if (board.state.moveCount>=90) {
        clearHash();
    }
    time_up = terminated = 0;
    time_added = 0;
    root_total_mat = board.getMaterial(White).value() + board.getMaterial(Black).value();
    start_time = getTime();
}


int RootSearch::check_time(const Board &board,int ply, int rank,
int iteration_depth)
{
    if (was_terminated()) {
        return 1;
    }

    if (controller->command_function && controller->check_input(board)) {
        terminate_now();
        if (controller->talkLevel == Trace)
            cout << "(debug) terminating, command input received" << endl;
        return 1;
    }
#if defined(SMP) && defined(SMP_STATS)
    stats->samples++;
    stats->threads += controller->pool->activeCount();
#endif
    time_t current_time = (time_t)(getTime());
    if (type_of_search != Fixed_Depth) {
        // update at least the part of the stats structure that
        // is displayed in analysis mode.
        stats->elapsed_time = current_time - start_time;
        if (current_time - (int)get_start_time() > (int)get_time_limit()) {
            if (controller->talkLevel == Trace) {
                terminate_now();
                cout << "(debug) terminating, time up" << endl;
            }
            return 1;
        }
        if (type_of_search == Fixed_Time) {
            return 0;
        }
        unsigned n = theLog->current();

        if (xtra_time > 0 &&
            type_of_search == Time_Limit &&
            (stats->elapsed_time > time_target/2) &&
        time_added == 0) {
            // see if our score has dropped a lot since our
            // last search
            if (n >= 2 && !(*theLog)[n-2].get_book_info().is_valid() &&
                (stats->value < 5*PAWN_VALUE) &&
                (stats->value > -PAWN_VALUE*5) &&
            (stats->value < (*theLog)[n-2].score() - PAWN_VALUE/3)) {
                // search more time because our score is dropping
                time_added++;
                if (controller->talkLevel == Trace) {
                    cout << "(debug) adding time" << endl;
                }
            }
        }
        // Allow terminating early if this is a recapture, appeared
        // better than everything else at initial sort time, and
        // still looks good.
        if (!time_up && easy && iteration_depth >= 5) {
            // do not do this if this is the first move out of book
            if (n>=2 && !(*theLog)[n-2].get_book_info().is_valid() &&
                time_target >= 300 &&
                // Don't do this if our score is dropping ..
                (stats->value > (*theLog)[n-2].score() - PAWN_VALUE/4) &&
                // require that we have consumed at least 1/3 of the time
                // limit:
            (current_time - get_start_time())*3 > time_target) {
                if (getBest() == easy_move &&  PieceMoved(getBest()) != King) {
                    time_up = 1;
                    if (controller->talkLevel == Trace) {
                        cout << "(debug) terminating, easy move" << endl;
                    }
                    terminate_now();
                    return 1;
                }
            }
        }
    }
    if (controller->uci && (current_time-last_time >= 200)) {
        extern int hashSize;
        cout << "info";
        if (stats->elapsed_time>30) cout << " nps " <<
                (long)((100L*stats->num_nodes)/stats->elapsed_time);
        cout << " nodes " << stats->num_nodes << " hashfull " << (1000*stats->new_hash_inserts)/hashSize << endl;
        last_time = current_time;
    }
    if ( time_up) {
        if (controller->talkLevel == Trace)
            cout << "(debug) terminating, time up" << endl;
        terminate_now();
    }
    return time_up;
}


Move Search::getBest() const
{
    return root->stack->best;
}


int Search::draw_score(const Board &board)
{
    return controller->draw_score(board,
				  root->computerSide,root->root_total_mat);
}


void RootSearch::update_stats(int score, int alpha, int beta, int multipv_count)
{
    time_t end_time = getTime();
    stats->elapsed_time = end_time - get_start_time();
    stats->multipv_count = multipv_count;
    // Note: the search routine counts depth internally starting
    // at 0, but we report it starting at 1 (as most other programs do).
    stats->depth = iteration_depth+1;
    stats->value = score;
    // Leave the previous value intact if we have failed low.
    if (!(stats->value == alpha && iteration_depth > 0)) {
        stats->display_value = stats->value;
    }
    stats->state = state;
    Board board_copy(root->initial_board);

    // note: retain previous best line if we do not have one here
    if (IsNull(node->pv[0])) {
#ifdef _TRACE
        cout << "(debug) warning: pv is null\n";
#endif
        return;
    }
    else if (node->pv_length == 0) {
        return;
    }
    node->best = node->pv[0];                     // ensure "best" is non-null
    ASSERT(!IsNull(node->best));
    *(stats->best_line_image) = '\0';
    stats->best_line[0] = NullMove;
    int i = 0;
    char *p = stats->best_line_image;
#ifdef _TRACE
    cout << "best line:" << endl;
#endif
    while (i < node->pv_length && !IsNull(node->pv[i])) {
        ASSERT(i<Constants::MaxPly);
        stats->best_line[i] = node->pv[i];
        char move_image[20];
        *move_image = '\0';
#ifdef _TRACE
        MoveImage(node->pv[i],cout); cout << ' ' << (flush);
#endif
        ASSERT(legal_move(board_copy,stats->best_line[i]));
        if (controller->uci) {
            // UCI format for moves
            Notation::UCIMoveImage(stats->best_line[i],move_image);
        }
        else {
            // use SAN
            Notation::Image(board_copy,stats->best_line[i],move_image);
        }
        int len = (int)strlen(move_image);
        if (p+len+1-stats->best_line_image > 255)
            break;
        strcpy(p,move_image);
        p+= len;
        strcpy(p," ");
        p++;
        board_copy.DoMove(stats->best_line[i]);
        ++i;
        int rep_count;
        if (is_draw(board_copy,rep_count,0)) {
            break;
        }
        if (node->pv_length < 2 && controller->use_hash_table) {
            // get the next move from the hash table, if possible
            // (for pondering)
            Position_Info entry;
            Lock(hash_lock);
            Position_Info::ValueType result =
                searchHash(board_copy,board_copy.HashCode(rep_count),
                0,0,0,
                iteration_depth,entry);
            Unlock(hash_lock);
            if (result != Position_Info::NoHit) {
                Move hashMove = entry.best_move(board_copy);
                if (!IsNull(hashMove)) {
                    stats->best_line[i] = hashMove;
                    *p = '\0';
                    if (controller->uci) {
                        // UCI format for moves
                        Notation::UCIMoveImage(hashMove,move_image);
                    }
                    else {
                        // use SAN
                        Notation::Image(board_copy,hashMove,move_image);
                    }
                    int len = (int)strlen(move_image);
                    if (p+len+1-stats->best_line_image > 255)
                        break;
                    strcpy(p,move_image);
                    p+=len;
                    ++i;
                }
                break;
            }
        }
    }
#ifdef _TRACE
    cout << endl;
#endif
    *p = '\0';

    stats->best_line[i] = NullMove;
#ifdef _TRACE
    cout << "best line image: " << stats->best_line_image << endl;
#endif
}


uint64 SearchController::get_numnodes() const
{
    // return # of nodes visited
    return rootSearch ? rootSearch->stats->num_nodes : 0;
}


void RootSearch::terminate_now()
{
    if (controller->get_talk_level() == Trace)
        cout << "(debug) terminating search" << endl;
    time_up = terminated = terminate = 1;
    stopAllThreads();
}


void RootSearch::iterate()
{
    // Searching may alter the board, so operate on a copy:
    Board board = initial_board;

    Time_Check_Interval = 10000;

    computerSide = board.Side();

    terminate = 0;
    time_check_counter = 0;
    last_time = 0;

    ply0first = NullMove;
    int value = Scoring::INVALID_SCORE;
    int tb_hit = 0, tb_pieces = 0;
    if (options.search.use_tablebases) {
        const Material &wMat = board.getMaterial(White);
        const Material &bMat = board.getMaterial(Black);
        int tb_score;
        tb_pieces = wMat.men() + bMat.men();
        if(tb_pieces <= EGTBMenCount) {
            stats->tb_probes++;
            tb_hit = probe_tb(board, tb_score, 0);
            if (tb_hit) {
                value = tb_score;
                stats->tb_hits++;
            }
            if (tb_hit && controller->talkLevel == Trace)
                cout << "(debug) tb hit, score=" << value << endl;
        }
    }
    stats->value = stats->display_value = Scoring::evalu8(board,this);
    if (type_of_search == Fixed_Time || type_of_search == Time_Limit)
        ply_limit = Constants::MaxPly;

    // Incrementally search the board to greater depths - stop when
    // ply limit, time limit, interrupt, or a terminating condition
    // is reached.
    node->best = NullMove;
    for (iteration_depth = 0;
        iteration_depth <= ply_limit && !terminate && !time_up;
        iteration_depth++) {
        if (!controller->explicit_excludes) num_excludes = 0;
        for (int multipv_count=0; multipv_count < options.search.multipv; multipv_count++) {
            int lo_window, hi_window;
	    if (iteration_depth == 0) {
	      lo_window = -Constants::BIG; hi_window = Constants::BIG;
	    }
	    else {
               lo_window = Util::Max(-Constants::BIG,value - SEARCH_WINDOW / 2);
               hi_window = Util::Min(value + SEARCH_WINDOW / 2,
                Constants::BIG);
	    }
            if (controller->get_talk_level() == Trace && is_background_search()) {
                cout << "(debug) " << iteration_depth << ". move=";
                MoveImage(node->best,cout); cout << " score=" << node->score << endl;
            }

#ifdef _TRACE
            cout << "iteration " << iteration_depth << " window = [" <<
                lo_window << "," << hi_window << "]" << endl;
#endif
            int failhigh = 0;
            int faillow = 0;
            int oldvalue = -Constants::BIG;
            Move oldbest = NullMove;
            for (int count = 0; 1; ++count,oldvalue = value,oldbest = getBest()) {
                if (!failhigh && !faillow)
                    ply0first = node->pv[0];
                value = ply0_search(lo_window, hi_window,
                    DEPTH_INCREMENT*iteration_depth,
                    excludes,num_excludes,multipv_count);
                update_stats(value,lo_window,hi_window,multipv_count);
                if (iteration_depth == 0 && Scoring::is_legal_draw(board)) {
                    if (controller->talkLevel == Trace)
                        cout << "(debug) draw, terminating" << endl;
                    terminate_now();
                }
                // check for forced move, but only at depth 2
                // (so we get a ponder move if possible). But exit immediately
                // if a tb hit because deeper search will hit the q-search and
                // the score will be inaccurate. Do not terminate here if a
                // resign score is returned (search deeper to get an accurate
                // score).
                else if (ply0_move_count == 1 &&
                    (iteration_depth >= 2) &&
                    (!options.search.can_resign ||
                    ((100*stats->display_value)/PAWN_VALUE >
                options.search.resign_threshold))) {
                    if (controller->talkLevel == Trace)
                        cout << "(debug) single legal move, terminating" << endl;
                    terminate_now();
                }
                if (!was_terminated() && (state == Checkmate || state == Stalemate)) {
                    if (controller->talkLevel == Trace)
                        cout << "(debug) terminating due to checkmate or statemate, state="
                            << (int)state << endl;
                    terminate_now();
                    break;
                }
                if (stats->elapsed_time > 100) {
                    long denom = bkgrnd_search ? 200L : 500L;
                    Time_Check_Interval = (int)(
                        (denom*stats->num_nodes)/(36L*stats->elapsed_time));
                }
                if (!terminate) {
                    terminate = check_time(board,0,0,iteration_depth);
                    if (terminate && controller->talkLevel == Trace)
                        cout << "(debug) time up" << endl;
                    if (controller->terminate_function &&
                    controller->terminate_function(*stats)) {
                        if (controller->talkLevel == Trace)
                            cout << "(debug) terminating due to program or user input" << endl;
                        terminate_now();
                    }
                }
                if (count == 0 || ((value != oldvalue) || !MovesEqual(getBest(),oldbest))) {
                    show_status(board,getBest(),value <= lo_window,value >= hi_window,value < hi_window);
                }
                if (terminate) {
                    break;
                }
                else {
                    if (value < hi_window && value > lo_window) {
                        // score is in the bounds of the search window,
                        // no need to re-search
                        break;
                    }
                    if (value >= hi_window) {
#ifdef _TRACE
                        cout << "ply 0 high cutoff, re-searching ..." << endl;
#endif
                        // high cutoff, must re-search
                                                  // so we research this move
                        ply0first = node->last_move;
#ifdef TWO_STAGE_WINDOW
                        if (failhigh) {
#endif
                            hi_window = Constants::BIG;
#ifdef TWO_STAGE_WINDOW
                        }
                        else {
                            // try to expand window only a little
                            hi_window = Util::Min(node->score+2*PAWN_VALUE,Constants::BIG);
                        }
#endif
                        lo_window = Util::Min(value,Constants::BIG-Constants::MaxPly);
                        failhigh++;
                        faillow = 0;
                    }
                    else if (value <= lo_window && !failhigh) {
#ifdef _TRACE
                        cout << "ply 0 low cutoff, re-searching ..." << endl;
                        cout << "value = " << value << endl;
#endif
                        hi_window = lo_window;
#ifdef TWO_STAGE_WINDOW
                        if (faillow) {
#endif
                            lo_window = -Constants::BIG;
#ifdef TWO_STAGE_WINDOW
                        }
                        else {
                            // try to expand window only a little
                            lo_window = Util::Max(node->score-2*PAWN_VALUE,-Constants::BIG);
                        }
#endif
                        failhigh = 0;
                        faillow++;
                    }
                    else {
#ifdef _TRACE
                        cout << "using maximal window" << endl;
#endif
                        // Use infinite window - next search score should be in
                        // bounds.
                        hi_window = Constants::BIG;
                        lo_window = -Constants::BIG;
                        failhigh = faillow = 0;
                    }
                }
            }
#ifdef _TRACE
            cout << iteration_depth+1 << " ply search result: ";
            MoveImage(node->best,cout);
            cout << " value = ";
            char buf[20];
            Scoring::print_score(value,buf);
            cout << buf << endl;
#endif
            if (options.search.multipv > 1) {
                excludes[num_excludes++] = node->pv[0];
            }
            // allow early termination on a tablebase position, but not
            // if we have >=6 man tbs in use (because tb set may be
            // incomplete - in that case it is better to allow us to
            // search deeper those nodes that don't produce a tb hit).
            //
            if (tb_hit && tb_pieces<6 && iteration_depth>=3 && !IsNull(node->pv[0])) {
                if (controller->talkLevel == Trace)
                    cout << "(debug) terminating, tablebase hit" << endl;
#ifdef _TRACE
                cout << "terminating, tablebase hit" << endl;
#endif
                terminate_now();
                break;
            }
            if (value < (int)(iteration_depth*2) - Constants::BIG) {
                // We're either checkmated or we certainly will be, so
                // quit searching.
                if (controller->talkLevel == Trace)
                    cout << "(debug) terminating, low score" << endl;
#ifdef _TRACE
                cout << "terminating, low score" << endl;
#endif
                terminate_now();
                break;
            }
            else if (value >= (Constants::BIG - (int)(iteration_depth*2)
            - 1)) {
                if (iteration_depth>=2) {
                    if (controller->talkLevel == Trace)
                        cout << "(debug) terminating, high score" << endl;
#ifdef _TRACE
                    cout << "terminating, mate score" << endl;
#endif
                    terminate_now();
                    break;
                }
            }
        }
    }
}


int RootSearch::ply0_search(int alpha, int beta,
int depth, Move exclude [], int num_exclude, int multipv_count)
{

    // implements alpha/beta search for the top most ply.  We use
    // the negascout algorithm.

    time_check_counter++;
    stats->num_nodes++;

    int in_pv = 1;
    int in_check = 0;
    int initial_alpha = alpha;

    // At this point we need to know if we are in check or not.
    in_check = (board.CheckStatus() == InCheck);
    Board_State save_state = board.state;

    int try_score = alpha;
    //
    // re-sort the ply 0 moves.
    mg->reorder(node->best);
    mg->exclude(exclude,num_exclude);

    //
    // Search the next ply
    //
    NodeInfo *rootNode = (NodeInfo*)node;
#ifdef SMP
    node->parentNode = rootNode;
#endif
    node->pv[0] = NullMove;
    node->pv_length = 0;
    node->cutoff = 0;
    node->extensions = 0;
    node->num_try = 0;                            // # of legal moves tried
    node->beta = node->hibound = beta;
    node->alpha = alpha;
    node->score = node->alpha;
#ifdef MOVE_ORDER_STATS
    node->best_count = 0;
#endif
    node->pos_score = Scoring::INVALID_SCORE;
    node->fpruned_moves = 0;
#ifdef SMP
    node->ply = 0;
    node->depth = depth;
    node->split = NULL;
    node->splits = 0;
    node->mg = mg;
    // clear split point stack:
    split_stack_end = split_stack;
#endif
    int move_index = 0;
    const int canSplit = options.search.ncpus>1 &&
        depth >= SPLIT_DEPTH[board.getMaterial(board.Side()).material_level()];
    while (!rootNode->cutoff && !terminate) {
        Move move;
#ifdef SMP
        if ((move  = mg->NextMove(rootNode))==NullMove) break;
#else
        if ((move  = mg->NextMove())==NullMove) break;
#endif
	move_index++;
        rootNode->last_move = move;
        stats->mvleft = stats->mvtot-move_index;
        if (controller->uci && stats->elapsed_time) {
            cout << "info currmove ";
            char image[10];
            Notation::UCIMoveImage(move,image);
            cout << image;
            cout << " currmovenumber " << move_index;
            if (stats->elapsed_time >= 30) {
                cout << " nps " << (100L*stats->num_nodes)/stats->elapsed_time;
                cout << " nodes " << stats->num_nodes;
            }
            extern int hashSize;
            cout << " hashfull " << (1000*stats->new_hash_inserts)/hashSize;
            cout << endl;
        }
        if (IsMate(move)) continue;               // move leads to mate, skip it
        if (IsUsed(move)) continue;               // move was searched already, skip it
#ifdef _TRACE
        cout << "trying 0. ";
        MoveImage(move,cout);
        cout << " (" << move_index << "/" << ply0_move_count;
        cout << ")" << endl;
#endif
        node->last_move = move;
        board.DoMove(move);
        rootNode->num_try++;
#ifdef _TRACE
	cout << "window [" << -rootNode->hibound << ", " << -rootNode->alpha <<
	  "]" << endl;
#endif
        // note: no extensions at ply 0
        if (depth-DEPTH_INCREMENT >= 0) {
            try_score = -search(-rootNode->hibound, -rootNode->alpha,
                1, depth-DEPTH_INCREMENT);
        }
        else {
            try_score = -quiesce(-rootNode->hibound, -rootNode->alpha,
                1, 0);
        }
        if (terminate || rootNode->cutoff) {
            board.UndoMove(move,save_state);
            break;
        }
#ifdef _TRACE
        cout << "0. ";
        MoveImage(move,cout);
        cout << ' ' << try_score;
        if (in_pv) cout << " (pv)";
        cout << endl;
#endif
        if (!in_pv && 
        ((try_score > rootNode->alpha) && (try_score < rootNode->beta)) &&
	    !((node+1)->flags & EXACT)) {
            // We failed to get a cutoff and must re-search
#ifdef NEGASCOUT
	    int hibound = -try_score+1;
#else
	    int hibound = -node->alpha;
#endif
#ifdef _TRACE
            cout << "window = [" << -hibound << "," << rootNode->beta
                << "]" << endl;
            cout << "score = " << try_score << " - no cutoff, researching .." << endl;
#endif
            if (depth-DEPTH_INCREMENT >= 0)
                try_score=-search(
                    -rootNode->beta,hibound,1,depth-DEPTH_INCREMENT);
            else
                try_score=-quiesce(-rootNode->beta,hibound,1,0);
#ifdef _TRACE
            cout << "0. ";
            MoveImage(move,cout);
            cout << ' ' << try_score;
            cout << endl;
#endif
        }
        board.UndoMove(move,save_state);
        if (terminate || rootNode->cutoff) {
            break;
        }
        // If this move leads to being mated, remove it from the
        // ply 0 move list - there is no use searching it again
        if (try_score <= -Constants::BIG+(int)iteration_depth) {
            for (int i = 0; i < ply0_move_count; i++) {
                if (MovesEqual(ply0_moves[i],move)) {
#ifdef _TRACE
                    cout << "removing move ";
                    MoveImage(move,cout);
                    cout << endl;
#endif
                    SetMate(ply0_moves[i]);
                    continue;
                }
            }
        }
        if (try_score > rootNode->alpha) {
            if (update_root_move(board,rootNode,rootNode,move,try_score)) {
                // beta cutoff
                break;
            }
            if (!in_pv) {
                // pv change after 1st move
                //              show_status(board,move,0,0,0);
            }
        }
        rootNode->hibound = rootNode->alpha + 1;  // zero-width window
        in_pv = 0;
#ifdef SMP
        if (canSplit && maybe_split(board,rootNode,0,depth)) {
            break;
        }
#endif
    }
#ifdef SMP
    // We may have other threads still searching.
    maybe_wait(rootNode);
#endif
    if (rootNode->cutoff)
        return rootNode->score;
    if (in_pv && rootNode->score < initial_alpha) {
        // we have failed low and must reset the window
#ifdef _TRACE
        cout << "ply 0 alpha cutoff" << endl;
#endif
        if (controller->talkLevel == Debug) cout << "(debug) ply 0 alpha cutoff" << endl;
        return rootNode->score;
    }

    if (rootNode->num_try == 0) {
        // no moves were tried
        if (in_check) {
            if (ply0_move_count == 0) {           // mate
                rootNode->score = -(Constants::BIG);
                state = Checkmate;
            }
            else {
                // We generated some moves, and some were legal,
                // but we skipped them all.
                rootNode->score = evalu8(board,alpha,beta,0);
            }
        }
        else {                                    // stalemate
            state = Stalemate;
#ifdef _TRACE
            cout << "stalemate!" << endl;
#endif
            rootNode->score = draw_score(board);
        }
    }
#ifdef MOVE_ORDER_STATS
    if (rootNode->num_try && rootNode->alpha != initial_alpha) {
        stats->move_order_count++;
        if (rootNode->best_count<4) {
            stats->move_order[rootNode->best_count]++;
        }
    }
#endif
    ASSERT(rootNode->score >= -Constants::BIG && rootNode->score <= Constants::BIG);
    return rootNode->score;
}


int RootSearch::update_root_move(const Board &board,
NodeInfo *parentNode, NodeInfo *node,
Move move, int score)
{
    parentNode->best = move;
    ASSERT(parentNode->score >=-Constants::BIG && parentNode->score <=Constants::BIG);
    parentNode->score = score;
    // unlike higher plies, allow captures as killers here:
    context.set_killer(node->last_move, 0);
#ifdef MOVE_ORDER_STATS
    parentNode->best_count = node->num_try-1;
#endif
    parentNode->score = score;
    if (score >= parentNode->beta) {
#ifdef _TRACE
        if (master())
            cout << "ply 0 beta cutoff" << endl;
#endif
        // set pv to this move so it is searched first the next time
        parentNode->pv[0] = move;
        parentNode->pv_length = 1;
        parentNode->cutoff++;
        parentNode->score = score;
        if (controller->uci) {
            char buf[20];
            Scoring::print_score_uci(score,buf);
            cout << "info score " << buf << " lowerbound" << endl;
        }
        return 1;  // signal cutoff
    }
    parentNode->alpha = score;
    update_pv(board,parentNode,(node+1),move,0);
    return 0;   // no cutoff
}

// stop all threads associated with this search
void Search::stopAllThreads()
{
#ifdef _THREAD_TRACE
    log("waiting, discarding results");
#endif
    terminateSearch();
#ifdef SMP
    if (options.search.ncpus>1) {
        Lock(split_lock);
        if (controller->get_talk_level() == Trace)
            cout << "(debug) stopAllThreads: " << ti->index << endl;
        split_t *s = split_stack_end;
        ASSERT(split_stack_end-split_stack<=SPLIT_STACK_MAX_DEPTH);
        while (s != split_stack) {
            --s;
            const int max = s->slaves.size();
            for (int i = 0; i<max; i++) {
                (s->slaves[i])->work->stopAllThreads();
            }
        }
        Unlock(split_lock);
    }
#endif
}

#ifdef SMP
// Initialize the saved state from the current Search instance
void Search::Search_State::init(const Search *source) {
  ti = source->ti;
  parent = (Search*)(source->parent);
  board = source->board;
  nodeContents = (*(source->node));
  node = (source->node);
  split_stack_end = source->split_stack_end;
}

void Search::restore(Search::Search_State &state) {
  ti = state.ti;
  parent = state.parent;
  board = state.board;
  node = state.node;
  *node = state.nodeContents;
  split_stack_end = state.split_stack_end;
}

split_t *Search::split(NodeInfo *nodeToSplit,ThreadInfo *ti,int ply,int depth)
{
    Lock(split_lock);
    if (!nodeToSplit->split) {
        // This node didn't have a split point already, so make one,
        // if possible:
        ASSERT(parent->split_stack_end>=parent->split_stack);
        if (split_stack_end-split_stack >= SPLIT_STACK_MAX_DEPTH) {
            Unlock(split_lock);
            return NULL;   // can't split this search instance any more
        }
        ASSERT(split_stack);
        nodeToSplit->split = split_stack_end++;
        ASSERT(nodeToSplit->split->slaves.size()==0);
    }
    ASSERT(nodeToSplit->split);
    nodeToSplit->lock();
    nodeToSplit->split->ply = ply;
    nodeToSplit->split->depth = depth;
    nodeToSplit->split->slaves.add(ti);
    nodeToSplit->splits++;
    ti->state = ThreadInfo::Working;
    ti->parentNode = nodeToSplit;
    // ensure parent thread will wait
    parent->ti->reset();
    nodeToSplit->unlock();
    Unlock(split_lock);
    return nodeToSplit->split;
}


// return # of still active threads
int Search::join(NodeInfo *splitNode,ThreadInfo *ti)
{
#ifdef _THREAD_TRACE
    log("join",ti->index);
#endif
    Search *parent = ti->work->parent;
    Lock(parent->split_lock);
    splitNode->lock();
    // dissociate the thread from the parent
    ASSERT(splitNode->split);
    ArasanSet<ThreadInfo *,Constants::MaxCPUs> &slaves =
        splitNode->split->slaves;
    // remove ti from the list of slave threads in the parent
#ifdef _DEBUG
    ASSERT(slaves.remove(ti));
#else
    slaves.remove(ti);
#endif
    ASSERT(slaves.size()>=0 &&
        slaves.size()<MAX_CPUS);
    int remaining = slaves.size();
    splitNode->unlock();
    Unlock(parent->split_lock);
    return remaining;
}

int Search::maybe_split(const Board &board, NodeInfo *node,int ply,int depth)
{
    // Now that we have searched at least one valid move, we can
    // consider using multiple threads to search the rest (YBWC).
    int splits = 0;
    if (!terminate && node->mg->more() && 
        // (ply<3 || node->mg->getPhase() > Move_Generator::WINNING_CAPTURE_PHASE+1) &&
        (!options.search.use_tablebases ||
        board.getMaterial(White).men() +
        board.getMaterial(Black).men() > EGTBMenCount) &&
        controller->pool->checkAvailable()) {

        ThreadInfo *child_ti;
        int remaining = 100;
        ThreadInfo *slaves[Constants::MaxCPUs];
        // Keep trying to get/assign threads until no threads are
        // available or no more moves are available to search.
        while (remaining>1 && (child_ti = controller->pool->checkOut(this,node,ply,depth))!=NULL) {
            // A thread is available.
            if (!splits) {
                // Force all remaining moves to be generated, since the
                // Move_Generator class is not thread-safe otherwise (it
                // maintains a pointer to the board and when accessed by
                // multiple threads this pointer may not always be at the
                // current position). Also we want to know how many moves
                // remain.
                remaining = node->mg->GenerateAllMoves(node);
            }
            --remaining;
            slaves[ splits++] = child_ti;
#ifdef SMP_STATS
            root->stats->splits++;
#endif
            // A thread is available, copy state so the slave thread can
            // run independently of its master:
            child_ti->work->init(board,this,node,child_ti,ply,depth);
#ifdef _THREAD_TRACE
            log("split ply",ply);
	    log("depth",depth);
#endif
        }
        // now start the slave threads
        for (int i = 0; i < splits; i++) {
            // Start searching in the new thread:
            slaves[i]->start();
        }
    }
    // Now the master thread is still available to do work, either
    // on its own or on behalf of a slave thread ("helpful master").
    // Enter the idle_loop function, which will continue searching
    // until all work is complete.
    if (splits) idle_loop(node);
    return splits;
}


void Search::maybe_wait(NodeInfo *node)
{
    if (node->wasSplit()) {
        if (node->hasSplit()) {
            if (node->cutoff || terminate) {
                // We just need to stop the other threads.
                // We don't care about their results.
                stopAllThreads(node);
            }
            waitForThreadCompletion(node);
        }
        // there are now no slave threads, so pop the split point stack
        Lock(split_lock);
        ASSERT(node->split->slaves.size()==0);
        node->split = NULL;
	node->splits = 0;
        split_stack_end--;
        ASSERT(split_stack_end>=split_stack);
        Unlock(split_lock);
        // ensure this thread will wait once back in the idle loop
        ti->reset();
    }
}


// initialize a Search instance to prepare it for searching at a split point
void Search::init(const Board &aboard, Search *p, NodeInfo *n, ThreadInfo *child_ti,int ply, int depth)
{
    parent = p;
    board = aboard;
    // clear split stack, unless this is a master node, in which case
    // we push down on its split stack
    if (this->ti && this->ti->master) 
      split_stack_end = saved.split_stack_end;
    else
       split_stack_end = split_stack;
    this->ti = child_ti;
    terminate = p->terminate;
    if (parent == NULL) {
        root = (RootSearch*)this;
    }
    else {
        ASSERT(controller);
        root = controller->rootSearch;
    }
    // Copy search history.
    // Note: we do not copy all of it: just the previous node,
    // and the current node (all we need).
    // set node ptr to our position in the stack
    node = stack+ply;
    // We need very little state copied because the SMP search
    // function will mostly access its master node for state.
    node->ply = ply;
    node->depth = depth;
    node->hibound = (p->stack+ply)->hibound;
    node->beta = (p->stack+ply)->beta;
    if (ply>0) {
      // There is some info from the previous node in the stack
      // that the search needs to access, so copy it from the parent:
      (node-1)->extensions = (p->stack+ply-1)->extensions;
      (node-1)->last_move = (p->stack+ply-1)->last_move;
    }
    // set back pointer to parent
    node->parentNode = n;
}


#ifdef SMP
// stop all threads at or below node n
void Search::stopAllThreads(NodeInfo *n)
{
    Lock(split_lock);
    if (n->split == NULL) {
        Unlock(split_lock);
        return;  // no slave threads
    }
    split_t *s = (split_t*)split_stack_end;
    // in case of many threads, there may be multiple split points
    // at various depths in the current search
    split_t *top = (split_t*)n->split;
    while (s != top) {
        --s;
        ASSERT(s>=split_stack);
        const int max = s->slaves.size();
        for (int i = 0; i<max; i++) {
            // this call has no parameter because all searching in the
            // associated Search class is to stop
            (s->slaves[i])->work->stopAllThreads();
        }
    }
    Unlock(split_lock);
}


void Search::waitForThreadCompletion(NodeInfo *n)
{
    if (options.search.ncpus<=1) return;
    while (n->hasSplit()) {
#ifdef _TRACE
        indent(n->ply);cout << "waiting .. " << endl;
#endif
        // this node has an active split point
        ti->state = ThreadInfo::Waiting;
#ifdef _THREAD_TRACE
        log("waiting",ti->index);
#endif
        if (ti->wait()) break;
    }
    //ti->reset(); // ensure event state not signaled
    ti->state = ThreadInfo::Working;
#ifdef _THREAD_TRACE
    log("out of wait",ti->index);
#endif
}
#endif

void Search::idle_loop(NodeInfo *node)
{
    // This node is the master, so it is its own parent (score updates, pv etc.
    // are kept in the parentNode)
    node->parentNode = node;
    // Go ahead and try to grab/search moves in this thread, in parallel
    // with its slaves:
    searchSMP(ti);
    /** NOT WORKING YET
    // Now there's the chance that we can be a "helpful master" by
    // becoming (temporarily) a work thread for one of our slave threads.
    if (!terminate && !node->cutoff && node->hasSplit()) {
       // Node was split but has terminated before its slaves.
      cout << "helpful master" << endl;
#ifdef _THREAD_TRACE
      log("helpful master");
#endif
       // We are going to allow this search class to be "borrowed" for
       // use in another search, but when it returns we want our state
       // back. So save it here:
       saved.init(this);
       ti->master++;
       ti->reset();
       // Make this thread available to its slaves:
       ::idle_loop(ti,node->split);
       ti->master--;
       restore(saved);
    }
    **/
}
#endif  // SMP

void RootSearch::clearHashTables() {
    context.clear_killer();
    context.clear_history();
    Search::clearHashTables();
}

void Search::clearHashTables()
{
  node = stack;
  parent = this;
  int i ;
  for (i = 0; i < Constants::MaxPly; i++) {
    stack[i].init();
  }
    memset(pawn_hash_table,0xff,PAWN_HASH_SIZE*sizeof(Pawn_Entry));
    memset(king_pawn_hash_table,0xff,KING_PAWN_HASH_SIZE*sizeof(King_Pawn_Entry));
    memset(king_cover_hash_table,0xff,KING_COVER_HASH_SIZE*sizeof(King_Cover_Entry));
    memset(qsearch_cache,0xff,QSEARCH_CACHE_SIZE*sizeof(QSearchHashEntry));
    for (int i = 0; i < QSEARCH_CACHE_SIZE; i++) {
        qsearch_cache[i].best = NullMove;
        qsearch_cache[i].score = Scoring::INVALID_SCORE;
    }
}


Search::Search(SearchController *c
#ifdef SMP
	       , ThreadInfo *threadInfo
#endif
)
: parent(NULL),controller(c),root(c->rootSearch)
{
    node = stack;   // top of stack
    parent = this;  // true for main thread
    Scoring::init();
    LockInit(split_lock);
#ifdef SMP
    ti = threadInfo;    
#endif
    pawn_hash_table_allocated = malloc(PAWN_HASH_SIZE*sizeof(Pawn_Entry)+256);
    pawn_hash_table = (Pawn_Entry*)ALIGN_POINTER(pawn_hash_table_allocated,128);
    memset(pawn_hash_table,0xff,PAWN_HASH_SIZE*sizeof(Pawn_Entry));
    king_pawn_hash_table_allocated = malloc(KING_PAWN_HASH_SIZE*sizeof(King_Pawn_Entry)+256);
    king_pawn_hash_table = (King_Pawn_Entry*)ALIGN_POINTER(king_pawn_hash_table_allocated,128);
    memset(king_pawn_hash_table,0xff,KING_PAWN_HASH_SIZE*sizeof(King_Pawn_Entry));
    king_cover_hash_table_allocated = malloc(KING_COVER_HASH_SIZE*sizeof(King_Cover_Entry)+256);
    qsearch_cache_allocated = malloc(QSEARCH_CACHE_SIZE*sizeof(QSearchHashEntry)+256);
    king_cover_hash_table = (King_Cover_Entry*)ALIGN_POINTER(king_cover_hash_table_allocated,128);
    memset(king_cover_hash_table,0xff,KING_COVER_HASH_SIZE*sizeof(King_Cover_Entry));
    qsearch_cache = (QSearchHashEntry*)ALIGN_POINTER(qsearch_cache_allocated,128);
    memset(qsearch_cache,0xff,QSEARCH_CACHE_SIZE*sizeof(QSearchHashEntry));
    for (int i = 0; i < QSEARCH_CACHE_SIZE; i++) {
        qsearch_cache[i].best = NullMove;
        qsearch_cache[i].score = Scoring::INVALID_SCORE;
    }
}


void Search::init(const Board &aboard, Search *p)
{
    board = aboard;
#ifdef SMP
    split_stack_end = split_stack;
#endif
    parent = p;
    if (parent == NULL) {
        root = (RootSearch*)this;
    }
    else {
        ASSERT(controller);
        root = controller->rootSearch;
    }
}


Search::~Search()
{
    LockDestroy(split_lock);
    free(qsearch_cache_allocated);
    free(pawn_hash_table_allocated);
    free(king_pawn_hash_table_allocated);
    free(king_cover_hash_table_allocated);
}


void RootSearch::update_history(NodeInfo *parentNode, Move best, int cutoff, int depth, ColorType side)
{
    Lock(history_lock);
    for (int i=0; i<parentNode->num_try; i++) {
        ASSERT(i<Constants::MaxMoves);
        // safe to access this here because it is after slave thread completion:
        const Move &m = (const Move &)parentNode->done[i];
        if (TypeOfMove(m) == Normal && Capture(m) == EmptyPiece) {
            const int index = SearchContext::history_index(m,board.Side());
            ASSERT(index<8192);
            SearchContext::HistoryEntry *h = &root->context.History[index];
            if (MovesEqual(m,best)) {
                h->hits++;
                if (cutoff) {
                    int hist = root->context.History2[MakePiece(PieceMoved(m),side)][DestSquare(m)]++;
                    if (hist+1 > Constants::HISTORY_MAX) {
                        for (int i = 0; i < 16; i++) {
                            for (int j = 0; j < 64; j++) {
                                root->context.History2[i][j] /= 2;
                            }
                        }
                    }
                }
            }
            h->total++;
            if (h->total > Constants::HISTORY_MAX) {
                h->hits = h->hits/2 + 1;
                h->total = h->total/2 + 1;
            }
        }
    }
    Unlock(history_lock);
}


void Search::update_pv(const Board &board,NodeInfo *node,NodeInfo *fromNode,Move move, int ply)
{
#ifdef _TRACE
    if (master()) {
        indent(ply); cout << "update_pv, ply " << ply << endl;
    }
#endif
    node->pv[ply] = move;
    if (fromNode->pv_length) {
        memcpy((void*)(node->pv+ply+1),(void*)(fromNode->pv+ply+1),
            sizeof(Move)*fromNode->pv_length);
    }
    node->pv_length = fromNode->pv_length+1;
#ifdef _DEBUG
    Board board_copy(board);
    for (int i = ply; i < node->pv_length+ply; i++) {
		ASSERT(i<Constants::MaxPly);
#ifdef _TRACE
        if (master()) {
            MoveImage(node->pv[i],cout); cout << " " << (flush);
        }
#endif
        ASSERT(legal_move(board_copy,node->pv[i]));
        board_copy.DoMove(node->pv[i]);
    }
#endif
}


void Search::update_pv(const Board &board, Move m, int ply)
{
    update_pv(board,node,(node+1),m,ply);
#ifdef _DEBUG
    Board board_copy(board);
#ifdef _TRACE
    if (master()) {
        indent(ply); cout << "update_pv, ply " << ply << endl;
    }
#endif
    for (int i = ply; i < node->pv_length+ply; i++) {
#ifdef _TRACE
        if (master()) {
            MoveImage(node->pv[i],cout); cout << " " << (flush);
        }
#endif
        ASSERT(legal_move(board_copy,node->pv[i]));
        board_copy.DoMove(node->pv[i]);
    }
#endif
#ifdef _TRACE
    if (master()) cout << endl;
#endif
}


int Search::is_draw( const Board &board, int &rep_count, int ply)
{
    // First check for draw by repetition
    int entries = board.state.moveCount-2;
    rep_count = 0;
    // follow rule Crafty uses: 2 repeats if ply<=2, 1 otherwise:
    const int target = (ply<=2) ? 2 : 1;
    if (entries >= 0) {
        for (hash_t *rep_list=board.rep_list_head-3;
            entries>=0;
        rep_list-=2,entries-=2) {
            if (*rep_list == board.HashCode()) {
                rep_count++;
                if (rep_count >= target) {
#ifdef _TRACE
                    if (master()) {
                        indent(ply); cout << "rep count=" << rep_count << endl;
                    }
#endif
                    return 1;
                }
            }
        }
    }
#ifdef _TRACE
    if (rep_count && master()) {
        indent(ply); cout << "rep count=" << rep_count << endl;
    }
#endif
    // check the 50 move rule
    if (board.state.moveCount >= 100) {
        return 1;
    }
    if (board.getMaterial(White).value() <= KING_VALUE+(KNIGHT_VALUE*2) &&
        board.getMaterial(Black).value() <= KING_VALUE+(KNIGHT_VALUE*2)) {
      // check for insufficient material and other drawing situations
      return Scoring::material_draw(board) || Scoring::is_theoretical_draw(board);
    }
    else
      return 0;
}


int Search::evalu8(const Board & board,
int alpha, int beta, int ply)
{
    // Evaluate a board position.
    int rep_count;
    if (is_draw(board,rep_count,ply))
        return draw_score(board);
    int score = Scoring::material_score(board);
    int pos_score = Scoring::positional_score(board,this,-Constants::BIG,
        Constants::BIG);
    score += pos_score;
    return score;
}


int Search::quiesce(int ply,int depth)
{
    // recursive function, implements quiescence search.
    //
    ASSERT(ply < Constants::MaxPly);
    root->stats->num_nodes++;
    root->stats->num_qnodes++;
    root->time_check_counter++;
    node->pv_length = 0;
	int rep_count;
    if (ply >= Constants::MaxPly-1) {
      node->score = evalu8(board,node->alpha,node->beta,ply);
      if (node->score > node->alpha && node->score < node->beta) {
          node->pv_length = 0;
      }
      return node->score;
    }
    else if (is_draw(board,rep_count,ply)) {
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << "draw!" << endl;
        }
#endif
   	    node->flags |= EXACT;
	    node->score = draw_score(board);
            if (node->score > node->alpha && node->score < node->beta) {
               node->pv_length = 0;
	    }
            return node->score;
    }
#ifdef _TRACE
    if (master()) {
        indent(ply); cout << "window [" << node->alpha << ","
            << node->beta << "]" << endl;
    }
#endif
    if (board.CheckStatus((node-1)->last_move)==InCheck) {
        // If last move was a checking move, ensure that in making it we did
        // not move a pinned piece or move the king into check (normally we
        // would detect this by find the King can be captured, but if in
        // check we only generate evasions and will not find this).
        if (!board.wasLegal((node-1)->last_move))
            return -Illegal;
        else {
            return qsearch_check(ply,depth);
        }
    }
    else {
        return qsearch_no_check(ply,depth);
    }
}


int Search::qsearch_no_check(int ply, int depth)
{
    int stand_pat_score;
    // Establish a default score.  This score is returned if no
    // captures are generated, or if no captures generate a better
    // score (since we generally can choose whether or not to capture).
    int mat_score = Scoring::material_score(board);
    if (mat_score > node->beta + ROOK_VALUE) {
	// check legality of prev move but not at depth == 0 (because
	// regular search checked already)
        if (depth < 0 && !board.wasLegal((node-1)->last_move)) 
            return -Illegal;
        else {
#ifdef _TRACE
            if (master()) {
               indent(ply); cout << "** CUTOFF ** (material)" << endl;;
            }
#endif
            return mat_score;
	}
    }
    int bitscore;
    QSearchHashEntry *cache_entry  = &qsearch_cache[board.HashCode() %
        QSEARCH_CACHE_SIZE];
    if (board.getMaterial(White).material_level()==0 &&
        board.getMaterial(Black).material_level()==0 &&
    ((bitscore=Scoring::tryBitbase(board))!=Scoring::INVALID_SCORE)) {
        stand_pat_score = bitscore;
    }
    else {
        stand_pat_score = mat_score +
            Scoring::positional_score(board,this,node->alpha-mat_score,
            node->beta-mat_score,cache_entry);
    }
#ifdef _TRACE
    if (master()) {
        indent(ply);
        cout << "cap score = " << stand_pat_score << endl;
    }
#endif
    if (stand_pat_score > node->alpha) {
        if (stand_pat_score >= node->beta) {
#ifdef _TRACE
            if (master()) {
                indent(ply); cout << "**CUTOFF**" << endl;
            }
#endif
	    // check legality of prev move but not at depth == 0 (because
	    // regular search checked already)
            if (depth < 0 && !board.wasLegal((node-1)->last_move))
               return -Illegal;
            else {
	      return stand_pat_score;
	    }
        }
        node->alpha = stand_pat_score;
    }

    node->score = node->alpha;
    Move pv;
    if (cache_entry && cache_entry->hash_code == board.HashCode())
        pv = cache_entry->best;
    else
        pv = NullMove;

    int initial_alpha = node->alpha;
    int move_index = 0;
    int try_score;
    Board_State state(board.state);
    const int swag = QSEARCH_FORWARD_PRUNE_BIAS+stand_pat_score;
    if (!IsNull(pv)) {
#ifdef _TRACE
        if (master()) {
            indent(ply);
            cout << "qsearch cache hit: "; MoveImage(pv,cout);
            cout << endl;
        }
#endif
        if (Capture(pv) == King) {
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << "previous move illegal, exiting qsearch" << endl;
            }
#endif
            return -Illegal;
        }
        ASSERT(legal_move(board,pv));
        node->last_move = pv;
        board.DoMove(pv);
        try_score = -quiesce(-node->beta, -node->alpha, ply+1, depth-1);
        board.UndoMove(pv,state);
        if (try_score != Illegal) {
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << ply << ". ";
                MoveImage(pv,cout);
                cout << "(pv) " << try_score << endl;
            }
#endif
            if (try_score > node->alpha) {
                node->score = try_score;
                node->best = pv;
                if (try_score >= node->beta)
                    goto search_end;
                node->alpha = try_score;
                if (node->score >= Constants::BIG-1-ply)
                    goto search_end;              // mating move found
            }
        }
    }
    {
        Move *moves = (Move*)node->done;
        Move_Generator mg(board, &root->context, ply, NullMove, master());
        // generate all the capture moves
        int move_count = mg.GenerateCaptures(moves);
        root->stats->num_moves += move_count;
#ifdef _TRACE
        int move_count1 = move_count;
#endif
        move_count = Move_Ordering::order_captures(board, moves, move_count,
						   node->alpha,
						   stand_pat_score,ply);
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << move_count1 << " capture moves, ";
            cout << move_count << " after pruning." << endl;
        }
#endif

        while (move_index < move_count) {
            Move move = moves[move_index++];
            if (MovesEqual(move,pv)) continue;
            if (Capture(move) == King) {
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << "previous move illegal, exiting qsearch" << endl;
                }
#endif
                cache_entry->hash_code = board.HashCode();
                cache_entry->best = move;
                return -Illegal;
            }
            // We already did forward pruning in order_captures, but 
	    // if alpha has increased we have an opportunity to do it again.
            if (node->alpha != initial_alpha) {
                int est = Gain(move);
                if (swag + est < node->alpha) {
                    continue;
                }
                else {
                    est = est - PieceValue(PieceMoved(move));
                    if (est <= BISHOP_BONUS && Capture(move) != EmptyPiece) {
                        est = attack_estimate(board,move);
                        if (swag + est < node->alpha) {
                            continue;
                        }
                    }
                }
            }
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << "trying " << ply << ". ";
                MoveImage(move,cout);
                cout << endl;
            }
#endif
            node->last_move = move;
            board.DoMove(move);
            try_score = -quiesce(-node->beta, -node->alpha, ply+1, depth-1);
            board.UndoMove(move,state);
            if (try_score != Illegal) {
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << ply << ". ";
                    MoveImage(move,cout);
                    cout << ' ' << try_score << endl;
                }
#endif
                if (try_score > node->alpha) {
                    node->score = try_score;
                    node->best = move;
                    if (try_score >= node->beta)
                        goto search_end;
                    node->alpha = try_score;
                    if (node->score >= Constants::BIG-1-ply)
                        goto search_end;          // mating move found
                }
            }
#ifdef _TRACE
            else if (master()) {
                indent(ply);
                cout << ply << ". ";
                MoveImage(move,cout);
                cout << " (illegal)" << endl;
            }
#endif
        }
        if (node->score >= node->beta) {
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << "**CUTOFF**" << endl;
            }
#endif
            goto search_end;
        }
        if ((node->alpha == initial_alpha) &&
            board.getMaterial(board.Side()).material_level() > 8 &&
	    (depth >= 1-options.search.checks_in_qsearch)) {
            // Do checks in qsearch
            Move_Generator mg(board, &root->context, ply, NullMove, master());
            move_count = mg.GenerateChecks(moves);
            move_index = 0;
#ifdef _TRACE
            if (master()) {
                if (move_count) {
                    indent(ply);
                    cout << move_count << " checks generated" << endl;;
                }
            }
#endif
            while (move_index < move_count) {
                Move move = moves[move_index++];
                if (MovesEqual(move,pv)) continue;
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << "trying " << ply << ". ";
                    MoveImage(move,cout);
                    cout << endl;
                }
#endif
                node->last_move = move;
                board.DoMove(move);
                try_score = -quiesce(-node->beta, -node->alpha, ply+1, depth-1);
                board.UndoMove(move,state);
                if (try_score != Illegal) {
#ifdef _TRACE
                    if (master()) {
                        indent(ply);
                        cout << ply << ". ";
                        MoveImage(move,cout);
                        cout << ' ' << try_score << endl;
                    }
#endif
                    if (try_score > node->alpha) {
                        node->score = try_score;
                        node->best = move;
                        if (try_score >= node->beta) {
#ifdef _TRACE
                            indent(ply); cout << "**CUTOFF**" << endl;
#endif
                            goto search_end;
			}
                        node->alpha = try_score;
                        if (node->score >= Constants::BIG-1-ply)
                            goto search_end;      // mating move found
                    }
                }
#ifdef _TRACE
                else if (master()) {
                    indent(ply);
                    cout << ply << ". ";
                    MoveImage(move,cout);
                    cout << " (illegal)" << endl;
                }
#endif
            }
        }
    }
    search_end:
    if (!IsNull(node->best)) {
        cache_entry->hash_code = board.HashCode();
        cache_entry->best = node->best;
    }
    if (node->alpha != initial_alpha && node->score < node->beta) {
        if (IsNull(node->best))
            node->pv_length = 0;
        else {
            update_pv(board,node->best,ply);
        }
    }
    return node->score;
}


int Search::qsearch_check(int ply, int depth)
{
    int initial_alpha = node->alpha;
    int try_score;
    QSearchHashEntry *cache_entry = &qsearch_cache[board.HashCode() %
        QSEARCH_CACHE_SIZE];
    Move pvmove;
    if (cache_entry->hash_code == board.HashCode()) {
        pvmove = cache_entry->best;
    }
    else
        pvmove = NullMove;
    Move_Generator mg(board, &root->context, ply, pvmove, master());

    Move move;
    Board_State state = board.state;
    node->num_try = 0;
    while ((move = mg.NextEvasion()) != NullMove) {
        if (Capture(move) == King) {
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << "previous move illegal, exiting qsearch" << endl;
            }
#endif
            return -Illegal;
        }
        // omit non-capture evasions if we have already found a move
        // that gets us above alpha
        if (node->alpha > initial_alpha && !IsForced(move) &&
            Capture(move)==EmptyPiece && PromoteTo(move)==EmptyPiece) {
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << "skipping " << ply << ". ";
                MoveImage(move,cout);
                cout << endl;
            }
#endif
            continue;
        }
#ifdef _TRACE
        if (master()) {
            indent(ply);
            cout << "trying " << ply << ". ";
            MoveImage(move,cout);
            cout << endl;
        }
#endif
        node->last_move = move;
        board.DoMove(move);
        try_score = -quiesce(-node->beta, -node->alpha, ply+1, depth-1);
        board.UndoMove(move,state);
        if (try_score != Illegal) {
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << ply << ". ";
                MoveImage(move,cout);
                cout << ' ' << try_score << endl;
            }
#endif
            node->num_try++;
            if (try_score > node->alpha) {
                node->score = try_score;
                node->best = move;
                if (try_score >= node->beta)
                    break;
                node->alpha = try_score;
                if (node->score >= Constants::BIG-1-ply)
                    break;                        // mating move found
            }
        }
#ifdef _TRACE
        else if (master()) {
            indent(ply);
            cout << ply << ". ";
            MoveImage(move,cout);
            cout << " (illegal)" << endl;
        }
#endif
    }
    if (!IsNull(node->best)) {
        cache_entry->best = node->best;
        cache_entry->hash_code = board.HashCode();
        cache_entry->score = Scoring::INVALID_SCORE;
    }
    if (node->score >= node->beta) {
#ifdef _TRACE
        if (master()) {
            indent(ply);
            cout << "**CUTOFF**" << endl;
        }
#endif
        return node->score;
    }
    if (node->num_try == 0) {
        node->flags |= EXACT;
        node->score = -(Constants::BIG - ply);           // mate
	if (node->score > node->alpha) node->alpha = node->score;
    }
    if (node->alpha != initial_alpha && node->score < node->beta) {
        if (IsNull(node->best))
            node->pv_length = 0;
        else {
            update_pv(board,node->best,ply);
        }
    }
    return node->score;
}


static int FORCEINLINE passed_pawn_push(const Board &board, Move move)
{
    return PieceMoved(move) == Pawn && Rank(DestSquare(move),board.Side())==7;
}

static int FORCEINLINE passed_pawn_move(const Board &board, Move move) {
  extern CACHE_ALIGN Bitmap passedW[64];
  extern CACHE_ALIGN Bitmap passedB[64];
  if (PieceMoved(move) == Pawn) {
    if (Rank(DestSquare(move),board.Side())>=6) return 1;
    if (board.Side() == White) {
      return !Bitmap::And(board.pawn_bits[Black],passedW[StartSquare(move)]).is_clear() &&
	Bitmap::And(board.pawn_bits[Black],passedW[DestSquare(move)]).is_clear();    
    }
    else {
      return !Bitmap::And(board.pawn_bits[White],passedB[StartSquare(move)]).is_clear() &&
	Bitmap::And(board.pawn_bits[White],passedB[DestSquare(move)]).is_clear();    
    }
  }
  else
    return 0;
}


#ifdef SMP
void Search::searchSMP(ThreadInfo *ti)
{
    Move move;
    const int in_check = board.CheckStatus() == InCheck;
    Board_State state = board.state;
    const int &ply = node->ply;
    const int &depth = node->depth;
    NodeInfo *parentNode = ti->work->node->parentNode;
#ifdef _THREAD_TRACE
    log("searchSMP",ti->index);
#endif
    while (!parentNode->cutoff && !terminate) {
        move = in_check ?
            parentNode->mg->NextEvasion(parentNode) :
            parentNode->mg->NextMove(parentNode);
        if (IsNull(move)) break;
        if (IsUsed(move)) continue;
#ifdef _TRACE
        if (master() || ply==0) {
            indent(ply); cout << "trying " << ply << ". ";
            MoveImage(move,cout); cout << endl;
        }
#endif
        ASSERT(Capture(move) != King);
        int try_score;

        int extend = 0;
        node->last_move = move;
        CheckStatusType in_check_after_move = board.wouldCheck(move);
        if (ply > 0) {
            extend = calc_extensions(board,node,parentNode,
                in_check_after_move,ply,depth,move);
	    if (extend == PRUNE) {
#ifdef _TRACE
	      if (master()) {
		indent(ply); cout << "fwd pruned." << endl;
	      }
#endif
	      parentNode->fpruned_moves++;
 	      continue;
	    }
        }
        board.DoMove(move);
        if (!in_check && !board.wasLegal(move)) {
#ifdef _TRACE
           if (master()) {
               indent(ply); cout << "Illegal move!" << endl;
           }
#endif
           board.UndoMove(move,state);
           continue;
        }
        // In rare cases we have not applied the check extension prior to
        // the move: do so now
        if (ply > 0 && (in_check_after_move == CheckUnknown)) {
            in_check_after_move = board.CheckStatus();
            if (in_check_after_move == InCheck) {
                node->extensions |= CHECK;
                extend += CHECK_EXTENSION;
                root->stats->check_extensions++;
                if (ply > limits[root->iteration_depth]) extend /= 2;
                if (extend > DEPTH_INCREMENT) extend = DEPTH_INCREMENT;
            }
        }
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << "window [" << parentNode->alpha <<
                "," << node->hibound << "]" << endl;
        }
#endif
        if (depth+extend-DEPTH_INCREMENT >= 0) {
            try_score = -search(-node->hibound, -parentNode->alpha,
                ply+1, depth+extend-DEPTH_INCREMENT);
        }
        else {
            try_score = -quiesce(-node->hibound, -parentNode->alpha,
                ply+1, 0);
        }
        if (try_score == Illegal) {
#ifdef _TRACE
            if (master()) {
                indent(ply); cout << "Illegal move" << endl;
            }
#endif
            board.UndoMove(move,state);
            continue;
        }
#ifdef _TRACE
        if (master() || ply==0) {
            indent(ply); cout << ply << ". ";
            MoveImage(move,cout);
            cout << " " << try_score << endl;
        }
#endif
        if (terminate || parentNode->cutoff) {
#ifdef _TRACE
	  if (master()) {
  	     indent(ply); cout << "parent node cutoff" << endl;
	  }
#endif
            board.UndoMove(move,state);
            break;
        }
        // it is possible the parent node's alpha has changed, so
        // compare against that
        if (try_score > parentNode->alpha && try_score < node->beta &&
            !((node+1)->flags & EXACT)) {
#ifdef _TRACE
            if (master() || ply==0) {
	        indent(ply); cout << ply << ". ";
		MoveImage(move,cout);
                cout << "score = " << try_score << " - no cutoff, researching"; 
                cout << " (smp ";
                cout << ti->index << ')' << endl;
            }
#endif
            // failed to get cutoff, re-search with non-zero window
#ifdef NEGASCOUT
	    int hibound = -try_score+1;
#else
	    int hibound = -parentNode->alpha;
#endif
            if (depth+extend-DEPTH_INCREMENT >= 0) {
	      try_score = -search(-node->beta, hibound,
                    ply+1, depth+extend-DEPTH_INCREMENT);
            }
            else {
	      try_score = -quiesce(-node->beta, hibound,
                    ply+1, 0);
            }
        }
        if ((node->extensions & HPRUNE) &&
            !terminate && !parentNode->cutoff && try_score >= node->beta) {
#ifdef _TRACE
            if (master()) {
	       indent(ply); cout << "verifying LMR" << endl;
	    }
#endif
            // verify LMR
            if (depth-DEPTH_INCREMENT >= 0) {
                try_score = -search(-node->beta, -parentNode->alpha,
                    ply+1, depth-DEPTH_INCREMENT);
            }
            else {
                try_score = -quiesce(-node->beta, -parentNode->alpha,
                    ply+1, 0);
            }
        }
        board.UndoMove(move,state);
#ifdef _TRACE
        if (master() || ply==0) {
            indent(ply);
            cout << ply << ". ";
            MoveImage(move,cout);
            cout << " " << try_score << endl;
        }
#endif
        if (!terminate && !parentNode->cutoff) {
            parentNode->lock();
            ASSERT(parentNode->num_try<Constants::MaxMoves);
            parentNode->done[parentNode->num_try++] = move;
            if (try_score > parentNode->alpha) {
                // search produced a new best move or cutoff, update parent node
#if defined (_THREAD_TRACE) || defined(_TRACE)
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
	      std::ostringstream s;
#else
              strstream s;
#endif
                s << "new best smp, thread=" << ti->index << " ply= " << ply << '\0';
#ifdef _TRACE
		cout << s << endl;
#endif
#ifdef _THREAD_TRACE
                log(s.str());
#endif
#endif
                if (ply == 0)
                    ((RootSearch*)(ti->work->parent))->update_root_move(board,parentNode,node,move,try_score);
                else
                    ti->work->parent->update_move(board,parentNode,node,move,try_score,ply,depth);
            }
            if (try_score >= Constants::BIG-1-ply) {
                parentNode->unlock();
                break;                            // mating move found
            }
            node->hibound = parentNode->alpha+1;
            parentNode->unlock();
        }
    }
    // this search is done, make this thread available for
    // other tasks
}
#endif

int Search::update_move(const Board &board, NodeInfo *parentNode, NodeInfo *node, Move move, int score, int ply, int depth)
{
#ifdef SMP
    if (parentNode->split) parentNode->lock();
#endif
    parentNode->score = score;
    parentNode->best = move;
    int cutoff = 0;
    if (Capture(move) == EmptyPiece && PromoteTo(move) == InvalidPiece) {
        if (node->score >= parentNode->beta) {
            root->context.set_killer(move, ply);
        }
	// update history count for pruning/reduction
        const int index = SearchContext::history_index(move,board.Side());
        SearchContext::HistoryEntry *h = &root->context.History[index];
        h->hits++;
    }
#ifdef MOVE_ORDER_STATS
    parentNode->best_count = node->num_try-1;
#endif
    if (score >= parentNode->beta) {
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << "beta cutoff" << endl;
        }
#endif
        parentNode->cutoff++;
        cutoff++;
    }
    else {
        parentNode->alpha = score;
        // update pv from slave node to master
        update_pv(board,parentNode,(node+1),move,ply);
    }
#ifdef SMP
    if (parentNode->split) parentNode->unlock();
#endif
    return cutoff;
}


int Search::calc_extensions(const Board &board,
NodeInfo *node, NodeInfo *parentNode,
CheckStatusType in_check_after_move,
int ply,int depth,
const Move &move)
{
    // see if we should apply any extensions at this node.
    node->extensions = 0;
    int extend = 0;
    if (options.search.forced_extensions && IsForced(move)) {
        // one single reply to check
        extend += FORCED_EXTENSION;
        node->extensions |= FORCED;
        root->stats->forced_extensions++;
    }
    else if (options.search.check_extensions && in_check_after_move == InCheck) {
        node->extensions |= CHECK;
        extend += CHECK_EXTENSION;
        root->stats->check_extensions++;
    }
    else if (options.search.pawn_push_extensions &&
    passed_pawn_push(board,move)) {
        node->extensions |= PAWN_PUSH;
        extend += PAWN_PUSH_EXTENSION;
        root->stats->pawn_extensions++;
    }
    else if (options.search.recapture_extensions &&
        Capture(move) != EmptyPiece &&
        Capture(move) != Pawn &&
        Capture((node-1)->last_move) != EmptyPiece &&
        DestSquare(move) == DestSquare((node-1)->last_move) &&
        Util::Abs(PieceValue(Capture((node-1)->last_move))-
        PieceValue(Capture(move)))<=BISHOP_BONUS &&
        (board.getMaterial(board.Side()).material_level() >= 7)) {
        extend += RECAPTURE_EXTENSION;
        node->extensions |= RECAPTURE;
        root->stats->recap_extensions++;
    }
    if (options.search.threat_extensions && parentNode->threat) {
        extend += THREAT_EXTENSION;
        node->extensions |= THREAT;
        root->stats->threat_extensions++;
    }
    if (extend) {
       if (ply > limits[root->iteration_depth]) {
           extend /= 2;
       }
       if (extend > DEPTH_INCREMENT)
          return DEPTH_INCREMENT;
       else
          return extend;
    }
    if (board.CheckStatus() == InCheck || in_check_after_move == InCheck) {
      return 0;
    }
    // No extensions were triggered, see if we can reduce or prune.
    // Try late move reduction first.
    if (options.search.late_move_reductions && 
	depth > FUTILITY_RANGE &&
        CanHReduce(move) &&
        !passed_pawn_move(board,move)) {
        extend -= HISTORY_REDUCTION;
        node->extensions |= HPRUNE;
    }
    // Now try score-based pruning.
    //depth += extend;
#ifdef FUTILITY_PRUNE
    int fmargin;
    if (depth <= FUTILITY_RANGE && parentNode->num_try && extend <= 0) {
        fmargin = (depth>2*DEPTH_INCREMENT) ? EXTENDED_FUTILITY_MARGIN : FUTILITY_MARGIN;
        if (parentNode->pos_score == Scoring::INVALID_SCORE) {
	    int score;
            if (Scoring::is_draw(board)) {
	      parentNode->pos_score = draw_score(board);
	    }
            else if ((score = Scoring::tryBitbase(board))!= Scoring::INVALID_SCORE)
            {
	       parentNode->pos_score = score;
            }
	    else {
               QSearchHashEntry *cache_entry  = 
		 &qsearch_cache[board.HashCode() % QSEARCH_CACHE_SIZE];
               parentNode->pos_score = Scoring::material_score(board) + 
	         Scoring::positional_score(board, this,
		  		  -Constants::BIG,
				  Constants::BIG,
				  cache_entry);
	    }
        }
	int est = fmargin + parentNode->pos_score + Gain(move);
        if (est < parentNode->alpha) {
	   root->stats->futility_pruning++;
 	   return PRUNE;
        }
    }
#endif
    return extend;
}


// Recursive function, implements alpha/beta search below ply 0 but
// above the quiescence search.
//
// The current board is this->board.
// The search parameters such as depth, ply are in this->node.
//
int Search::search()
{
    int rep_count;
    int using_tb;
    int move_count;
#ifdef MOVE_ORDER_STATS
    node->best_count = 0;
#endif
    int ply = node->ply;
    int depth = node->depth;
    if (root->time_check_counter++ >= Time_Check_Interval) {
        root->time_check_counter = 0;
        terminate = root->check_time(board,ply,0,root->iteration_depth);
        if (terminate) {
            if (controller->talkLevel == Trace)
                cout << "(debug) terminating, time up" << endl;
            root->terminate_now();   // signal all searches to end
        }
    }
    ASSERT(ply < Constants::MaxPly);
    root->stats->num_nodes++;
    if (terminate) {
        return node->alpha;
    }
    else if (ply >= Constants::MaxPly-1) {
        return evalu8(board,node->alpha,node->beta,ply);
    }
    int initial_alpha = node->alpha;

    if (is_draw(board,rep_count,ply)) {
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << "draw!" << endl;
        }
#endif
        int score = draw_score(board);
        if (score > -(node-1)->beta && score < -(node-1)->alpha) {
            node->pv[ply] = NullMove;
            node->pv_length = 0;
        }
   	node->flags |= EXACT;
        return score;
    }
    Move hash_move = NullMove;
    using_tb = 0;
    if (options.search.use_tablebases) {
        const Material &wMat = board.getMaterial(White);
        const Material &bMat = board.getMaterial(Black);
        using_tb = (wMat.men() + bMat.men() <= EGTBMenCount) &&
            (ply <= 2+Util::Max(1,3*root->iteration_depth/4));
    }
    Position_Info hash_entry;
    Position_Info::ValueType result = Position_Info::NoHit;
    // Note: query the hash table before the tablebases, since TB info may
    // be cached.
    if (controller->use_hash_table) {
        // Search the hash table to see if we have hit this
        // position before.
        //
        Lock(hash_lock);
        // Note: we copy the hash entry .. so mods by another thread do not
        // alter the copy
        result = searchHash(board,board.HashCode(rep_count),
            node->alpha,node->beta,ply,depth,hash_entry);
        Unlock(hash_lock);
        root->stats->hash_searches++;
    }
    if (result != Position_Info::NoHit) {
        root->stats->hash_hits++;
        // if using TBs at this ply, do not pull a non-TB entry out of
        // the hashtable. Also do not allow cut of a full node from a
        // cut hash entry.
        if ((node->flags & VERIFY) ||
            (using_tb && depth != Constants::MaxPly*DEPTH_INCREMENT) ||
            (node->alpha+1<node->beta && !hash_entry.isFull())) {
            result = Position_Info::Invalid;
        }
        int value = hash_entry.value();
        switch (result) {
            case Position_Info::Valid:
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << "hash cutoff, type = E" <<
                        " alpha = " << node->alpha <<
                        " beta = " << node->beta <<
                        " value = " << value << endl;
                }
#endif
                if (value > -(node-1)->beta && value < -(node-1)->alpha) {
		    // parent node will consider this a new best line
                    hash_move = hash_entry.best_move(board);
                    if (IsNull(hash_move))
                        node->pv_length=0;
                    else {
                        node->pv[ply] = hash_move;
                        node->pv_length = 1;
                    }
#ifdef _TRACE
                    if (master()) {
                        indent(ply); cout << "best line[ply][ply] = ";
                        MoveImage(hash_move,cout);
                        cout << endl;
                    }
#endif
                }
   	        node->flags |= EXACT;
                return value;
            case Position_Info::UpperBound:
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << "hash cutoff, type = U" <<
                        " alpha = " << node->alpha <<
                        " beta = " << node->beta <<
                        " value = " << value << endl;
                }
#endif
                return value;                     // cutoff
            case Position_Info::LowerBound:
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << "hash cutoff, type = L" <<
                        " alpha = " << node->alpha <<
                        " beta = " << node->beta <<
                        " value = " << value << endl;
                }
#endif
                return value;                     // cutoff
            default:
                // Note: hash move & flags may be usable even if score is not usable
                hash_move = hash_entry.best_move(board);
                board.set_check_status(hash_entry.in_check() ?
                    InCheck : NotInCheck);
                break;
        }
    }
    if (using_tb && rep_count==0) {
        int tb_hit = 0, tb_score;
        root->stats->tb_probes++;
        tb_hit = probe_tb(board, tb_score, ply);
        if (tb_hit) {
#ifdef _TRACE
            if (master()) {
                indent(ply); cout << "EGTB hit: score " << tb_score << endl;
            }
#endif
            root->stats->tb_hits++;
            int score = tb_score;
            // insert TB info in hash table.
            if (score <= -Constants::BIG+200) {
                score -= ply;
            }
            else if (score >= Constants::BIG-200) {
                score += ply;
            }
#ifdef _TRACE
            if (master() && tb_score != score) {
                indent(ply); cout << "adjusted score " << score << endl;
            }
#endif
            // Put it in with a large depth so we will not
            // overwrite - this entry is "exact" at all
            // search depths, so effectively its depth is infinite.
            Hash_Entry new_pos(board.HashCode(rep_count),
                Constants::MaxPly*DEPTH_INCREMENT,
                controller->age,
                Position_Info::Valid,
                board.CheckStatus() == InCheck,
                score, 1, NullMove
                );
            Lock(hash_lock);
            storeHash(new_pos,*(root->stats));
            Unlock(hash_lock);
            node->score = tb_score;               // unadjusted score
            if (node->score > node->alpha && node->score < node->beta) {
                node->pv[ply] = NullMove;
                node->pv_length = 0;
            }
            node->flags |= EXACT;
            return node->score;
        }
    }
    // At this point we need to know if we are in check or not.
    int in_check =
        (board.CheckStatus((node-1)->last_move) == InCheck);
#ifdef _TRACE
    if (master() && in_check) { indent(ply); cout << "in_check=" << in_check << endl;}
#endif
#if defined(FUTILITY_PRUNE)
    node->pos_score = Scoring::INVALID_SCORE;
#endif
#ifdef FUTILITY_PRUNE
    node->fpruned_moves = 0;
#endif
#ifdef SMP
    node->split = NULL;
    node->splits = 0;
#endif
    move_count = 0;

    // Try to get a fast cutoff using a "null move".  Skip if the side
    // to move is in check or if material balance is low enough that
    // zugzwang is a possibility. Do not do null move if this is an
    // IID search, because it will only help us get a cutoff, not a move.
    // Also avoid null move near the 50-move draw limit.
    node->threat = 0;
    if (root->null_depth && !in_check && (node->beta==node->alpha+1) &&
        !(node->flags & (IID|VERIFY)) &&
        !IsNull((node-1)->last_move) &&
        board.state.moveCount <= 98 &&
        (board.getMaterial(board.Side()).pieceCount() >= 1+(depth<2*DEPTH_INCREMENT))) {
        node->last_move = NullMove;
        Board_State state = board.state;
        board.DoNull();
#ifdef _TRACE
        if (master()) {
            indent(ply);
            cout << "trying " << ply << ". " << "(null)" << endl;
        }
#endif
        int nu_depth = depth-root->null_depth-DEPTH_INCREMENT;
#ifdef ADAPTIVE_NULL
        if (depth >= 7*DEPTH_INCREMENT ||
            (depth >= 4*DEPTH_INCREMENT &&
            board.getMaterial(board.Side()).pieceCount() >= 2))
            nu_depth -= DEPTH_INCREMENT;          // R=3
#endif
        int nscore;
        if (nu_depth >= 0)
            nscore = -search(-node->beta, -node->alpha,
                ply+1, nu_depth);
        else
            nscore = -quiesce(-node->beta, -node->alpha,
                ply+1, 0);
#ifdef _TRACE
        if (master()) {
            indent(ply);
            cout << ply << ". " << "(null)" << ' ' << nscore << endl;
        }
#endif
        board.UndoNull(state);
        if (terminate) {
            node->score = node->alpha;
            goto search_end2;
        }
        else if (nscore >= node->beta) {          // cutoff
#ifdef VERIFY_NULL_SEARCH
            if (board.getMaterial(board.Side()).pieceCount() <= 2) {
                // in endgame, verify null cutoff with reduced-depth search
                // (idea from Dieter Buerssner)
                //nu_depth = depth-2*DEPTH_INCREMENT;
                if (nu_depth >= 0)
                    nscore = search(node->alpha, node->beta,
                        ply+1, nu_depth, VERIFY);
                else
                    nscore = quiesce(node->alpha, node->beta,
                        ply+1, 0);
                if (nscore == -Illegal) {
#ifdef _TRACE
                    indent(ply); cout << "previous move illegal" << endl;
#endif
                    return -Illegal;
                }
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    if (nscore>=node->beta)
                        cout << "null cutoff verified, score=" << nscore;
                    else
                        cout << "null cutoff not verified";
                    cout << endl;
                }
#endif
            }
            if (nscore >= node->beta)             // null cutoff
#endif
            {
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << "**CUTOFF**" << endl;
                }
#endif
                root->stats->num_null_cuts++;
                return nscore;
            }
        }
        if (nscore <= 200-Constants::BIG) {
            node->threat++;
        }
    }

    // Use "internal iterative deepening" to get an initial move to try if
    // there is no hash move .. an idea from Crafty (previously used by
    // Hitech).
    if (IsNull(hash_move) && depth >= 4*DEPTH_INCREMENT) {
        int d;
        d = Util::Min(depth-2*DEPTH_INCREMENT,depth/2);
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << "== deepening, depth = " << d
                << endl;
        }
#endif
        // Call search routine at lower depth to get a 1st move to try.
        // ("Internal iterative deepening").
        //
        // Note: we do not push down the node stack because we want this
        // search to have all the same parameters (including ply) as the
        // current search, just reduced depth + the IID flag set.
        node->flags |= IID;
        int alpha = node->alpha;
        node->num_try = 0;
        node->depth = d;
        int iid_score = -search();
        node->flags &= ~IID;
        // reset key params
        node->num_try = 0;
        node->cutoff = 0;
        node->depth = depth;
        node->alpha = node->score = alpha;
#ifdef SMP
        node->splits = 0;
#endif
        // set hash move to IID search result (may still be null)
        hash_move = node->best;
        // do not retain any pv information from the IID search
        // (can screw up non-IID pv).
        (node+1)->pv[ply+1] = NullMove;
        (node+1)->pv_length = 0;
	if (iid_score == Illegal || (node->flags & EXACT)) {
	   // previous move was illegal or was an exact score
#ifdef _TRACE
	  if (master()) {
   	     indent(ply);
	     cout << "== exact result from deeping" << endl;
	  }
#endif
	   return -iid_score;
	}
        if (terminate) {
            node->score = node->alpha;
            return node->alpha;
        }
#ifdef _TRACE
        if (master()) {
            indent(ply); cout << "== deepening done.";
        }
#endif

#ifdef _TRACE
        if (master()) {
            if (!IsNull(hash_move)) {
                indent(ply); cout << "  hash_move = ";
                MoveImage(hash_move,cout);
            }
            cout << endl;
        }
#endif
    }
    {
        Move_Generator mg(board, &root->context, ply, hash_move, master());
#ifdef SMP
        node->mg = &mg;
#endif
        Board_State state = board.state;
        int try_score;
#ifdef SMP
        // we do not split if in check because generally there will be few moves to search there.
        const int canSplit = options.search.ncpus>1 && !in_check &&
            depth >= SPLIT_DEPTH[board.getMaterial(board.Side()).material_level()];
#endif
        //
        // Now we are ready to loop through the moves from this position
        //
        int first = 1;
        while (!node->cutoff && !terminate) {
            Move move;
#ifdef SMP
#ifdef _DEBUG
            ASSERT(node->splits == 0);
#endif
#endif
            // we do not bother to lock here, because if we have split
	    // this node, we don't come back to this part of the loop
            move = in_check ? mg.NextEvasion() : mg.NextMove();
            if (IsNull(move)) break;
            if (IsUsed(move)) continue;
            if (Capture(move)==King) {
                return -Illegal;                  // previous move was illegal
            }
            move_count++;
            root->stats->num_moves++;
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << "trying " << ply << ". ";
                MoveImage(move,cout); 
               if (first) cout << "(pv)";
                cout << " [" << node->alpha << "," << node->hibound << "]";
                cout << " d:" << depth << endl;
            }
#endif
            node->last_move = move;
            CheckStatusType in_check_after_move = board.wouldCheck(move);
            int extend = calc_extensions(board,node,node,in_check_after_move,
                ply,depth,move);
	    if (extend == PRUNE) {
#ifdef _TRACE
	      if (master()) {
		indent(ply); cout << "fwd pruned." << endl;
	      }
#endif
	      node->fpruned_moves++;
	      continue;
	    }
            board.DoMove(move);
            if (!in_check && !board.wasLegal(move)) {
#ifdef _TRACE
               if (master()) {
                  indent(ply); cout << "Illegal move!" << endl;
               }
#endif
               board.UndoMove(move,state);
               continue;
            }
            // in rare cases we may not have applied the check extension
            // before the move, so do it here
            if (in_check_after_move == CheckUnknown) {
                in_check_after_move = board.CheckStatus();
                if (in_check_after_move == InCheck) {
                    node->extensions |= CHECK;
                    extend += CHECK_EXTENSION;
                    root->stats->check_extensions++;
                    if (ply > limits[root->iteration_depth]) extend /= 2;
                    if (extend > DEPTH_INCREMENT) extend = DEPTH_INCREMENT;
                }
            }
            ASSERT(in_check_after_move != CheckUnknown);
            ASSERT((in_check_after_move == InCheck) ==
                board.any_attacks(board.KingPos(
                board.Side()),board.OppositeSide()));
            if (depth+extend-DEPTH_INCREMENT >= 0) {
                try_score = -search(-node->hibound, -node->alpha,
                    ply+1, depth+extend-DEPTH_INCREMENT);
            }
            else {
                try_score = -quiesce(-node->hibound, -node->alpha,
                    ply+1, 0);
            }
            if (try_score == Illegal) {
#if defined(_TRACE)
                if (master()) {
                    indent(ply); cout << "Illegal move" << endl;
                }
#endif
                board.UndoMove(move,state);
                continue;
            }
            if (!first &&
		(try_score > node->alpha) && (try_score < node->beta) && 
                !((node+1)->flags & EXACT) && !terminate) {
                // We failed to get a cutoff and must re-search
#ifdef NEGASCOUNT
		int hibound = -try_score+1;
#else
		int hibound = -node->alpha;
#endif
#ifdef _TRACE
                if (master()) {
		  indent(ply); cout << ply << ". ";
		  MoveImage(move,cout);
		  cout << " score = " << try_score << " - no cutoff, researching .." << endl;
                    indent(ply); cout << "window = [" << -hibound << "," << node->beta
                        << "]" << endl;
                }
#endif
                if (depth+extend-DEPTH_INCREMENT >= 0)
                    try_score=-search(-node->beta,hibound,ply+1,depth+extend-DEPTH_INCREMENT);
                else
                    try_score=-quiesce(-node->beta,hibound,ply+1,0);
#ifdef _TRACE
                if (master()) {
                    indent(ply);
                    cout << ply << ". ";
                    MoveImage(move,cout);
                    cout << ' ' << try_score << endl;
                }
#endif
            }
            if ((node->extensions & HPRUNE) && try_score >= node->beta) {
#ifdef _TRACE
                if (master()) {
		  indent(ply); cout << "verifying LMR" << endl;
		}
#endif
                // verify LMR
                if (depth-DEPTH_INCREMENT >= 0) {
                    try_score = -search(-node->beta, -node->alpha,
                        ply+1, depth-DEPTH_INCREMENT);
                }
                else {
                    try_score = -quiesce(-node->beta, -node->alpha,
                        ply+1, 0);
                }
            }
            board.UndoMove(move,state);
            if (terminate) {
                break;
            }
#ifdef _TRACE
            if (master()) {
                indent(ply);
                cout << ply << ". ";
                MoveImage(move,cout);
                cout << ' ' << try_score;
                if (first) cout << " (pv)";
                cout << endl;
            }
#endif
            first = 0;
            ASSERT(node->num_try<Constants::MaxMoves);
            node->done[node->num_try++] = move;
            ASSERT(node->num_try<Constants::MaxMoves);
            if (try_score > node->alpha) {
                if (update_move(board,node,node,move,try_score,ply,depth)) {
                    // cutoff
                    break;
                }
            }
            if (try_score >= Constants::BIG-1-ply) {
                node->cutoff++;
                break;                            // mating move found
            }
            // zero-width window for the rest of the moves
            node->hibound = node->alpha+1;
#ifdef SMP
            if (canSplit && maybe_split(board,node,ply,depth)) {
                break;
            }
#endif
        }                                         // end move loop
#ifdef _TRACE
        if (node->score >= node->beta && master()) {
            indent(ply);
            cout << "**CUTOFF**" << endl;
        }
#endif
#ifdef SMP
        // We may have other threads still searching.
        maybe_wait(node);
#endif
        if (node->score >= node->beta) {
	  if (node->score > initial_alpha) {
	    // we have previously updated the PV, but now we
	    // will cutoff, so remove it
	    node->pv_length = 0;
	  }
	}
        if (terminate) {
            node->score = node->alpha;
            goto search_end2;
        }
        if (node->num_try == 0
#ifdef FUTILITY_PRUNE
            && node->fpruned_moves == 0
#endif
        ) {
            // no moves were tried
            if (in_check) {
#ifdef _TRACE
                if (master()) {
                    indent(ply); cout << "mate" << endl;
                }
#endif
                node->score = -(Constants::BIG - ply);
                node->pv_length = 0;
		node->flags |= EXACT;
                goto search_end2;
            }
            else {                                // stalemate
#ifdef _TRACE
                if (master()) {
                    indent(ply); cout << "stalemate!" << endl;
                }
#endif
                node->score = draw_score(board);
                if (node->score > -(node-1)->beta && node->score < -(node-1)->alpha) {
                    node->pv[ply] = NullMove;
                    node->pv_length = 0;
                }
		node->flags |= EXACT;
                return node->score;
            }
        }
    }
    if (node->num_try && (initial_alpha != node->alpha)) {
        root->update_history(node,node->best,
            node->cutoff,depth,
            board.Side());
    }

    // don't insert into the hash table if we are terminating - we may
    // not have an accurate score.
    if (controller->use_hash_table && !terminate) {
        if (node->best == NullMove) node->best = hash_move;
        // store the position in the hash table, if there's room
        int value = node->score;
        Position_Info::ValueType val_type;
        // Adjust mate scores to reflect current ply. But only
        // if the score is in bounds.
        if (value > initial_alpha && value < node->beta) {
            if (value < -Constants::BIG+200) {
                value -= ply;
            }
            else if (value > Constants::BIG-200) {
                value += ply;
            }
        }
#ifdef _TRACE
        char typeChar;
#endif
        if (value <= initial_alpha) {
            val_type = Position_Info::UpperBound;
#ifdef _TRACE
            typeChar = 'U';
#endif
            // We don't have a "best" move, because all moves
            // caused alpha cutoff.  But if there was a hash
            // move or an initial move produced by internal
            // interative deepening, save it in the hash table
            // so it will be tried first next time around.
            node->best = hash_move;
        }
        else if (value >= node->beta) {
            val_type = Position_Info::LowerBound;
#ifdef _TRACE
            typeChar = 'L';
#endif
        }
        else {
#ifdef FUTILITY_PRUNE
            val_type = node->fpruned_moves ? Position_Info::Invalid : Position_Info::Valid;
#else
            val_type = Position_Info::Valid;
#endif
#ifdef _TRACE
            typeChar = 'E';
#endif
        }
#ifdef _TRACE
        if (master()) {
            indent(ply);
            cout << "storing type=" << typeChar <<
                " ply=" << ply << " depth=" << depth << " value=" << value <<
                " move=";
            MoveImage(node->best,cout);
            cout << endl;
        }
#endif
        Hash_Entry new_pos(board.HashCode(rep_count), depth,
            controller->age,
            val_type,
            board.CheckStatus() == InCheck,
            node->score, initial_alpha+1<node->beta,node->best);
        Lock(hash_lock);
        storeHash(new_pos,*(root->stats));
        Unlock(hash_lock);
    }
    search_end2:
#ifdef _TRACE
    indent(ply); cout << "num_try=" << node->num_try << endl;
#endif
#ifdef MOVE_ORDER_STATS
    if (node->num_try && node->score != initial_alpha) {
        root->stats->move_order_count++;
        ASSERT(node->best_count>=0);
        if (node->best_count<4) {
            root->stats->move_order[node->best_count]++;
        }
    }
#endif
#ifdef FUTILITY_PRUNE
    root->stats->futility_pruning += node->fpruned_moves;
#endif
    int score = node->score;
    ASSERT(score >= -Constants::BIG && score <= Constants::BIG);
    return score;
}
