// Copyright 1994-2008 by Jon Dart.  All Rights Reserved.

#include "scoring.h"
#include "types.h"
#include "bearing.h"
#include "constant.h"
#include "util.h"
#include "pawnentr.h"
#include "bhash.h"
#include "config.h"
#include "debug.h"
#include "bhash.h"
#include "bitprobe.h"
#include "bearing.h"
#include "globals.h"
#include "tbprobe.h"
#include "search.h"
extern "C"
{
#include <stdio.h>
};

#include <memory.h>

//#define EVAL_DEBUG
//#define PAWN_DEBUG
//#define LAZY_DEBUG
//#define DEBUG_PAWN_RACE
//#define DEBUG_HASH_SCORES
//#define DEBUG_HASH

enum GamePhase { OPENING, MIDDLEGAME, ENDGAME_TRANSITION, ENDGAME };

#ifdef DEBUG_PAWN_RACE
extern int probe_tb(const Board &b, int &score, int ply);
#endif

static const int LAZY_MARGIN = 280;
static const int ENDGAME_LAZY_MARGIN = 192;

#ifdef LAZY_DEBUG
#include "boardio.h"
#undef USE_LAZY_EVAL
long ks_w,ks_b,bc,wc;
#endif

#ifdef EVAL_STATS
long position_hash_probes = 0L, position_hash_hits = 0L;
long pawn_hash_probes = 0L, pawn_hash_hits = 0L;
long king_pawn_hash_probes = 0L, king_pawn_hash_hits = 0L;
long king_cover_hash_probes = 0L, king_cover_hash_hits = 0L;
long lazy_cut_chances = 0L, lazy_cuts = 0L;
#endif

static const CACHE_ALIGN int KnightScoresB[64] =
{
  -12, -9, -9, -9, -9, -9, -9, -12,
  -5, 0, 2, 2, 2, 2, 0, -5,
  -3, 2, 4, 2, 2, 4, 2, -3,
  -5, 2, 2, 4, 4, 2, 2, -5,
  -5, 3, 4, 6, 6, 4, 3, -5,
  -5, 5, 7, 6, 6, 7, 5, -5,
  -5, 0, 2, 2, 2, 2, 0, -5,
  -8, 0, 0, 0, 0, 0, 0, -8
};

static const int KnightScoresW[64] =
{
  -8, 0, 0, 0, 0, 0, 0, -8,
  -5, 0, 2, 2, 2, 2, 0, -5,
  -5, 5, 7, 6, 6, 7, 5, -5,
  -5, 3, 4, 6, 6, 4, 3, -5,
  -5, 2, 2, 4, 4, 2, 2, -5,
  -3, 2, 4, 2, 2, 4, 2, -3,
  -5, 0, 2, 2, 2, 2, 0, -5,
-12, -9, -9, -9, -9, -9, -9, -12
};

static const int BishopPositionB[] =
{
    -6, -9, -9, -9, -9, -9, -9, -6,
    0, 3, 0, 2, 2, 0, 3, 0,
    0, 0, 3, 2, 2, 3, 0, 0,
    0, 0, 2, 3, 3, 2, 0, 0,
    0, 2, 0, 3, 3, 0, 2, 0,
    2, 0, 3, 0, 0, 3, 0, 2,
    0, 3, 0, 0, 0, 0, 3, 0,
    3, 0, 0, 0, 0, 0, 0, 3
};

static const int BishopPositionW[] =
{
    3, 0, 0, 0, 0, 0, 0, 3,
    0, 3, 0, 0, 0, 0, 3, 0,
    2, 0, 3, 0, 0, 3, 0, 2,
    0, 2, 0, 3, 3, 0, 2, 0,
    0, 0, 2, 3, 3, 2, 0, 0,
    0, 0, 3, 2, 2, 3, 0, 0,
    0, 3, 0, 2, 2, 0, 3, 0,
    -6, -9, -9, -9, -9, -9, -9, -6
};

static const int PawnCenterScores[] =
{
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 4, 4, 0, 0, 0,
    0, 0, 5, 7, 7, 5, 0, 0,
    0, 0, 3, 4, 4, 3, 0, 0,
    0, 0, -1, -2, -2, -1, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

static const int KingCenterScores[] =
{
    0, 0, 1, 1, 1, 1, 0, 0,
    0, 1, 2, 3, 3, 2, 1, 0,
    1, 2, 6, 8, 8, 6, 2, 1,
    1, 3, 8, 10, 10, 8, 3, 1,
    1, 3, 8, 10, 10, 8, 3, 1,
    1, 2, 6, 8, 8, 6, 2, 1,
    0, 1, 2, 3, 3, 2, 1, 0,
    0, 0, 1, 1, 1, 1, 0, 0
};

static int KingEndgameScoresW[64] = {
  9, 12, 13, 15, 15, 13, 12, 9,
  9, 12, 13, 15, 15, 13, 12, 9,
  6, 9, 10, 12, 12, 10, 9, 6,
  0, 3, 4, 6, 6, 4, 3, 0,
 -6, -3, -1, 0, 0, -1, -3, -6,
-12, -9, -7, -6, -6, -7, -9, -12,
-18, -15, -13, -12, -12, -13, -15, -18,
-24, -21, -19, -18, -18, -19, -21, -24};


static const int KingEndgameScoresB[64] = {
-24, -21, -19, -18, -18, -19, -21, -24,
-18, -15, -13, -12, -12, -13, -15, -18,
-12, -9, -7, -6, -6, -7, -9, -12,
 -6, -3, -1, 0, 0, -1, -3, -6,
  0, 3, 4, 6, 6, 4, 3, 0,
  6, 9, 10, 12, 12, 10, 9, 6,
  9, 12, 13, 15, 15, 13, 12, 9,
  9, 12, 13, 15, 15, 13, 12, 9};

static const int QueenCenterScoresW[64] =
{
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 3, 4, 4, 3, 0, 0,
    0, 0, 3, 5, 5, 3, 0, 0,
    0, 0, 3, 5, 5, 3, 0, 0,
    0, 0, 3, 2, 2, 3, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    -2, -2, -2, -2, -2, -2, -2, -2
};

static const int QueenCenterScoresB[64] =
{
    -2, -2, -2, -2, -2, -2, -2, -2,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 3, 2, 2, 3, 0, 0,
    0, 0, 3, 5, 5, 3, 0, 0,
    0, 0, 3, 5, 5, 3, 0, 0,
    0, 0, 3, 4, 4, 3, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

static const int OutpostScoresB[64] =
{
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 6, 10, 10, 6, 0, 0,
    0, 10, 10, 12, 12, 10, 10, 0,
    0, 6, 6, 6, 6, 6, 6, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

static const int OutpostScoresW[64] =
{
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 6, 6, 6, 6, 6, 6, 0,
    0, 10, 10, 12, 12, 10, 10, 0,
    0, 0, 6, 10, 10, 6, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0
};

static const int KBNKScores[64] =
{
    -20, -15, -10, -3, 3, 10, 15, 20,
    -15, -10, -3, 0, -1, 3, 10, 15,
    -10, -3, 0, 0, 0, -1, 3, 10,
    -3, -3, 0, -2, -2, 0, -1, 3,
    3, -1, 0, -2, -2, 0, 0, -3,
    10, 3, -1, 0, 0, 0, -3, -10,
    15, 10, 3, -1, 0, -3, -10, -15,
    20, 15, 10, 3, -3, -10, -15, -20
};

static Bitmap kingCoverW0[64];
static Bitmap kingCoverW[64];
static Bitmap kingCoverW2[64];
static Bitmap kingCoverB0[64];
static Bitmap kingCoverB[64];
static Bitmap kingCoverB2[64];

// Weights for positional scoring components.  A plus value means that
// the item being scored is valuable to the player; a minus value means
// that a penalty is extracted.  In most cases, scoring components are
// simply summed together to get a total score, but some are folded
// in, in a more complex way.

// material terms
static const int RB_TRADE[] = { 0, -PAWN_VALUE/4, (3*PAWN_VALUE)/8, 
				PAWN_VALUE/2,
				PAWN_VALUE/2, PAWN_VALUE/2, PAWN_VALUE/2 };
static const int RBN_TRADE[] = { 0, PAWN_VALUE/2, PAWN_VALUE/4, PAWN_VALUE/4,
				 PAWN_VALUE/4, PAWN_VALUE/4, PAWN_VALUE/4 };

// development terms
static const int BISHOP_MOBILITY[] =
{
    -12, -10, -8, -6, -2, 2, 4, 6, 8, 10, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12
};
static const int BAD_BISHOP = -4;
static const int BISHOP_PAIR = PAWN_VALUE/2;
static const int BISHOP_PAIR_PAWN_ADJUST[8] =
  {PAWN_VALUE/8, PAWN_VALUE/8, PAWN_VALUE/8, PAWN_VALUE/8,
   0, -PAWN_VALUE/8, -PAWN_VALUE/8, -PAWN_VALUE/8};
static const int KNIGHT_MOBILITY[9] = {-8, -4, -2, 0, 1, 3, 5, 8, 8};
static const int ROOK_ON_7TH_RANK = 25;
static const int LOW_ROOK_MOBILITY[2] = {-8,-4};
static const int DOUBLED_ROOKS = 8;               // on open file
static const int ROOK_ON_OPEN_FILE = 12;
static const int QUEEN_OUT = -4;
// castling terms
static const int KCASTLE = 11;
static const int QCASTLE = 8;
static const int CAN_CASTLE = -2;
static const int CANT_CASTLE = -16;
static const int CANT_CASTLE_K_SIDE = -12;
static const int CANT_CASTLE_Q_SIDE = -8;
static const int CASTLING_SCORES[6] = {
    CAN_CASTLE, CANT_CASTLE_Q_SIDE, CANT_CASTLE_K_SIDE,
    KCASTLE, QCASTLE, CANT_CASTLE
};

// pawn structure terms
static const int WEAK = -4;
static const int WEAK2 = -4;
static const int WEAK_ON_OPEN_FILE = -8;
static const int SPACE=1;
// isolated pawn
static const int ISOLATED_PAWN[9] = 
  {0, -4, -8, -13, -18, -24, -30, -36, -42 };
static const int DOUBLED_PAWNS = -10;
static const int TRIPLED_PAWNS = -10; // in addition to double penalty

static const int PASSED_PAWN[8] =                 // score by rank
{
    0, 9, 15, 24, 33, 48, 75, 0
};
static const int HIDDEN_PASSED_PAWN = 15;
static const int PASSED_PAWN_BLOCK[8] =
{ 0, 9/3, 15/3, 24/3, 33/3, 48/4, 10, 0 };
// outside passed pawns
static const int OUTSIDE_PP[16] =
{
  30, 24, 18, 14, 12, 12, 10, 10,
  10, 10, 8, 8, 8, 8, 8, 8
};

static const int SUPPORTED_PASSER6 = 24;
static const int SUPPORTED_PASSER7 = 48;

static const int ROOK_BEHIND_PP = 25;
static const int ROOK_AHEAD_OF_PP = -10;
static const int CONNECTED_PASSERS[8] =
{0, 5, 7, 11, 17, 25, 35, 0};
static const int ADJACENT_PASSERS = 8;
static const int RIM_PASSED_PAWN = -4;            // pawn on h-file or a-file
static const int PAWN_DUO = 5;                    // a la Kmoch
static const int ADV_UNSUPPORTED_PAWN = -4;       // an advanced but unsupported pawn

// king safety terms
static const int KING_COVER0 = 5;
static const int KING_COVER1 = 18;
static const int KING_COVER2 = 9;
static const int KING_COVER3 = 9;
static const int KING_OFF_BACK_RANK = -10;
static const int KING_NEAR_OPEN_FILE = -22;
static const int KING_NEAR_HALF_OPEN_FILE = -14;
static const int KING_PIECE_ATTACKS[7] =
  { 0, 0, 1, 1, 2, 4, 1 };
static int KING_PAWN_PROXIMITY[8] =
  {-16, -16, -10, -4, 0, 0, 0, 0 };
static int FRIENDLY_ROOK=8;
static const int KING_ATTACK_MATERIAL_SCALE[32] = {
  0, 0, 0, 0, 0, 0, 0, 0, 
  0, 0, 0, 0, 2, 4, 7, 11,
 14, 16, 18, 20, 22, 24, 26, 27, 
 28, 29, 29, 30, 30, 31, 32, 32};
static const int KING_ATTACK_MULTIPLIER = 8;
static const int KING_ATTACK_FACTOR = 100;
static const int KING_PROXIMITY_FACTOR = 100;
static const int KING_COVER_FACTOR = 80;
#define BOOST
static const int KING_ATTACK_MAX_BOOST = PAWN_VALUE;
static const int KING_ATTACK_BOOST_THRESHOLD = 32;


// king position terms - valid only in endgame
static const int OPPOSITION = 6;                  // twice this for direct opposition

// piece safety terms
static const int BISHOP_TRAPPED = -100;           // bishop trapped by pawn
// general endgame terms
static const int KING_PAWN_DISTANCE = 4;
// KP vs. K endgame
static const int UNCATCHABLE_PAWN = 2*PAWN_VALUE;
// a pawn that cannot be captured by the opposing king even without
// the assistance of its own king:
static const int UNCATCHABLE_PAWN3 = 48;
static const int KING_AHEAD_OF_PAWN = 12;
static const int DISTANCE_FROM_PAWN = -4;

static int first_ones_8bit[256];
static Bitmap aheadW[64];
static Bitmap aheadB[64];
CACHE_ALIGN Bitmap passedW[64];
CACHE_ALIGN Bitmap passedB[64];
static Bitmap backwardW[64];
static Bitmap backwardB[64];
static Bitmap weakW[64];
static Bitmap weakB[64];
static Bitmap outpostW[64];
static Bitmap outpostB[64];
static Bitmap sq_king_wtm[64],sq_king_btm[64];
static Bitmap right_side_mask[8], left_side_mask[8];
// area near king
static Bitmap king_file_proximity[8];
static Bitmap king_rank_proximity[8];
static Bitmap king_knight_proximity[64];
static byte king_file_danger[8];
static Bitmap king_prox[64];
static Bitmap w_pawn_attacks[64], b_pawn_attacks[64];
static Bitmap rook_pawn_mask;
static Bitmap abcd_mask, efgh_mask;
static int done_init = 0;
static int opposition[64][64];
int distances[64][64];

// Don't call it "distance" - conflicts with MS template library
static inline int distance1(int x, int y)
{
    return distances[x][y];
}


// Return all squares attacked by a pawn of color "side" at square "sq".
static FORCEINLINE Bitmap pawn_attacks( const Board &board, Square sq, ColorType side)
{
    return Bitmap::And(Bearing::pawn_attacks[sq][side],board.pawn_bits[side]);
}


#ifdef EVAL_STATS
void Scoring::clearStats()
{
    position_hash_probes = position_hash_hits = 0L;
    pawn_hash_probes = pawn_hash_hits = 0L;
    king_pawn_hash_probes = king_pawn_hash_hits = 0L;
    king_cover_hash_probes = king_cover_hash_hits = 0L;
    lazy_cut_chances = lazy_cuts = 0L;
}
#endif

int Scoring::sq_king_test(Square oppkp,Square pawnpos,ColorType side,
ColorType sideToMove)
{
    int oppkp_rank = Rank(oppkp,side);
    int rankdist = Rank(pawnpos,side) - oppkp_rank;
    int sqkingdist;
    if (sideToMove == side) {
        if (rankdist > 0) {
            // If the king is behind us and we are on move we can't be caught.
            return 1;
        }
        else {
            sqkingdist = distance1(oppkp,Square(File(pawnpos),8,side))-(8-Rank(pawnpos,side));
            if (sqkingdist>0)
                return 1;
        }
    }
    else {
        if (rankdist > 1) {
            // pawn > 1 square ahead
            return 1;
        }
        else {
            sqkingdist = distance1(oppkp,Square(File(pawnpos),8,side))-(9-Rank(pawnpos,side));
            if (sqkingdist>0)
                return 1;
        }
    }
    return 0;
}

static void init_bitmaps()
{
    int i, r;

    for (i = 0; i < 64; i++) {
        kingCoverW[i] = kingCoverB[i] = kingCoverW2[i] = kingCoverB2[i] =
            kingCoverW0[i] = kingCoverB0[i] = Bitmap(0);
        if (File(i) != 8) {
            kingCoverW0[i].set(i+1);
            kingCoverB0[i].set(i+1);
        }
        if (File(i) != 1) {
            kingCoverW0[i].set(i-1);
            kingCoverB0[i].set(i-1);
        }
        if (Rank(i,White) != 8) {
            kingCoverW[i].set(i-8);
            if (File(i) != 8) {
                kingCoverW[i].set(i-7);
            }
            if (File(i) != 1) {
                kingCoverW[i].set(i-9);
            }
        }
        kingCoverW[H1].set(F2);
        kingCoverW[A1].set(C2);
        kingCoverB[H8].set(F7);
        kingCoverB[A8].set(C7);
	if (i>=16)
            kingCoverW2[i].set(i-16);
        if (Rank(i,White) < 7) {
            if (File(i) != 8) {
  	        ASSERT(i-7-8>=0);
                kingCoverW2[i].set(i-7-8);
            }
            if (File(i) != 1) {
  	        ASSERT(i-9-8>=0);
                kingCoverW2[i].set(i-9-8);
            }
        }
        if (Rank(i,Black) != 8) {
            ASSERT(i+8<=63);
            kingCoverB[i].set(i+8);
            if (File(i) != 8) {
	        ASSERT(i+9<=63);
                kingCoverB[i].set(i+9);
            }
            if (File(i) != 1) {
                ASSERT(i+7<=63);
                kingCoverB[i].set(i+7);
            }
        }
	if (i<48)
	    kingCoverB2[i].set(i+16);
        if (Rank(i,Black) < 7) {
            if (File(i) != 8) {
                ASSERT(i+9+8<=63);
                kingCoverB2[i].set(i+9+8);
            }
            if (File(i) != 1) {
                ASSERT(i+7+8<=63);
                kingCoverB2[i].set(i+7+8);
            }
        }
        kingCoverW2[H1].set(F3);
        kingCoverW2[H2].set(F3);
        kingCoverW2[A1].set(C3);
        kingCoverW2[A2].set(C3);
        kingCoverB2[H8].set(F6);
        kingCoverB2[H7].set(F6);
        kingCoverB2[A8].set(C6);
        kingCoverB2[A7].set(C6);
        for (int j = 0; j < 64; j++) {
            int file_distance, rank_distance;
            Square kp = i;
            Square oppkp = j;
            file_distance=Util::Abs(File(kp)-File(oppkp));
            rank_distance=Util::Abs(Rank(kp,Black)-Rank(oppkp,Black));
            if (file_distance == 0 && rank_distance == 2) {
                opposition[i][j] = 2;             // direct opposition
            }
            else if (file_distance == 0 && rank_distance % 2 == 0)
                opposition[i][j] = 1;             // remote opposition
            else if ((file_distance == 2) && (rank_distance == 2))
                opposition[i][j] = 1;
            else
                opposition[i][j] = 0;
            if (file_distance > rank_distance)
                distances[i][j] = file_distance;
            else
                distances[i][j] = rank_distance;
        }
    }
    Square sq;
    for (sq = 0; sq < 64; sq++) {
        for (int j=0; j<64; j++) {
            if (Scoring::sq_king_test(sq,j,White,White))
                sq_king_wtm[sq].set(j);
            if (Scoring::sq_king_test(sq,j,White,Black))
                sq_king_btm[sq].set(j);
        }

        int rank = Rank(sq,White);
        int file = File(sq);
        if (file == 1 || file == 8)
            rook_pawn_mask.set(sq);
        for (r = rank; r > 1; r--) {
            Square sq2 = Square(file,r,White);
            if (file != 8) {
                sq2 = Square(file+1,r,White);
                backwardW[sq].set((int)sq2);
            }
            if (file != 1) {
                sq2 = Square(file-1,r,White);
                backwardW[sq].set((int)sq2);
            }
        }
        weakW[sq] = backwardW[sq];
        if (file != 8) {
            weakW[sq].clear(Square(file+1,rank,White));
        }
        if (file != 1)
            weakW[sq].clear(Square(file-1,rank,White));
        for (r = rank + 1; r < 8; r++) {
            Square sq2 = Square(file,r,White);
            passedW[sq].set(sq2);
            aheadW[sq].set(sq2);
            if (file != 8) {
                sq2 = Square(file+1,r,White);
                passedW[sq].set(sq2);
                outpostW[sq].set(sq2);
            }
            if (file != 1) {
                sq2 = Square(file-1,r,White);
                passedW[sq].set(sq2);
                outpostW[sq].set(sq2);
            }
        }
        rank = Rank(sq,Black);
        for (r = rank; r > 1; r--) {
            Square sq2 = Square(file,r,Black);
            if (file != 8) {
                sq2 = Square(file+1,r,Black);
                backwardB[sq].set(sq2);
            }
            if (file != 1) {
                sq2 = Square(file-1,r,Black);
                backwardB[sq].set(sq2);
            }
        }
        weakB[sq] = backwardB[sq];
        if (file != 8) {
            weakB[sq].clear(Square(file+1,rank,Black));
        }
        if (file != 1) {
            weakB[sq].clear(Square(file-1,rank,Black));
        }
        for (r = rank + 1; r < 8; r++) {
            Square sq2 = Square(file,r,Black);
            passedB[sq].set((int)sq2);
            aheadB[sq].set(sq2);
            if (file != 8) {
                sq2 = Square(file+1,r,Black);
                passedB[sq].set(sq2);
                outpostB[sq].set(sq2);
            }
            if (file != 1) {
                sq2 = Square(file-1,r,Black);
                passedB[sq].set(sq2);
                outpostB[sq].set(sq2);
            }
        }
    }
    for (sq = 0; sq < 64; sq++) {
        Bitmap dests(Bearing::king_attacks[sq]);
        king_prox[sq] = dests;
        king_prox[sq].set(sq);
        king_knight_proximity[sq] = Bearing::knight_attacks[sq];
        Square sq2;
        while (dests.iterate(sq2)) {
            king_knight_proximity[sq].Or(
                Bearing::knight_attacks[sq2]);
        }
        w_pawn_attacks[sq].clear();
        b_pawn_attacks[sq].clear();
        int file = File(sq); int rankw = WhiteRank(sq);
        int fmin = Util::Max(1,file-2);
        int fmax = Util::Min(8,file+2);
        //if (file == 1) fmax=4;
        //if (file == 8) fmin=5;
        for (int f = fmin; f <= fmax; f++) {
            int r;
            if (Util::Abs(f-file)<2) {
                for (r = rankw; r <= Util::Min(rankw+3,8); r++) {
                    w_pawn_attacks[sq].set(Square(f,r,White));
                }
                for (r = rankw; r >= Util::Max(rankw-3,1); r--) {
                    b_pawn_attacks[sq].set(Square(f,r,White));
                }
            }
            else {
                for (r = rankw; r <= Util::Min(rankw+3,8); r++) {
                    w_pawn_attacks[sq].set(Square(f,r,White));
                }
                for (r = rankw; r >= Util::Max(rankw-3,1); r--) {
                    b_pawn_attacks[sq].set(Square(f,r,White));
                }
            }
        }
    }

    int x;
    for (x=1; x <8; x++) {
        for (int y=x-1;y>=0;y--)
            left_side_mask[x].Or(Bearing::file_mask[y]);
    }
    for (x=6; x >=0; x--) {
        for (int y=x+1;y<8;y++)
            right_side_mask[x].Or(Bearing::file_mask[y]);
    }
    for (x = 0; x < 8; x++) {
        king_file_danger[x] = 0;
        king_file_proximity[x] = Bearing::file_mask[x];
        king_rank_proximity[x] = Bearing::rank_mask[x];
        king_file_danger[x] |= 1<<x;
        if (x == 0) {
            king_file_proximity[x].data |= Bearing::file_mask[2].data;
            king_file_danger[x] |= 1<<2;
        }
        else {
            king_file_danger[x] |= 1<<(x-1);
            if (x==2) king_file_danger[x] |= 1<<0;
            if (x==5) king_file_danger[x] |= 1<<7;
            king_file_proximity[x].data |= Bearing::file_mask[x-1].data;
            king_rank_proximity[x].data |= Bearing::rank_mask[x-1].data;
        }
        if (x==7) {
            king_file_proximity[x].data |= Bearing::file_mask[5].data;
            king_file_danger[x] |= 1<<5;
        }
        else {
            king_file_danger[x] |= 1 <<(x+1);
            king_file_proximity[x].data |= Bearing::file_mask[x+1].data;
            king_rank_proximity[x].data |= Bearing::rank_mask[x+1].data;
        }
    }
    Bitmap ghfiles = Bitmap::Or(Bearing::file_mask[7],Bearing::file_mask[6]);
    Bitmap right_mask = ghfiles;
    right_mask.Or(Bearing::file_mask[5]);
    Bitmap left_mask = Bitmap::Or(Bearing::file_mask[0],Bearing::file_mask[1]);
    left_mask.Or(Bearing::file_mask[2]);
    abcd_mask = left_mask;
    abcd_mask.Or(Bearing::file_mask[3]);
    efgh_mask = right_mask;
    efgh_mask.Or(Bearing::file_mask[4]);
    for (i = 0; i < 256; i++) {
        int y =0;
        Bitmap bm(i);
        for (int j = 0; j<8; j++) {
            if (bm.is_set(j)) {
                y = j;
                break;
            }
        }
        first_ones_8bit[i] = y;
    }
}


void Scoring::init()
{
    if (!done_init) {
        init_bitmaps();
        done_init++;
    }
}


void Scoring::cleanup()
{
}


static int KBPDrawW(const Board &board)
{
    Bitmap pb(board.pawn_bits[White]);
    if (OnFile(pb,AFILE)) {
        // on the a file
        // queen square is white
        if (!TEST_MASK(board.bishop_bits[White],Bearing::white_squares)) {
            // wrong color bishop
            Square kp = board.KingPos(Black);
            if (kp == A1 || Bearing::king_attacks[A1].is_set(kp)) {
                return 1 /*TRUE*/;
            }
        }
    }
    else if (OnFile(pb,HFILE)) {
        // on the h file
        // queen square is black
        if (!TEST_MASK(board.bishop_bits[White],Bearing::black_squares)) {
            // wrong color bishop
            Square kp = board.KingPos(Black);
            if (kp == H1 || Bearing::king_attacks[H1].is_set(kp)) {
                return 1 /*TRUE*/;
            }
        }
    }
    return 0;
}


static int KBPDrawB(const Board &board)
{
    Bitmap pb(board.pawn_bits[Black]);
    if (OnFile(pb,AFILE)) {
        // on the a file
        // queen square is black
        if ((Bitmap::And(board.bishop_bits[Black],Bearing::black_squares)).is_clear()) {
            // wrong color bishop
            Square kp = board.KingPos(White);
            if (kp == A8 || Bearing::king_attacks[A8].is_set(kp)) {
                return 1 /*TRUE*/;
            }
        }
    }
    else if (OnFile(pb,HFILE)) {
        // on the h file
        // queen square is white
        if ((Bitmap::And(board.bishop_bits[Black],Bearing::white_squares)).is_clear()) {
            // wrong color bishop
            Square kp = board.KingPos(White);
            if (kp == H8 || Bearing::king_attacks[H8].is_set(kp)) {
                return 1;
            }
        }
    }
    return 0;
}

static int adjustMaterialScore(const Board &board, ColorType side)
{
    const Material &ourmat = board.getMaterial(side);
    const Material &oppmat = board.getMaterial(OppositeColor(side));
    int score = 0;
    if (ourmat.bishop_count() >=2 && oppmat.bishop_count() < 2) {
        int pawns = ourmat.pawn_count() + oppmat.pawn_count();
        score += (BISHOP_PAIR + BISHOP_PAIR_PAWN_ADJUST[pawns/2]);
#ifdef EVAL_DEBUG
        cout << "bishop pair (" << ColorImage(side) << ") : " <<
            (BISHOP_PAIR + BISHOP_PAIR_PAWN_ADJUST[pawns/2]) << endl;
#endif
    }
    // Evaluate uneven trades. Penalties/bonuses loosely based on
    // Larry Kaufman article.
    int trade_score = 0;
    if (Util::Abs(ourmat.value()-oppmat.value())<=5*PAWN_VALUE) {
        int oppMinors = oppmat.bishop_count() + oppmat.knight_count();
        int ourMinors = ourmat.bishop_count() + ourmat.knight_count();
        if (ourMinors >= oppMinors) {
            int ourQ = ourmat.queen_count();
            int oppQ = oppmat.queen_count();
            int ourR = ourmat.rook_count();
            int oppR = oppmat.rook_count();
            if (ourQ == oppQ && ourR == oppR) {
                // knight or bishop vs. pawns
                if (ourMinors == oppMinors+1) {
                    if (ourmat.infobits() == Material::KB ||
                        ourmat.infobits() == Material::KN) {
                        // we don't have mating material (no pawns)
                        trade_score -= PAWN_VALUE;
                    }
                    else                          // small bonus for piece
                        trade_score += PAWN_VALUE/4;
                }
		else if (ourMinors == oppMinors+2) {
		   // 2 minors vs. pawns
		  if (ourmat.infobits() == Material::KNN) {
                    // we don't have mating material (no pawns)
		    trade_score -= PAWN_VALUE;
		  }
		  else {
		    trade_score += PAWN_VALUE/2;
		  }
		}
            }
            else if (ourQ != oppQ) {
	      if (ourQ == 0) {
                 if (ourR==oppR+1 && ourMinors==oppMinors+1) {
                    // penalize Q traded for Rook + Knight or bishop
                    trade_score -= PAWN_VALUE/2;
                 }
                 else if (ourR==oppR && ourMinors==oppMinors+2) {
		    // Queen vs. 2 minors + pawns 
		    trade_score -= PAWN_VALUE/2;
		 }
	      }
            }
            else {                                // rooks un-equal
                // 2 minors vs. Rook. Slight bonus for minors (worth > 1
	        // pawn over Rook).
                if (ourMinors == oppMinors+2) {    // rook vs. 2 minors
		  trade_score += RBN_TRADE[Util::Min(7,oppmat.pieceCount())];
                }
                // rook vs. minor
                else if (ourR+1==oppR && ourMinors==oppMinors+1) {
		  if (oppmat.pawn_count() + ourmat.pawn_count() == 0) {
   		      // R vs minor w. no pawns is generally a draw
		      // so is RB vs RR. So give bonus for side with minor.
		     if (ourmat.infobits() == Material::KR ||
			 ourmat.infobits() == Material::KRR) {
			 trade_score = 2*PAWN_VALUE;
		     }
		  }
		  else { // minor vs. Rook, w. pawns
		    // side with minor actually gets bonus. Because, per
                    // Kaufman, this is less good for the Rook side
                    // than material balance alone would indicate. But we
		    // give a bit larger score for the Rook than Kaufman.
                    trade_score += RB_TRADE[Util::Min(7,ourmat.pieceCount())];
		  }
                }
            }
        }
        else {                                    // Q vs. 2 R
            if (oppmat.queen_count() == ourmat.queen_count()+1 &&
            ourmat.rook_count() == oppmat.rook_count()+2) {
                if (!((oppmat.infobits() & Material::PieceMask) == Material::KQ &&
                (ourmat.infobits() & Material::PieceMask) == Material::KRR)) {
                    trade_score -= PAWN_VALUE;    // bonus for queen if other pieces
                }
            }
        }
    }
    else {
        int mat_total = oppmat.material_level()+ourmat.material_level();
        // this catches cases where material value is even but a piece
        // has been traded for pawns (usually a bad idea unless pawns
        // are far advanced). It also gives a bonus for large material
        // imbalance.
        int bonus = (ourmat.material_level()-oppmat.material_level())*8;
        if (bonus<-50) bonus = -50;
        else if (bonus>50) bonus = 50;
        if (mat_total < 32) {
            if (bonus>0) bonus = (bonus+32-mat_total);
            else bonus = (bonus-32+mat_total);
        }
        score += bonus;
#ifdef EVAL_DEBUG
        cout << "score material adjust (" << ColorImage(side) <<
            ") = " << bonus << endl;
#endif
    }
    score += trade_score;
    return score;
}

static int adjustMaterialScoreNoPawns( const Board &board, ColorType side ) {
   // pawnless endgames. Generally drawish in many cases.
   const Material &ourmat = board.getMaterial(side);
   const Material &oppmat = board.getMaterial(OppositeColor(side));
   int score = 0;
   if (ourmat.infobits() == Material::KQ) {
      if (oppmat.infobits() == Material::KRR) {
	 score -= (QUEEN_VALUE-2*ROOK_VALUE); // even
      }
      else if (oppmat.infobits() == Material::KRB) {
	 score -= QUEEN_VALUE-(ROOK_VALUE+BISHOP_VALUE); // even
      }
      else if (oppmat.infobits() == Material::KRN) { // close to even
	 score -= QUEEN_VALUE-(ROOK_VALUE+BISHOP_VALUE)+PAWN_VALUE/4;
      }
   }
   else if (ourmat.infobits() == Material::KR) {
     if (oppmat.infobits() == Material::KBN) { // close to even
       score += (KNIGHT_VALUE+BISHOP_VALUE-ROOK_VALUE)-PAWN_VALUE/4;
     }
     else if (oppmat.infobits() == Material::KB){ // even
       score -= (ROOK_VALUE-BISHOP_VALUE);
     }
     else if (oppmat.infobits() == Material::KN){ // close to even
       score -= (ROOK_VALUE-KNIGHT_VALUE)+PAWN_VALUE/4;
     }
   }
   else if (ourmat.infobits() == Material::KRR) {
     if (oppmat.infobits() == Material::KRB) { // even
       score -= ROOK_VALUE-BISHOP_VALUE;
     }
     else if (oppmat.infobits() == Material::KRN) { // close to even
       score -= ROOK_VALUE-KNIGHT_VALUE-PAWN_VALUE/4;
     }
     else if (oppmat.infobits() == Material::KRBN) { // close to even
       score += (KNIGHT_VALUE+BISHOP_VALUE-ROOK_VALUE)-PAWN_VALUE/4;
     }
   }
   return score;
}


static int MaterialScore( const Board &board, ColorType side )
{
    // Computes material score, from the perspective of 'side'.

    const Material &ourmat = board.getMaterial(side);
    const Material &oppmat = board.getMaterial(OppositeColor(side));
    int mdiff =  (int)(ourmat.value() - oppmat.value());
    int score = 0;
#ifdef EVAL_DEBUG
    cout << "mdiff=" << mdiff << endl;
#endif
    // check for bishop endgame - drawish
    int our_pieces = ourmat.infobits() & Material::PieceMask;
    int opp_pieces = oppmat.infobits() & Material::PieceMask;
    if (our_pieces == Material::KB && opp_pieces == Material::KB) {
        if (Util::Abs(ourmat.pawn_count() - oppmat.pawn_count()) < 4)
            mdiff -= mdiff/4;
    }
    if (our_pieces != opp_pieces) {
#ifdef EVAL_DEBUG
      int tmp = score;
#endif
      if (ourmat.no_pawns() && oppmat.no_pawns()) {
	 score += adjustMaterialScoreNoPawns(board,side) -
	   adjustMaterialScoreNoPawns(board,OppositeColor(side));
      }
      else {
         score += adjustMaterialScore(board,side) -
	          adjustMaterialScore(board,OppositeColor(side));
      }
#ifdef EVAL_DEBUG
    if (score-tmp) cout << "trade score = " << score-tmp << endl;
#endif
    }
    return score+mdiff;
}


// evaluate outside passed pawns.
static void eval_outside_pp(const Board &board,Pawn_Entry *pawn_hash_entry)
{
    // This code is based on similar code in Crafty. See also
    // Fine's Basic Chess Endings, pp. 35-36.

    byte first_passer_w = pawn_hash_entry->w_pawn_entry.first_passer;
    byte last_passer_w = pawn_hash_entry->w_pawn_entry.last_passer;
    byte first_passer_b = pawn_hash_entry->b_pawn_entry.first_passer;
    byte last_passer_b = pawn_hash_entry->b_pawn_entry.last_passer;

    pawn_hash_entry->b_pawn_entry.outside =
        pawn_hash_entry->w_pawn_entry.outside = 0;
    if (first_passer_w == 9 && first_passer_b == 9) {
        return;
    }
    Bitmap all_pawns(Bitmap::Or(board.pawn_bits[White],board.pawn_bits[Black]));
    if (first_passer_w != 9 && !board.getMaterial(Black).no_pawns()) {
        if ((first_passer_w < first_passer_b-1) ||
        (last_passer_w > last_passer_b+1)) {
            if (first_passer_w <= 4) {
                if (!Bitmap::And(all_pawns,right_side_mask[first_passer_w-1]).is_clear()
                &&  Bitmap::And(board.pawn_bits[Black],left_side_mask[first_passer_w-1]).is_clear()) {
                    if (!pawn_hash_entry->w_pawn_entry.protected_pp || pawn_hash_entry->b_pawn_entry.protected_pp) {
                        pawn_hash_entry->w_pawn_entry.outside++;
                        goto step2;
                    }
                }
            }
            if (last_passer_w >= 5) {
                if (!Bitmap::And(all_pawns,left_side_mask[last_passer_w-1]).is_clear()
                &&  Bitmap::And(board.pawn_bits[Black],right_side_mask[last_passer_w-1]).is_clear()) {
                    if (!pawn_hash_entry->w_pawn_entry.protected_pp || pawn_hash_entry->b_pawn_entry.protected_pp) {
                        pawn_hash_entry->w_pawn_entry.outside++;
                    }
                }
            }
        }
    }
    step2:
    if (first_passer_b != 9 && !board.getMaterial(White).no_pawns()) {
        if ((first_passer_b < first_passer_w-1) ||
        (last_passer_b > last_passer_w+1)) {
            if (first_passer_b <= 4) {
                if (!Bitmap::And(all_pawns,right_side_mask[first_passer_b-1]).is_clear()
                &&  Bitmap::And(board.pawn_bits[White],left_side_mask[first_passer_b-1]).is_clear()) {
                    if (!pawn_hash_entry->b_pawn_entry.protected_pp || pawn_hash_entry->w_pawn_entry.protected_pp) {
                        pawn_hash_entry->b_pawn_entry.outside++;
                        goto step3;
                    }
                }
            }
            if (last_passer_b >= 5) {
                if (!Bitmap::And(all_pawns,left_side_mask[last_passer_b-1]).is_clear()
                &&  Bitmap::And(board.pawn_bits[White],right_side_mask[last_passer_b-1]).is_clear()) {
                    if (!pawn_hash_entry->b_pawn_entry.protected_pp || pawn_hash_entry->w_pawn_entry.protected_pp) {
                        pawn_hash_entry->b_pawn_entry.outside++;
                    }
                }
            }
        }
    }
    step3:
    return;
}


static int pawn_structure_score(const Board &board, ColorType side,
Pawn_Entry::PawnData &entr)
{
    // This function computes a pawn structure score that depends
    // only on the location of pawns (of both colors). It also fills
    // in the pawn hash entry.
    //
    memset(&entr,'\0',sizeof(Pawn_Entry::PawnData));
    entr.first_passer = 9;
    entr.last_passer = 0;
    entr.open_files = 0xff;

    int score = 0;
    int isolated_count = 0;
    int incr = (side == White) ? -8 : 8;
    const ColorType oside = OppositeColor(side);
    entr.opponent_pawn_attacks = board.all_pawn_attacks(oside);
    Piece enemy_pawn = MakePiece(Pawn,oside);
    Piece our_pawn = MakePiece(Pawn,side);
    Bitmap bi(board.pawn_bits[side]);
    Bitmap all_pawns(board.pawn_bits[White]);
    all_pawns.Or(board.pawn_bits[Black]);
    Square sq;
    int lside = 0, rside = 0;
    int max_passer_rank = 0;
#ifdef PAWN_DEBUG
    int total_space = 0;
#endif
    int locked_pawns = 0;
    entr.blocked = Bitmap(0);
    while (bi.iterate(sq)) {
#ifdef PAWN_DEBUG
        int tmp = score;
        cout << ColorImage(side) << " pawn on " << FileImage(sq) <<
            RankImage(sq);
#endif
        int file = File(sq);
        int rank = Rank(sq,side);
        if (rank == 2) ++entr.pawns_on_2nd_rank;
        if (file < 4) {
            lside++;
        }
        else if (file > 5) {
            rside++;
        }
        if (SquareColor(sq) == White)
            entr.w_square_pawns++;
        else
            entr.b_square_pawns++;
        entr.open_files &= (byte)~(1<<(file-1));
        int backward = 0 /*FALSE*/;
        int passed = 0 /*FALSE*/;
        int weak = 0 /*FALSE*/;
        int doubled = 0 /*FALSE*/;
        int isolated;
        if (side == White) {
            entr.center_score += PawnCenterScores[sq];
        }
        else {
            entr.center_score += PawnCenterScores[63-(int)sq];
        }
        Bitmap result;
        if (side == White)
            result = Bitmap::And(board.pawn_bits[side],backwardW[sq]);
        else
            result = Bitmap::And(board.pawn_bits[side],backwardB[sq]);
        backward = result.is_clear();
        if (side == White)
            result = Bitmap::And(board.pawn_bits[oside],passedW[sq]);
        else
            result = Bitmap::And(board.pawn_bits[oside],passedB[sq]);
        passed = result.is_clear();
        if (file == AFILE) {
            isolated = !OnFile(board.pawn_bits[side],BFILE);
        }
        else if (file == HFILE) {
            isolated = !OnFile(board.pawn_bits[side],GFILE);
        }
        else {
            isolated = !OnFile(board.pawn_bits[side],file-1) &&
                !OnFile(board.pawn_bits[side],file+1);
        }
        Bitmap doubles;
        if (side == White) 
           doubles = Bitmap::And(board.pawn_bits[side],Bearing::file_mask_up[sq]);
        else
           doubles = Bitmap::And(board.pawn_bits[side],Bearing::file_mask_down[sq]);
        if (!doubles.is_clear()) {
#ifdef PAWN_DEBUG
            cout << " doubled";
#endif
            passed = 0; // our own pawn is ahead of us, don't count as passed
            doubled++;
            score += DOUBLED_PAWNS;
            if (doubles.bit_count() > 1) score += TRIPLED_PAWNS;
        }
        if (isolated && !passed && !doubled) {
            isolated_count++;
#ifdef PAWN_DEBUG
            cout << " isolated";
#endif
        }
        if (Rank(sq,side)<6) {
            int defenders = pawn_attacks(board,sq,side).bit_count();
            if (board[sq+incr] != enemy_pawn) {
                if (defenders == 0) {
                    Square sq2(sq + incr);
                    int patcks = pawn_attacks(board,sq2,oside).bit_count() -
                        pawn_attacks(board,sq2,side).bit_count();
                    if (patcks > 0) {
#ifdef PAWN_DEBUG
                        cout << " weak";
#endif
                        score += WEAK;
                        weak++;
                    }
                    if (patcks > 1) {             // "extra weak" pawn
                        score += WEAK2;
#ifdef PAWN_DEBUG
                        cout << " extra weak";
#endif
                        locked_pawns++;
                    }
                }
            }
            else
                locked_pawns++;
            if (defenders && pawn_attacks(board,sq,oside).is_clear()) {
	      entr.blocked.set(sq);
	    }
        }
        if (!passed && (weak || backward || isolated)) {
            entr.weak_files = (byte)(entr.weak_files | (byte)(1<<(file-1)));
            if (!OnFile(board.pawn_bits[oside],file)) {
                entr.weakopen++;
#ifdef PAWN_DEBUG
                cout << " weak/open";
#endif
            }
        }
        if (!isolated && !weak && !backward && !passed && rank > 2) {
            //score += UNSUPPORTED_PAWN;
            if (rank >4 && pawn_attacks(board,sq,side).is_clear()) {
                score += ADV_UNSUPPORTED_PAWN;
#ifdef PAWN_DEBUG
                cout << " unsupported" << endl;
#endif
            }
        }
        if (passed) {
#ifdef PAWN_DEBUG
            cout << " passed";
#endif
            entr.passers[entr.passers_count++] = sq;
            ASSERT(entr.passers_count <= 8);
            ASSERT(Rank(sq,side)<8);
            if (rank>max_passer_rank) {
                entr.max_passer_rank = rank;
                max_passer_rank = rank;
                entr.max_passer_rank_sq = sq;
            }
            if (file > entr.last_passer)
                entr.last_passer = file;
            if (file < entr.first_passer)
                entr.first_passer = file;
            score += PASSED_PAWN[rank-1];
            if (side == White) {
                Bitmap atcks(Bearing::pawn_attacks[sq][White]);
                atcks.And(board.pawn_bits[White]);
                if (!atcks.is_clear())
                    entr.protected_pp = 1;
            }
            else {
                Bitmap atcks(Bearing::pawn_attacks[sq][Black]);
                atcks.And(board.pawn_bits[Black]);
                if (!atcks.is_clear())
                    entr.protected_pp = 1;
            }
            int j;
            for (j=0;j<entr.passers_count-1;j++) {
                Square psq = entr.passers[j];
                int psqfile = File(psq);
                if (psqfile == file-1 || psqfile == file+1) {
                    int psqrank = Rank(psq,side);
                    int rdiff = rank - psqrank;
                    // score is based on the more advanced
                    // of the 2 pawns
                    if (rdiff == 0 || rdiff == 1) {
                        score += CONNECTED_PASSERS[rank-1];
#ifdef PAWN_DEBUG
                        cout <<  " connected passer";
#endif
                    }
                    else if (rdiff == -1) {
                        score += CONNECTED_PASSERS[psqrank-1];
#ifdef PAWN_DEBUG
                        cout <<  " connected passer";
#endif
                    }
                    else
                        score += ADJACENT_PASSERS;
                }
            }
            if (file ==1 || file == 8) {
	      score += RIM_PASSED_PAWN;
#ifdef PAWN_DEBUG
	      cout << " rim=" << RIM_PASSED_PAWN;
#endif
	    }
        }
        else if (Rank(side,sq) >= 5) {
            // A "hidden" passed pawn can advance and turn into a
            // passer, since it is part of a 2 vs. 1 pawn majority.
            // Crafty has similar scoring for this, but this code
            // is a little more general:
            int hidden = 0;
            if (File(sq) == HFILE && !OnFile(all_pawns,FFILE)) {
                if (side == White) {
                    Bitmap b = Bitmap::And(board.pawn_bits[White],Bearing::file_mask[GFILE-1]);
                    Square gpawn = b.first_one();
                    if (gpawn != InvalidSquare) {
                        int gahead = Bitmap::And(aheadW[gpawn],board.pawn_bits[Black]).bit_count();
                        int hahead = Bitmap::And(aheadW[sq],board.pawn_bits[Black]).bit_count();
                        hidden = (gahead == 0 && hahead == 1) ||
                            (gahead == 1 && hahead == 0);
                    }
                }
                else {
                    Bitmap b = Bitmap::And(board.pawn_bits[Black],Bearing::file_mask[GFILE-1]);
                    Square gpawn = b.first_one();
                    if (gpawn != InvalidSquare) {
                        int gahead = Bitmap::And(aheadB[gpawn],board.pawn_bits[White]).bit_count();
                        int hahead = Bitmap::And(aheadB[sq],board.pawn_bits[White]).bit_count();
                        hidden = (gahead == 0 && hahead == 1) ||
                            (gahead == 1 && hahead == 0);
                    }
                }
            }
            else if (File(sq) == AFILE && !OnFile(all_pawns,CFILE)) {
                if (side == White) {
                    Bitmap b = Bitmap::And(board.pawn_bits[White],Bearing::file_mask[BFILE-1]);
                    Square bpawn = b.first_one();
                    if (bpawn != InvalidSquare) {
                        int bahead = Bitmap::And(aheadW[bpawn],board.pawn_bits[Black]).bit_count();
                        int aahead = Bitmap::And(aheadW[sq],board.pawn_bits[Black]).bit_count();
                        hidden = (bahead == 0 && aahead == 1) ||
                            (bahead == 1 && aahead == 0);
                    }
                }
                else {
                    Bitmap b = Bitmap::And(board.pawn_bits[Black],Bearing::file_mask[BFILE-1]);
                    Square bpawn = b.first_one();
                    if (bpawn != InvalidSquare) {
                        int bahead = Bitmap::And(aheadB[bpawn],board.pawn_bits[White]).bit_count();
                        int aahead = Bitmap::And(aheadB[sq],board.pawn_bits[White]).bit_count();
                        hidden = (bahead == 0 && aahead == 1) ||
                            (bahead == 1 && aahead == 0);
                    }
                }
            }
            if (hidden) {
                score += HIDDEN_PASSED_PAWN;
#ifdef PAWN_DEBUG
                cout <<  " hidden passer";
#endif
            }
        }
        int duo = 0;
        if (rank > 3 && file != 8 && board[sq+1] == our_pawn) {
#ifdef PAWN_DEBUG
            cout << " duo";
#endif
            score += PAWN_DUO;
            duo++;
        }
        if (!passed && rank >= 4) {
            int space = SPACE*(rank-3);
            if (duo || !pawn_attacks(board,sq,side).is_clear())
                space += (SPACE*(rank-3))/2;
            if (rank >= 5 && board[sq+incr] == enemy_pawn)
                // cramping enemy pawn
                space += (SPACE*(rank-3))/2;
            score += space;
#ifdef PAWN_DEBUG
            cout << " space=" <<space;
            total_space += space;
#endif
        }
#ifdef PAWN_DEBUG
        cout << " total = " << score-tmp << endl;
#endif
    }
    if (locked_pawns>=6) entr.locked++;
    ASSERT(isolated_count<=8);

    score += ISOLATED_PAWN[isolated_count];
#ifdef PAWN_DEBUG
    cout << ColorImage(side) << " isolated pawn penalty = " << ISOLATED_PAWN[isolated_count] << endl;
    cout << ColorImage(side) << " pawn space score = " << total_space << endl;
#endif
    ASSERT(entr.passers_count <= board.getMaterial(side).pawn_count());

    entr.score = score;
    return score;
}


// Calculate endgame-related info for a position and cache it
// in "kp_hash_entry"
//
static void calc_endgame(const Board &board, Pawn_Entry::PawnData *w_pawn_entry,
Pawn_Entry::PawnData *b_pawn_entry, Search::King_Pawn_Entry *kp_hash_entry)
{
    int score = 0;
    int k_pos = 0; Square kp = board.KingPos(White);
    // evaluate king position. Big bonus for centralizing king or (if
    // pawns are all on one side) being near pawns.
    // Similar to how Crafty does it.
    Bitmap all_pawns(Bitmap::Or(board.pawn_bits[White],board.pawn_bits[Black]));
    if (!all_pawns.is_clear()) {
       k_pos += KingEndgameScoresW[kp];
       if (Bitmap::And(abcd_mask,all_pawns).is_clear()) {
           if (File(kp)>DFILE)
             k_pos += 4;
           else
             k_pos -= 4;
       }
       else if (Bitmap::And(efgh_mask,all_pawns).is_clear()) {
           if (File(kp)<=DFILE)
             k_pos += 4;
           else
             k_pos -= 4;
       }
    }
    kp_hash_entry->white_king_position  = k_pos;
    k_pos = 0; kp = board.KingPos(Black);
    if (!all_pawns.is_clear()) {
       k_pos += KingEndgameScoresB[kp];
       if (Bitmap::And(abcd_mask,all_pawns).is_clear()) {
           if (File(kp)>DFILE)
             k_pos += 4;
           else
             k_pos -= 4;
       }
       else if (Bitmap::And(efgh_mask,all_pawns).is_clear()) {
           if (File(kp)<=DFILE)
             k_pos += 4;
           else
             k_pos -= 4;
       }
    }
    kp_hash_entry->black_king_position = k_pos;
    kp_hash_entry->white_endgame_pawn_proximity =
        kp_hash_entry->black_endgame_pawn_proximity = (unsigned char)0;
    kp_hash_entry->w_uncatchable = kp_hash_entry->b_uncatchable = (byte)0;
    if (Bitmap::And(all_pawns,left_side_mask[4]).is_clear() ||
    Bitmap::And(all_pawns,right_side_mask[3]).is_clear()) {
        // bonus for king near pawns
        Bitmap it(all_pawns);
        Square sq;
        while (it.iterate(sq)) {
            kp_hash_entry->white_endgame_pawn_proximity += (4-distance1(board.KingPos(White),sq));
            kp_hash_entry->black_endgame_pawn_proximity += (4-distance1(board.KingPos(Black),sq));
        }
#ifdef EVAL_DEBUG
        cout << "king/pawn proximity (endgame) = " <<
            kp_hash_entry->white_endgame_pawn_proximity -
            kp_hash_entry->black_endgame_pawn_proximity << endl;
#endif
    }
    int i;
    for (i = 0; i < w_pawn_entry->passers_count; i++) {
        if (lookupBitbase(board.KingPos(White),w_pawn_entry->passers[i],
        board.KingPos(Black),White,board.Side())) {
#ifdef PAWN_DEBUG
            cout << "White pawn on " <<  FileImage(w_pawn_entry->passers[i]) <<
                RankImage(w_pawn_entry->passers[i]) << " is uncatchable" << endl;
#endif
            kp_hash_entry->w_uncatchable |= (byte)(1<<i);
        }
        int rank = Rank(w_pawn_entry->passers[i],White);
        int file = File(w_pawn_entry->passers[i]);
        if (rank == 6 && (File(board.KingPos(White)) == file-1 || File(board.KingPos(White)) == file+1) &&
            Rank(board.KingPos(White),White) >= rank)
            score += SUPPORTED_PASSER6;
        else if (rank == 7 && (File(board.KingPos(White)) == file-1 || File(board.KingPos(White)) == file+1) &&
            Rank(board.KingPos(White),White) >= rank)
            score += SUPPORTED_PASSER7;
    }
    for (i = 0; i < b_pawn_entry->passers_count; i++) {
        if (lookupBitbase(board.KingPos(Black),b_pawn_entry->passers[i],
        board.KingPos(White),Black,board.Side())) {
#ifdef PAWN_DEBUG
            cout << "Black pawn on " << FileImage(b_pawn_entry->passers[i]) << RankImage(b_pawn_entry->passers[i]) << " is uncatchable" << endl;
#endif
            kp_hash_entry->b_uncatchable |= (byte)(1<<i);
        }
        int rank = Rank(b_pawn_entry->passers[i],Black);
        int file = File(b_pawn_entry->passers[i]);
        if (rank == 6 && (File(board.KingPos(Black)) == file-1 || File(board.KingPos(Black)) == file+1) &&
            Rank(board.KingPos(Black),Black) >= rank)
            score -= SUPPORTED_PASSER6;
        else if (rank == 7 && (File(board.KingPos(Black)) == file-1 || File(board.KingPos(Black)) == file+1) &&
            Rank(board.KingPos(Black),Black) >= rank)
            score -= SUPPORTED_PASSER7;
    }
    kp_hash_entry->score = score;
}


static int calc_cover(const Board &board, Square kp, ColorType side) {
    int cover = -60;
    if (side == White) {
       Bitmap cover0(kingCoverW0[kp]);
       cover0.And(board.pawn_bits[White]);
       cover += cover0.bit_count()*KING_COVER0;
       Bitmap cover1(kingCoverW[kp]);
       cover1.And(board.pawn_bits[White]);
       cover += cover1.bit_count()*KING_COVER1;
       Bitmap cover2(kingCoverW2[kp]);
       cover2.And(board.pawn_bits[White]);
       cover += cover2.bit_count()*KING_COVER2;
    } else {
       Bitmap cover0(kingCoverB0[kp]);
       cover0.And(board.pawn_bits[Black]);
       cover += cover0.bit_count()*KING_COVER0;
       Bitmap cover1(kingCoverB[kp]);
       cover1.And(board.pawn_bits[Black]);
       cover += cover1.bit_count()*KING_COVER1;
       Bitmap cover2(kingCoverB2[kp]);
       cover2.And(board.pawn_bits[Black]);
       cover += cover2.bit_count()*KING_COVER2;
    }
    cover = Util::Min(0,cover);
    return cover;
}

// Calculate a "cover" score (pawn cover around the king, for
// king safety calculation).
//
static void calc_cover(const Board &board,Square kp,ColorType side,
const Pawn_Entry &pawn_entry,Search::King_Cover_Entry &cover)
{
    cover.cover = 0;
    int kpfile = File(kp);
    int k_cover = 0, q_cover = 0;
    int kprank = Rank(kp,side);
    int king_out = 0;
    if (kprank > 1)
      king_out = KING_OFF_BACK_RANK;
    if (kprank > 2)
      king_out *= 2;
    cover.cover += king_out;
    if (kpfile >= 4) {
       k_cover = calc_cover(board,kp,side);
       for (int i=6;i<=8;i++) {
         if (pawn_entry.FileOpen(i)) 
           k_cover += KING_NEAR_OPEN_FILE;
	 else if (pawn_entry.FileOpen(i,side))
	   k_cover += KING_NEAR_HALF_OPEN_FILE;
      }
    }
    if (kpfile <= 5) {
       q_cover = calc_cover(board,kp,side);
       for (int i=1;i<=3;i++) {
         if (pawn_entry.FileOpen(i)) 
           q_cover += KING_NEAR_OPEN_FILE;
	 else if (pawn_entry.FileOpen(i,side))
	   q_cover += KING_NEAR_HALF_OPEN_FILE;
      }
    } 
    if (kpfile>=6)
      cover.cover = k_cover;
    else if (kpfile<=3)
      cover.cover = q_cover;
    else {
      cover.cover = (k_cover + q_cover)/3;
      if (pawn_entry.FileOpen(DFILE))
        cover.cover += KING_NEAR_OPEN_FILE;
      if (pawn_entry.FileOpen(EFILE))
        cover.cover += KING_NEAR_OPEN_FILE;
    }

    // check for enemy pawns near the king
    cover.pawn_proximity = 0;
    if (side == White) {
        Bitmap patcks(w_pawn_attacks[kp]);
        patcks.And(board.pawn_bits[Black]);
        if (!patcks.is_clear()) {
            Square sq;
            while (patcks.iterate(sq)) {
                ASSERT(Rank(sq,White)-Rank(kp,White)>=0);
                int x = KING_PAWN_PROXIMITY[Rank(sq,White)-Rank(kp,White)];
                if (TEST_MASK(Bearing::pawn_attacks[sq][White],board.pawn_bits[White])) {
		   x += x/2;
		}
                else if (board[sq+8]==whitePawn) {
                    x /= 2;
                }
#ifdef EVAL_DEBUG
		cout << "pawn prox (to W. king), sq " << sq << " =" << x << endl;
#endif
                cover.pawn_proximity -= x;
            }
        }
    }
    else {
        Bitmap patcks(b_pawn_attacks[kp]);
        patcks.And(board.pawn_bits[White]);
        if (!patcks.is_clear()) {
            Square sq;
            while (patcks.iterate(sq)) {
                ASSERT(Rank(sq,Black)-Rank(kp,Black)>=0);
                int x = KING_PAWN_PROXIMITY[Rank(sq,Black)-Rank(kp,Black)];
                if (TEST_MASK(Bearing::pawn_attacks[sq][Black],board.pawn_bits[Black])) {
		            x += x/2;
		}
                else if (board[sq-8]==blackPawn) {
                    x /= 2;
                }
#ifdef EVAL_DEBUG
		cout << "pawn prox (to B. king), sq " << sq << " =" << x << endl;
#endif
                cover.pawn_proximity -= x;
            }
        }
    }
}


int Scoring::evalu8( const Board &board, Search *search) {
    int endgame = board.getMaterial(White).material_level() +
        board.getMaterial(Black).material_level() <= 6;
    if (endgame) {
        int score;
        if (is_draw(board)) {
#ifdef EVAL_DEBUG
            cout << "draw score" << endl;
#endif
            return 0;
        }
        else if ((score = tryBitbase(board))!= Scoring::INVALID_SCORE) {
#ifdef EVAL_DEBUG
            cout << "bitbase score=" << score << endl;
#endif
            return score;
        }
    }
#ifdef EVAL_DEBUG
    int mat_score = MaterialScore(board,board.Side());
    cout << "material score = " << mat_score << endl;
    return mat_score +
        positional_score(board,search,-Constants::BIG,Constants::BIG);
#else
    return MaterialScore(board,board.Side()) +
        positional_score(board,search,-Constants::BIG,Constants::BIG);
#endif
}


int Scoring::is_draw(const Board &board)
{
    return (repetition_draw(board) || material_draw(board) ||
        is_theoretical_draw(board));
}


int Scoring::is_legal_draw(const Board &board)
{
    return (repetition_draw(board) || material_draw(board));
}


int Scoring::is_theoretical_draw(const Board &board)
{
    const Material &mat1 = board.getMaterial(White);
    const Material &mat2 = board.getMaterial(Black);
    if (mat1.value() > KING_VALUE+(KNIGHT_VALUE*2) ||
        mat2.value() > KING_VALUE+(KNIGHT_VALUE*2))
        return 0 /*FALSE*/;
    // Check for K + P vs rook pawn
    if ((unsigned)mat1.infobits() == Material::KP &&
    mat2.king_only()) {
        if (!Bitmap::And(rook_pawn_mask,
        board.pawn_bits[White]).is_clear()) {
            Square kp = board.KingPos(White);
            Square kp2 = board.KingPos(Black);
            Square psq = (board.pawn_bits[White].first_one());
            return lookupBitbase(kp,psq,kp2,White,board.Side())==0;
        }
    }
    else if ((unsigned)mat2.infobits() == Material::KP &&
        mat1.king_only()) {
        if (!Bitmap::And(rook_pawn_mask,
        board.pawn_bits[Black]).is_clear()) {
            Square kp = board.KingPos(Black);
            Square kp2 = board.KingPos(White);
            Square psq = (board.pawn_bits[Black].first_one());
            return lookupBitbase(kp,psq,kp2,Black,board.Side())==0;
        }
    }
    // Check for wrong bishop + rook pawn vs king.  Not very common but
    // we don't want to get it wrong.
    else if (mat1.infobits() == Material::KBP && mat2.king_only()) {
        return KBPDrawW(board);
    }
    else if (mat2.infobits() == Material::KBP && mat1.king_only()) {
        return KBPDrawB(board);
    }
    // check for KNN vs K
    if (((unsigned)mat1.infobits() == Material::KNN &&
        mat2.king_only()) ||
        ((unsigned)mat2.infobits() == Material::KNN &&
        mat1.king_only()))
        return 1 /*TRUE*/;

    return 0 /*FALSE*/;
}


#ifdef DEBUG_PAWN_RACE
static void check_white_win(const Board &board, const char *n)
{
    int score2;
    if (probe_tb(board, score2, 0)) {
        if (board.Side() == White) {
            if (score2 < Constants::BIG-100)
                cout << "error (" << n << "): " << board << endl;
        }
        else {
            if (score2 > -Constants::BIG+100)
                cout << "error (" << n << "): " << board << endl;
        }
    }
}


static void check_black_win(const Board &board, const char *n)
{
    int score2;
    if (probe_tb(board, score2, 0)) {
        if (board.Side() == White) {
            if (score2 > -Constants::BIG+100)
                cout << "error (" << n << "): " << board << endl;
        }
        else {
            if (score2 < Constants::BIG-100)
                cout << "error (" << n << "): " << board << endl;
        }
    }
}
#endif

// Evaluate passed pawn races, from the standpoint of the White side
static int score_pawn_race(const Board &board,Pawn_Entry *entr)
{
    int score = 0;
    Square black_passer_race_sq = entr->b_pawn_entry.max_passer_rank_sq;
    Square white_passer_race_sq = entr->w_pawn_entry.max_passer_rank_sq;
    // remove positions where King is in check or pawn can be captured
    if (board.Side() == White && Bearing::pawn_attacks[white_passer_race_sq][Black].is_set(black_passer_race_sq)) {
        return 0;
    }
    if (board.Side() == Black && Bearing::pawn_attacks[black_passer_race_sq][White].is_set(white_passer_race_sq)) {
        return 0;
    }
    if (board.Side() == Black && Bearing::pawn_attacks[white_passer_race_sq][Black].is_set(board.KingPos(Black))) {
        return 0;
    }
    if (board.Side() == White && Bearing::pawn_attacks[black_passer_race_sq][White].is_set(board.KingPos(White))) {
        return 0;
    }
    int Wpassed = Bitmap::And(board.pawn_bits[Black],passedW[white_passer_race_sq]).is_clear();
    int Bpassed = Bitmap::And(board.pawn_bits[White],passedB[black_passer_race_sq]).is_clear();
    if (!Bpassed || !Wpassed) return 0;
    int black_passer_race_rank = entr->b_pawn_entry.max_passer_rank;
    int white_passer_race_rank = entr->w_pawn_entry.max_passer_rank;
    int uw = Scoring::uncatchable(board.Side(),white_passer_race_sq,
        board.KingPos(White),board.KingPos(Black),White);
    int ub = Scoring::uncatchable(board.Side(),black_passer_race_sq,
        board.KingPos(Black),board.KingPos(White),Black);
    int uw2,ub2;
    if (board.Side()==White) {
        uw2 = uw && sq_king_wtm[board.KingPos(Black)].is_set(white_passer_race_sq);
    }
    else {
        uw2 = uw && sq_king_btm[board.KingPos(Black)].is_set(white_passer_race_sq);
    }
    if (board.Side()==Black) {
        ub2 = ub && sq_king_wtm[Flip[board.KingPos(White)]].is_set(Flip[black_passer_race_sq]);
    }
    else {
        ub2 = ub && sq_king_btm[Flip[board.KingPos(White)]].is_set(Flip[black_passer_race_sq]);
    }
    // Check for one side has uncatchable pawn, other pawn is blocked
    if (uw2 && Wpassed && aheadB[black_passer_race_sq].is_set(board.KingPos(White))) {
        score += UNCATCHABLE_PAWN;                // White win
#ifdef DEBUG_PAWN_RACE
        check_white_win(board,"1");
#endif
        return score;
    }
    else if (ub2 && Bpassed && aheadW[white_passer_race_sq].is_set(board.KingPos(Black))) {
        score -= UNCATCHABLE_PAWN;                // Black win
#ifdef DEBUG_PAWN_RACE
        check_black_win(board,"2");
#endif
        return score;
    }
    // evaluate pawn race
    // Decrement rank of passer if own King obstructs it (but not if
    // King is on 7th or 8th)
    if (File(board.KingPos(Black))==File(black_passer_race_sq) &&
        Rank(board.KingPos(Black),Black) > black_passer_race_rank &&
        Rank(board.KingPos(Black),Black)<7)
        --black_passer_race_rank;
    if (File(board.KingPos(White))==File(white_passer_race_sq) &&
        Rank(board.KingPos(White),White) > white_passer_race_rank &&
        Rank(board.KingPos(White),White)<7)
        --white_passer_race_rank;
    Square white_queen_square = Square(File(white_passer_race_sq),8,White);
    Square black_queen_square = Square(File(black_passer_race_sq),8,Black);
    if (ub2) {
        int diff = black_passer_race_rank+(board.Side()==Black)-white_passer_race_rank;
        if (diff >= 3) {
#ifdef DEBUG_PAWN_RACE
            check_black_win(board,"3a");
#endif
            return -UNCATCHABLE_PAWN;
        }
        else if (!Bearing::king_attacks[white_queen_square].is_set(board.KingPos(White))) {
#ifdef DEBUG_PAWN_RACE
            check_black_win(board,"3b");
#endif
            return -UNCATCHABLE_PAWN;
        }
    }
    else if (uw2) {
        int diff = white_passer_race_rank+(board.Side()==White)-black_passer_race_rank;
        if (diff >= 3) {
#ifdef DEBUG_PAWN_RACE
            check_white_win(board,"4a");
#endif
            return UNCATCHABLE_PAWN;
        }
        else if (!Bearing::king_attacks[black_queen_square].is_set(board.KingPos(Black))) {
#ifdef DEBUG_PAWN_RACE
            check_white_win(board,"4b");
#endif
            return UNCATCHABLE_PAWN;
        }
    }
    else if (white_passer_race_rank == black_passer_race_rank) {
        if (board.Side() == White) {
            // White will queen first
            // See if White will check the black king or
            // if the White queen will stop the Black
            // passer.
            if ((board.Clear(white_queen_square,
                board.KingPos(Black)) && distance1(board.KingPos(Black),white_queen_square)>1) ||
                (board.Clear(black_queen_square,
                white_queen_square) &&
            distance1(board.KingPos(Black),black_queen_square)>1)) {
                if (uw) {
                    score += UNCATCHABLE_PAWN;
#ifdef DEBUG_PAWN_RACE
                    check_white_win(board,"5");
#endif
                }
                else
                    return 0;
            }
            // See if Black will queen with check
            else if (board.Clear(black_queen_square,
            board.KingPos(White))) {
                if (ub && distance1(board.KingPos(White),black_queen_square)>1) {
                    score -= UNCATCHABLE_PAWN;
                    // There are still some rare cases where this is a draw
#ifdef DEBUG_PAWN_RACE
                    check_black_win(board,"6");
#endif
                }
                else
                    return 0;
            }
        }
        else {
            // Black will queen first.
            // See if Black will check the White king or
            // if the Black queen will stop the White
            // passer.
            if ((board.Clear(black_queen_square,
                board.KingPos(White)) && distance1(board.KingPos(White),black_queen_square)>1) ||
                (board.Clear(white_queen_square,
                black_queen_square) &&
            distance1(board.KingPos(White),white_queen_square)>1)) {
                if (ub) {
                    score -= UNCATCHABLE_PAWN;
#ifdef DEBUG_PAWN_RACE
                    check_black_win(board,"7");
#endif
                }
                else
                    return 0;
            }
            // See if White will queen with check
            else if (board.Clear(white_queen_square,
            board.KingPos(Black))) {
                if (uw2 && distance1(board.KingPos(Black),white_queen_square)>1) {
                    score += UNCATCHABLE_PAWN;
                    // There are still some rare cases where this is a draw
#ifdef DEBUG_PAWN_RACE
                    check_white_win(board,"8");
#endif
                }
                else
                    return 0;
            }
        }
    }
    return score;
}


static int score_endgame(const Board &board, ColorType side, Pawn_Entry::PawnData *hash_entry, Search::King_Pawn_Entry *kp_hash_entry)
{
    const ColorType oside = OppositeColor(side);
    const Square kp = board.KingPos(side);
    const Square oppkp = board.KingPos(oside);
    int score = 0;
    const Material &my_mat = board.getMaterial(side);
    const Material &opp_mat = board.getMaterial(OppositeColor(side));
    if (board.Side() != side) {
        int x = opposition[kp][oppkp];
        if (x > 0) {
            score += OPPOSITION*x;
#ifdef EVAL_DEBUG
            cout << ColorImage(side) << " has opposition, score=" << OPPOSITION*x << endl;
#endif
        }
    }
    if ((my_mat.infobits() == Material::KBN) &&
    opp_mat.king_only()) {
        // KBNK endgame, special case.
        const int opprank = Rank(oppkp,White);
        const int oppfile = File(oppkp);
        Bitmap bishops = board.bishop_bits[side];
        Square sq = bishops.first_one();
        // try to force king to a corner where mate is possible.
        Square evalsq;
        if (SquareColor(sq) == White) {
            evalsq = Square(9-opprank, oppfile, White);
        }
        else {
            evalsq = oppkp;
        }
        score += KBNKScores[evalsq];
        // keep the kings close
        score += 10-distance1(board.KingPos(White),board.KingPos(Black));
        return score;
    }
    else if ((my_mat.king_only() && opp_mat.pawn_count() > 0 &&
        (opp_mat.infobits() == Material::KR ||
        opp_mat.infobits() == Material::KQ)) ||
        (my_mat.infobits() == Material::KQ &&
        (opp_mat.infobits() == Material::KN ||
        opp_mat.infobits() == Material::KB))) {
        // bare K vs. R and/or Q, or Q vs 1 minor piece
        // "side" has the material advantage.
        // Encourage moving the opposing king to the edge.
        score += 2*(10-KingCenterScores[board.KingPos(oside)]);
        // Keep the kings close.
        score += 15-3*distance1(board.KingPos(White),board.KingPos(Black));
        return score;
    }
    else if ((my_mat.infobits() == Material::KP) &&
             opp_mat.king_only()) {
        // KPK endgame
        Square passer = hash_entry->passers[0];
        int cant_catch = (side == White) ?
            (int)kp_hash_entry->w_uncatchable :
        (int)kp_hash_entry->b_uncatchable;
        if (cant_catch) {
            // The opposing king can't catch the pawn before
            // it queens
            score += (UNCATCHABLE_PAWN*Rank(passer,side))/7;
        }
        else {
            // This should be a draw with correct play, so
            // bring the score nearer zero:
            score -= 50;
            int rank = Rank(passer,side);
            int file_dist = Util::Abs(File(passer) - File(kp));
            int rank_dist = Rank(kp,side)-rank;
            // strongly encourage putting the king ahead of the
            // pawn
            if (rank_dist > 0 && file_dist <= 1)
                score += KING_AHEAD_OF_PAWN;
            else if (rank_dist<=0)                // king behind pawn
                score += 8-rank*2;
            //also encourage staying near pawn
            score += DISTANCE_FROM_PAWN*distance1(kp,passer);
        }
        return score;
    }

#ifdef EVAL_DEBUG
    if (my_mat.pawn_count() || opp_mat.pawn_count()) {
      int tmp = score;
      if (side == White) {
          score += (int)kp_hash_entry->white_king_position;
      }
      else {
          score += (int)kp_hash_entry->black_king_position;
      }
      cout << "king position score " << ColorImage(side) << " = " << score-tmp << endl;
    }
#else
    if (my_mat.pawn_count() || opp_mat.pawn_count()) {
      score += (side == White) ? (int)kp_hash_entry->white_king_position :
      (int)kp_hash_entry->black_king_position;
    }
#endif
    int uncatchables = 0;
    // King/Pawn interactions
    for (int i = 0; i < hash_entry->passers_count; i++) {
        Square passer = hash_entry->passers[i];
        int uncatchable = (side == White) ?
           (int)(kp_hash_entry->w_uncatchable & (byte)(1<<i)) :
           (int)(kp_hash_entry->b_uncatchable & (byte)(1<<i)) ;
        if (uncatchable) {
            ++uncatchables;
        } else {
          // Encourage escorting a passer with the King, ideally with King in
          // front
          if (Util::Abs(File(board.KingPos(side))-File(passer))<=1 && Rank(board.KingPos(side),side)>=Rank(passer,side)) score += KING_AHEAD_OF_PAWN/2;
          if (Util::Abs(File(board.KingPos(OppositeColor(side)))-File(passer))<=1 && Rank(board.KingPos(OppositeColor(side)),side)>=Rank(passer,side)) score -=  KING_AHEAD_OF_PAWN/2;
	}
    }
    if (uncatchables) {
       // give a large bonus for uncatchable pawns only if opponent has no
       // pieces, otherwise we are not sure to win:
       int bonus = (opp_mat.no_pieces()) ?
	 UNCATCHABLE_PAWN :
	 6*Util::Max(0,4-opp_mat.pieceCount());
#ifdef EVAL_DEBUG
        cout << ColorImage(side) << " uncatchables=" << uncatchables << endl;
#endif
        score += bonus*Util::Min(uncatchables,2);
#ifdef EVAL_DEBUG
        cout << ColorImage(side) << " uncatchable bonus=" << bonus*Util::Min(uncatchables,2) << endl;
#endif
    }
    // encourage staying near passed pawns
#ifdef EVAL_DEBUG
    int tmp = score;
#endif
    score += (side == White) ?
       kp_hash_entry->white_endgame_pawn_proximity :
       kp_hash_entry->black_endgame_pawn_proximity;
#ifdef EVAL_DEBUG
    cout << ColorImage(side) << 
       " endgame proximity bonus =" << score-tmp << endl;
#endif
    return score;
}

static int pieceScoreW(const Board &board,const Pawn_Entry &pawn_entry,
Search::King_Cover_Entry *cover, GamePhase phase)
{
    const Pawn_Entry::PawnData *hash_entry_w = &pawn_entry.w_pawn_entry;
    const Pawn_Entry::PawnData *hash_entry_b = &pawn_entry.b_pawn_entry;
    int score = 0, kattack=0;
    int haveWbishop = 0 /*FALSE*/;
    int haveBbishop = 0 /*FALSE*/;
    Bitmap pieces(board.occupied[White]);
    pieces.andNot(board.pawn_bits[White]);
    pieces.clear(board.KingPos(White));
    Square sq;
    const Square okp = board.KingPos(Black);
    const int kfile = File(okp);
    Bitmap rook_attacks(king_file_proximity[kfile-1]);
    int atks=0, natks=0, qatks=0, major_attacks=0, minor_attacks = 0;
    int cover1 = cover->cover;
    int proximity = 0;
    if (kfile <= DFILE) {
        if (pawn_entry.FileOpen(AFILE)) rook_attacks.Or(Bearing::file_mask[AFILE-1]);
    }
    if (kfile >= EFILE) {
        if (pawn_entry.FileOpen(HFILE)) rook_attacks.Or(Bearing::file_mask[HFILE-1]);
    }
    proximity = cover->pawn_proximity;
#ifdef EVAL_DEBUG
    cout << "pawn attacks on king (White): " << proximity << endl;
#endif

    while (pieces.iterate(sq)) {
        Piece p = board[sq];
#ifdef EVAL_DEBUG
        int tmp;
#endif
        switch(TypeOfPiece(p)) {
            case Knight:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                int center = KnightScoresW[sq];
                score += center;
                // Compute knight mobility.  We award a bonus for
                // each square to which the knight can move, that
                // is not occupied or attacked by an enemy pawn.
                Bitmap ka(Bearing::knight_attacks[sq]);
                ka.andNot(board.occupied[White]);
                ka.andNot(hash_entry_w->opponent_pawn_attacks);
                ASSERT(ka.bit_count()<=8);
                score += KNIGHT_MOBILITY[ka.bit_count()];
                // evaluate a knight outpost
                if (OutpostScoresW[sq] &&
                Bitmap::And(outpostW[sq],board.pawn_bits[Black]).is_clear()) {
		    int outpost = 0;
                    if (hash_entry_b->opponent_pawn_attacks.is_set(sq)) {
                                                  // defended by pawns
                        outpost += OutpostScoresW[sq];
                    }
                    if (phase >= ENDGAME_TRANSITION) outpost /= 2;
#ifdef EVAL_DEBUG
		    if (outpost) cout << "outpost(" << SquareImage(sq) << "): " << outpost << endl;
#endif
                    score += outpost;
                }
#ifdef EVAL_DEBUG
                cout << "White Knight on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                if (king_knight_proximity[okp].is_set(sq)) {
                    atks += KING_PIECE_ATTACKS[Knight];
                    natks++; minor_attacks++;
                }
            }
            break;
            case Rook:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                int r = 0;
                if (rook_attacks.is_set(sq)) {
                    int sq2 = sq;
                    const int incr = (Rank(sq,White)<Rank(okp,White)) ? -8 : 8;
                    while (OnBoard(sq2+incr)) {
                        sq2 += incr;
                        if (board[sq2] == whiteRook || board[sq2] == whiteQueen) {
                            continue;
                        }
                        int d;
                        if (incr>0) {
                            if ((d=distance1(okp,sq2))<=2) {
                                atks += KING_PIECE_ATTACKS[Rook];
                                r++;
                                break;
                            }
                        }
                        else if ((d=distance1(okp,sq2))<=2 && !hash_entry_b->blocked.is_set(sq2)) {
                            if (board[sq2]!=whitePawn) {
                                atks += KING_PIECE_ATTACKS[Rook];
                                r++;
                            }
                            break;
                        }
                        if (board[sq2] != EmptyPiece) {
                            break;
                        }
                    }
                }
                if (king_rank_proximity[Rank(okp,Black)-1].is_set(sq)) {
                    if (distance1(sq,okp)<=2) {
                        atks += KING_PIECE_ATTACKS[Rook];
                        r++;
                    }
                    else if (File(okp)>File(sq)) {
                        if (board.Clear(sq,Square(Util::Max(1,File(okp)-1),Rank(sq,White),White))) {
                            atks += KING_PIECE_ATTACKS[Rook];
                            r++;
                        }
                    }
                    else {
                        if (board.Clear(sq,Square(Util::Min(8,File(okp)+1),Rank(sq,White),White))) {
                            atks += KING_PIECE_ATTACKS[Rook];
                            r++;
                        }
                    }
                }
                natks += (r>0);
		major_attacks += (r>0);
                const int file = File(sq);
                int mobl = 0;
                Square sq2 = sq;
                while (mobl<2 && File(sq2--)!=1) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && File(sq2++)!=8) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
		sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=8) {
		    sq2 += 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
		sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=1) {
		    sq2 -= 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                if (mobl<2) {
                    score += LOW_ROOK_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                    cout << "low rook mobility(w): " << LOW_ROOK_MOBILITY[mobl] << endl;
#endif
                }
                if (board.getMaterial(Black).material_level() >= 4 && pawn_entry.FileOpen(file,White) &&
                    (pawn_entry.FileOpen(file,Black) ||
                (hash_entry_b->weak_files & (1<<(file-1))))) {
                    score += ROOK_ON_OPEN_FILE;

                    if (phase < ENDGAME_TRANSITION && !Bitmap::And(board.rook_bits[White],Bearing::file_mask_up[sq]).is_clear()) {
                        if (!Bitmap::And(FileAttacksUp(board,sq),board.rook_bits[White]).is_clear()) {
                            score += DOUBLED_ROOKS;
                        }
                    }
                }
                int rank = Rank(sq,White);
                if (rank == 7 && (Rank(okp,Black)==1 || hash_entry_b->pawns_on_2nd_rank)) {
                    score += ROOK_ON_7TH_RANK;
                }
#ifdef EVAL_DEBUG
                cout << "White Rook on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            }
            break;
            case Bishop:
            {
#ifdef EVAL_DEBUG
                int tmp = score;
#endif
                switch (phase) {
                    case OPENING:
		    case MIDDLEGAME:
                        score += BishopPositionW[sq]; break;
                    case ENDGAME_TRANSITION:
                        score += BishopPositionW[sq]/2; break;
                    case ENDGAME:
                        break;
                }
                if (SquareColor(sq) == White) haveWbishop++; else haveBbishop++;
                Bitmap mobility(0);
                int mobl = 0;
                int i = sq;
                int k_proximity = 0;
                for (;i>=0;i-=7) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i != sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != whiteQueen) {
                                if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i) == 8) break;
                }
                i = sq;
                for (;i>=0;i-=9) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i != sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != whiteQueen) {
                                if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i) == 1) break;
                }
                i = sq;
                for (;i<=63;i+=9) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i!=sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != whiteQueen) {
                                if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i) == 8) break;
                }
                i = sq;
                for (;i<=63;i+=7) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i!=sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != whiteQueen) {
                                if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i)==1) break;
                }

                if (k_proximity) {
                    atks += KING_PIECE_ATTACKS[Bishop];
                    natks++; minor_attacks++;
                }
                mobl -= Bitmap::And(mobility,pawn_entry.w_pawn_entry.opponent_pawn_attacks).bit_count();
                ASSERT(mobl>=0);
                score += BISHOP_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                cout << "B mobility (" << SquareImage(sq) << ")=" <<
                    BISHOP_MOBILITY[mobl] << endl;
#endif
#ifdef EVAL_DEBUG
                cout << "White Bishop on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                break;
            }
            case Queen:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                if (phase<ENDGAME_TRANSITION) score += QueenCenterScoresW[sq];
		if (phase == OPENING && Rank(sq,White)>3) {
		  Bitmap back(board.knight_bits[White] | board.bishop_bits[White] &
			      Bearing::rank_mask[7]);
#ifdef EVAL_DEBUG
		  int tmp = score;
#endif
		  score += QUEEN_OUT*back.bit_count();
#ifdef EVAL_DEBUG
		  if (score != tmp) cout << "queen out (Black): " << score-tmp << endl;
#endif
		}
                if (Rank(sq,White)<=Rank(okp,White)) {
                    const int *data = BishopSquares[sq];
                    while (*data != 255) {
                        if (king_prox[okp].is_set((Square)*data)) {
                            qatks++; break;
                        }
                        else if (board[(Square)*data] != EmptyPiece) break;
                        data++;
                    }
                    data = BishopSquares[sq]+8;
                    while (*data != 255) {
                        if (king_prox[okp].is_set((Square)*data)) {
                            qatks++;
                            break;
                        }
                        else if (board[(Square)*data] != EmptyPiece) break;
                        data++;
                    }
                }
                else {
                    const int *data = BishopSquares[sq]+16;
                    while (*data != 255) {
                        if (king_prox[okp].is_set((Square)*data)) {
                            qatks++; break;
                        }
                        else if (board[(Square)*data] != EmptyPiece) break;
                        data++;
                    }
                    data = BishopSquares[sq]+24;
                    while (*data != 255) {
                        if (king_prox[okp].is_set((Square)*data)) {
                            qatks++;
                            break;
                        }
                        else if (board[(Square)*data] != EmptyPiece) break;
                        data++;
                    }
                }
                if (rook_attacks.is_set(sq)) {
                    int sq2 = sq;
                    const int incr = (Rank(sq,White)<Rank(okp,White)) ? -8 : 8;
                    while (OnBoard(sq2+incr)) {
                        sq2 += incr;
                        if (board[sq2] == whiteRook || board[sq2] == whiteQueen) {
                            continue;
                        }
                        if (incr > 0) {
                            if (distance1(okp,sq2)<=1) {
                                qatks++;
                                break;
                            }
                        }
                        else if (distance1(okp,sq2)<=2 && !hash_entry_b->blocked.is_set(sq2)) {
                            if (board[sq2] != whitePawn) {
                                qatks++;
                            }
                            break;
                        }
                        else if (board[sq2] != EmptyPiece) {
                            break;
                        }
                    }
                }
                if (king_rank_proximity[Rank(okp,Black)-1].is_set(sq) &&
                distance1(sq,okp)>2) {
                    if (File(okp)>File(sq)) {
                        int sq2 = sq;
                        while (File(sq2) != 8) {
                            sq2++;
                            if (board[sq2]==whiteRook || board[sq2]==whiteQueen)
                                continue;
                            if (distance1(sq2,okp)<=2) {
                                qatks++;
                                break;
                            }
                            else if (board[sq2] != EmptyPiece) {
                                break;
                            }
                        }
                    }
                    else {
                        int sq2 = sq;
                        while (File(sq2) != 1) {
                            sq2--;
                            if (board[sq2]==whiteRook || board[sq2]==whiteQueen)
                                continue;
                            if (distance1(sq2,okp)<=2) {
                                qatks++;
                                break;
                            }
                            else if (board[sq2] != EmptyPiece) {
                                break;
                            }
                        }
                    }

                }
                natks += (qatks>0);
                if (qatks) atks += KING_PIECE_ATTACKS[Queen];
            }
#ifdef EVAL_DEBUG
            cout << "White Queen on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            break;
            default: break;
        }
    }
    if (haveWbishop && !haveBbishop) {
        int whitePawns = hash_entry_w->w_square_pawns;
        int blackPawns = hash_entry_w->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (whitePawns > blackPawns+2)) {
#ifdef EVAL_DEBUG
            cout << "bad bishop (White) : " << BAD_BISHOP << endl;
#endif
            score += BAD_BISHOP*(whitePawns-blackPawns-2);
        }
    }
    else if (haveBbishop && !haveWbishop) {
        int whitePawns = hash_entry_w->w_square_pawns;
        int blackPawns = hash_entry_w->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (blackPawns > whitePawns+2)) {
#ifdef EVAL_DEBUG
            cout << "bad bishop (White) : " << BAD_BISHOP << endl;
#endif
            score += BAD_BISHOP*(blackPawns-whitePawns-2);
        }
    }
#ifdef EVAL_DEBUG
    cout << "piece score (White) : " << score << endl;
    int tmpScore = score;
#endif

    int k = king_file_danger[File(board.KingPos(Black))-1] & (pawn_entry.w_pawn_entry.open_files & pawn_entry.b_pawn_entry.open_files);
    while (k) {
        int f = first_ones_8bit[k];
        if (TEST_MASK(board.rook_bits[Black],Bearing::file_mask[f])) {
#ifdef EVAL_DEBUG
            cout << "friendly Rook (W): " << FRIENDLY_ROOK << endl;
#endif
            cover1 += FRIENDLY_ROOK;
        }
        k &= ~(1<<f);
    }
    if (minor_attacks && (major_attacks || qatks)) atks++;
    if (major_attacks && qatks) atks += 2;
    if (proximity/16 && (major_attacks || qatks)) atks++;
    if (atks>=20) atks=19;
#ifdef EVAL_DEBUG
    cout << "king cover (Black) : " << cover1 << endl;
    cout << "king attack count (Black) : " << atks << endl;
    cout << "proximity (Black) : " << proximity << endl;
#endif
    int attack = (KING_ATTACK_FACTOR*KING_ATTACK_MULTIPLIER*atks)/100 + 
      (KING_PROXIMITY_FACTOR*proximity)/100;
    cover1 = (KING_COVER_FACTOR*cover1)/100;
    int boost = 0;
#ifdef BOOST
    if (cover1 < -KING_ATTACK_BOOST_THRESHOLD) {
       boost = Util::Min(KING_ATTACK_MAX_BOOST,-attack*(cover1+KING_ATTACK_BOOST_THRESHOLD)/96);
    }
#endif
    kattack = attack - cover1 + boost;
#ifdef EVAL_DEBUG
    if (boost) cout << "king attack boost (Black) : " << boost << endl;
    cout << "pre-scaled king attack score (Black) : " << kattack << endl;
#endif
    kattack = (kattack*KING_ATTACK_MATERIAL_SCALE[board.getMaterial(White).material_level()])/32;
#ifdef EVAL_DEBUG
    cout << "scaled king attack score (Black) : " << kattack << endl;
#endif
    score += kattack;

    return score;
}


static int pieceScoreWEndgame(const Board &board,const Pawn_Entry &pawn_entry,
Search::King_Cover_Entry *cover)
{
    const Pawn_Entry::PawnData *hash_entry_w = &pawn_entry.w_pawn_entry;
    const Pawn_Entry::PawnData *hash_entry_b = &pawn_entry.b_pawn_entry;
    int score = 0;
    int haveWbishop = 0 /*FALSE*/;
    int haveBbishop = 0 /*FALSE*/;
    Bitmap pieces(board.occupied[White]);
    pieces.andNot(board.pawn_bits[White]);
    pieces.clear(board.KingPos(White));
    Square sq;

    while (pieces.iterate(sq)) {
        Piece p = board[sq];
#ifdef EVAL_DEBUG
        int tmp;
#endif
        switch(TypeOfPiece(p)) {
            case Knight:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                int center = KnightScoresW[sq];
                if (board.getMaterial(White).material_level() < 6) center /= 2;
                score += center;
                // Compute knight mobility.  We award a bonus for
                // each square to which the knight can move, that
                // is not occupied or attacked by an enemy pawn.
                Bitmap ka(Bearing::knight_attacks[sq]);
                ka.andNot(board.occupied[White]);
                ka.andNot(hash_entry_w->opponent_pawn_attacks);
                ASSERT(ka.bit_count()<=8);
                score += KNIGHT_MOBILITY[ka.bit_count()];

#ifdef EVAL_DEBUG
                cout << "White Knight on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            }
            break;
            case Rook:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                const int file = File(sq);
                int mobl = 0;
                Square sq2 = sq;
                while (mobl<2 && File(sq2--)!=1) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && File(sq2++)!=8) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=8) {
		    sq2 += 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=1) {
		    sq2 -= 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_w->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                if (mobl<2) {
                    score += LOW_ROOK_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                    cout << "low rook mobility(w): " << LOW_ROOK_MOBILITY[mobl] << endl;
#endif
                }
                if (board.getMaterial(Black).material_level() >= 4 && pawn_entry.FileOpen(file,White) &&
                    (pawn_entry.FileOpen(file,Black) ||
                (hash_entry_b->weak_files & (1<<(file-1))))) {
                    score += ROOK_ON_OPEN_FILE;

                }
                Square okp = board.KingPos(board.OppositeSide());
                if (Rank(sq,White) == 7 && ( Rank(okp,Black)==1 || hash_entry_b->pawns_on_2nd_rank)) {
                    score += ROOK_ON_7TH_RANK;
                }
#ifdef EVAL_DEBUG
                cout << "White Rook on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            }
            break;
            case Bishop:
            {
#ifdef EVAL_DEBUG
                int tmp = score;
#endif
                if (SquareColor(sq) == White) haveWbishop++; else haveBbishop++;
                Bitmap mobility(0);
                int mobl = 0;
                const int *data = BishopSquares[sq];
                for (;*data != 255;data++) {
                    Square i = *data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                data = BishopSquares[sq]+8;
                for (;*data != 255;data++) {
                    Square i = *data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                data = BishopSquares[sq]+16;
                for (;*data != 255;data++) {
                    Square i = *data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                data = BishopSquares[sq]+24;
                for (;*data != 255;data++) {
                    Square i = *data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==Black && board[i] != blackPawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                mobl -= Bitmap::And(mobility,pawn_entry.w_pawn_entry.opponent_pawn_attacks).bit_count();
                ASSERT(mobl>=0);
                score += BISHOP_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                cout << "B mobility (" << SquareImage(sq) << ")=" <<
                    BISHOP_MOBILITY[mobl] << endl;
#endif
#ifdef EVAL_DEBUG
                cout << "White Bishop on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                break;
            }
            case Queen:
            {
                break;
            }
            break;
            default: break;
        }
    }
    if (haveWbishop && !haveBbishop) {
        int whitePawns = hash_entry_w->w_square_pawns;
        int blackPawns = hash_entry_w->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (whitePawns > blackPawns+2)) {
#ifdef EVAL_DEBUG
            cout << "bad bishop (White) : " << BAD_BISHOP << endl;
#endif
            score += BAD_BISHOP*(whitePawns-blackPawns-2);
        }
    }
    else if (haveBbishop && !haveWbishop) {
        int whitePawns = hash_entry_w->w_square_pawns;
        int blackPawns = hash_entry_w->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (blackPawns > whitePawns+2)) {
#ifdef EVAL_DEBUG
            cout << "bad bishop (White) : " << BAD_BISHOP << endl;
#endif
            score += BAD_BISHOP*(blackPawns-whitePawns-2);
        }
    }
#ifdef EVAL_DEBUG
    cout << "piece score (White) : " << score << endl;
#endif
    return score;
}


static int pieceScoreB(const Board &board, const Pawn_Entry &pawn_entry,
		       Search::King_Cover_Entry * cover, GamePhase phase)
{
    const Pawn_Entry::PawnData *hash_entry_w = &pawn_entry.w_pawn_entry;
    const Pawn_Entry::PawnData *hash_entry_b = &pawn_entry.b_pawn_entry;
    int score = 0, kattack=0;
    int haveWbishop = 0 /*FALSE*/;
    int haveBbishop = 0 /*FALSE*/;
    Bitmap pieces(board.occupied[Black]);
    pieces.andNot(board.pawn_bits[Black]);
    pieces.clear(board.KingPos(Black));
    Square sq;
    const Square okp = board.KingPos(White);
    const int kfile = File(okp);
    Bitmap rook_attacks(king_file_proximity[kfile-1]);
    int atks=0, natks=0, qatks=0, major_attacks=0, minor_attacks = 0;
    int proximity = cover->pawn_proximity;
    int cover1 = cover->cover;
    if (kfile <= DFILE) {
        if (pawn_entry.FileOpen(AFILE)) rook_attacks.Or(Bearing::file_mask[AFILE-1]);
    }
    if (kfile >= EFILE) {
        if (pawn_entry.FileOpen(HFILE)) rook_attacks.Or(Bearing::file_mask[HFILE-1]);
    }
#ifdef EVAL_DEBUG
    cout << "pawn attacks on king (Black): " << proximity << endl;
#endif
    while (pieces.iterate(sq)) {
#ifdef EVAL_DEBUG
        int tmp;
#endif
        Piece p = board[sq];
        switch(TypeOfPiece(p)) {
            case Knight:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                int center = KnightScoresB[sq];
                score += center;
                // Compute knight mobility.  We award a bonus for
                // each square to which the knight can move, that
                // is not occupied or attacked by an enemy pawn.
                Bitmap ka(Bearing::knight_attacks[sq]);
                ka.andNot(board.occupied[Black]);
                ka.andNot(hash_entry_b->opponent_pawn_attacks);
                ASSERT(ka.bit_count()<=8);
                score += KNIGHT_MOBILITY[ka.bit_count()];
                // evaluate a knight outpost
                if (OutpostScoresB[sq] && 
                Bitmap::And(outpostB[sq],board.pawn_bits[White]).is_clear()) {
		    int outpost = 0;
                    if (hash_entry_w->opponent_pawn_attacks.is_set(sq)) {
                                                  // defended by pawns
                        outpost += OutpostScoresB[sq];
                    }
                    if (phase >= ENDGAME_TRANSITION) outpost /= 2;
#ifdef EVAL_DEBUG
		    if (outpost) cout << "outpost(" << SquareImage(sq) << "): " << outpost << endl;
#endif
                    score += outpost;
                }
#ifdef EVAL_DEBUG
                cout << "Black Knight on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                if (king_knight_proximity[okp].is_set(sq)) {
                    atks += KING_PIECE_ATTACKS[Knight];
                    natks++; minor_attacks++;
                }
            }
            break;
            case Rook:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                int r = 0;
                if (rook_attacks.is_set(sq)) {
                    int sq2 = sq;
                    const int incr = (Rank(sq,Black)<Rank(okp,Black)) ? 8 : -8;
                    while (OnBoard(sq2 + incr)) {
                        sq2 += incr;
                        if (board[sq2] == blackRook || board[sq2] == blackQueen) {
                            continue;
                        }
                        if (incr<0) {
                            if (distance1(okp,sq2)<=2) {
                                atks += KING_PIECE_ATTACKS[Rook];
                                r++;
                                break;
                            }
                        }
                        else if (distance1(okp,sq2)<=2 && !hash_entry_w->blocked.is_set(sq2)) {
                            if (board[sq2] != blackPawn) {
                                atks += KING_PIECE_ATTACKS[Rook];
                                r++;
                            }
                            break;
                        }
                        if (board[sq2] != EmptyPiece) {
                            break;
                        }
                    }
                }
                if (king_rank_proximity[Rank(okp,Black)-1].is_set(sq)) {
                    if (distance1(sq,okp)<=2) {
                        atks += KING_PIECE_ATTACKS[Rook];
                        r++;
                    }
                    else if (File(okp)>File(sq)) {
                        if (board.Clear(sq,Square(Util::Max(1,File(okp)-1),Rank(sq,White),White))) {
                            atks += KING_PIECE_ATTACKS[Rook];
                            r++;
                        }
                    }
                    else {
                        if (board.Clear(sq,Square(Util::Min(8,File(okp)+1),Rank(sq,White),White))) {
                            atks += KING_PIECE_ATTACKS[Rook];
                            r++;
                        }
                    }
                }
                natks += (r>0);
		major_attacks += (r>0);
                const int file = File(sq);
                int mobl = 0;
                Square sq2 = sq;
                while (mobl<2 && File(sq2--)!=1) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && File(sq2++)!=8) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=8) {
		  sq2 += 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=1) {
  		    sq2 -= 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                if (mobl<2) {
                    score += LOW_ROOK_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                    cout << "low rook mobility(b): " << LOW_ROOK_MOBILITY[mobl] << endl;
#endif
                }
                if (board.getMaterial(White).material_level() >= 4 && pawn_entry.FileOpen(file,Black) &&
                    (pawn_entry.FileOpen(file,White) ||
                (hash_entry_w->weak_files & (1<<(file-1))))) {
                    score += ROOK_ON_OPEN_FILE;

                    if (phase < ENDGAME_TRANSITION && !Bitmap::And(board.rook_bits[Black],Bearing::file_mask_down[sq]).is_clear()) {
                        if (!Bitmap::And(FileAttacksDown(board,sq),board.rook_bits[Black]).is_clear()) {
                            score += DOUBLED_ROOKS;
                        }
                    }
                }
                int rank = Rank(sq,Black);
                if (rank == 7 && ( Rank(okp,White)==1 || hash_entry_w->pawns_on_2nd_rank)) {
                    score += ROOK_ON_7TH_RANK;
                }
            }
#ifdef EVAL_DEBUG
            cout << "Black Rook on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            break;
            case Bishop:
            {
#ifdef EVAL_DEBUG
                int tmp = score;
#endif
                switch (phase) {
                    case OPENING:
		    case MIDDLEGAME:
                        score += BishopPositionB[sq]; break;
                    case ENDGAME_TRANSITION:
                        score += BishopPositionB[sq]/2; break;
                    case ENDGAME:
                        break;
                }
                if (SquareColor(sq) == White) haveWbishop++; else haveBbishop++;
                Bitmap mobility(0);
                int mobl = 0;
                int i = sq;
                int k_proximity = 0;
                for (;i<=63;i+=9) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i!=sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != blackQueen) {
                                if (PieceColor(board[i])==White && board[i] != whitePawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i) == 8) break;
                }
                i = sq;
                for (;i<=63;i+=7) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i!=sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != blackQueen) {
                                if (PieceColor(board[i])==White && board[i] != whitePawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i)==1) break;
                }
                i = sq;
                for (;i>=0;i-=7) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i != sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != blackQueen) {
                                if (PieceColor(board[i])==White && board[i] != whitePawn) {
                                    mobility.set(i);
                                    ++mobl;
                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i) == 8) break;
                }
                i = sq;
                for (;i>=0;i-=9) {
                    if (king_prox[okp].is_set(i)) k_proximity++;
                    if (i != sq) {
                        if (board[i] != EmptyPiece) {
                            if (board[i] != blackQueen) {
                                if (PieceColor(board[i])==White && board[i] != whitePawn) {
                                    mobility.set(i);
                                    ++mobl;

                                }
                                break;
                            }
                        }
                        mobility.set(i);
                        ++mobl;
                    }
                    if (File(i) == 1) break;
                }
                if (k_proximity) {
                    atks += KING_PIECE_ATTACKS[Bishop];
		    minor_attacks++;
                    natks++;
                }
                mobl -= Bitmap::And(mobility,pawn_entry.b_pawn_entry.opponent_pawn_attacks).bit_count();
                ASSERT(mobl>=0);
                score += BISHOP_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                cout << "B mobility (" << SquareImage(sq) << ")=" <<
                    BISHOP_MOBILITY[mobl] << endl;
#endif
#ifdef EVAL_DEBUG
                cout << "Black Bishop on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                break;
            }
            case Queen:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                if (phase<ENDGAME_TRANSITION) score += QueenCenterScoresB[sq];
		if (phase == OPENING && Rank(sq,Black)>3) {
		  Bitmap back(board.knight_bits[Black] | board.bishop_bits[Black] &
			      Bearing::rank_mask[0]);
#ifdef EVAL_DEBUG
		  int tmp = score;
#endif
		  score += QUEEN_OUT*back.bit_count();
#ifdef EVAL_DEBUG
		  if (score != tmp) cout << "queen out (Black): " << score-tmp << endl;
#endif
		}
                if (Rank(sq,Black)<=Rank(okp,Black)) {
                    const int *data = BishopSquares[sq]+16;
                    while (*data != 255) {
                        if (king_prox[okp].is_set((Square)*data)) {
                            qatks++;
                            break;
                        }
                        else if (board[(Square)*data] != EmptyPiece) break;
                        data++;
                    }
                    data = BishopSquares[sq]+24;
                    while (*data != 255) {
                        Square i = (Square)*data;
                        if (king_prox[okp].is_set(i)) {
                            qatks++;
                            break;
                        }
                        else if (board[i] != EmptyPiece) break;
                        data++;
                    }
                }
                else {
                    const int *data = BishopSquares[sq];
                    while (*data != 255) {
                        if (king_prox[okp].is_set((Square)*data)) {
                            qatks++;
                            break;
                        }
                        else if (board[(Square)*data] != EmptyPiece) break;
                        data++;
                    }
                    data = BishopSquares[sq]+8;
                    while (*data != 255) {
                        Square i = (Square)*data;
                        if (king_prox[okp].is_set(i)) {
                            qatks++;
                            break;
                        }
                        else if (board[i] != EmptyPiece) break;
                        data++;
                    }
                }
                if (rook_attacks.is_set(sq)) {
                    int sq2 = sq;
                    const int incr = (Rank(sq,Black)<Rank(okp,Black)) ? 8 : -8;
                    while (OnBoard(sq2+incr)) {
                        sq2 += incr;
                        if (board[sq2] == blackRook || board[sq2] == blackQueen) {
                            continue;
                        }
                        int d;
                        if (incr<0) {
                            if (distance1(okp,sq2)<=2) {
                                qatks++;
                                break;
                            }
                        }
                        else if ((d=distance1(okp,sq2))<=2 && !hash_entry_w->blocked.is_set(sq2)) {
                            if (board[sq2]!=blackPawn) {
                                qatks++;
                            }
                            break;
                        }
                        if (board[sq2] != EmptyPiece) {
                            break;
                        }
                    }
                }
                if (king_rank_proximity[Rank(okp,Black)-1].is_set(sq) &&
                distance1(sq,okp)>2) {
                    if (File(okp)>File(sq)) {
                        int sq2 = sq;
                        while (File(sq2) != 8) {
                            sq2++;
                            if (board[sq2]==blackRook || board[sq2]==blackQueen)
                                continue;
                            if (distance1(sq2,okp)<=2) {
                                qatks++;
                                break;
                            }
                            else if (board[sq2] != EmptyPiece) {
                                break;
                            }
                        }
                    }
                    else {
                        int sq2 = sq;
                        while (File(sq2) != 1) {
                            sq2--;
                            if (board[sq2]==blackRook || board[sq2]==blackQueen)
                                continue;
                            if (distance1(sq2,okp)<=2) {
                                qatks++;
                                break;
                            }
                            else if (board[sq2] != EmptyPiece) {
                                break;
                            }
                        }
                    }

                }
                natks += (qatks>0);
                if (qatks) atks += KING_PIECE_ATTACKS[Queen];
#ifdef EVAL_DEBUG
                cout << "Black Queen on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                break;
            }
            default: break;
        }
    }
    if (haveWbishop && !haveBbishop) {
        int whitePawns = hash_entry_b->w_square_pawns;
        int blackPawns = hash_entry_b->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (whitePawns > blackPawns + 2)) {
            score += BAD_BISHOP*(whitePawns-blackPawns-2);
#ifdef EVAL_DEBUG
            cout << "bad bishop (Black) : " << BAD_BISHOP << endl;
#endif
        }
    }
    else if (haveBbishop && !haveWbishop) {
        int whitePawns = hash_entry_b->w_square_pawns;
        int blackPawns = hash_entry_b->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (blackPawns > whitePawns + 2)) {
            score += BAD_BISHOP*(blackPawns-whitePawns-2);
#ifdef EVAL_DEBUG
            cout << "bad bishop (Black) : " << BAD_BISHOP << endl;
#endif
        }
    }
#ifdef EVAL_DEBUG
    cout << "piece score (Black) : " << score << endl;
    int tmpScore = score;
#endif
    byte k = king_file_danger[File(board.KingPos(White))-1] & (pawn_entry.w_pawn_entry.open_files & pawn_entry.b_pawn_entry.open_files);
    while (k) {
        int f = first_ones_8bit[k];
        if (TEST_MASK(board.rook_bits[White],Bearing::file_mask[f])) {
#ifdef EVAL_DEBUG
            cout << "friendly Rook (W): " << FRIENDLY_ROOK << endl;
#endif
            cover1 += FRIENDLY_ROOK;
        }
        k &= ~(1<<f);
    }
    if (minor_attacks && (major_attacks || qatks)) atks++;
    if (major_attacks && qatks) atks += 2;
    if (proximity/16 && (major_attacks || qatks)) atks++;
#ifdef EVAL_DEBUG
    cout << "king cover (White) : " << cover1 << endl;
    cout << "king attack count (White) : " << atks << endl;
    cout << "proximity (White) : " << proximity << endl;
#endif

    if (atks>=20) atks=19;
    int attack = (KING_ATTACK_FACTOR*KING_ATTACK_MULTIPLIER*atks)/100 + 
      (KING_PROXIMITY_FACTOR*proximity)/100;
    cover1 = (KING_COVER_FACTOR*cover1)/100;
    int boost = 0;
#ifdef BOOST
    if (cover1 < -KING_ATTACK_BOOST_THRESHOLD) {
       boost = Util::Min(KING_ATTACK_MAX_BOOST,-attack*(cover1+KING_ATTACK_BOOST_THRESHOLD)/96);
    }
#endif
    kattack = attack - cover1 + boost;
#ifdef EVAL_DEBUG
    if (boost) cout << "king attack boost (White) : " << boost << endl;
    cout << "pre-scaled king attack score (White) : " << kattack << endl;
#endif
    kattack = (kattack*KING_ATTACK_MATERIAL_SCALE[board.getMaterial(Black).material_level()])/32;
#ifdef EVAL_DEBUG
    cout << "scaled king attack score (White) : " << kattack << endl;
#endif
    score += kattack;
    return score;
}


static int pieceScoreBEndgame(const Board &board, const Pawn_Entry &pawn_entry,
Search::King_Cover_Entry * cover)
{
    const Pawn_Entry::PawnData *hash_entry_w = &pawn_entry.w_pawn_entry;
    const Pawn_Entry::PawnData *hash_entry_b = &pawn_entry.b_pawn_entry;
    int score = 0;
    int haveWbishop = 0 /*FALSE*/;
    int haveBbishop = 0 /*FALSE*/;
    Bitmap pieces(board.occupied[Black]);
    pieces.andNot(board.pawn_bits[Black]);
    pieces.clear(board.KingPos(Black));
    Square sq;

    while (pieces.iterate(sq)) {
#ifdef EVAL_DEBUG
        int tmp;
#endif
        Piece p = board[sq];
        switch(TypeOfPiece(p)) {
            case Knight:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                int center = KnightScoresB[sq];
                if (board.getMaterial(Black).material_level() < 6) center /= 2;
                score += center;
                // Compute knight mobility.  We award a bonus for
                // each square to which the knight can move, that
                // is not occupied or attacked by an enemy pawn.
                Bitmap ka(Bearing::knight_attacks[sq]);
                ka.andNot(board.occupied[Black]);
                ka.andNot(hash_entry_b->opponent_pawn_attacks);
                ASSERT(ka.bit_count()<=8);
                score += KNIGHT_MOBILITY[ka.bit_count()];
#ifdef EVAL_DEBUG
                cout << "Black Knight on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            }
            break;
            case Rook:
            {
#ifdef EVAL_DEBUG
                tmp = score;
#endif
                const int file = File(sq);
                int mobl = 0;
                Square sq2 = sq;
                while (mobl<2 && File(sq2--)!=1) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                sq2 = sq;
                while (mobl<2 && File(sq2++)!=8) {
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
		sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=8) {
		    sq2 += 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
		sq2 = sq;
                while (mobl<2 && Rank(sq2,Black)!=1) {
		    sq2 -= 8;
                    if (board[sq2] != EmptyPiece) break;
                    if (!hash_entry_b->opponent_pawn_attacks.is_set(sq2)) {
                        mobl++;
                    }
                }
                if (mobl<2) {
                    score += LOW_ROOK_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                    cout << "low rook mobility(b): " << -LOW_ROOK_MOBILITY[mobl] << endl;
#endif
                }
                if (board.getMaterial(White).material_level() >= 4 && pawn_entry.FileOpen(file,Black) &&
                    (pawn_entry.FileOpen(file,White) ||
                (hash_entry_w->weak_files & (1<<(file-1))))) {
                    score += ROOK_ON_OPEN_FILE;
                }
                Square okp = board.KingPos(board.OppositeSide());
                if (Rank(sq,Black) == 7 && ( Rank(okp,White)==1 || hash_entry_w->pawns_on_2nd_rank)) {
                    score += ROOK_ON_7TH_RANK;
                }
            }
#ifdef EVAL_DEBUG
            cout << "Black Rook on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
            break;
            case Bishop:
            {
#ifdef EVAL_DEBUG
                int tmp = score;
#endif
                if (SquareColor(sq) == White) haveWbishop++; else haveBbishop++;
                Bitmap mobility(0);
                int mobl = 0;
                const int *data = BishopSquares[sq];
                for (;*data!=255;data++) {
                    Square i = (Square)*data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==White && board[i] != whitePawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                data = BishopSquares[sq]+8;
                for (;*data!=255;data++) {
                    Square i = (Square)*data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==White && board[i] != whitePawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                data = BishopSquares[sq]+16;
                for (;*data!=255;data++) {
                    Square i = (Square)*data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==White && board[i] != whitePawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                data = BishopSquares[sq]+24;
                for (;*data!=255;data++) {
                    Square i = (Square)*data;
                    if (board[i] != EmptyPiece) {
                        if (PieceColor(board[i])==White && board[i] != whitePawn) {
                            mobility.set(i);
                            ++mobl;
                        }
                        break;
                    }
                    mobility.set(i);
                    ++mobl;
                }
                mobl -= Bitmap::And(mobility,pawn_entry.b_pawn_entry.opponent_pawn_attacks).bit_count();
                ASSERT(mobl>=0);
                score += BISHOP_MOBILITY[mobl];
#ifdef EVAL_DEBUG
                cout << "B mobility (" << SquareImage(sq) << ")=" <<
                    BISHOP_MOBILITY[mobl] << endl;
#endif
#ifdef EVAL_DEBUG
                cout << "Black Bishop on " << SquareImage(sq) << " : " << score-tmp << endl;
#endif
                break;
            }
            case Queen:
            {
                break;
            }
            default: break;
        }
    }
    if (haveWbishop && !haveBbishop) {
        int whitePawns = hash_entry_b->w_square_pawns;
        int blackPawns = hash_entry_b->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (whitePawns > blackPawns + 2)) {
            score += BAD_BISHOP*(whitePawns-blackPawns-2);
#ifdef EVAL_DEBUG
            cout << "bad bishop (Black) : " << BAD_BISHOP << endl;
#endif
        }
    }
    else if (haveBbishop && !haveWbishop) {
        int whitePawns = hash_entry_b->w_square_pawns;
        int blackPawns = hash_entry_b->b_square_pawns;
        if ((whitePawns + blackPawns > 4) && (blackPawns > whitePawns + 2)) {
            score += BAD_BISHOP*(blackPawns-whitePawns-2);
#ifdef EVAL_DEBUG
            cout << "bad bishop (Black) : " << BAD_BISHOP << endl;
#endif
        }
    }
#ifdef EVAL_DEBUG
    cout << "piece score (Black) : " << score << endl;
#endif
    return score;
}

int Scoring::positional_score( const Board &board,
                               Search *search, int alpha, 
                               int beta, QSearchHashEntry *entry ) {
#ifdef DEBUG_HASH_SCORES
  int hashScore = INVALID_SCORE;
#endif
#ifdef EVAL_STATS
    position_hash_probes++;
#endif
    int use_lazy = 1;
    if (entry && entry->hash_code == board.HashCode() &&
         entry->score != INVALID_SCORE) {
#ifdef EVAL_STATS
        position_hash_hits++;
#endif
#ifdef DEBUG_HASH_SCORES
        hashScore = entry->score;
        use_lazy = 0;
#else
	//#ifdef _TRACE
	//        cout << " *score hash hit* ";
	//#endif
        return entry->score;
#endif
    }
    if (entry) {
      // always replace existing entry. 
      if (entry->hash_code != board.HashCode()) {
         entry->best = NullMove;
         entry->hash_code = board.HashCode();
      }
      entry->score = INVALID_SCORE; // may be set valid later
    }
    int score = 0;
    int w_material_level = board.getMaterial(White).material_level();
    int b_material_level = board.getMaterial(Black).material_level();
    int level = Util::Min(w_material_level,b_material_level);
    GamePhase phase;
    if (level < 13)
        phase = ENDGAME;
    else if (level < 16)
        phase = ENDGAME_TRANSITION;
    else if (level < 24)
        phase = MIDDLEGAME;
    else
        phase = OPENING;
#ifdef EVAL_DEBUG
    cout << "White material level = " << w_material_level << endl;
    cout << "Black material level = " << b_material_level << endl;
    static const char *names[4] = {"Opening","Middlegame","Endgame Transition","Endgame"};
    cout << "phase = " << names[phase] << endl;;
#endif

    // Query the pawn hash table to see if we have found a position
    // before that has the same pawn structure.  If so, we can skip
    // some of the scoring calculations.
    Pawn_Entry *pawn_hash_probe =
        &search->pawn_hash_table[quickmod(board.pawn_hash(),Search::PAWN_HASH_SIZE)];
#ifdef EVAL_STATS
    pawn_hash_probes++;
#endif
    if (pawn_hash_probe->hc != board.pawn_hash()) {
        // pawn position not found, calculate the data we need
        pawn_structure_score(board,White,pawn_hash_probe->w_pawn_entry);
        pawn_structure_score(board,Black,pawn_hash_probe->b_pawn_entry);
        eval_outside_pp(board,pawn_hash_probe);
        pawn_hash_probe->hc = board.pawn_hash();
    }
    else {
#ifdef EVAL_STATS
        pawn_hash_hits++;
#endif
#ifdef DEBUG_HASH
        cout << "pawn hash hit, score(w) = " << pawn_hash_entry->w_pawn_entry.score <<
            ", score(b)= " << pawn_hash_entry->b_pawn_entry.score << endl;
#endif
    }
    Pawn_Entry &pawn_hash_entry =
        search->pawn_hash_table[quickmod(board.pawn_hash(),Search::PAWN_HASH_SIZE)];

    ASSERT(pawn_hash_entry.w_pawn_entry.passers_count <= board.getMaterial(White).pawn_count());
    ASSERT(pawn_hash_entry.b_pawn_entry.passers_count <= board.getMaterial(Black).pawn_count());

    Pawn_Entry::PawnData *hash_entry_w = &pawn_hash_entry.w_pawn_entry;
    Pawn_Entry::PawnData *hash_entry_b = &pawn_hash_entry.b_pawn_entry;
#ifdef EVAL_DEBUG
    cout << "----" << endl << "white pawn score = " << hash_entry_w->score << endl;
    cout << "black pawn score = " << hash_entry_b->score << endl;
#endif
    score += hash_entry_w->score - hash_entry_b->score;
    if (phase < ENDGAME_TRANSITION) {
        score += hash_entry_w->center_score - hash_entry_b->center_score;
    }
#ifdef EVAL_DEBUG
    int tmp = score;
#endif
    if (!board.rooksqueens[White].is_clear())
        score -= (int)hash_entry_b->weakopen*WEAK_ON_OPEN_FILE;
    if (!board.rooksqueens[Black].is_clear())
        score += (int)hash_entry_w->weakopen*WEAK_ON_OPEN_FILE;
#ifdef EVAL_DEBUG
    cout << "weak pawns (open files): " << score-tmp << endl;
    tmp = score;
#endif
    // outside passed pawn scoring, based on cached pawn data
    if (pawn_hash_entry.w_pawn_entry.outside && !pawn_hash_entry.b_pawn_entry.outside) {
        score += OUTSIDE_PP[ b_material_level/2];
    }
    else if (pawn_hash_entry.b_pawn_entry.outside && !pawn_hash_entry.w_pawn_entry.outside) {
        score -= OUTSIDE_PP[w_material_level/2];
    }
#ifdef EVAL_DEBUG
    cout << "outside pawn score = " << score-tmp << endl;
    tmp = score;
#endif
    if (board[A7] == whiteBishop) {
        if (board[B6] == blackPawn &&
        board.any_attacks(B6,Black)) {
            score += BISHOP_TRAPPED;
        }
    }
    if (board[H7] == whiteBishop) {
        if (board[G6] == blackPawn &&
        board.any_attacks(G6,Black)) {
            score += BISHOP_TRAPPED;
        }
    }
    if (board[A2] == blackBishop) {
        if (board[B3] == whitePawn &&
        board.any_attacks(B3,White)) {
            score -= BISHOP_TRAPPED;
        }
    }
    if (board[H2] == blackBishop) {
        if (board[G3] == whitePawn &&
        board.any_attacks(G3,White)) {
            score -= BISHOP_TRAPPED;
        }
    }
#ifdef EVAL_DEBUG
    if (tmp-score!=0)
        cout << "bishop trapped = " << score-tmp << endl;
    tmp = score;
#endif
    if (phase < ENDGAME_TRANSITION) {
        score += CASTLING_SCORES[(int)board.CastleStatus(White)] -
            CASTLING_SCORES[(int)board.CastleStatus(Black)];
    }
    else if (phase == ENDGAME_TRANSITION) {
        score += (CASTLING_SCORES[(int)board.CastleStatus(White)] -
            CASTLING_SCORES[(int)board.CastleStatus(Black)])/2;
    }
#ifdef EVAL_DEBUG
    cout << "castling score= " << score-tmp << endl;
#endif
    // passed pawn scoring
    int endgame_score = 0;
    int i;
    ASSERT(hash_entry_w->passers_count<9);
    static const int blockers[8] = {0,0,1,1,0,0,1,0};
    // score passed pawns less if large material.
    for (i = 0; i < hash_entry_w->passers_count; i++) {
        Square sq = hash_entry_w->passers[i];
        ASSERT(OnBoard(sq));
        int rank = Rank(sq,White);
		ASSERT(rank<8);
        int pp_block = 0;
        Square sq2 = sq-8;
        if (!IsEmptyPiece(board[sq2]) && PieceColor(board[sq2]) == Black) {
#ifdef PAWN_DEBUG
            int tmp = score;
#endif
            if (blockers[TypeOfPiece(board[sq2])])
                score -= PASSED_PAWN_BLOCK[rank-1];
            else
                score -= PASSED_PAWN_BLOCK[rank-1]/2;
#ifdef PAWN_DEBUG
            cout << "White pawn on " << SquareImage(sq) << " blocked, score=" << score-tmp << endl;
#endif
            ++pp_block;
        }
        if (!board.rooksqueens[White].is_clear()) {
            if (!Bitmap::And(FileAttacksDown(board,sq),board.rooksqueens[White]).is_clear()) {
                int bonus = ROOK_BEHIND_PP;
                if (pp_block) bonus /= 2;
                endgame_score += bonus;
            }
            // give penalty if rook is ahead of a passer
            if (!Bitmap::And(Bearing::file_mask_up[sq],board.rook_bits[White]).is_clear()) {
                endgame_score += ROOK_AHEAD_OF_PP;
#ifdef EVAL_DEBUG
                cout << "White rook ahead of passer: " << ROOK_AHEAD_OF_PP << endl;
#endif
            }
            // also give penalty if protecting a passer with a rook from the side.
            if (!Bitmap::And(Bearing::rank_mask[Rank(sq,Black)-1],board.rook_bits[White]).is_clear() &&
            !Bitmap::And(RankAttacks(board,sq),board.rook_bits[White]).is_clear()) {
                endgame_score += ROOK_AHEAD_OF_PP;
#ifdef EVAL_DEBUG
                cout << "White rook side-protecting passer: " << ROOK_AHEAD_OF_PP << endl;
#endif
            }
        }
    }
    ASSERT(hash_entry_b->passers_count<9);
    for (i = 0; i < hash_entry_b->passers_count; i++) {
        Square sq = hash_entry_b->passers[i];
        ASSERT(OnBoard(sq));
        int rank = Rank(sq,Black);
	ASSERT(rank<8);
        int pp_block = 0;
        Square sq2 = sq+8;
        if (!IsEmptyPiece(board[sq2]) && PieceColor(board[sq2]) == White) {
#ifdef PAWN_DEBUG
            int tmp = score;
#endif
            if (blockers[TypeOfPiece(board[sq2])])
                score += PASSED_PAWN_BLOCK[rank-1];
            else
                score += PASSED_PAWN_BLOCK[rank-1]/2;
#ifdef PAWN_DEBUG
            cout << "Black pawn on " << SquareImage(sq) << " blocked, score=" << tmp-score << endl;
#endif
            pp_block++;
        }
        if (!board.rooksqueens[Black].is_clear()) {
            if (!Bitmap::And(FileAttacksUp(board,sq),board.rooksqueens[Black]).is_clear()) {
                int bonus = ROOK_BEHIND_PP;
                if (pp_block) bonus /= 2;
                endgame_score -= bonus;
            }
            // give penalty if rook is ahead of a passer
            if (!Bitmap::And(Bearing::file_mask_down[sq],board.rook_bits[Black]).is_clear()) {
                endgame_score -= ROOK_AHEAD_OF_PP;
#ifdef EVAL_DEBUG
                cout << "Black rook ahead of passer: " << -ROOK_AHEAD_OF_PP << endl;
#endif
            }
            // also give penalty if protecting a passer with a rook from the side.
            if (!Bitmap::And(Bearing::rank_mask[rank-1],board.rook_bits[Black]).is_clear() &&
            !Bitmap::And(RankAttacks(board,sq),board.rook_bits[Black]).is_clear()) {
                endgame_score -= ROOK_AHEAD_OF_PP;
#ifdef EVAL_DEBUG
                cout << "Black rook side-protecing passer: " << -ROOK_AHEAD_OF_PP << endl;
#endif
            }
        }
    }
    Search::King_Cover_Entry *w_cover = NULL;
    Search::King_Cover_Entry *b_cover = NULL;
    if (phase != ENDGAME) {
        hash_t kp_hash = board.pawnHashCodeW;
        kp_hash ^= board.pawnHashCodeB;
        kp_hash ^= hash_codes[board.KingPos(White)][whiteKing];
        w_cover = &search->king_cover_hash_table[quickmod(kp_hash,Search::KING_COVER_HASH_SIZE)];
#ifdef EVAL_STATS
        king_cover_hash_probes++;
#endif
        if (w_cover->hc != kp_hash) {
            calc_cover(board,board.KingPos(White),White,pawn_hash_entry,*w_cover);
            w_cover->hc = kp_hash;
        }
        else {
#ifdef EVAL_STATS
            king_cover_hash_hits++;
#endif
#ifdef DEBUG_HASH
            cout << "pawn cover hash hit (w), cover=" << w_cover->cover << endl;
#endif
        }
        kp_hash = board.pawnHashCodeB;
        kp_hash ^= board.pawnHashCodeW;
        kp_hash ^= hash_codes[board.KingPos(Black)][blackKing];
        b_cover = &search->king_cover_hash_table[quickmod(kp_hash,Search::KING_COVER_HASH_SIZE)];
#ifdef EVAL_STATS
        king_cover_hash_probes++;
#endif
        if (b_cover->hc != kp_hash) {
            calc_cover(board,board.KingPos(Black),Black,pawn_hash_entry,*b_cover);
            b_cover->hc = kp_hash;
        }
        else {
#ifdef EVAL_STATS
            king_cover_hash_hits++;
#endif
#ifdef DEBUG_HASH
            cout << "pawn cover hash hit (b), cover=" << b_cover->cover << endl;
#endif
        }
    }
#if defined(USE_LAZY_EVAL) || defined(LAZY_DEBUG)
    if (use_lazy) {
    // lazy cutoff
#ifdef EVAL_STATS
    lazy_cut_chances++;
#endif
    int lazy_score = INVALID_SCORE;
    int est_score = score;
    int margin = LAZY_MARGIN;
    if (phase != ENDGAME) {
        if (board.Side() == White) {
            lazy_score = est_score;
        }
        else {
            lazy_score = -est_score;
        }
        if (Util::Max(w_material_level,b_material_level)<=16) margin -= margin/4;
#ifndef LAZY_DEBUG
        if (lazy_score >= beta+margin ||
        lazy_score <= alpha-margin) {
#ifdef EVAL_STATS
            lazy_cuts++;
#endif
            return lazy_score;
        }
#endif
    }
    }
#endif
    if (phase == ENDGAME) {
        score += pieceScoreWEndgame(board,pawn_hash_entry,b_cover)-
                 pieceScoreBEndgame(board,pawn_hash_entry,w_cover);
    }
    else
        score += pieceScoreW(board,pawn_hash_entry,b_cover,phase)-
                 pieceScoreB(board,pawn_hash_entry,w_cover,phase);

    if (phase < ENDGAME_TRANSITION) {
        score = (score/4)*4;
        score = (board.Side() == White) ? score : -score;
#ifdef DEBUG_HASH_SCORES
        if (entry && entry->hash_code == board.HashCode() &&
            hashScore != INVALID_SCORE && hashScore != score) {
	  cout << "realScore = " << score << " hashScore=" << hashScore << endl;
        }
#endif
        if (entry) {
           entry->score = score;
	}
#ifdef LAZY_DEBUG
        if (lazy_score != INVALID_SCORE) {
#ifdef EVAL_STATS
            if (lazy_score + margin <= alpha) lazy_cuts++;
            else if (lazy_score - margin >= beta) lazy_cuts++;
#endif
            if (lazy_score + margin <= alpha && score_entry.score > alpha) {
                cout << "alpha error(1):" << board << " score = " << score_entry.score << " lazy score=" << lazy_score << endl;
                cout << "king safety(w) = " << ks_w << " king safety(b) = " << ks_b << endl;
                cout << "w cover = " << wc << " b cover = " << bc << endl;
            } else if (lazy_score - margin >= beta &&
                        score_entry.score < beta) {
                cout << "beta error(1): " << board << " score = " << score_entry.score << " lazy score=" << lazy_score << endl;
                cout << "king safety(w) = " << ks_w << " king safety(b) = " << ks_b << endl;
                cout << "w cover = " << wc << " b cover = " << bc << endl;
            }
        }
#endif
        return score;
    }
    const hash_t kp_hash = BoardHash::kingPawnHash(board);
    Search::King_Pawn_Entry &kp_hash_entry = search->king_pawn_hash_table[quickmod(kp_hash,Search::KING_PAWN_HASH_SIZE)];
#ifdef EVAL_STATS
    king_pawn_hash_probes++;
#endif
    if (kp_hash_entry.hc != kp_hash) {
        kp_hash_entry.hc = kp_hash;
        calc_endgame(board,hash_entry_w,hash_entry_b,&kp_hash_entry);
#ifdef EVAL_DEBUG
        cout << "endgame score = " << kp_hash_entry.score << endl;
#endif
    }
    else {
#ifdef EVAL_STATS
        king_pawn_hash_hits++;
#endif
#ifdef DEBUG_HASH
        cout << "endgame hash hit, score=" << kp_hash_entry.score << endl;
#endif
    }
    score += kp_hash_entry.score;
#if defined(LAZY_DEBUG) || defined(USE_LAZY_EVAL)
    if (use_lazy) {
#ifdef LAZY_DEBUG
    int endgame_lazy = INVALID_SCORE;
#endif
    // lazy cutoff
    if (w_material_level+b_material_level>0 && Util::Abs(hash_entry_w->passers_count-hash_entry_b->passers_count)<=1) {
        int opp_mat,lazy_score;
        if (board.Side() == White) {
            lazy_score = score; opp_mat = b_material_level;
        }
        else {
            lazy_score = -score; opp_mat = w_material_level;
        }
        if ((lazy_score >= beta+ENDGAME_LAZY_MARGIN) ||
        (lazy_score <= alpha-ENDGAME_LAZY_MARGIN)) {
#ifdef LAZY_DEBUG
            endgame_lazy = lazy_score;
#else
#ifdef EVAL_STATS
            lazy_cuts++;
#endif
            return lazy_score;
#endif
        }
    }
    }
#endif
#ifdef EVAL_DEBUG
    int endw = score_endgame(board,White,hash_entry_w,&kp_hash_entry);
    int endb = score_endgame(board,Black,hash_entry_b,&kp_hash_entry);
    cout << "endgame score(w) = " << endw << endl;
    cout << "endgame score(b) = " << endb << endl;
    endgame_score += endw-endb;
#else
    endgame_score += score_endgame(board,White,hash_entry_w,&kp_hash_entry) -
        score_endgame(board,Black,hash_entry_b,&kp_hash_entry);
#endif
    int scale = w_material_level+b_material_level;
    int endgame_score_scaled = (endgame_score*(64-scale))/64;
#ifdef EVAL_DEBUG
    cout << "net endgame score = " << endgame_score << " (" <<
        endgame_score_scaled << " scaled)" << endl;
    tmp = score;
#endif
    score += endgame_score_scaled;
    int race_score = 0;
    // The following pawn race code is used only when each side has
    // passed pawns and at least one side has no more than one:
    if (board.getMaterial(White).no_pieces() && board.getMaterial(Black).no_pieces() &&
        ((hash_entry_b->passers_count==1 && hash_entry_w->passers_count>=1) ||
    (hash_entry_w->passers_count==1 && hash_entry_b->passers_count>=1))) {
        race_score = score_pawn_race(board,&pawn_hash_entry);
        // if White appears to be winning but Black has extra passers don't
        // count as a certain win, or vice-versa but do give a bonus.
        if (race_score >= PAWN_VALUE && hash_entry_w->passers_count<hash_entry_b->passers_count)
            race_score = UNCATCHABLE_PAWN3;
        else if (race_score <= -PAWN_VALUE && hash_entry_b->passers_count<hash_entry_w->passers_count)
            race_score = -UNCATCHABLE_PAWN3;
    }
    else if (board.getMaterial(White).no_pieces() && board.getMaterial(Black).no_pieces() &&
    (hash_entry_b->passers_count>=1 && hash_entry_w->passers_count>=1)) {
        // give bonus if pawns appear to be uncatchable
        int i;
        for (i = 0; i < hash_entry_b->passers_count; i++) {
            int ub = Scoring::uncatchable(board.Side(),hash_entry_b->passers[i],
                board.KingPos(Black),board.KingPos(White),Black);
            int ub2;
            if (board.Side()==Black) {
                ub2 = ub && sq_king_wtm[Flip[board.KingPos(White)]].is_set(Flip[hash_entry_b->passers[i]]);
            }
            else {
                ub2 = ub && sq_king_btm[Flip[board.KingPos(White)]].is_set(Flip[hash_entry_b->passers[i]]);
            }
            if (ub2) race_score -= UNCATCHABLE_PAWN3;
        }
        for (i = 0; i < hash_entry_w->passers_count; i++) {
            int uw = Scoring::uncatchable(board.Side(),hash_entry_w->passers[i],
                board.KingPos(White),board.KingPos(Black),White);
            int uw2;
            if (board.Side()==White) {
                uw2 = uw && sq_king_wtm[board.KingPos(Black)].is_set(hash_entry_w->passers[i]);
            }
            else {
                uw2 = uw && sq_king_btm[board.KingPos(Black)].is_set(hash_entry_w->passers[i]);
            }
            if (uw2) race_score += UNCATCHABLE_PAWN3;
        }
    }
    score += race_score;
    score = (score/2)*2;                          // round
#ifdef EVAL_DEBUG
    cout << "pawn race score = " << race_score << endl;
#endif
    int s = (board.Side() == White) ? score : -score;
#ifdef DEBUG_HASH_SCORES
    int realScore = s;
    if (entry) {
        entry->score = s;
    }
    if (entry && 
        hashScore != INVALID_SCORE &&
        realScore != hashScore) {
        cout << "realScore = " << realScore << " hashScore=" << hashScore << " endgame=" << endgame << endl;
    }
    return realScore;
#else
#if defined(LAZY_DEBUG)
    if (lazy_score != INVALID_SCORE) {
#ifdef EVAL_STATS
        if (lazy_score + margin <= alpha) lazy_cuts++;
        else if (lazy_score - margin >= beta) lazy_cuts++;
#endif
        if (lazy_score + margin <= alpha &&
        score_entry.score > alpha) {
            cout << "alpha error(2): " << board << " score = " << score_entry.score << endl;
        } else if (lazy_score - margin >= beta &&
        score_entry.score < beta) {
            cout << "beta error(2): " << board << " score = " << score_entry.score << endl;
        }
    }
    if (endgame_lazy != INVALID_SCORE) {
        int margin = ENDGAME_LAZY_MARGIN;
#ifdef EVAL_STATS
        if (endgame_lazy + margin <= alpha) lazy_cuts++;
        else if (endgame_lazy - margin >= beta) lazy_cuts++;
#endif
        if (endgame_lazy + margin <= alpha &&
        score_entry.score > alpha) {
            cout << "alpha error(3): " << board << " score = " << score_entry.score << endl;
        } else if (endgame_lazy - margin >= beta &&
        score_entry.score < beta) {
            cout << "beta error(3): " << board << " score = " << score_entry.score << endl;
        }
    }
#endif
    if (entry) {
        entry->score = s;
    }
    return s;
#endif
}


int Scoring::material_score( const Board &board )
// returns a material score
{
    return MaterialScore(board,board.Side());
}


int Scoring::repetition_draw( const Board &board )
{
    return board.rep_count() >= 2;
}


int Scoring::material_draw( const Board &board)
{
    // check for insufficient material per FIDE Rules of Chess
    const Material &mat1 = board.getMaterial(White);
    const Material &mat2 = board.getMaterial(Black);
    if (mat1.pawn_count() || mat2.pawn_count()) {
        return 0 /*FALSE*/;
    }
    if ((mat1.value() <= KING_VALUE + BISHOP_VALUE) &&
    (mat2.value() <= KING_VALUE + BISHOP_VALUE)) {
        if (mat1.king_only() || mat2.king_only())
            // K vs K, or K vs KN, or K vs KB
            return 1 /*TRUE*/;
        else if (mat1.infobits() == Material::KN &&
        mat2.infobits() == Material::KN) {
            // KN vs KN
	  return 0;
        }
        else {                                    /* KBKB */
            // drawn if same-color bishops
            if (TEST_MASK(board.bishop_bits[White],Bearing::black_squares)) 
	      return TEST_MASK(board.bishop_bits[Black],Bearing::black_squares);
            else if (TEST_MASK(board.bishop_bits[White],Bearing::white_squares))
	      return TEST_MASK(board.bishop_bits[Black],Bearing::white_squares);
        }
    }
    return 0;
}


void Scoring::print_score( int score, char *char_buf )
{
    if (score <= -Constants::BIG+200) {
        int movesToMate = (Constants::BIG+score+1)/2;
        sprintf(char_buf,"-Mate%d",movesToMate);
    }
    else if (score >= Constants::BIG-200) {
        int movesToMate = (Constants::BIG-score+1)/2;
        sprintf(char_buf,"+Mate%d",movesToMate);
    }
    else
        sprintf(char_buf,"%+6.2f",(1.0F*score)/64.0F);
}


void Scoring::print_score_uci(int score, char *score_buf)
{
    if (score <= -Constants::BIG+200) {
        int movesToMate = (Constants::BIG+score+1)/2;
        sprintf(score_buf,"mate -%d",movesToMate);
    }
    else if (score >= Constants::BIG-200) {
        int movesToMate = (Constants::BIG-score+1)/2;
        sprintf(score_buf,"mate %d",movesToMate);
    }
    else {
        sprintf(score_buf,"cp %d",(100*score)/PAWN_VALUE);
    }
}


int Scoring::tryBitbase(const Board &board)
{
    const Material &wMat = board.getMaterial(White);
    const Material &bMat = board.getMaterial(Black);
    int score = 0;
    if ((unsigned)wMat.infobits() == Material::KP &&
    bMat.king_only()) {
        Square passer = board.pawn_bits[White].first_one();
        Square kp = board.KingPos(White);
        if (lookupBitbase(board.KingPos(White),passer,
            board.KingPos(Black),White,board.Side()))
            score += UNCATCHABLE_PAWN;
        else
            return 0;
        int rank = Rank(passer,White);
        score += PASSED_PAWN[rank-1];
        int file_dist = Util::Abs(File(passer) - File(kp));
        int rank_dist = Rank(kp,White)-rank;
        // strongly encourage putting the king ahead of the
        // pawn
        if (rank_dist > 0 && file_dist <= 1)
            score += KING_AHEAD_OF_PAWN;
        else if (rank_dist<=0)                    // king behind pawn
            score += 8-rank*2;
        //also encourage staying near pawn
        score += DISTANCE_FROM_PAWN*distance1(kp,passer);
        return board.Side() == White ? score : -score;
    }
    else if ((unsigned)bMat.infobits() == Material::KP &&
    wMat.king_only()) {
        Square passer = board.pawn_bits[Black].first_one();
        Square kp = board.KingPos(Black);
        if (lookupBitbase(board.KingPos(Black),passer,
            board.KingPos(White),Black,board.Side()))
            score -= UNCATCHABLE_PAWN;
        else
            return 0;
        int rank = Rank(passer,Black);
        score -= PASSED_PAWN[rank-1];
        int file_dist = Util::Abs(File(passer) - File(kp));
        int rank_dist = Rank(kp,Black)-rank;
        // strongly encourage putting the king ahead of the
        // pawn
        if (rank_dist > 0 && file_dist <= 1)
            score -= KING_AHEAD_OF_PAWN;
        else if (rank_dist<=0)                    // king behind pawn
            score -= 8-rank*2;
        //also encourage staying near pawn
        score -= DISTANCE_FROM_PAWN*distance1(kp,passer);
        return board.Side() == White ? score : -score;
    }
    return INVALID_SCORE;
}
