// Copyright 1999, 2005, 2008 by Jon Dart.  All Rights Reserved.

#include "posnbook.h"
#include "globals.h"
#include "util.h"
#include "types.h"
#include <stdio.h>

#define NUM_ENTRIES 32768

LockDefine(posn_hash_lock);

struct PositionBookEntry {
  hash_t hashcode;
  int16 value;
  int16 depth;
  byte flags;
  signed char start, dest, promotion;
};

PositionBook::PositionBook()
: pBook(NULL),
  pPage(NULL),
  book_file(NULL),
  hFileMapping(NULL),
  open(0),newBook(0)
{                 
   learn_size = NUM_ENTRIES*sizeof(PositionBookEntry)+sizeof(Book_Header);
   int newBook = 0;
   string book_path(derivePath(POSITION_BOOK_NAME));
   book_file = bu.openFile(book_path.c_str(),false);
   if (book_file == NULL) {
       // create new learning file
       if (theLog) theLog->write("creating position.bin\n");
       book_file = bu.createFile(book_path.c_str(),learn_size
            );
         ++newBook;
   }
   LockInit(posn_hash_lock);
   init(book_path.c_str(),book_file);
   
}

void PositionBook::init(const char *book_path,FILE_HANDLE book_file) 
{
   if (book_file != NULL)
   {
      hFileMapping = bu.createMap(book_file,false);
   }
   if (hFileMapping)
   {
      pBook = bu.map(hFileMapping,0,learn_size,false);
      if (newBook)
      {
         *((char*)pBook) = Book_Version;
      }
   }

   open = pBook != NULL;
   if (!open)
      return;

   char *p = (char*)pBook;

   if (*p != (byte)Book_Version)
   {
      cout << "Creating new position.bin" << endl;
      bu.unmap(pBook,4096);
      bu.closeMap(hFileMapping);
      bu.closeFile(book_file);
      open = 0;
      pBook = NULL;
      book_file = bu.createFile(book_path,learn_size);
      ++newBook;
      init(book_path,book_file);
   }
}


PositionBook::~PositionBook()
{
   if (pBook)
      bu.unmap(pBook,4096);
   if (hFileMapping)
      bu.closeMap(hFileMapping);
   if (book_file)
      bu.closeFile(book_file);
   LockDestroy(posn_hash_lock);
}

void PositionBook::add(const Board &b,const Hash_Entry &info)
{
   PositionBookEntry pbi;
   pbi.value = (int16)info.value();
   pbi.depth = (int16)info.depth();
   pbi.flags = info.flags();
   pbi.start = (signed char)StartSquare(info.best_move(b));
   pbi.dest = (signed char)DestSquare(info.best_move(b));
   pbi.promotion = (signed char)PromoteTo(info.best_move(b));
   pbi.hashcode = info.hash_code();
   hash_t index = quickmod(info.hash_code(),NUM_ENTRIES);
   char *p = (char*)pBook + sizeof(Book_Header) + index*sizeof(PositionBookEntry);
   // TBD: portable byte order
   memcpy(p,&info,sizeof(PositionBookEntry));
}

int PositionBook::size() const
{
   return NUM_ENTRIES; // currently fixed
}
        
int PositionBook::lookup(const Board &b,hash_t hashCode,Position_Info &result) {
   hash_t index = quickmod(hashCode,NUM_ENTRIES);
   PositionBookEntry *p = 
    (PositionBookEntry*)((char*)pBook + sizeof(Book_Header) + index*sizeof(PositionBookEntry));
   Lock(posn_hash_lock);
   if (p->hashcode == hashCode) {
     Position_Info pi(p->value,p->depth,p->flags,
       CreateMove(b,(Square)p->start,(Square)p->dest,(PieceType)p->promotion));
     result = pi;
     Unlock(posn_hash_lock);
     return 1;
   }
   else {
     Unlock(posn_hash_lock);
     return 0;
   }
}
