// Copyright 1994, 1995, 1999, 2005 by Jon Dart.  All Rights Reserved.

#ifndef _PINFO_H
#define _PINFO_H

#include "board.h"
#include "chess.h"
#include <stddef.h>

extern const hash_t rep_codes[3];

class Hash_Entry;

class Position_Info {
    friend class Hash_Entry;
public:

    enum ValueType { Invalid, NoHit, Valid, UpperBound, LowerBound};

    enum FLAGS { FORCED_FLAG = 0x80, IN_CHECK_FLAG = 0x40,
                 FULL_FLAG = 0x20, AGE_FLAG = 0x18 };

    enum { DEPTH_MASK = 0x3ff, PROC_MASK = 0xfc00 };

    enum { PROC_INC = 0x0400 };

    Position_Info(int val = 0,int d = 0,
                  unsigned int flag = 0,Move best_move = NullMove) 
       :my_value(val),
        my_depth(d),
        my_flags((byte)flag),
        start(StartSquare(best_move)),
        dest(DestSquare(best_move)),
        promotion(PromoteTo(best_move)) {
    }

    ValueType type() const
    {
        return (ValueType)(my_flags & 0x7);
    }
    
    int value() const
    {
        return (int)my_value;
    }

    void setValue(int new_value) {
        my_value = (int16)new_value;
    }
    
    void copy(const Hash_Entry &source) {
      memcpy(&my_value,&((Position_Info*)&source)->my_value,sizeof(Position_Info));
    }

    int forced() const
    {
         return (my_flags & FORCED_FLAG) != 0;
    }
    
    int in_check() const
    {
        return (my_flags & IN_CHECK_FLAG) != 0;
    }

    int isFull() const
    {
        return (my_flags & FULL_FLAG) != 0;
    }

    int age() const
    {
        return (my_flags & AGE_FLAG) >> 3;
    }

    void setFull(int b) {
        if (b) my_flags |= (byte)FULL_FLAG;
        else my_flags &= (byte)~FULL_FLAG;
    }

    int depth() const {
        return (int)(my_depth & DEPTH_MASK);
    }

    byte flags() const {
        return my_flags;
    }

    Move best_move(const Board &b) const {
       if (start == InvalidSquare)
          return NullMove;
       else {
          Move m = CreateMove(b,(Square)start,(Square)dest,
            (PieceType)promotion);
          if (forced()) SetForced(m);
          return m;
       }
    }


private:
    int16 my_value;
    int16 my_depth; 
    byte my_flags;
    signed char start, dest, promotion;
};    

class Hash_Entry : public Position_Info {

    // this class represents one entry in the hash table.
    friend class Search;
    
    public:

    Hash_Entry( hash_t hashCode, const int depth,
                              byte age,
	           	      Position_Info::ValueType hashflags,
                              int in_check,
			      int value,
 		              int full,
			      Move best_move)
   :  
      Position_Info(value,
                    depth,
                    (IsForced(best_move)<<7) | (in_check<<6) | (age<<3) | (full<<5) | hashflags,
                    best_move),
      my_hashcode(hashCode)
   {}

   void update(const Position_Info &info, int age) {
       memcpy(&my_value,&info,sizeof(Position_Info));
       my_flags &= (uint16)~Position_Info::AGE_FLAG;
       my_flags |= (age<<3);
    }
    
    hash_t hash_code() const  {
        return my_hashcode;
    }

    void update_best_move(const Hash_Entry &h) {
       if (h.start != InvalidSquare) {
         start = h.start;
         dest = h.dest;
         promotion = h.promotion;
       }
    }

    private:

    hash_t my_hashcode;
};

unsigned long hash_code(const Hash_Entry &p);

struct Dual_Hash_Entry {
   Hash_Entry replace;
   Hash_Entry depth;
};

#endif
