// Copyright 1996 by Jon Dart.  All Rights Reserved.

// Utility to read a PGN file and convert it into the opening book.
// Input is a raw PGN file.

#include "board.h"
#include "legal.h"
#include "movegen.h"
#include "options.h"
#include "movearr.h"
#include "notation.h"
#include "chessio.h"
#include "eco.h"
#include "movearr.h"
#include "strpair.h"
#include "bearing.h"
#include "globals.h"
#include "util.h"
#include "chessio.h"

extern "C"
{
#include <stddef.h>
#include <string.h>
#include <ctype.h>
};
#include <fstream>
#include <iostream>
#include <ctype.h>
#include <map>
#include <vector>

using namespace std;

class Game_Info {

    // this class represents one entry in the hash table.

    public:

    Game_Info();

    Game_Info( const hash_t hashcode)
    : my_hashcode(hashcode)
    {
    }

    Game_Info &operator = (const Game_Info &p)
    {
        if (this != &p)
        {
           my_hashcode = p.my_hashcode;
        }
        return *this;
    }
           
    Game_Info( const Game_Info &p)
      : my_hashcode(p.my_hashcode)
    {
    }
    
    int operator == (const Game_Info &c) {
        return (my_hashcode == c.my_hashcode);
    }
    
    hash_t hash_code() const
    {
        return my_hashcode;
    }

    private:

    hash_t my_hashcode;
};

struct compareGI
{
  bool operator()(const hash_t &key1, const hash_t &key2) const
  {
    return key1 < key2;
  }
};

int replace = 1;
static map< uint64, Game_Info *,compareGI>* hashTable = NULL;
static int sample = 0;
static int max_ply = 40;
static const int MARGIN = 75;

static void show_usage()
{
	 cerr << "Usage: pgn2book -s <sample> -m <minimum ELO> -p <max ply> pgn_file1 pgn_file2 ..." << endl;
}

void write_moves(ostream &game_file, ArasanVector<StringPair> &hdrs, 
                const MoveArray &moves, const char *eco,
                const char *result)
{
    Board board;
    int max = Util::Min(moves.num_moves(),max_ply);
    char buf[80];
    *buf = '\0';
    int last = 0;
    int num_image_size;
    for (int i = 0; i < max; i++)
    {
       const MoveRecord &mr = moves[i];
       num_image_size = 0;
       char img[20];
       Notation::Image(board,mr.move(),img);
       const int image_size = (int)strlen(img);
       if (last + image_size + num_image_size + 1 >= MARGIN)
       {
	    buf[last] = '\0';
	    game_file << buf << endl;
    	    last = 0;
   	    *buf = '\0';
       }
       if (last != 0)
	    strcat(buf," ");
       strcat(buf,img);
       last = (int)strlen(buf);
       board.DoMove(mr.move());
    }
    if (last)
    {	
       buf[last + strlen(result) + 1] = '\0';
       game_file << buf;
    }
    game_file << endl;
}

int CDECL main(int argc, char **argv)
{
   init_options(argv[0]);
   options.book.book_enabled = options.log_enabled = 0;
   init_globals(argv[0],0,0);
   atexit(cleanup_globals);
   
   Board board;
   ECO eco;
   int arg = 1;
   unsigned minMoves = 30;
   int minELO = 0;

   if (argc ==1)
   {
      show_usage();
	  return -1;
   }
   else
   {
        for (;arg < argc;++arg)
        {
           if (argv[arg][0] == '-')
           {
              switch(argv[arg][1])
              {
                 case 'p':
                   ++arg;
                   max_ply = atoi(argv[arg]);
                   break;
                 case 's':
                   ++arg;
                   sample = atoi(argv[arg]);
                   break;
  	         case 'm':
		   ++arg;
		   minELO = atoi(argv[arg]);
		   break; 
                 default:
                   cerr << "Unrecognized switch : " << argv[arg][1] << endl;
                   break;
              }
           }
           else
             break;
        }
        for (;arg < argc;arg++) { 
	ifstream pgn_file( argv[arg], ios::in);
	int c;
        int count = 0;
        hash_t game_hash;
        ColorType side;
        string result, white, black;
	if (!pgn_file.good()) {
	  cerr << "could not open file " << argv[arg] << endl;
          exit(-1);
	}
        else
	{
            hashTable = new map< uint64, Game_Info *,compareGI>;
            ArasanVector<StringPair> hdrs;
            while (!pgn_file.eof())
            {
               long first;
               MoveArray moves;
               board.Reset();
               side = White;
               while (pgn_file.good() && (c = pgn_file.get()) != EOF)
               {
                   if (c=='[')
                   {
                      pgn_file.putback(c);
                      break;
                   }
               }
               hdrs.removeAll();
               ChessIO::collect_headers(pgn_file,hdrs,first);
               
               game_hash = 0;
               for (;;)
               {
                   static char buf[2048];
                   static char num[20];
                   ChessIO::Token tok = ChessIO::get_next_token(pgn_file,buf,2048);
                   if (tok == ChessIO::Eof)
                      break;
                   if (tok == ChessIO::Number)
                   {
                      strncpy(num,buf,20);
                   }
                   else if (tok == ChessIO::GameMove)
                   {
                      // parse the move
                      Move m = Notation::Value(board,side,buf);
                      if (IsNull(m) ||
                        !legal_move(board,StartSquare(m),
                            DestSquare(m)))
                      {
                          // echo to both stdout and stderr
                          fprintf(stderr,"Illegal move: %s\n",buf);
                          cout << "Illegal move: " << buf << endl;
                          break;
                      }
                      else
                      {
			  Board_State bs = board.state;
                          moves.add_move(board,bs,m,"");
                          
                          board.DoMove(m);
                          game_hash ^= board.HashCode();
                      }
                      side = OppositeColor(side);
                  }
                  else if (tok == ChessIO::Unknown)
                  {
                      fprintf(stderr,"Unrecognized text: %s\n",buf);
                  }
                  else if (tok == ChessIO::Comment)
                  {
                      // ignored for now
                  }
                  else if (tok == ChessIO::Result)
                  {
                       result = buf;
                       break;
                  }
               }
               // done with a game
               map<uint64,Game_Info *,compareGI>::const_iterator it =
                 hashTable->find(game_hash);
               Game_Info *g;
               if (it == hashTable->end())
                 g = NULL;
               else
                 g = (*it).second;
               if (g == NULL)
               {
                   g = new Game_Info(game_hash);
                   (*hashTable)[game_hash] = g;
               }
               else
               {
                   //cout << "duplicate" << endl;
                   continue; // duplicate game
              }
               if (moves.num_moves() < minMoves)
               {
                   continue;
               }
               if (sample != 0)
               {
                  count += sample;
                  if (count < 100)
                     continue;
                  count = count % 100;
               }
               // output header
               string white,black,ecoC,event,site,round,date,opening,source,whiteELOStr, blackELOStr;
               ChessIO::get_header(hdrs,"ECO",ecoC);
               ChessIO::get_header(hdrs,"White",white);
               ChessIO::get_header(hdrs,"Black",black);
               ChessIO::get_header(hdrs,"Event",event);
               ChessIO::get_header(hdrs,"Site",site);
               ChessIO::get_header(hdrs,"Round",round);
               ChessIO::get_header(hdrs,"Date",date);
               ChessIO::get_header(hdrs,"Opening",opening);
               ChessIO::get_header(hdrs,"Source",source);
	       if (minELO > 0) {
                 if (ChessIO::get_header(hdrs,"WhiteElo",whiteELOStr) &&
		   ChessIO::get_header(hdrs,"BlackElo",blackELOStr)) {
                   int whiteElo=0, blackElo=0;
                   if (sscanf(whiteELOStr.c_str(),"%d",&whiteElo) &&
	  	       sscanf(blackELOStr.c_str(),"%d",&blackElo)) {
		      if (whiteElo < minELO || blackElo < minELO) 
                         continue;
		   }
		 }
                 else // no ELO info available 
		   continue;
	       }
               string header = "; ";
               if (white.length() >0)
               {
                  header += white;
                  header += '-';
                  header += black;
               }
               else
               {
                  if (opening.length() > 0)
                  {
                     header += opening;
                     header += ' ';
                  }
                  header += source;
               }
               if (event.length() > 0 && event != "?")
               {
                  header += ' ';
                  if (event.length() > 50)
                     header += event.substr(0,50);
                  else
                     header += event;
               }
               if (site.length() > 0 && site != "?")
               {
                  header += ' ';
                  if (site.length() > 50)
                    header += site.substr(0,50);
                  else
                    header += site;
               }
               if (round.length() > 0 && round != "?")
               {
                  header += " (";
                  header += round;
                  header += ") ";
               }
               if (date.length() > 0 && date != "?")
               {
                  header += ' ';
                  header += date.substr(0,4);
               }
               if (result.length() > 0 && result != "*")
               {
                  header += " (";
                  header += result;
                  header += ")";
               }
               if (ecoC.length() > 0)
               {
                  header += " [";
                  header += ecoC;
                  header += "]";
               }
               cout << "-" << endl;
               cout << header << endl;
               write_moves(cout,hdrs,moves,ecoC.c_str(),result.c_str());
            }
            pgn_file.close();
	}
        }
   }

   return 0;
}
