// Copyright 1994, 1995 by Jon Dart.  All Rights Reserved.

#include "notation.h"
#include "board.h"
#include "bearing.h"
#include "movegen.h"
#include "debug.h"
#ifdef _DEBUG
#include "legal.h"
#endif
#include <ctype.h>
#include <string.h>

// This module handles I/O of standard algebraic notation (SAN).

Move Notation::MoveValue( const Board &board, const char *str, const ColorType color )
{
    const char *p = str;
    Square source, dest;

    if (strcmp(str,"O-O")==0) {
        if (color == White)
            return CreateMove(E1,G1,King,EmptyPiece,InvalidPiece,KCastle);
        else
            return CreateMove(E8,G8,King,EmptyPiece,InvalidPiece,KCastle);
    }
    else if (strcmp(str,"O-O-O")==0) {
        if (color == White)
            return CreateMove(E1,C1,King,EmptyPiece,InvalidPiece,QCastle);
        else
            return CreateMove(E8,C8,King,EmptyPiece,InvalidPiece,QCastle);
    }

    while (isspace(*p)) p++;
    source = SquareValue(p);
    p += 2;
    if (*p == '-' || *p == 'x') p++;
    dest = SquareValue(p);
    if ((source == InvalidSquare) || (dest == InvalidSquare))
        return NullMove;
    else {
        p += 2;
        PieceType promotion = InvalidPiece;
        if (*p == '=') {
            // promotion
            p++;
            promotion = PieceCharValue(*p);
            // check for promotion to valid piece:
            switch (promotion) {
                case EmptyPiece:
                case Pawn:
                    return NullMove;
                case Knight:
                case Bishop:
                case Rook:
                case Queen:
                    break;
                case King:
                case InvalidPiece:
                    return NullMove;
            }
        }
        return CreateMove(board,source,dest,promotion);
    }
}

void Notation::UCIMoveImage(const Move &move, char *image) {
   int j = 0;
   image[j++] = FileImage(StartSquare(move));
   image[j++] = RankImage(StartSquare(move));
   image[j++] = FileImage(DestSquare(move));
   image[j++] = RankImage(DestSquare(move));
   if (TypeOfMove(move) == Promotion) {
      ASSERT(PromoteTo(move)<16);
      image[j++] = tolower(PieceImage(TypeOfPiece(PromoteTo(move))));
   }
   image[j++] = '\0';
}


void Notation::
Image(const Board & b, const Move & m, char *image)
{
    if (IsNull(m)) {
        strcpy(image,"(null)");
        return;
    }

    PieceType p = PieceMoved(m);
    ASSERT(p != EmptyPiece);
    int k;
    if (TypeOfMove(m) == KCastle) {
        strcpy(image, "O-O");
        k = 3;
    }
    else if (TypeOfMove(m) == QCastle) {
        strcpy(image, "O-O-O");
        k = 5;
    }
    else {
        k = 0;
        if (p == Pawn) {
            if (Capture(m) == EmptyPiece) {
                image[k] = FileImage(DestSquare(m));
                k++;
                image[k] = RankImage(DestSquare(m));
                k++;
            }
            else {
                image[k] = FileImage(StartSquare(m));
                k++;
                image[k] = 'x';
                k++;
                image[k] = FileImage(DestSquare(m));
                k++;
                image[k] = RankImage(DestSquare(m));
                k++;
            }
            if (TypeOfMove(m) == Promotion) {
                image[k] = '=';
                k++;
                image[k] = PieceImage(PromoteTo(m));
                k++;
            }
        }
        else {
            image[k] = PieceImage(p);
            k++;
            Bitmap attacks = 
              b.calc_attacks(DestSquare(m), b.Side());
            unsigned n = attacks.bit_count();
            int dups = 0;
            int filedups = 0;
            int rankdups = 0;
            int files[9];
            int ranks[9];

            if (n > 1) {
                Square sq;
                while (attacks.iterate(sq)) {
                    if (TypeOfPiece(b[sq]) == p) {
                        files[dups] = File(sq);
                        if (files[dups] == File(StartSquare(m)))
                            filedups++;
                        ranks[dups] = Rank(sq,White);
                        if (ranks[dups] == Rank(StartSquare(m),White))
                            rankdups++;
                        ++dups;
                    }
                }
            }
            if (dups > 1) {
                // need to disambiguate move.
                if (filedups == 1) {
                    image[k] = FileImage(StartSquare(m));
                    k++;
                }
                else if (rankdups == 1) {
                    image[k] = RankImage(StartSquare(m));
                    k++;
                }
                else {
                    // need both rank and file to disambiguate
                    image[k] = FileImage(StartSquare(m));
                    k++;
                    image[k] = RankImage(StartSquare(m));
                    k++;
                }
            }
            if (Capture(m) != EmptyPiece) {
                image[k] = 'x';
                k++;
            }
            image[k] = FileImage(DestSquare(m));
            k++;
            image[k] = RankImage(DestSquare(m));
            k++;
        }
    }
    Board board_copy(b);
	ASSERT(legal_move(board_copy,m));
    board_copy.DoMove(m);
    if (board_copy.CheckStatus() == InCheck) {
        Move moves[Constants::MaxMoves];
        Move_Generator mg(board_copy,NULL,0,NullMove,0);
        if (mg.GenerateEvasions(moves))
            image[k++] = '+';
        else
            image[k++] = '#';                     // mate
    }
    image[k] = '\0';
}


Move Notation::
Value(const Board & board,
const ColorType side, const char *image)
{
    const char *p;
    int rank = 0;
    int file = 0;

    PieceType piece = InvalidPiece;
    PieceType promotion = InvalidPiece;
    Square dest, start = InvalidSquare;
    int capture = 0;

    for (p = image; isspace(*p); ++p);
    if (!isalpha(*p) && *p != '0')
        return NullMove;
    if (toupper(*p) == 'O' || *p == '0') {
        // castling, we presume
        char tmp[10];

        strncpy(tmp, p, 9);
        // look for brain-dead variants of castling like "O-O-0"
        char *q = tmp;
        while ((q = strchr(q,'0')) != NULL) {
            *q++ = 'O';
        }
        char *check;
        if ((check = strchr(tmp, '+')) != NULL)
            *check = '\0';
        else if ((check = strchr(tmp, '#')) != NULL)
            *check = '\0';
        for (q = tmp; *q; q++)
            *q = toupper(*q);
        return MoveValue(board, tmp, side);
    }
    int have_start = 0;
    if (isupper(*p)) {
        piece = PieceCharValue(*p);
        ++p;
    }
    else {
        piece = Pawn;
        file = *p - 'a' + 1;
        // allow "dc4" as in Informant, instead of dxc4
        if (*(p+1) == 'x' || isalpha(*(p+1))) {
            capture = 1;
            ++p;
        }
        else if (isdigit(*(p+1))) {
            const char *q = p+2;
            if (*q == 'x' || *q == '-') {
                // long algebraic notation
                have_start++;
                start = SquareValue(p);
                p = q;
            }
        }
    }
    if (piece == InvalidPiece)
        return NullMove;
    if (piece != Pawn) {
        // look for disambiguating rank or file, e.g. 'b' in "Nbd7".

        if (isalpha(*p) && isdigit(*(p+1)) &&
        (strchr(p,'-') || strchr(p,'x'))) {
            // looks like long algebraic notation
            start = SquareValue(p);
            p+=2;
            have_start++;
        }
        else if (isdigit(*p)) {
            rank = *p - '0';
            ++p;
        }
        else if (*p != 'x' && isalpha(*p) && isalpha(*(p + 1))) {
            file = *p - 'a' + 1;
            ++p;
        }
    }

    if (*p == 'x') {
        capture = 1;
        ++p;
    }
    else if (*p == '-')
        ++p;
    // remainder of move should be a square identifier, e.g. "g7"
    dest = SquareValue(p);
    if (IsInvalid(dest))
        return NullMove;
    p += 2;
    if (*p == '=') {
        promotion = PieceCharValue(*(p + 1));
        if (piece != Pawn || promotion == InvalidPiece)
            return NullMove;
    }
    else if (piece == Pawn && isupper(*p)) {
        // Quite a few "PGN" files have a8Q instead of a8=Q.
        promotion = PieceCharValue(*p);
        if (promotion == InvalidPiece)
            return NullMove;
    }

    // Informant does not use "x" for captures.  Assume that if the destination
    // is occupied, this is a capture move.
    if (board[dest] != EmptyPiece)
        capture = 1;
    // Do a sanity check on capture moves:
    if (capture && !IsEmptyPiece(board[dest]) && PieceColor(board[dest]) == board.Side())
        return NullMove;

    // ok, now we need to figure out where the start square is.

    int dups = 0;

    if (!have_start) {

        if (capture && piece == Pawn && IsEmptyPiece(board[dest]) &&
        Rank(dest,board.Side()) != 8) {
            // en passant capture, special case
            int start_rank = (board.Side() == White) ?
                Rank(dest,White) - 1 :
            Rank(dest,White) + 1;

            start = Square(file, start_rank, White);
            dups = 1;
        }
        else if (piece == Pawn && board[dest] == EmptyPiece) {
            start = Square(file,Rank(dest,board.Side())-1,board.Side());   
            if (board[start] == EmptyPiece && Rank(dest,board.Side())==4) {
                start = Square(file,Rank(dest,board.Side())-2,board.Side());  
            }
            if (board[start] == EmptyPiece) return NullMove;
            dups = 1;
        }
        else {
            Bitmap attacks = board.calc_attacks(dest,side);
            Square maybe;
			while (attacks.iterate(maybe)) {
                if (TypeOfPiece(board[maybe]) == piece &&
                PieceColor(board[maybe]) == board.Side()) {
                    if (file && File(maybe) != file)
                        continue;
                    else if (rank && Rank(maybe,White) != rank)
                        continue;
                    else if (PieceColor(board[maybe]) == board.Side()) {
                        // Possible move to this square.  Make sure it is legal.

                        Board board_copy(board);
                        Move emove = CreateMove(board,maybe,dest,promotion);
                        board_copy.DoMove(emove);
                        if (!board_copy.any_attacks(
                            board_copy.KingPos(board_copy.OppositeSide()),
                        board_copy.Side())) {
                            ++dups;
                            start = maybe;
                        }
                    }
                }
            }
        }
    }
    if (dups == 1 || have_start) {
		if (start == InvalidSquare || board[start] == EmptyPiece)
			return NullMove;
		else
            return CreateMove(board, start, dest, promotion);
    }
    else                                          // ambiguous move
        return NullMove;
}
