// Copyright 1994-2007 by Jon Dart.  All Rights Reserved.

#include "moveord.h"
#include "movegen.h"
#include "scoring.h"
#include "constant.h"
#include "attacke.h"
#include "bearing.h"
#include "util.h"
#include "globals.h"
#include "config.h"

#include <iostream>

inline void swap( Move moves[], const int i, const int j)
{
   Move tmp = moves[j];
   moves[j] = moves[i]; 
   moves[i] = tmp;
}

int
Move_Ordering::order_captures(Board & board,
                              Move moves[], const int num_moves,
                              const int alpha, const int material, int ply)
{
   int scores[Constants::MaxMoves];
   int n=0;
   int bias = QSEARCH_FORWARD_PRUNE_BIAS + material;
  
   for (int i = 0; i < num_moves; i++)
   {
      PieceType capture = Capture(moves[i]);
      Square loc = StartSquare(moves[i]);
      if (capture != EmptyPiece)
      {
         int est;
         int start_val = (int)PieceValue(board[loc]);
         if (capture == King) { // king capture
            // There is no point sorting the
            // rest of the moves. We return only
            // this move, which signals that
            // the previous move was illegal.
            moves[0] = moves[i];
            return 1;
         }
         else {
	   est = Gain(moves[i]);
         }
         // Futility check: if capturing the piece "for free" will not
         // get our score up to alpha, discard the move
         if (bias + est >= alpha)
         {
            est -= start_val;
            if (est > BISHOP_BONUS) 
            {
               // Always consider capturing a higher-valued piece
               scores[i] = est + (int)capture;
               if (i>n){ 
                  scores[n] = scores[i];
                  moves[n] = moves[i];
               }
               ++n;
            }
            else
            {
               // This capture appears to lose or be even, but we call
               // attack_estimate to make sure
	       est = attack_estimate(board,moves[i]);
	       /**#ifdef _TRACE
               indent(ply); cout << "move "; MoveImage(moves[i],cout);
               cout <<  " est=" << est << endl;
	       #endif**/
               if (est >= -BISHOP_BONUS && bias + est >= alpha) {
                  scores[i] = est + (int)capture;
                  if (i>n){ 
                    scores[n] = scores[i];
                    moves[n] = moves[i];
                  }
                  ++n;
               }
            }
         }
      }
      else if (TypeOfMove(moves[i]) == Promotion)
      {
         scores[i] = PieceValue(PromoteTo(moves[i])) - PAWN_VALUE;
         if (scores[i] + PAWN_VALUE + bias  >= alpha) {
             if (i>n){ 
                 scores[n] = scores[i];
                 moves[n] = moves[i];
             }
             ++n;
         }
      }
   }
   if (n>1){ 
      // this is generally faster than calling sort_moves, because
      // n is small.
      int swapped = 1;
      while (swapped) {
        swapped = 0;
        for (int i = 0; i < n-1; i++) {
           if (scores[i]<scores[i+1]) {
              ++swapped;
              swap(moves,i,i+1);
              int tmp = scores[i];
              scores[i] = scores[i+1];
              scores[i+1] = tmp;
           }
        }
      }
   }
   return n;
}

