// Copyright 1992-2007 by Jon Dart. All Rights Reserved.
//
#ifndef _MOVE_GENERATOR_H
#define _MOVE_GENERATOR_H

#include "board.h"
#include "constant.h"
#include "searchc.h"
#include "search.h"

class Move_Generator {
    public:
		
    enum Phase {HASH_MOVE_PHASE, WINNING_CAPTURE_PHASE,
      KILLER1_PHASE, KILLER2_PHASE, HISTORY_PHASE,
      LOSERS_PHASE, LAST_PHASE };

    Move_Generator( const Board &ABoard, 
                    SearchContext *,
                    unsigned ply, 
                    Move pvMove, 
                    int trace);
	
    // Generate the next move, in sorted order, NullMove if none left
    virtual Move NextMove() {
       if (index >= batch_count) {
          if ((batch_count = getBatch(batch,index)) == 0) 
             return NullMove;
       }
       return batch[index++];
    }

    virtual ~Move_Generator() {
    }

    // Generate the next check evasion, NullMove if none left
    virtual Move NextEvasion();

#ifdef SMP
    virtual Move NextMove(NodeInfo *node) {
      if (node->wasSplit()) {
	 node->lock();
         Move m = NextMove();
         node->unlock();
         return m;
      }
      else
	return NextMove();
    }

    virtual Move NextEvasion(NodeInfo *node) {
      if (node->wasSplit()) {
	 node->lock();
         Move m = NextEvasion();
         node->unlock();
         return m;
      }
      else
	return NextEvasion();
    }
#endif

#ifdef SMP
   virtual int GenerateAllMoves(NodeInfo *);
#endif

    // Generate only non-capturing moves.
    int GenerateNonCaptures(Move *moves);

    // fill array "moves" with moves, starting at "index", returns
    // # of moves (0 if no more). This tries to generate moves in stages,
    // so it should be called repeatedly until the return value is zero.
    // 		
    // The moves returned are generally "pseudo-legal", i.e. they may
    // involve moves into check.  However, if the side to move is in
    // check, then all moves returned are strictly legal.		
    //
    int GenerateAllMoves(Move *moves,int repeatable);

    // Only generates captures, promotions, and check evasions.
    //
    int GenerateCaptures( Move *moves );
		
    // Return a list of the moves that may be used to escape
    // check.  Unlike the regular move generation routine, all
    // moves are checked for legality.
    //
    unsigned GenerateEvasions(Move * moves);

    // Generate non-capturing checking moves
    unsigned GenerateChecks(Move * moves);

    unsigned Moves_Generated()
    {
       return moves_generated;
    }

    Phase getPhase() const {
      return (Phase)phase;
    }

    int more() const {
      return phase<LAST_PHASE || index < batch_count;
    }
		
 protected:
		
    int getBatch(Move *&batch,int &index);
    int OrderWinningCaptures (int captures, Move hashMove);

    int getEvasions();

    unsigned GenerateEvasionsCaptures(Move * moves);
    unsigned GenerateEvasionsNonCaptures(Move * moves);
    unsigned GenerateEvasions(Move * moves,
			      const Bitmap &mask);

    const Board &board;
    SearchContext *context;
    int ply;
    int moves_generated;
    int phase, winners, losers_count,index,batch_count,forced;
    Move hashMove;
    int non_captures_count,history_count;
    Bitmap king_attacks; // for evasions
    int num_attacks; // for evasions
    Square source; // for evasions;
    Move *batch;
    Move losers[100];
    Move moves[Constants::MaxMoves];
    Move killer1,killer2,killer3,killer4;
    int master;
        
    inline void SetMove( const Square source, const Square dest,
       const PieceType promotion,
       Move *moves, unsigned &NumMoves);

};


class Root_Move_Generator : public Move_Generator {
 public:
    Root_Move_Generator(const Board &ABoard, 
                        SearchContext *,
	 	        Move *moves,
                        Move pvMove, 
		        int trace);

    // Generate the next move, in sorted order, NullMove if none left
    virtual Move NextMove() {
      ASSERT(index<=batch_count);
      if (index < batch_count)
        return batch[index++];
      else
	return NullMove;
    }

    // Generate the next check evasion, NullMove if none left
    virtual Move NextEvasion() {
      return NextMove();
    }


#ifdef SMP
    virtual int GenerateAllMoves(NodeInfo *);

    virtual Move NextMove(NodeInfo *node) {
	if (node->wasSplit()) {
           node->lock();
           Move m = NextMove();
           node->unlock();
           return m;
	}
	else
	   return NextMove();
    }

    virtual Move NextEvasion(NodeInfo *node) {
	if (node->wasSplit()) {
           node->lock();
           Move m = NextEvasion();
           node->unlock();
           return m;
	}
	else
	   return NextEvasion();
    }
#endif

    void reorder(Move pvMove);

    void exclude(Move *exclude, int num_exclude);

    int move_count() const {
      return batch_count;
    }

    Move first() const {
      return batch[0];
    }

    int hasEasyMove() const {
      return easy;
    }

 private:
    int order();
    int score(const Move & move);

	int easy;

};


#endif
