// Copyright 1994-2008 by Jon Dart. All Rights Reserved.
//
#include "movegen.h"
#include "bearing.h"
#include "attacke.h"
#include "util.h"
#include "debug.h"
#include "moveord.h"
#include "config.h"
#include "bearing.h"
#include "globals.h"
#include <iostream>
#include <fstream>

extern const int Direction[2];

static inline void swap( Move moves[], const int i, const int j){
    Move tmp = moves[j];
    moves[j] = moves[i];
    moves[i] = tmp;
}


static int IsValidMove(const Board &board, Move killer) {
    PieceType pieceMoved = PieceMoved(killer);
    Square start = StartSquare(killer); Square dest = DestSquare(killer);
    // generally killers are not captures but can be at ply 0
    if (Capture(killer) != EmptyPiece || TypeOfMove(killer) == Promotion)
        return 0;
    if (TypeOfPiece(board[start]) == pieceMoved && PieceColor(board[start]) == board.Side()
    && board[dest] == EmptyPiece) {
        switch(pieceMoved) {
            case Knight:
                return 1;
            case King:
            {
                if (TypeOfMove(killer) == KCastle) {
                    Square kp = board.KingPos(board.Side());
                    CastleType CS = board.CastleStatus(board.Side());
                    return board[kp] == MakePiece(King,board.Side()) &&
                        ((CS == CanCastleEitherSide) ||
                        (CS == CanCastleKSide)) &&
                        board[kp + 1] == EmptyPiece &&
                        board[kp + 2] == EmptyPiece &&
                        board.CheckStatus() != InCheck &&
                        !board.any_attacks(kp + 1,board.OppositeSide()) &&
                        !board.any_attacks(kp + 2,board.OppositeSide());
                }
                else if (TypeOfMove(killer) == QCastle) {
                    Square kp = board.KingPos(board.Side());
                    CastleType CS = board.CastleStatus(board.Side());
                    return board[kp] == MakePiece(King,board.Side()) &&
                        ((CS == CanCastleEitherSide) ||
                        (CS == CanCastleQSide)) &&
                        board[kp - 1] == EmptyPiece &&
                        board[kp - 2] == EmptyPiece &&
                        board[kp - 3] == EmptyPiece &&
                        board.CheckStatus() != InCheck &&
                        !board.any_attacks(kp - 1,board.OppositeSide()) &&
                        !board.any_attacks(kp - 2,board.OppositeSide());
                }
                else
                    return 1;
            }
            case Rook:
            {
                int d = Util::Abs(Bearing::Directions[start][dest]);
                return (d==1 || d==8) && board.Clear(start,dest);
            }
            case Bishop:
            {
                int d = Util::Abs(Bearing::Directions[start][dest]);
                return (d==7 || d==9) && board.Clear(start,dest);
            }
            case Queen:
            {
                int d = Util::Abs(Bearing::Directions[start][dest]);
                return (d==1 || d==8 || d==7 || d==9) && board.Clear(start,dest);
            }
            case Pawn:
            {
                if (board.Side() == White) {
                    if (start-8==dest) return 1;
                    else return (board[start-8] == EmptyPiece && Rank(start,White)==2 && start-16==dest);
                }
                else {
                    if (start+8==dest) return 1;
                    else return (board[start+8] == EmptyPiece && Rank(start,Black)==2 && start+16==dest);
                }
            }
            default:
                break;
        }
    }
    return 0;
}

Root_Move_Generator::Root_Move_Generator(const Board &ABoard, 
                      SearchContext *s,
		      Move *moves,
                      Move pvMove, 
		      int trace)
  : Move_Generator(ABoard,s,0,pvMove,trace)
{
   batch = moves;
   batch_count = Move_Generator::GenerateAllMoves(batch,0);
   batch_count = order();
   if (board.CheckStatus()==InCheck && batch_count == 1)
      SetForced(batch[0]);
   phase = LAST_PHASE;
}

#ifdef SMP
int Root_Move_Generator::GenerateAllMoves(NodeInfo *) {
  // There's nothing to be done here since we generated moves in the
  // constructor. Just return the number of moves remaining.
  return batch_count-index;
}
#endif

int Root_Move_Generator::score(const Move & move)
{
   int val = 0;
   Square dest_square = DestSquare(move);
   Piece capture = board[dest_square];
   if (!IsEmptyPiece(capture))
   {
     if (TypeOfPiece(capture) == King) {
       val = KING_VALUE;
     }
     else {
       int est = attack_estimate(board,move);
       if (est > 0)
          val += est;
       else // losing capture
          val += 10 + (est/16);
     }
   }
   else if (TypeOfMove(move) == Promotion) {
       val += PieceValue(PromoteTo(move)) - PieceValue(Pawn);
   }
   else if (TypeOfMove(move) == KCastle || TypeOfMove(move)== QCastle)
     val += 5;
   else { // favor moves forward
     val += Rank(DestSquare(move),board.Side())-Rank(StartSquare(move),board.Side());
     if (PieceMoved(move) == Pawn)
       val += Rank(DestSquare(move),board.Side());
   }
   Board b(board);
   Board_State state(b.state);
   b.DoMove(move);
   if (b.any_attacks(b.KingPos(b.OppositeSide()),b.Side())) {
     val = -10000;
   }
   else if (b.CheckStatus() == InCheck)
     val += 15; // opponent in check;
   b.UndoMove(move,state);
#ifdef _TRACE
   MoveImage(move,cout); cout << " score = " << val << endl;
#endif
   return val;
}

int Root_Move_Generator::order()
{
   int scores[Constants::MaxMoves];

   int i;
   for (i = 0; i < batch_count; i++)
   {
      scores[i] = score(batch[i]);
   }
   Move_Ordering::sort_moves(batch, scores, batch_count);
   int real_num_moves = batch_count;

   easy = 1;
   for (i=batch_count; i >=0; --i)
   {
      if (scores[i] == -10000) {
         --real_num_moves;
      }
      else if (i>0 && scores[0]-100 < scores[i]) {
	easy = 0;
      }
   }
   return real_num_moves;
}


void Root_Move_Generator::reorder(Move pvMove) {
   int ply0move_count = batch_count;
   index = 0; // reset so we will fetch moves again
   phase = 0;
   if (ply0move_count <= 1)
     return;
   int i; int j = 0;
   // Pv first, then killers. Killers at this ply (ply 0)
   // are moves that have previously exceeded alpha.
   //
   for (i = 0; i < ply0move_count; i++)
   {
      ClearUsed(batch[i]);
      if (EqualMove(batch[i],pvMove)) {
	/**#ifdef _TRACE
	if (master) {
           cout << "move ";
           MoveImage(batch[i],cout);
           cout << " (pv)"<< endl;
	}
	#endif**/
         swap(batch,i,0); ++j;
      }
   }
   for (i=j; i< ply0move_count;i++) {
      if (EqualMove(batch[i],context->Killers1[0])) {
         swap(batch,i,j); j++;
         //break; // causes VC++98 compiler error
      }
   }
   for (i=j; i< ply0move_count;i++) {
      if (EqualMove(batch[i],context->Killers2[0])) {
         swap(batch,i,j); 
         break;
      }
   }
}

void Root_Move_Generator::exclude(Move *exclude, int num_exclude) {
    ASSERT(num_exclude<MAX_PV);
    for (int i = 0; i < num_exclude; i++) {
       for (int j = 0; j < batch_count; j++) {
           if (MovesEqual(batch[j],exclude[i])) {
               SetUsed(batch[j]);
           }
       }
    }
}


int Move_Generator::OrderWinningCaptures (int captures, Move hashMove) {
    int j = 0;
    int scores[100];
    for (int i = index; i < index+captures; i++) {
        const Square start = StartSquare(moves[i]);
        const PieceType capture = Capture(moves[i]);
        if (MovesEqual(hashMove,moves[i])) {
            SetUsed(moves[i]);
        }
        else if (!IsEmptyPiece(capture)) {
            int start_val;
            start_val = PieceValue(board[start]);
            int capture_val = PieceValue(capture);
            int gain;
            if (capture_val - start_val > 10) {
                gain = capture_val - start_val;
                if (TypeOfMove(moves[i]) == Promotion) gain += (PieceValue(PromoteTo(moves[i])) - PAWN_VALUE);
            }
            else
                gain = attack_estimate(board,moves[i]);
#ifdef _TRACE
            if (master) {
                for (int k=0; k < ply; k++) cout << ' ';
                cout << "move ";
                MoveImage(moves[i],cout);
                cout << " est = " << gain << endl;
            }
#endif
            if (gain >= -10) {
                if (i != j) {
                    swap(moves,i,j);
                }
                scores[j] = gain;
                ++j;
            }
            else {
	      ASSERT(losers_count<100);
              losers[losers_count++] = moves[i];
            }
        }
        else if (TypeOfMove(moves[i]) == Promotion) {
            int score = PieceValue(PromoteTo(moves[i])) - PAWN_VALUE;
#ifdef _TRACE
            if (master) {
                for (int k=0; k < ply; k++) cout << ' ';
                cout << "move ";
                MoveImage(moves[i],cout);
                cout << " est = " << score << endl;
            }
#endif
            scores[j] = score;
            swap(moves,i,j);
            ++j;
        }
        else if (TypeOfMove(moves[i]) == EnPassant) {
            int gain = attack_estimate(board,moves[i]);
            if (gain >= 0) {
                scores[j] = gain;
                swap(moves,i,j);
                ++j;
            }
            else
                losers[losers_count++] = moves[i];
        }
    }
    int NumMoves = j-index;
    if (NumMoves > 1) {
        Move_Ordering::sort_moves(moves,scores,NumMoves);
    }
    return NumMoves;
}


Move Move_Generator::NextEvasion() {
    if (index >= batch_count) {
          if ((batch_count = getEvasions()) == 0) 
             return NullMove;
    }
    return batch[index++];
}

int Move_Generator::getEvasions() {
  batch_count = 0; index = 0; 
  while (phase < 4 && !batch_count) {
    switch(phase) {
        case 0: {
            ++phase;
            if (hashMove != NullMove) {
#ifdef _TRACE
              if (master) {
                 indent(ply); 
	 	 cout << "phase 0: ";
		 MoveImage(hashMove,cout); 
		 cout << endl;
              }
#endif
	      moves[0] = hashMove;
	      index = 0;
	      batch = moves;
	      batch_count = 1;
	    }
	    break;
        }
        case 1: {
#ifdef _TRACE
            if (master) {
                indent(ply); cout << "phase 1: ";
            }
#endif
            batch_count = GenerateEvasionsCaptures(moves);
            if (batch_count) {
                batch_count = OrderWinningCaptures(batch_count,NullMove);
            }
            winners = batch_count;
#ifdef _TRACE
            if (master) {
                cout << " winners=" << batch_count << " losers=" << losers_count << endl;
            }
#endif
            batch = moves;
            for (int i = 0; i < batch_count; i++) {
                if (MovesEqual(hashMove,batch[i])) {
		    SetUsed(batch[i]);
                    break;
                }
            }
            ++phase;
            break;
        }
        case 2:
        {
            batch_count = GenerateEvasionsNonCaptures(moves);
#ifdef _TRACE
            if (master) {
                indent(ply); cout << "phase 2: non-captures: " << batch_count << endl;
            }
#endif
            // We do not consider an evasion move "forced" if it is a winning
            // capture (even if there is only one singular move), since many
            // checks will be refuted by a simple capture of the checking piece.
            forced = (winners == 0) && (losers_count + batch_count == 1);

            if (forced && batch_count) SetForced(moves[0]);

            batch = moves;
            for (int i = 0; i < batch_count; i++) {
                if (MovesEqual(hashMove,batch[i])) {
		    SetUsed(batch[i]);
                    break;
                }
            }
            index = 0;
            ++phase;
            break;
        }
        case 3:
        {
#ifdef _TRACE
            if (master) {
                indent(ply); cout << "phase 3: losers: " << losers_count << endl;
            }
#endif
            batch = losers;
            batch_count = losers_count;
            for (int i = 0; i < batch_count; i++) {
                if (MovesEqual(hashMove,batch[i])) {
		    SetUsed(batch[i]);
                    break;
                }
            }
            if (forced && batch_count) SetForced(losers[0]);
            index = 0;
            ++phase;
            break;
        }
    }
  }
  return batch_count;
}


int Move_Generator::getBatch(Move *&batch,int &index) {
    unsigned NumMoves  = 0;
    batch = moves;
    for (; NumMoves == 0 && phase<=LOSERS_PHASE; ++phase) {
#ifdef _TRACE
        if (master) {
            indent(ply); cout << "phase = " << phase << endl;
        }
#endif
        switch(phase) {
            case HASH_MOVE_PHASE:
            {
                if (!IsNull(hashMove)) {
                    *moves = hashMove;
#ifdef _TRACE
                    if (master) {
                      indent(ply); cout << "hash move = ";
	              MoveImage(hashMove,cout); 
	              cout << endl;
                   }
#endif
                    index = 0;
                    NumMoves = 1;
                }
                break;
            }
            case WINNING_CAPTURE_PHASE:
            {
                int captures = Move_Generator::GenerateCaptures(moves);
                index = 0;
                NumMoves = OrderWinningCaptures(captures,hashMove);
#ifdef _TRACE
		indent(ply); cout << "winners=" << NumMoves << endl;
#endif
                break;
            }
            case KILLER1_PHASE:
            {
                killer1 = killer2 = killer3 = killer4 = NullMove;
                if (context) {
                   context->get_killers(ply,killer1,killer2);
                   if (ply >=2)
                       context->get_killers(ply-2,killer3,killer4);
                   else
                       killer3 = killer4 = NullMove;
                }
                if (!IsNull(killer1) && !MovesEqual(hashMove,killer1)) {
                    if (IsValidMove(board,killer1)) {
                        moves[NumMoves++] = killer1;
                        index = 0;
                    }
                }
                if (ply>=2 && !IsNull(killer3) && !MovesEqual(hashMove,killer3) &&
                !MovesEqual(killer1,killer3)) {
                    if (IsValidMove(board,killer3)) {
                        moves[NumMoves++] = killer3;
                        index = 0;
                    }
                }
            }
            break;
            case KILLER2_PHASE:
            {
                if (!IsNull(killer2) && !MovesEqual(hashMove,killer2) &&
                (ply<2||!MovesEqual(killer2,killer3))) {
                    if (IsValidMove(board,killer2)) {
                        moves[NumMoves++] = killer2;
                        index = 0;
                    }
                }
                if (ply>=2 && !IsNull(killer4) &&
                    !MovesEqual(hashMove,killer4) &&
                    !MovesEqual(killer1,killer4) &&
                !MovesEqual(killer2,killer4)) {
                    if (IsValidMove(board,killer4)) {
                        moves[NumMoves++] = killer4;
                        index = 0;
                    }
                }
            }
            break;
            case HISTORY_PHASE:
            {
                NumMoves = non_captures_count = GenerateNonCaptures(moves);
                index = 0;
                history_count = 0;
                if (NumMoves) {
                    long max = -1L; int maxi=-1;
                    int maxi2 =-1;
                    for (unsigned i = 0; i < NumMoves; i++) {
                        if (MovesEqual(hashMove,moves[i])) {
                            SetUsed(moves[i]);
                            continue;
                        }
                        else if (MovesEqual(killer1,moves[i]) ||
                        MovesEqual(killer2,moves[i])) {
                            SetUsed(moves[i]);
                            continue;
                        }
                        else if ((MovesEqual(killer3,moves[i]) ||
                        MovesEqual(killer4,moves[i]))) {
                            SetUsed(moves[i]);
                            continue;
                        }
                        if (context) {
                            int pscore = context->history_score_for_pruning(moves[i],board.Side());
                            if (TypeOfMove(moves[i])==Normal && pscore < 6*Constants::HISTORY_MAX/10) {
                                SetHReduce(moves[i]);
                                if (pscore < Constants::HISTORY_MAX/4) {
				  SetHPrune(moves[i]);
				}
                            }
                            int score = context->history_score_for_ordering(moves[i],board.Side());
                            if (score >= max) {
                                if (maxi >=0) {
                                    maxi2 = maxi;
                                }
                                maxi = i;
                                max = score;
                            }
                        }
                    }
                    if (maxi != -1) {
                        ClearHPrune(moves[maxi]);
			ClearHReduce(moves[maxi]);
                        swap(moves,history_count++,maxi);
                    }
                    if (maxi2 != -1) {
                        ClearHPrune(moves[maxi2]);
			ClearHReduce(moves[maxi]);
                        swap(moves,history_count++,maxi2);
                    }
                }
                NumMoves = non_captures_count;
                index = 0;
                break;
            }
            case LOSERS_PHASE:
            {
                if (losers_count) {
                    batch = losers;
                }
                NumMoves = losers_count;
                index = 0;
                break;
            }
        }                                         // end switch
    }                                             // end for
#ifdef DEBUG
    for (int = 0; i < NumMoves; i++)
        if (Capture(batch[i])==King) ASSERT(0);
#endif
    return NumMoves;
}


int Move_Generator::GenerateNonCaptures(Move *moves) {
    int NumMoves = 0;
    // castling moves
    const ColorType side = board.Side();
    CastleType CS = board.CastleStatus(side);
    if ((CS == CanCastleEitherSide) ||
    (CS == CanCastleKSide)) {
        const Square kp = board.KingPos(side);
#ifdef _DEBUG
        if (side == White) {
            ASSERT(kp == 60);
            ASSERT(board[kp+3] == whiteRook);
        }
        else {
            ASSERT(kp == 4);
            ASSERT(board[kp+3] == blackRook);
        }
#endif
        if (board[kp + 1] == EmptyPiece &&
            board[kp + 2] == EmptyPiece &&
            board.CheckStatus() == NotInCheck &&
            !board.any_attacks(kp + 1,OppositeColor(side)) &&
            !board.any_attacks(kp + 2,OppositeColor(side)))
            // can castle
            moves[NumMoves++] = CreateMove(kp, kp+2, King, 
                EmptyPiece, InvalidPiece, KCastle);
    }
    if ((CS == CanCastleEitherSide) ||
    (CS == CanCastleQSide)) {
        const Square kp = board.KingPos(side);
        if (board[kp - 1] == EmptyPiece &&
            board[kp - 2] == EmptyPiece &&
            board[kp - 3] == EmptyPiece &&
            board.CheckStatus() == NotInCheck  &&
            !board.any_attacks(kp - 1,OppositeColor(side)) &&
            !board.any_attacks(kp - 2,OppositeColor(side)))
            // can castle
            moves[NumMoves++] = CreateMove(kp, kp-2, King, EmptyPiece, 
					   InvalidPiece, QCastle);
    }
    // non-pawn moves:
    Bitmap pieces(board.occupied[board.Side()]);
    pieces.andNot(board.pawn_bits[board.Side()]);
    Square loc;
    while (pieces.iterate(loc)) {
        switch(TypeOfPiece(board[loc])) {
            case Knight:
            {
                Bitmap dests(Bearing::knight_attacks[loc]);
                dests.andNot(board.all_occupied);
                Square sq;
                while (dests.iterate(sq)) {
                    moves[NumMoves++] =
                        CreateMove(loc,sq,Knight);
                }
                break;
            }
            case King:
            {
                Bitmap dests(Bearing::king_attacks[loc]);
                dests.andNot(board.all_occupied);
                dests.andNot(Bearing::king_attacks[board.KingPos(board.OppositeSide())]);
                Square sq;
                while (dests.iterate(sq)) {
                    moves[NumMoves++] =
                        CreateMove(loc,sq,King);
                }
                break;
            }
            case Bishop:
            {
                const int *data = BishopSquares[loc];
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Bishop);
                    ++data;
                }
                data = BishopSquares[loc] + 8;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Bishop);
                    ++data;
                }
                data = BishopSquares[loc] + 16;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Bishop);
                    ++data;
                }
                data = BishopSquares[loc] + 24;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Bishop);
                    ++data;
                }
                break;
            }
            case Rook:
            {
                const int *data = RookSquares[loc];
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Rook);
                    if (Capture(moves[NumMoves-1])!=EmptyPiece) cout << "error" << endl;
                    ++data;
                }
                data = RookSquares[loc] + 8;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Rook);
                    ++data;
                }
                data = RookSquares[loc] + 16;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Rook);
                    ++data;
                }
                data = RookSquares[loc] + 24;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    moves[NumMoves++] =
                        CreateMove(loc,(Square)*data,Rook);
                    ++data;
                }
                break;
            }
            case Queen:
            {
                int j;
                for (j = 0; j < 4; j++) {
                    const int *data = BishopSquares[loc] + (j*8);
                    while (*data != 255 && board[*data] == EmptyPiece) {
                        moves[NumMoves++] =
                            CreateMove(loc,(Square)*data,Queen);
                        ++data;
                    }
                }
                for (j = 0; j < 4; j++) {
                    const int *data = RookSquares[loc] + (j*8);
                    while (*data != 255 && board[*data] == EmptyPiece) {
                        moves[NumMoves++] =
                            CreateMove(loc,(Square)*data,Queen);
                        ++data;
                    }
                }
                break;
            }
        }
    }
    // pawn moves
    if (board.Side() == White) {
        Bitmap pawns(board.pawn_bits[White]);
        pawns.shr8();
        pawns.andNot(board.all_occupied);
        pawns.andNot(Bearing::rank_mask[0]);
        Square sq;
        while (pawns.iterate(sq)) {
            moves[NumMoves++] = CreateMove(sq+8,sq,Pawn);
            if (Rank(sq,White)==3 && board[sq-8] == EmptyPiece)
                moves[NumMoves++] = CreateMove(sq+8,sq-8,Pawn);
        }
    }
    else {
        Bitmap pawns(board.pawn_bits[Black]);
        pawns.shl8();
        pawns.andNot(board.all_occupied);
        pawns.andNot(Bearing::rank_mask[7]);
        Square sq;
        while (pawns.iterate(sq)) {
            moves[NumMoves++] = CreateMove(sq-8,sq,Pawn);
            if (Rank(sq,Black)==3 && board[sq+8] == EmptyPiece)
                moves[NumMoves++] = CreateMove(sq-8,sq+8,Pawn);
        }
    }
    return NumMoves;
}


int Move_Generator::GenerateCaptures(Move * moves) {
    unsigned NumMoves;
    NumMoves = 0;
    Square loc;

    if (board.Side() == White) {
        Bitmap pawns(board.pawn_bits[White]);
        Bitmap pawns1 = pawns;
        pawns1.shr(7);
        pawns1.andNot(Bitmap(MAKELONGLONG(0x01010101,0x01010101)));
        pawns1.And(board.occupied[Black]);
        Square dest;
        while (pawns1.iterate(dest)) {
            Square loc = dest+7;
            int candidate = (Rank(loc,White) == 7);
            if (candidate) {
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Queen,Promotion);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Knight,Promotion);
                if (ply == 0) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Rook,Promotion);
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Bishop,Promotion);
                }
            }
			else {
	            ASSERT(board[dest] != EmptyPiece);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]));
			}
        }
        pawns1 = pawns;
        pawns1.shr(9);
        pawns1.andNot(Bitmap(MAKELONGLONG(0x80808080,0x80808080)));
        pawns1.And(board.occupied[Black]);
        while (pawns1.iterate(dest)) {
            Square loc = dest+9;
            int candidate = (Rank(loc,White) == 7);
            if (candidate) {
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Queen,Promotion);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Knight,Promotion);
                if (ply == 0) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Rook,Promotion);
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Bishop,Promotion);
                }
            }
            else {
	        ASSERT(board[dest] != EmptyPiece);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]));
            }
        }
        pawns1 = pawns;
        pawns1.And(Bearing::rank_mask[1]);
        pawns1.shr8();
        pawns1.andNot(board.all_occupied);
        while (pawns1.iterate(dest)) {
            Square loc = dest + 8;
            moves[NumMoves++] =
                CreateMove(loc,dest,Pawn,EmptyPiece,Queen,Promotion);
            moves[NumMoves++] =
                CreateMove(loc,dest,Pawn,EmptyPiece,Knight,Promotion);
            if (ply == 0) {
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,EmptyPiece,Rook,Promotion);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,EmptyPiece,Bishop,Promotion);
            }
        }
        if (!IsInvalid(board.EnPassantSq())) {
            Square epsq = board.EnPassantSq();
            ASSERT(TypeOfPiece(board[epsq])==Pawn);
            if (File(epsq) != 8 && board[epsq + 1] == whitePawn) {
                Square dest = epsq - 8;
                if (board[dest] == EmptyPiece)
                    moves[NumMoves++] =
                        CreateMove(epsq+1,dest,Pawn,Pawn,InvalidPiece,
                        EnPassant);
            }
            if (File(epsq) != 1 && board[epsq - 1] == whitePawn) {
                Square dest = epsq - 8;
                if (board[dest] == EmptyPiece)
                    moves[NumMoves++] =
                        CreateMove(epsq-1,dest,Pawn,Pawn,InvalidPiece,
                        EnPassant);
            }
        }
    }
    else {
        Bitmap pawns(board.pawn_bits[Black]);
        Bitmap pawns1 = pawns;
        pawns1.shl(7);
        pawns1.andNot(Bitmap(MAKELONGLONG(0x80808080,0x80808080)));
        pawns1.And(board.occupied[White]);
        Square dest;
        while (pawns1.iterate(dest)) {
            Square loc = dest-7;
            int candidate = (Rank(loc,Black) == 7);
            if (candidate) {
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Queen,Promotion);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Knight,Promotion);
                if (ply == 0) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Rook,Promotion);
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Bishop,Promotion);
                }
            }
			else {
	            ASSERT(board[dest] != EmptyPiece);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]));
			}
        }
        pawns1 = pawns;
        pawns1.shl(9);
        pawns1.andNot(Bitmap(MAKELONGLONG(0x01010101,0x01010101)));
        pawns1.And(board.occupied[White]);
        while (pawns1.iterate(dest)) {
            Square loc = dest-9;
            int candidate = (Rank(loc,Black) == 7);
            if (candidate) {
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Queen,Promotion);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Knight,Promotion);
                if (ply == 0) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Rook,Promotion);
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]),Bishop,Promotion);
                }
            }
            else {
	        ASSERT(board[dest] != EmptyPiece);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,TypeOfPiece(board[dest]));
            }
        }
        pawns1 = pawns;
        pawns1.And(Bearing::rank_mask[6]);
        pawns1.shl8();
        pawns1.andNot(board.all_occupied);
        while (pawns1.iterate(dest)) {
            Square loc = dest - 8;
            moves[NumMoves++] =
                CreateMove(loc,dest,Pawn,EmptyPiece,Queen,Promotion);
            moves[NumMoves++] =
                CreateMove(loc,dest,Pawn,EmptyPiece,Knight,Promotion);
            if (ply == 0) {
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,EmptyPiece,Rook,Promotion);
                moves[NumMoves++] =
                    CreateMove(loc,dest,Pawn,EmptyPiece,Bishop,Promotion);
            }
        }

        if (!IsInvalid(board.EnPassantSq())) {
            Square epsq = board.EnPassantSq();
            ASSERT(TypeOfPiece(board[epsq])==Pawn);
            if (File(epsq) != 8 && board[epsq + 1] == blackPawn) {
                Square dest = epsq + 8;
                if (board[dest] == EmptyPiece)
                    moves[NumMoves++] =
                        CreateMove(epsq+1,dest,Pawn,Pawn,InvalidPiece,
                        EnPassant);
            }
            if (File(epsq) != 1 && board[epsq - 1] == blackPawn) {
                Square dest = epsq + 8;
                if (board[dest] == EmptyPiece)
                    moves[NumMoves++] =
                        CreateMove(epsq-1,dest,Pawn,Pawn,InvalidPiece,
                        EnPassant);
            }
        }
    }

    Bitmap pieces(board.occupied[board.Side()]);
    pieces.andNot(board.pawn_bits[board.Side()]);
    while (pieces.iterate(loc)) {
        const ColorType side = PieceColor(board[loc]);
        switch (TypeOfPiece(board[loc])) {
            case EmptyPiece:
            case Pawn:
                break;
            case Knight:
            {
                Bitmap dests(Bearing::knight_attacks[loc]);
                if (side == White)
                    dests.And(board.occupied[Black]);
                else
                    dests.And(board.occupied[White]);
                Square dest;
                while (dests.iterate(dest)) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Knight,TypeOfPiece(board[dest]));
                }
            }
            break;
            case Bishop:
            {
                Square dest;
                Bitmap attacks(DiagAtcksA1(board,loc));
                attacks.Or(DiagAtcksA8(board,loc));
                attacks.And(board.occupied[board.OppositeSide()]);
                while (attacks.iterate(dest)) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Bishop,TypeOfPiece(board[dest]));
                }
            }
            break;
            case Rook:
            {
                Square dest;
                Bitmap rank_attacks(RankAttacks(board,loc));
                rank_attacks.Or(FileAttacks(board,loc));
                rank_attacks.And(board.occupied[board.OppositeSide()]);
                while (rank_attacks.iterate(dest)) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Rook,TypeOfPiece(board[dest]));
                }
            }
            break;
            case Queen:
            {
                Square dest;
                Bitmap attacks(DiagAtcksA1(board,loc));
                attacks.Or(DiagAtcksA8(board,loc));
                attacks.Or(RankAttacks(board,loc));
                attacks.Or(FileAttacks(board,loc));
                attacks.And(board.occupied[board.OppositeSide()]);
                while (attacks.iterate(dest)) {
                    moves[NumMoves++] =
                        CreateMove(loc,dest,Queen,TypeOfPiece(board[dest]));
                }
            }
            break;
            case King:
            {
                Bitmap dests(Bearing::king_attacks[loc]);
                if (side == White)
                    dests.And(board.occupied[Black]);
                else
                    dests.And(board.occupied[White]);
                dests.andNot(Bearing::king_attacks[board.KingPos(
                    board.OppositeSide())]);
                Square sq;
                while (dests.iterate(sq)) {
                    moves[NumMoves++] =
                        CreateMove(loc,sq,King,TypeOfPiece(board[sq]));
                }
            }
            break;
            case InvalidPiece:
                break;
        }
    }
    moves_generated += NumMoves;
    return NumMoves;
}


Move_Generator::Move_Generator( const Board &ABoard,
                                SearchContext *s, 
                                unsigned curr_ply, Move pvMove,
                                int trace)
:
board(ABoard),
context(s),
ply(curr_ply),
moves_generated(0),
phase(HASH_MOVE_PHASE),
losers_count(0),
index(0),
batch_count(0),
forced(0),
hashMove(pvMove),
master(trace)
{
}


unsigned Move_Generator::GenerateEvasions(Move * moves) {
    unsigned n = GenerateEvasionsCaptures(moves);
    n += GenerateEvasionsNonCaptures(moves+n);
    return n;
}


unsigned Move_Generator::GenerateEvasionsNonCaptures(Move * moves) {
    int num_moves = 0;
    const Square kp = board.KingPos(board.Side());
    if (num_attacks == 1) {
        // try to interpose a piece
        if (sliding(board[source])) {
            Bitmap btwn_squares;
            board.between(source,kp,btwn_squares);
            if (!btwn_squares.is_clear()) {
                // blocking pawn moves
                if (board.Side() == White) {
                    Bitmap pawns(board.pawn_bits[White]);
                    pawns.shr8();
                    pawns.andNot(board.all_occupied);
                    Bitmap pawns1(pawns);
                    pawns.And(btwn_squares);
                    Square sq;
                    while (pawns.iterate(sq)) {
                        if (!Bearing::is_pinned(board, board.Side(), whitePawn, sq+8, sq)) {
                            if (Rank(sq,board.Side()) == 8) {
                                // interposition is a promotion
                                moves[num_moves++] = CreateMove(
                                    sq+8, sq, Pawn, EmptyPiece, Queen,Promotion);
                                moves[num_moves++] = CreateMove(
                                    sq+8, sq, Pawn, EmptyPiece, Rook, Promotion);
                                moves[num_moves++] = CreateMove(
                                    sq+8, sq, Pawn, EmptyPiece, Knight, Promotion);
                                moves[num_moves++] = CreateMove(
                                    sq+8, sq, Pawn, EmptyPiece, Bishop, Promotion);
                            }
                            else {
                                moves[num_moves++] = CreateMove(sq+8, sq, Pawn);
                            }
                        }
                    }
                    pawns1.And(Bearing::rank_mask[5]);
                    if (!pawns1.is_clear()) {
                        pawns1.shr8();
                        pawns1.andNot(board.all_occupied);
                        pawns1.And(btwn_squares);
                        while (pawns1.iterate(sq)) {
                            if (!Bearing::is_pinned(board, board.Side(), whitePawn, sq+16, sq))
                                moves[num_moves++] = CreateMove(sq+16,sq,Pawn);
                        }
                    }
                }
                else {
                    Bitmap pawns(board.pawn_bits[Black]);
                    pawns.shl8();
                    pawns.andNot(board.all_occupied);
                    Bitmap pawns1(pawns);
                    pawns.And(btwn_squares);
                    Square sq;
                    while (pawns.iterate(sq)) {
                        if (!Bearing::is_pinned(board, board.Side(), blackPawn, sq-8, sq)) {
                            if (Rank(sq,board.Side()) == 8) {
                                // interposition is a promotion
                                moves[num_moves++] = CreateMove(
                                    sq-8, sq, Pawn, EmptyPiece, Queen, Promotion);
                                moves[num_moves++] = CreateMove(
                                    sq-8, sq, Pawn, EmptyPiece, Rook, Promotion);
                                moves[num_moves++] = CreateMove(
                                    sq-8, sq, Pawn, EmptyPiece, Knight, Promotion);
                                moves[num_moves++] = CreateMove(
                                    sq-8, sq, Pawn, EmptyPiece, Bishop, Promotion);
                            }
                            else {
                                moves[num_moves++] = CreateMove(sq-8, sq, Pawn);
                            }
                        }
                    }
                    pawns1.And(Bearing::rank_mask[2]);
                    if (!pawns1.is_clear()) {
                        pawns1.shl8();
                        pawns1.andNot(board.all_occupied);
                        pawns1.And(btwn_squares);
                        while (pawns1.iterate(sq)) {
                            if (!Bearing::is_pinned(board, board.Side(), blackPawn, sq-16, sq))
                                moves[num_moves++] = CreateMove(sq-16,sq,Pawn);
                        }
                    }
                }
                // other blocking pieces
                Bitmap pieces(board.occupied[board.Side()]);
                pieces.andNot(board.pawn_bits[board.Side()]);
                Square loc;
                while (pieces.iterate(loc)) {
                    switch (TypeOfPiece(board[loc])) {
                        case EmptyPiece:
                        case Pawn:
                            break;
                        case Knight:
                        {
                            Bitmap dests(Bearing::knight_attacks[loc]);
                            dests.And(btwn_squares);
                            Square sq;
                            while (dests.iterate(sq)) {
                                if (!Bearing::is_pinned(board, board.Side(), board[loc], loc, sq))
                                    moves[num_moves++] =
                                        CreateMove(loc,sq,Knight,TypeOfPiece(board[sq]));
                            }
                        }
                        break;
                        case Bishop:
                        {
                            Square dest;
                            Bitmap a1(Bearing::diag_a1_mask[loc]);
                            a1.Or(Bearing::diag_a8_mask[loc]);
                            a1.And(btwn_squares);
                            while (a1.iterate(dest)) {
                                if (board.Clear(loc,dest) &&
                                    !Bearing::is_pinned(board, board.Side(), board[loc], loc, dest))
                                    moves[num_moves++] =
                                        CreateMove(loc,dest,Bishop,TypeOfPiece(board[dest]));
                            }
                        }
                        break;
                        case Rook:
                        {
                            Square dest;
                            Bitmap r(Bearing::file_mask[File(loc)-1]);
                            r.Or(Bearing::rank_mask[Rank(loc,Black)-1]);
                            r.And(btwn_squares);
                            while (r.iterate(dest)) {
                                if (board.Clear(loc,dest) && !Bearing::is_pinned(board, board.Side(), board[loc], loc, dest))
                                    moves[num_moves++] =
                                        CreateMove(loc,dest,Rook,TypeOfPiece(board[dest]));
                            }
                        }
                        break;
                        case Queen:
                        {
                            Square dest;
                            Bitmap r(Bearing::file_mask[File(loc)-1]);
                            r.Or(Bearing::rank_mask[Rank(loc,Black)-1]);
                            r.Or(Bearing::diag_a1_mask[loc]);
                            r.Or(Bearing::diag_a8_mask[loc]);
                            r.And(btwn_squares);
                            while (r.iterate(dest)) {
                                if (board.Clear(loc,dest) &&
                                    !Bearing::is_pinned(board, board.Side(), board[loc], loc, dest))
                                    moves[num_moves++] =
                                        CreateMove(loc,dest,Queen,TypeOfPiece(board[dest]));
                            }
                        }
                        break;
                        case King:
                            break;
                        case InvalidPiece:
                            break;
                    }
                }
            }
        }
    }
    // generate evasive moves that do not capture
    num_moves += GenerateEvasions(moves+num_moves,
				  Bitmap::Not(board.all_occupied));
    return num_moves;
}

unsigned Move_Generator::GenerateEvasions(Move * moves,
  const Bitmap &mask) {
    Square kp = board.KingPos(board.Side());
    int num_moves = 0;
    Bitmap b(Bearing::king_attacks[kp]);
    b.andNot(board.all_pawn_attacks(board.OppositeSide()));
    b.andNot(Bearing::king_attacks[board.KingPos(board.OppositeSide())]);
    b.And(mask);
    Square sq;
    while (b.iterate(sq)) {
	if (num_attacks > 1 || sq != source) {
            // We need to do some extra checking, since the board
            // info on attacks reflects the state before the move,
            // and we need to be sure that the destination square
            // is not attacked after the move.
            int illegal = 0;
            Bitmap it(king_attacks);
            Square atck_sq;
            while (it.iterate(atck_sq)) {
                Piece attacker = board[atck_sq];
                if (sliding(attacker)) {
                    int dir = Bearing::Directions[atck_sq][sq];
                    // check for movement in the direction of the
                    // attacker:
					if (dir != 0 && (kp + dir == sq)) {
                        illegal = 1;
						break;
					}
                }
            }
            if (!illegal && !board.any_attacks(sq, board.OppositeSide())) {
                moves[num_moves++] = CreateMove(kp, sq, King, TypeOfPiece(board[sq]));
            }
        }
    }
    return num_moves;
}

unsigned Move_Generator::GenerateEvasionsCaptures(Move * moves) {
    int num_moves = 0;
    const Square kp = board.KingPos(board.Side());
    king_attacks = board.calc_attacks(kp, board.OppositeSide());
    ASSERT(!king_attacks.is_clear());
    num_attacks = king_attacks.bit_count();
    if (num_attacks == 1) {
        // try to capture checking piece
        source = (Square)king_attacks.first_one();

        ASSERT(source != InvalidSquare);
 	    const PieceType sourcePiece = TypeOfPiece(board[source]);
        Bitmap atcks(board.calc_attacks(source,board.Side()));
        Square sq;
        while (atcks.iterate(sq)) {
            PieceType capturingPiece = TypeOfPiece(board[sq]);
            if (capturingPiece == King) {
                // We can capture with the king only if the piece
                // checking us is undefended.  But always allow a
                // capture *of* the king - for illegal move detection.
                if (sourcePiece == King ||
                !board.any_attacks(source, board.OppositeSide())) {
                    moves[num_moves++] = CreateMove(sq, source, King, sourcePiece);
                }
            }
            else {
                if (!Bearing::is_pinned(board, board.Side(), board[sq], sq, source)) {
                    if (capturingPiece == Pawn &&
                    Rank(source,board.Side()) == 8) {
                        moves[num_moves++] = CreateMove(
                            sq, source, Pawn, sourcePiece, Queen, Promotion);
                        moves[num_moves++] = CreateMove(
                            sq, source, Pawn, sourcePiece, Rook, Promotion);
                        moves[num_moves++] = CreateMove(
                            sq, source, Pawn, sourcePiece, Knight, Promotion);
                        moves[num_moves++] = CreateMove(
                            sq, source, Pawn, sourcePiece, Bishop, Promotion);
                    }
                    else
                        moves[num_moves++] = CreateMove(sq, source, 
							TypeOfPiece(board[sq]), sourcePiece);
                }
            }
        }
        // Bearing::Attack does not return en passant captures, so try
        // this as a special case
        if (board.EnPassantSq() == source) {
            Square dest = source + RankIncr * Direction[board.Side()];
            Piece myPawn = MakePiece(Pawn,board.Side());
            if (File(source) != 8 && board[source + 1] == myPawn) {
                Piece tmp = board[source];
                Piece &place = (Piece &)board[source];
                place = EmptyPiece;               // imagine me gone
                if (!Bearing::is_pinned(board, board.Side(), board[source+1], source + 1, dest))
                    moves[num_moves++] = CreateMove(source + 1, dest, Pawn, Pawn, InvalidPiece, EnPassant);
                place = tmp;
            }
            if (File(source) != 1 && board[source - 1] == myPawn) {
                Piece tmp = board[source];
                Piece &place = (Piece &)board[source];
                place = EmptyPiece;               // imagine me gone
                if (!Bearing::is_pinned(board, board.Side(), board[source-1], source - 1, dest))
                    moves[num_moves++] = CreateMove(source - 1, dest, Pawn, Pawn, InvalidPiece, EnPassant);
                place = tmp;
            }
        }
    }
    // try evasions that capture pieces beside the attacker
    num_moves += GenerateEvasions(moves+num_moves,
				  board.occupied[board.OppositeSide()]);

    return num_moves;
}


unsigned Move_Generator::GenerateChecks(Move * moves) {
    // Note: doesn't at present generate castling moves that check
    const Square kp = board.KingPos(board.OppositeSide());
    Bitmap pieces(board.occupied[board.Side()]);
    pieces.andNot(board.pawn_bits[board.Side()]);
    Square loc;
    int NumMoves = 0;
    while (pieces.iterate(loc)) {
        switch(TypeOfPiece(board[loc])) {
            case Knight:
            {
                Bitmap dests(Bearing::knight_attacks[loc]);
                dests.andNot(board.all_occupied);
                dests.And(Bearing::knight_attacks[kp]);
                Square sq;
                while (dests.iterate(sq)) {
                    moves[NumMoves++] =
                        CreateMove(loc,sq,Knight);
                }
                break;
            }
            case King:
                break;
            case Bishop:
            {
                Bitmap dests(Bearing::diag_a1_mask[kp]);
                dests.Or(Bearing::diag_a8_mask[kp]);
                const int *data = BishopSquares[loc];
                Bitmap b;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Bishop);
                    data++;
                }
                data = BishopSquares[loc] + 8;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Bishop);
                    data++;
                }
                data = BishopSquares[loc] + 16;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Bishop);
                    data++;
                }
                data = BishopSquares[loc] + 24;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Bishop);
                    data++;
                }
                break;
            }
            case Rook:
            {
                Bitmap dests(Bearing::file_mask[File(kp)-1]);
                dests.Or(Bearing::rank_mask[Rank(kp,Black)-1]);
                const int *data = RookSquares[loc];
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Rook);
                    data++;
                }
                data = RookSquares[loc] + 8;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Rook);
                    data++;
                }
                data = RookSquares[loc] + 16;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Rook);
                    data++;
                }
                data = RookSquares[loc] + 24;
                while (*data != 255 && board[*data] == EmptyPiece) {
                    if (dests.is_set(*data) && board.Clear(kp,*data))
                        moves[NumMoves++] =
                            CreateMove(loc,*data,Rook);
                    data++;
                }
                break;
            }
            case Queen:
            {
                int j;
                Bitmap dests(Bitmap::Or(Bearing::file_mask[File(kp)-1],Bearing::rank_mask[Rank(kp,Black)-1]));
                dests.Or(Bearing::diag_a1_mask[kp]);
                dests.Or(Bearing::diag_a8_mask[kp]);
                for (j = 0; j < 4; j++) {
                    const int *data = BishopSquares[loc] + (j*8);
                    while (*data != 255 && board[*data] == EmptyPiece) {
                        if (dests.is_set(*data) && board.Clear(kp,*data))
                            moves[NumMoves++] =
                                CreateMove(loc,*data,Queen);
                        data++;
                    }
                }
                for (j = 0; j < 4; j++) {
                    const int *data = RookSquares[loc] + (j*8);
                    while (*data != 255 && board[*data] == EmptyPiece) {
                        if (dests.is_set(*data) && board.Clear(kp,*data))
                            moves[NumMoves++] =
                                CreateMove(loc,*data,Queen);
                        data++;
                    }
                }
                break;
            }
        }
    }
    // pawn moves
    if (board.Side() == White) {
        Square sq;
        Bitmap pawns(board.pawn_bits[White]);
        pawns.shr8();
        pawns.andNot(board.all_occupied);
        pawns.andNot(Bearing::rank_mask[0]);
        Bitmap pawns1(pawns);
        pawns.And(Bearing::pawn_attacks[kp][White]);
        while (pawns.iterate(sq)) {
            moves[NumMoves++] = CreateMove(sq+8,sq,Pawn);
        }
        pawns1.And(Bearing::rank_mask[5]);
        pawns1.shr8();
        pawns1.andNot(board.all_occupied);
        pawns1.And(Bearing::pawn_attacks[kp][White]);
        while (pawns1.iterate(sq)) {
            moves[NumMoves++] = CreateMove(sq+16,sq,Pawn);
        }
    }
    else {
        Square sq;
        Bitmap pawns(board.pawn_bits[Black]);
        pawns.shl8();
        pawns.andNot(board.all_occupied);
        pawns.andNot(Bearing::rank_mask[7]);
        Bitmap pawns1(pawns);
        pawns.And(Bearing::pawn_attacks[kp][Black]);
        while (pawns.iterate(sq)) {
            moves[NumMoves++] = CreateMove(sq-8,sq,Pawn);
        }
        pawns1.And(Bearing::rank_mask[2]);
        pawns1.shr8();
        pawns1.andNot(board.all_occupied);
        pawns1.And(Bearing::pawn_attacks[kp][Black]);
        while (pawns1.iterate(sq)) {
            moves[NumMoves++] = CreateMove(sq-16,sq,Pawn);
        }
    }
    return NumMoves;
}


#ifdef SMP
int Move_Generator::GenerateAllMoves(NodeInfo *node) {
   // Force the remaining moves to be generated - do it incrementally
   // so we get the correct move ordering and flags:
   int count = 0;
   Move m;
   if (board.CheckStatus() == InCheck) {
      while ((m=NextEvasion()) != NullMove) {
         node->split->moves[count++] = m;
      }
   } else {
      while ((m=NextMove()) != NullMove) {
         node->split->moves[count++] = m;
      }
   }
   batch_count = count;
   // technically this is volatile but we are accessing it here
   // pre-split:
   batch = (Move *)node->split->moves;
   index = 0;
   phase = LAST_PHASE+1;
   return count;
}
#endif

int Move_Generator::GenerateAllMoves(Move *moves,int repeatable) {
    unsigned NumMoves  = 0;
    if (board.CheckStatus() == InCheck) {
        NumMoves = GenerateEvasions(moves);
    }
    else {
        NumMoves += GenerateCaptures(moves+NumMoves);
        NumMoves += GenerateNonCaptures(moves+NumMoves);
    }
    if (repeatable) {
        int scores[Constants::MaxMoves];
        for (unsigned i = 0; i < NumMoves; i++) {
            scores[i] = (int)StartSquare(moves[i]) + (int)(DestSquare(moves[i]) << 7);
            switch (PromoteTo(moves[i])) {
                case Queen:
                    scores[i] &= 0xB000;
                    break;
                case Rook:
                    scores[i] &= 0x8000;
                    break;
                case Bishop:
                    scores[i] &= 0x4000;
                    break;
                default:
                    break;
            }
        }
        Move_Ordering::sort_moves(moves, scores, NumMoves);
    }
    return NumMoves;
}
