// Copyright 1994 by Jon Dart.  All Rights Reserved.

#ifndef _MATERIAL_H
#define _MATERIAL_H

#include "chess.h"
#include "debug.h"
#include "util.h"

class Material
{
	friend class Board;
public:
	Material():
        info(0), total(0), level(0)
	{
	}
	
        enum {KP = 0x100001};
        enum {KBP = 0x100101};
        enum {KBN = 0x100110};
        enum {KRB = 0x101100};
        enum {KRN = 0x101010};
        enum {KRBN = 0x101110};
        enum {KNN = 0x100020};
        enum {KRR = 0x102000};
        enum {KB  = 0x100100};
        enum {KN  = 0x100010};
        enum {KR  = 0x101000};
        enum {KQ  = 0x110000};

	enum {PieceMask = 0x1ffff0};
        
        FORCEINLINE void add_piece(const PieceType p)
        {
           info += masks[(int)p];
           total += PieceValue(p);
           level += levels[(int)p];
           count++;
           ASSERT(level >=0);
        }

	FORCEINLINE void remove_piece(const PieceType p)
        {
          info -= masks[(int)p];
          total -= PieceValue(p);
          level -= levels[(int)p];
          count--;
          ASSERT(level >=0);
        }

        FORCEINLINE void add_pawn()
        {
           info += masks[(int)Pawn];
           total += PAWN_VALUE;
           ++count; 
        }

	FORCEINLINE void remove_pawn()
        {
          info -= masks[(int)Pawn];
          total -= PAWN_VALUE;
          --count;
        }

	// return the total material value:
	uint32 value() const
	{
		return total;
	}
	
	uint32 infobits() const
	{
		return info;
	}
	
	// return the total number of pieces + pawns
	int men() const {
	  return count;
        }
        
	// return the number of pieces (excluding the King)
	int pieceCount() const {
	  return count-pawn_count()-1;
        }
        
        // return the number of pawns
        int pawn_count() const
        {
                return (info & 0xf);
        }
        
        int queen_count() const
        {
                return (info & 0xf0000) >> 16;
        }

        int has_queen() const
        {
                return (info & 0xf0000);
        }

        int rook_count() const
        {
                return (info & 0xf000) >> 12;
        }
        
        int has_rook() const
        {
                return (info & 0xf000);
        }

        int knight_count() const
        {
                return (info & 0xf0) >> 4;
        }
        
        int has_knight() const
        {
                 return (info & 0xf0);
        }

        int bishop_count() const
        {
                return (info & 0xf00) >> 8;
        }

        int has_bishop() const
        {
                 return (info & 0xf00);
        }

	// True if only king and pawns:
	bool no_pieces() const
	{
		return (info & (0xffffffff-0x100000-0xf)) == 0;
	}
        
        // True if no pawns
        bool no_pawns() const
        {
                return (info & 0xf) == 0;
        }
	
	// True if bare king:
	bool king_only() const
	{
		return (info & 0xfffff) == 0;
	}
        
        // This provides a rough measure of the material
        // (used mostly for king safety calculation)
        int material_level() const {
	  return Util::Min(31,level);
        }
	
private:
	void clear();

	int32 info;
	uint32 total;
        int32 level;
        int count;
        static const int32 masks[8];
        static const int32 masks2[8];
        static const int32 levels[8];
};

#endif

