// Copyright 1995 by Jon Dart
// Stand-alone console executable to build "ecodata.h" file
// from "eco" text file.

#include "board.h"
#include "notation.h"
#include "legal.h"
#include "movearr.h"
#include "movegen.h"
#include "globals.h"
#include <iostream>
#include <fstream>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

void charout(unsigned char c)
{
    if (c == '"')
    {
        cout << '\\' << '"';
    }
    else if (c == '\\')
    {
        cout << '\\' << '\\';
    }
    else if (c >= ' ' && (int)c < 127)
    {
        cout << c;
    }
    else
    {
        cout << '\\';
	cout.fill('0');
	int oldwidth = cout.width();
	cout.width(3);
	cout << (oct) << (unsigned)c;
	cout.width(oldwidth);
	cout.fill(' ');
    }
}

static void write_64(char *buf, hash_t h)
{
    struct llong
    {
       int32 lo;
       int32 hi;
    };
    union
    {
        llong split;
        hash_t i64;
    } x;

    x.i64 = h;
    sprintf(buf,"MAKELONGLONG(0x%0x,0x%0x)",x.split.hi,x.split.lo);
}

static int skip_space(ifstream &eco_file)
{
    char c = EOF;
    while (!eco_file.eof())
    {
       c = eco_file.get();
       if (!isspace(c) && (c!='\n'))
       {
	   break;
       }
    }
    return c;
}

void do_eco(ifstream &eco_file)
{
    ColorType side = White;
    Board board;
    int movecount = 0;
    char c;
    // Collect the ECO code
    c = skip_space(eco_file);
    static char code[20];
    int i = 0;
    if (isalpha(c) && isupper(c))
    {
        while (!eco_file.eof() && !isspace(c) && i < 19)
	{
           code[i++] = c;
	   c = eco_file.get();
	}
    }
    code[i] = '\0';
    if (i == 0)
        return;
    cout << '{' << '"' << code << '"' << ", ";
    static char name[255];
    char *name_ptr = name;
    *name_ptr = '\0';
    // Follow the opening line
    unsigned int x;
    while (eco_file.good() && !eco_file.eof())
    {
       // skip to next move
       while (!eco_file.eof())
       {
	  if (c == '\n' || c == '"')
	      break;
	  else if (isalpha(c))
	  {
	     break;
	  }
          c = eco_file.get();
       }
       if (c == '"')
       {
          while (!eco_file.eof())
          {
	     c = eco_file.get();
	     if (c == '"')
                 break;
             else
                 *name_ptr++ = c;
          }
	  *name_ptr = '\0';
          while (!eco_file.eof() && c != '\n')
             c = eco_file.get();
       }
       if (c == '\n')
          break;
       // collect the move text
       char movebuf[20];
       int i = 0;
       while (i < 19 && !eco_file.eof() && 
       	      !isspace(c) && (c != '\n'))
       {
          movebuf[i] = c; ++i;
	  c = eco_file.get();
       }
       if (i == 0)
       	  break;
       movebuf[i] = '\0';
       // parse the move
       Move m = Notation::Value(board,side,movebuf);
       if (IsNull(m) ||
       	   !legal_move(board,StartSquare(m),DestSquare(m)))
       {
           cerr << "Illegal or invalid move: " << movebuf << endl;
           cout << "Illegal or invalid move: " << movebuf << endl;
	   break;
       }
       else
       {
           ++movecount;
       	   x = StartSquare(m);
//	   charout((char)x);
	   x = DestSquare(m);
//	   charout((char)x);
	   board.DoMove(m);
       }
       side = OppositeColor(side);
    }
    char buf[32];
    write_64(buf,board.HashCode());
    cout << buf << " ,";
    if (*name)
       cout << '"' << name << '"';
    else
       cout << '"' << '"';
    cout << "},";
    cout << endl;
}

int CDECL main(int argc, char **argv)
{
   if (argc ==1)
   {
	 cerr << "Usage: makeeco eco_file" << endl;
	 return -1;
   }
   init_options(argv[0]);
   options.book.book_enabled = options.log_enabled = 0;
   init_globals(argv[0],0,0);
   atexit(cleanup_globals);
   
   ifstream eco_file( argv[argc-1], ios::in);
   if (!eco_file.good())
   {
         cerr << "Cannot open file: " << argv[argc-1] << endl;
	 return -1;
   }
   Board b;
   cout << "// This is a machine-generated file.  Do not edit." << endl;
   cout << endl;   
   cout << "#include \"types.h\"" << endl << endl;
   cout << "struct eco_data" << endl;
   cout << "{" << endl;
   cout << "\tconst char *eco;" << endl;
   cout << "\thash_t hash_code;" << endl;
   cout << "\tconst char *opening_name;" << endl;
   cout << "};" << endl;
   cout << endl;
   cout << "const struct eco_data eco_codes[] =" << endl;
   char buf[32];
   write_64(buf,b.HashCode());
   cout << "{{" << '"' << "A00" << '"' << ", " << 
       buf << ", " << 
     '"' << '"' << "}," << endl;
   while (eco_file.good() && !eco_file.eof())
   {
       do_eco(eco_file);
   }
   cout << "{0,0," << '"' << '"' << "}};" << endl;
   return 0;
}
