// Copyright 1996-2004 by Jon Dart.  All Rights Reserved.

// Stand-alone executable to build the binary opening book from
// a text file.

// The book file is a binary file consisting of one or more "pages".
// Each page is a self-contained data structure consisting of
// three portions: a header, a hash table, and an array of book
// entries.  The page size and hash table size are configurable.

// By default each page is 65536 bytes, with a hash table of 2043
// entries.  Each page can hold around 4,000 half-moves.

// The header is defined in bookdefs.h.  It currently contains 4
// fields:
//   version number
//   number of pages
//   page size (bytes)
//   hash table size (number of entries)

// The hash table is an array of unsigned indexes into the book
// entry array.  The high-order 32 bits of the hash code for the
// position modulo the hash table size is used to look up in this
// array the first book entry for the position.

// The book entries themselves are structures defined in bookentr.h.
// Each one is 17 bytes and contains the full hash code, the move
// index, win/loss info for the move, a weight indicating how
// often the computer should play the move, and some learning information.
// Hash entries for each chain in the hash table are stored contiguously
// in the book, and the weight byte for the last entry in each chain has
// its hi bit set.

#include "board.h"
#include "bookentr.h"
#include "bookdefs.h"
#include "hash.h"
#include "bhash.h"
#include "movegen.h"
#include "notation.h"
#include "types.h"
#include "debug.h"
#include "bearing.h"
#include "globals.h"
#include "util.h"
extern "C"
{
#include <string.h>
#include <ctype.h>
#include <stdio.h>
};

#include <fstream>
#include <iostream>
#include <map>

static const int Buffer_Size = 4096;
// max. moves per page
static unsigned max_moves;

enum ResultType {White_Win, Black_Win, DrawResult, UnknownResult};
ResultType last_result = UnknownResult;
ResultType tmp_result;

// number of pages in the book.
static unsigned bookPages = 1; // default
// size of a single page
static unsigned pageSize = 65536;
// hash table size (number of entries).  Each entry is 2 bytes
// Default is 1/15th the number of entries in a page
static unsigned int hashSize;

static char *output_name;

class Book_Entry2;

struct book_info
{
    unsigned next_free;
    unsigned *hash_table; 
    Book_Entry2 **book_moves; // array of Book_Entry2 *
};

static int minFrequency = 0;

// During opening book construction, we add some extra members to
// each book entry type.
class Book_Entry2  : public Book_Entry
{
public:
     Book_Entry2( hash_t hc, byte rec, byte mv_indx,
                  Book_Entry2 *nxt, bool basic, bool last = false);

     Book_Entry2 *next;
     bool basic;
     unsigned int count; // number of times position has occurred
     unsigned int white_wins; // wins for White
     unsigned int black_wins; // wins for Black
     unsigned int next_index;
};

Book_Entry2::Book_Entry2( hash_t hc, byte rec, byte mv_indx,
                          Book_Entry2 *nxt, bool bas, bool last)
:Book_Entry(hc,rec,mv_indx,last),next(nxt),basic(bas)
{
   white_wins = black_wins = 0;
   count = 1;
   set_basic(bas);
}

struct compareBE
{
  bool operator()(const hash_t &key1, const hash_t &key2) const
  {
    return key1 < key2;
  }
};

static map <uint64, Book_Entry2 *,compareBE>* hashTable = NULL;

static book_info *book_data;


static void
add_move(const Board & board, int move_index, int recommend, bool is_basic)
{
     map<uint64,Book_Entry2 *,compareBE>::const_iterator it =
         hashTable->find(board.HashCode());
     Book_Entry2 *be;
     if (it == hashTable->end())
        be = NULL;
     else
        be = (*it).second;
     if (be == NULL)
     {
        Book_Entry2 *new_entry = new Book_Entry2(board.HashCode(), recommend,
                                             move_index, NULL, is_basic, false);
        (*hashTable)[board.HashCode()] = new_entry;
#ifdef _TRACE
      cout << "inserting: " << 
           " h:" << (hex) << Bitmap(board.HashCode()).hivalue() << 
           Bitmap(board.HashCode()).lovalue() << (dec) << 
           " index = " << (int)move_index << " basic = " << (int)is_basic << 
           " recommend=" << (int)recommend << endl;
#endif
     }
     else {
        Book_Entry2 *p = be;
        int found = 0;
        while (p && !found) {
           if (p->move_index == move_index) {
              ++found; break;
           }
           p = p->next;
        }
        if (found) {
          p->count++;
          if (last_result == White_Win)
             p->white_wins++;
          else if (last_result == Black_Win)
             p->black_wins++;
#ifdef _TRACE
      cout << "updating: " << 
           " h:" << (hex) << Bitmap(board.HashCode()).hivalue() << 
           Bitmap(board.HashCode()).lovalue() << (dec) <<
           " index = " << (int)move_index << " basic = " << (int)is_basic << 
           " winloss=" << p->white_wins-p->black_wins << 
           " count=" << p->count << endl;
#endif
        }
        else {
#ifdef _TRACE
      cout << "inserting: " << 
           " h:" << (hex) << Bitmap(board.HashCode()).hivalue() << 
           Bitmap(board.HashCode()).lovalue() << (dec) << 
           " index = " << (int)move_index << " basic = " << (int)is_basic << 
           " recommend=" << (int)recommend << endl;
#endif
           Book_Entry2 *new_entry = new Book_Entry2(board.HashCode(), recommend,
                move_index, p, is_basic, false);
           new_entry->next = (*hashTable)[board.HashCode()];
           (*hashTable)[board.HashCode()] = new_entry;
        }
     }
}

static void
add_move(Book_Entry2 *new_entry)
{
   hash_t hc = new_entry->hash_code();
   unsigned page = (unsigned)(hc % bookPages);
   // use the high order bits of the hash code as the table probe
   unsigned probe = (unsigned) (hc>>32) % hashSize;
   unsigned hit = book_data[page].hash_table[probe];
   new_entry->next_index = hit;
   {
      if (book_data[page].next_free >= max_moves)
      {
         cerr << "\nError - Too many moves in book " << 
              "(page " << page << ")" << endl;
         exit(-1);
      }
      unsigned next = book_data[page].next_free;
      book_data[page].book_moves[next] = new_entry;
      book_data[page].hash_table[probe] = next;

#ifdef _TRACE
      cout << "inserting pg:" << page << " probe:" << probe << 
           " h:" << (hex) << Bitmap(new_entry->hash_code()).hivalue() << 
           Bitmap(new_entry->hash_code()).lovalue() <<
           (dec) << " i:" <<
           (int)new_entry->get_move_index() << " slot:" << next <<
           " w:" << new_entry->get_recommend() << endl;
#endif
      book_data[page].next_free++;
   }
}

static void write_page(int page,ofstream &book_file)
{
   byte *write_buffer = new byte[Buffer_Size];
   assert(write_buffer);

   // write the header
   struct Book_Header hdr;
   
   hdr.version = Book_Version;
   hdr.num_pages = bookPages;
   hdr.page_capacity = pageSize;
   hdr.hash_table_size = hashSize;
   char buf[Header_Size];
   buf[0] = hdr.version;
   buf[1] = hdr.num_pages;
   // fix endian-ness for multi-byte values:
   *((uint32*)(buf+2)) = swapEndian32((byte*)&hdr.page_capacity);
   *((uint32*)(buf+6)) = swapEndian32((byte*)&hdr.hash_table_size);
   book_file.write(buf,Header_Size);

   // Re-work the hash table.  Right now it holds pointers to the
   // head of each hash chain, linked together with "next" pointers.
   // We are going to rearrange the chains so they are contiguous
   // in the book.  Then the hash table will hold indexes to the start
   // of each block of hash entries - these indexes are counts of
   // how many Book_Entry objects lie before the start of the desired
   // block.

#ifdef _TRACE
   cout << "write page " << page << endl;
#endif
   int running_count = 0;
   unsigned i;
   for (i = 0; i < hashSize; i++)
   {
      unsigned ht_entry = book_data[page].hash_table[i];
      if (ht_entry == INVALID)
      {
         book_file.put((char)(INVALID % 256));
         book_file.put((char)(INVALID / 256));
      }
      else
      {
         book_file.put((char)(running_count % 256));
         book_file.put((char)(running_count / 256));
#ifdef _TRACE
	 //         cout << "slot " << i << " writing hash entry " << running_count << endl;
#endif

         unsigned indx = ht_entry;
         Book_Entry2 *entry = NULL;
         while (indx != INVALID)
         {
           entry = book_data[page].book_moves[indx];
           //cout  << "Hash = " << (hex) << Bitmap(entry->hash_code()).hivalue() << 
           //Bitmap(entry->hash_code()).lovalue() <<
           //" white wins = " << entry->white_wins <<
           //" black wins = " << entry->black_wins << endl;
           if (Bitmap::And(Bitmap(entry->hash_code()),Bitmap(1)).is_clear())
           {
              // white is on move
              entry->set_winloss((((int)entry->white_wins-(int)entry->black_wins)*100)/(int)entry->count);
           }
           else
           {
              entry->set_winloss((((int)entry->black_wins-(int)entry->white_wins)*100)/(int)entry->count);
           }
           entry->set_frequency(entry->count);
           entry->learn_score = 0.0F;
           indx = entry->next_index;
           ++running_count;
         }
         if (entry)
           entry->set_last();
      }
   }

   // write the book moves

   unsigned index = 0;
   for (i = 0; i < hashSize; i++)
   {
      unsigned ht_entry = book_data[page].hash_table[i];
      while (ht_entry != INVALID)
      {
         Book_Entry2 *book_entr = 
             book_data[page].book_moves[ht_entry];
         Book_Entry2 new_entry = *book_entr;
         // fix multi-byte values for endian-ness:
         new_entry.my_hash_code = swapEndian64((byte*)&(book_entr->my_hash_code));
         new_entry.learn_score = (float)swapEndian32((byte*)&(book_entr->learn_score));
         if (index + (unsigned)Entry_Size <= (unsigned)Buffer_Size)
         {
                memcpy(write_buffer + index,
                           &new_entry, Entry_Size);
                index += Entry_Size;
         } 
         else
         {
                unsigned to_go = Buffer_Size - index;
                byte *buf = (byte*)&new_entry;
                if (to_go)
                {
                   memcpy(write_buffer + index,buf,to_go);
                }
                book_file.write((const char*)write_buffer, Buffer_Size);
                memcpy(write_buffer, buf + to_go, Entry_Size - to_go);
                index = Entry_Size - to_go;
         }
         Book_Entry2 *old_entry = book_entr;
         ht_entry = book_entr->next_index;
         delete old_entry;
      }
   }
   if (index)
      book_file.write((const char*)write_buffer, index);
   long bookSize = book_file.tellp();
   while (bookSize % pageSize != 0L)
   {
      book_file.put('\0');
      bookSize++;
   }
   delete[] write_buffer;

   //   cout << "page " << page << ": " <<
   //    book_data[page].next_free << " moves." << endl;
}

static void write_book()
{
   ofstream book_file(output_name,
                       ios::out | ios::trunc | ios::binary);
   long total_moves = 0L;
   for (unsigned i = 0; i < bookPages && book_file.good(); i++)
   {
      write_page(i,book_file);
      total_moves += book_data[i].next_free;
   }
   cout << total_moves << " total moves." << endl;
   book_file.close();
}

int CDECL main(int argc, char **argv)
{
   char book_name[MAX_PATH];
   
   init_options(argv[0]);
   options.log_enabled = 0;
   init_globals(argv[0],0,0);
   atexit(cleanup_globals);

   hashTable =  new map< uint64, Book_Entry2*,compareBE>;

   output_name = NULL;
   int arg = 1;
   hashSize = 0;
   while (arg < argc)
   {
      if (*argv[arg] == '-')
      {
         char c = argv[arg][1];
         switch(c)
         {
         case 'p': /* page size */
           ++arg;
           pageSize = atoi(argv[arg]);
           if (pageSize == 0)
           {
              cerr << "Illegal page size" << endl;
              exit(-1);
           }
           break;
         case 'o': /* output file name */
           ++arg;
           output_name = strdup(argv[arg]);
           break;
         case 'n': /* number of pages (power of 2)*/
           ++arg;
           bookPages = atoi(argv[arg]);
           if (bookPages == 0)
           {
              cerr << "Illegal size specified for -n" << endl;
              exit(-1);
           }
           else if (bookPages > 255) {
              cerr << "Error: max book pages is 255" << endl;
              exit(-1);
           }
           break;
        case 'm':
           ++arg;
           minFrequency = atoi(argv[arg]);
           break;
        case 'h': /* hash size */         
           ++arg;
           hashSize = atoi(argv[arg]);
           if (hashSize == 0)
           {
              cerr << "Illegal hash size" << endl;
              exit(-1);
           }
           break;
         default:
           cerr << "Illegal switch: " << c << endl;
           cerr << "Usage:" << endl;
           cerr << "makebook -n <bits> -p <page size> -h <hash size>" << endl;
           cerr << "         -m <min frequency> -o <output file> <input file>" << endl;
           exit(-1);
         }
         ++arg;
      }
      else
          break;
   }
   if (hashSize == 0) // no hash size specified
      hashSize = Util::Max((pageSize/Entry_Size)/15,1);
   max_moves = (pageSize - hashSize*2 - Header_Size)/Entry_Size;
   if (max_moves < 5)
   {
      cerr << "Insufficient page size.  Use a larger value for -p"
          << endl;
      exit(-1);
   }
   else if (max_moves*Entry_Size < pageSize/4)
   {
      cerr << "Warning: hash table will consume > 1/4 of page." << endl;
      cerr << "Suggest you use a larger page size or smaller hash table size." << endl;
   }

   if (arg < argc)
      strcpy(book_name, argv[arg]);
   else
      strcpy(book_name, "book.txt");
   if (!output_name)
   {
      output_name = (char*)strdup("book.bin");
   }
   // Initialize the "book_data" structures that will hold
   // the opening book info during book construction.
   
   book_data = new book_info[bookPages];
   unsigned i;
   for (i = 0; i < bookPages; i++)
   {
      book_data[i].book_moves = new Book_Entry2 * [max_moves];
      book_data[i].hash_table = new unsigned[hashSize];
      book_data[i].next_free = 0;
   }
   
   for (unsigned j = 0; j < bookPages; j++)
   {
      for (i = 0; i < hashSize; i++)
      {
         book_data[j].hash_table[i] = INVALID;
      }
      for (i = 0; i < max_moves; i++)
      {
         book_data[j].book_moves[i] = NULL;
      }
   }
   int dots = 0;
   bool is_basic = true;
   while (arg < argc)
   {
   strcpy(book_name, argv[arg++]);
   fstream infile;

   infile.open(book_name, ios::in);
   if (!infile.good())
   {
      cerr << "Can't open book file: " << argv[arg-1] << endl;
      return -1;
   }

   char buf[256];
   char movebuf[20];
   unsigned line = 0;
   char *q;
   Board board;
   Board tmpboard;
   int recommend;
   Move moves[Constants::MaxMoves];

   while (!infile.eof())
   {
      recommend = 50;
      infile.getline(buf, 256);
#ifdef _TRACE
      cout << buf << endl; 
#endif
      line++;
      char *p = buf;
      while (isspace(*p))
             p++;
      switch (*p)
      {
      case '\0':
            continue;
      case ';':
            {
               if (strstr(p,"1/2-1/2"))
                   last_result = DrawResult;
               else if (strstr(p,"0-1") || strstr(p,"0:1"))
                   last_result = Black_Win;
               else if (strstr(p,"1-0") || strstr(p,"1:0"))
                   last_result = White_Win;
               else 
                   last_result = UnknownResult;
               continue;
            }
      case 'm':
            tmpboard = board;
            tmp_result = last_result;
            continue;
      case 'r':
            last_result = tmp_result;
            board = tmpboard;
            continue;
      case '-':
            last_result = UnknownResult;
            board.Reset();
	    break;
      default:
          {
            while (*p)
            {
               while (isspace(*p))
                      p++;
               if (*p == '\0')
                  break;
               q = movebuf;
               int count = 0;
               while (!isspace(*p) && *p && count < 19)
               {
                      *q++ = *p++;
                      ++count;
               }
               *q = '\0';
               Move move = Notation::Value(board, board.Side(), movebuf);
               if (IsNull(move))
               {
                      cerr << endl << "Illegal move in file, line " << 
                         line << " (" << movebuf << ")" << endl;
                      infile.close();
                      return -1;
               }
#ifdef _TRACE
               cout << movebuf << endl;
#endif
               // check (pseudo) legality:
               Move_Generator mg(board, NULL, 0, NullMove, 0);
               int found = 0;
               int move_indx = 0;

               int n = mg.GenerateAllMoves(moves, 1 /* repeatable */);

               for (int i = 0; i < n; i++)
               {
                      if (MovesEqual(moves[i],move))
                      {
                         move_indx = i;
                         found++;
                         break;
                      }
               }
               if (!found)
               {
                       cerr << endl << "Illegal move in file, line " << line << " ("
                          << movebuf << ")" << endl;
                       infile.close();
                       return -1;
               }
               dots++;
	       /**               if (dots % 50 == 0)
                       cout << '.' << (flush);
               if (dots > 10000)
               {
                       cout << endl;
                       dots = 1;
		       }**/
               recommend = 50;
               while (isspace(*p))
                       p++;
               if (*p)
               {
                       if ((*p >= '0') && (*p <= '9'))
                       {
                           recommend = 10*(*p - '0');
                           p++;
                       }
               }
               add_move(board, move_indx, recommend,is_basic);
               board.DoMove(move);
            }
          }
      }
   }
   infile.close();
   is_basic = false;
   }

   // Iterate through the hash table, picking out moves that meet
   // the "minFrequency" test

   map< uint64, Book_Entry2 *,compareBE>::const_iterator it = hashTable->begin();
   while (it != hashTable->end()) {
     Book_Entry2* be = (*it).second;
     while (be) {
        if ((be->count >= minFrequency) || be->basic) {
           add_move(be);
        }
        be = be->next;
     }
     ++it;
   }

   cout << endl;
   write_book();
   cout << "Total book size: " <<
       pageSize*bookPages/1024 << "K. ";
   cout << "Capacity is about " <<
       max_moves*bookPages << " moves." << endl;
   
   return 0;
}

























































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































