// Copyright 1994, 1995, 2000 by Jon Dart.  All Rights Reserved.
// $Revision: 1.4 $

#ifndef _LOG_H
#define _LOG_H

#include "board.h"
#include "search.h"
#include "bookinfo.h"
#include <fstream>
#include <string.h>
#include <string>
#include "arasvec.h"

using namespace std;

class Log;

class Log_Entry
{
     // Maintains info on a move made so far in the game.            

     public:
     
     // create a log entry. 
     // "move_image" is the string representation of the move.
     Log_Entry(const Board_State &state,
               const Move &move,
               const char *move_image,
               uint64 move_count,
               int score,
               int depth);
             
     // create a null log entry.  Used only to initialize storage.
     Log_Entry();
             
     ~Log_Entry();
     
     const Move &move() const
     {
         return my_move;
     }
     
     const char *image() const
     {
         return my_image.c_str();
     }
     
     const Board_State &state() const
     {
        return my_state;
     }
     
     const char *result() const
     {
       return my_result.c_str();
     }
     
     void setResult( const char *result)
     {
       my_result = result;
     }
     
     int score() const
     {
         return my_score;
     }
     
     uint64 move_count() const
     {
         return my_move_count;
     }
     
     const Book_Info &get_book_info() const
     {
         return bi;
     }
     
     void set_book_info(const Book_Info &book_inf)
     {
         bi = book_inf;
     }
     
     int depth() const
     {
         return my_depth;
     }
    
     int operator == ( const Log_Entry &l ) const;
     int operator != ( const Log_Entry &l ) const;

     bool operator < (const Log_Entry &le) const
     {
        return my_state.hashCode < le.my_state.hashCode;
     }
     

     private:
             
     Board_State my_state;
     Move my_move;
     string my_image;
     uint64 my_move_count;
     int my_score;
     int my_depth;
     string my_result;
     Book_Info bi;
};

class Log : public ArasanVector<Log_Entry>
{
     // maintains a log of moves made in the game so far.  Unlike the
     // Move_Array (see movearr.h), moves are not added and removed
     // during the search process.  The log maintains a size, which
     // is the total number of moves ever added, and a current position,
     // which is normally == to its size, but may be less if the
     // user has "taken back" moves.

     public:         
        
     Log();
     // create a log.
             
     virtual ~Log();         

     // Add a move to the log.  If "toFile" is true, also record it
     // in the log file.  This is called before the move has been
     // made.
     void add_move( Board &board,
                    const Move &emove,
                    const char *move_image,
                    const Statistics *stats,
                    Book_Info *book_inf,
                    int toFile);
             
     // remove the most recently added move to the log.
     void remove_move()
     {
        remove_last();
        --my_current;
     }
             
     // Return the number of the last move made.  "Backing up"
     // through the moves changes current w/o changing num_moves.        
     unsigned current() const
     {
             return my_current;
     }
     
     // Return the total number of moves made.
     unsigned num_moves() const
     {
       return length();
     }
     
     // Decrement the "current" move by one.
     int back_up();
     
     // Advance the "current" move pointer.
     int go_forward();
     
     // Reset the "current" position to the start of the game, w/o
     // altering the file or clearing the log.    
     void reset();
     
     // return the last move in the log.  The log must be non-empty.
     const Move &last_move() const;

     // return the nth move in the log.  0 <= n <= num_moves - 1.
     const Move &move( int n ) const
     {
        return (*this)[n].move();
     }
        
     // remove everything from the log
     void clear();
             
     void write_header();

     void write(const char *);

     void write_eol();
     
     void setResult(const char *result);
     
     const char *getResult() const;
             
     void setEnabled(int enable) {
        enabled = enable;
     }

private:

     void flush();
     int my_current;
     ofstream log_file;     
     char buf[256];
     int enabled;
};

#endif

