// Copyright 1994, 1995, 2000 by Jon Dart.  All Rights Reserved.

#if defined(_MSC_VER)
#pragma optimize("g",off)
#endif
#include "log.h"
#include "notation.h"
#include "globals.h"
#include "debug.h"
#include "scoring.h"
#include <stddef.h>
#include <string.h>
#include <ctype.h>
#ifdef _WIN32
   #include <windows.h>
   #include <io.h>
#endif
#include <stdio.h>

#define LOG_FILE_NAME "arasan.log"

Log_Entry::Log_Entry(const Board_State &state, const Move &move,
                     const char *move_image, uint64 move_count, int score,int depth)
: my_state(state),
my_move(move),
my_image(move_image),
my_move_count(move_count),
my_score(score),
my_depth(depth),
my_result(""),
bi()
{
}

Log_Entry::Log_Entry()
{
}

Log_Entry::~Log_Entry()
{
}

static char *time_image(time_t time)
{
   static char buf[80];
   time /= 100; // time is in centiseconds.
   int hrs = (int)(time / 3600);
   int mins = (int)((time - (hrs*3600))/60);
   int secs = (int)(time - (hrs*3600) - (mins*60));
   sprintf(buf,"%02d:%02d:%02d",hrs,mins,secs);
   return buf;
}

Log::Log()
{
   my_current = 0;
   *buf = '\0';
   enabled = 0;
   string log_path;
   if (options.log_enabled)
   {
      if (options.log_pathname == "") {
         // find the last part of the program pathname
	log_path = derivePath(LOG_FILE_NAME);
      }
      else {
	log_path = options.log_pathname;
      }
 
      if (options.log_append) {
        log_file.open(log_path.c_str(),ios::out|ios::app);
      }
      else {
        log_file.open(log_path.c_str(),ios::out|ios::in|ios::trunc);
      }
      if (!log_file.is_open() || !log_file.good())
      {
#ifdef _WIN32
         MessageBox(NULL,"Can't open log file.  Game moves will not be saved.","",MB_OK);
#else
	 cerr << "Warning: can't open log file " << log_path << endl;
#endif
       }
       else
          ++enabled;
   }
}

Log::~Log()
{
   if (enabled && log_file.good())
   {
      log_file << buf << endl;
      log_file.close();
   }
}

void Log::add_move( Board &board, const Move &emove,
                    const char *move_image, const Statistics *stats,
                    Book_Info *info,
                    int toFile)
{
   // adding a move clears the "result" field
   setResult("");

   Board_State state = board.state;

   Log_Entry entry(
       state, emove, move_image,
       stats ? stats->num_moves : 0,
       stats ? stats->display_value : 0,
       stats ? stats->depth : 0);
   if (info) {
      entry.set_book_info(*info);
   }

   // moves are always added at the "current" position.

   if (my_current >= length())
   {
     append(entry);
   }
   else
   {
     set(my_current,entry);
   }
   my_current++;
   // Adding a move at a given point removes any moves
   // after it in the array.
   if (length() > my_current)
   {
      truncate(my_current);
   }
   char num[10];
   *buf = '\0';
   sprintf(num,"%d. ",(num_moves()-1)/2 + 1);
   strcat(buf,num);
   strcat(buf,move_image);
   int len = (int)strlen(buf);
   char *p = buf + len;
   for (int i = 0; i < 15-len; i++)
      *p++ = ' ';
   *p = '\0';
   if (stats)
   {
      char statbuf[256];
      if (stats->num_moves == 0)
         strcpy(statbuf," (book)");
      else
      {
         char score_buf[20];
         Scoring::print_score(stats->display_value,score_buf);
#ifdef _WIN32
         sprintf(statbuf,"\t%s    %d\t%I64d\t%s",
#else
         sprintf(statbuf,"\t%s    %d\t%lld\t%s",
#endif
                 time_image(stats->elapsed_time),
                 stats->depth,
                 stats->num_nodes,score_buf);
                 
         char *p = strchr((char*)stats->best_line_image,' ');
         // We skip the first move of the pv. and only show the
         // predicted continuation
         if (p != NULL)
         {
            while (isspace(*p)) p++;
            strcat(statbuf,"\t");
            char *q = statbuf+strlen(statbuf);
            char *breakpt = NULL;
            int count = 0;
            // copy up to 40 chars of the PV
            while (*p && count < 40)
            {
               *q = *p;
               if (isspace(*p)) breakpt = q;
               q++; p++; ++count;
            }
            if (breakpt) 
              *breakpt = '\0';
            else
              *q = '\0';
         }
      }
      if (stats->value == 9999)
      {
         // side to move has delivered checkmate
         strcat(statbuf," mate");
         if (board.Side() == White)
            setResult("1-0");
         else
            setResult("0-1");
      }
      else if (stats->state == Stalemate)
      {
         strcat(statbuf," stalemate");
         setResult("1/2-1/2");
      }
      else if (stats->state == Resigns)
      {
         if (board.Side() == White)
            setResult("0-1");
         else
            setResult("1-0");
         strcat(statbuf," resigns");
      }
      else if (stats->state == Draw)
      {
         strcat(statbuf," draw");
         setResult("1/2-1/2");
      }
      strncat(buf,statbuf,255-strlen(buf));
      buf[255] = '\0'; // ensure null termination
   }
   assert(strlen(buf)<256);
   if (!enabled || !toFile || !log_file.is_open())
      return;
   log_file << buf << endl;
   *buf = '\0';
}

const Move &Log::last_move() const
{
   ASSERT(current());
   return move(current()-1);
}

void Log::setResult(const char *result)
{
   if (!empty())
   {
      last().setResult(result);
   }
}

const char *Log::getResult() const
{
   if (!empty())
   {
      return last().result();
   }
   else
      return "";
}

int Log::back_up()
{
   if (my_current >0)
   {
      --my_current;
      return 1;
   }
   else
      return 0;
}

int Log::go_forward()
{
   if (my_current < length())
   {
      ++my_current;
      return 1;
   }
   else
      return 0;
}

void Log::reset()
{
   my_current = 0;
}

void Log::clear()
{
   write_eol();
   removeAll();
   my_current = 0;
}

void Log::write_header()
{
   if (enabled && log_file.is_open())
   {
      flush();
      static char header1[] = "Arasan version ";
      static char header2[] = "   move          time     depth\tnodes\tvalue\tpredicted";
      log_file << header1 << Arasan_Version << " game log" << endl;
      log_file << header2 << endl;
   }
}

void Log::write(const char *s)
{
   if (enabled && log_file.is_open())
   {
      flush();
      log_file << s;
   }
}

void Log::write_eol()
{
   if (enabled && log_file.is_open())
      log_file << endl;
}

void Log::flush()
{
   if (enabled && strlen(buf) > 0 && log_file.is_open())
   {
      log_file << buf << endl;
   }
   *buf = '\0';
}
