// Copyright 1994-2002, 2004 by Jon Dart.  All Rights Reserved.

#include "globals.h"
#include "book.h"
#include "util.h"
#include "scoring.h"
#include "config.h"

void learn(const Board &board, int rep_count)
{
   // First, do position learning. If our score has dropped or
   // jumped recently, store the position in the position learning
   // book (basically, a permanent hash table).

   if (theLog->current() == 0) return;
   const Log_Entry &last_entry = (*theLog)[theLog->current()-1];
   
   int last_score = last_entry.score();
   int last_depth = last_entry.depth();
   const Book_Info &book_info = last_entry.get_book_info();
   if (options.learning.position_learning && !book_info.is_valid() && theLog->current() >= 3)
   {
      const Log_Entry &prev = (*theLog)[theLog->current()-3];
      const Book_Info &book_info = prev.get_book_info();
	  int diff1 = 0;
      if (!book_info.is_valid()) { 
        int prev_score = prev.score();
        diff1 = Util::Abs(last_score - prev_score);
      }
      int diff2 = 0;
      if (theLog->current()>=5) {
        const Log_Entry &prev = (*theLog)[theLog->current()-5];
        const Book_Info &book_info = prev.get_book_info();
        if (!book_info.is_valid()) {
          int prev_score = prev.score();
          diff2 = Util::Abs(last_score - prev_score);
        }
      }
      int score_threshold = (64*options.learning.position_learning_threshold)/100;
      if (!book_info.is_valid() && 
        last_depth > options.learning.position_learning_minDepth &&
        theLog->current() <= POSITION_MAX_PLY &&
        (diff1 > score_threshold || diff2 > score_threshold))
      {
         // last 2 or more moves were not from book, and score has changed
         // significantly.
         Hash_Entry pos(board.HashCode(rep_count),last_depth*DEPTH_INCREMENT,
                   0, /* age, N/A */
                   Position_Info::Valid,
                   board.CheckStatus() == InCheck, /*in_check*/
		   last_score,1,
                   last_entry.move());
         position_book->add(board,pos);
         char msg[200];
         char score_buf[20];
         Scoring::print_score(last_score,score_buf);
         sprintf(msg,"learning position, score = %s depth = %d\n",
           score_buf,last_depth);
         theLog->write(msg);
      }
   }

   // If we are a few moves out of the opening book, and if 
   // our score is considerably off zero, update the book to
   // avoid or steer towards this position in the future.
   //
   if (!options.learning.score_learning) return;
   int count = options.learning.score_learning_horizon;
   int last_book_move = -1;
   int min_score = Constants::BIG, max_score = -Constants::BIG;
   int i;
   for (i = theLog->current()-1; i>=0 && count>=0; i-=2,--count)
   {
      const Log_Entry &entry = (*theLog)[i];
      const Book_Info &book_info = entry.get_book_info();
      if (book_info.is_valid()) // it was a book move
      {
         if (count > 0) return; // not at learning threshold
         last_book_move = i;
         break;
      }
      else
      {
         if (entry.score() < min_score)
            min_score = entry.score();
         if (entry.score() > max_score)
            max_score = entry.score();
      }
   }
   if (last_book_move < 0) return;
   if ((last_depth < options.learning.score_learning_minDepth) ||
       (Util::Abs(last_score) < options.learning.score_learning_threshold))
   {
      return; // we don't have an accurate score, or the
              // most recent score is close to even
   }
   theLog->write("learning"); theLog->write_eol();
   float learn_factor = (last_score*1.0F)/64.0F;
   int first = 1;
   for (i = last_book_move; i >= 0; i-=2)
   {
      const Log_Entry &entry = (*theLog)[i];
      const Book_Info &book_info = entry.get_book_info();
      if (book_info.is_valid())
      {
         if (!first)
         {
            learn_factor /= 1.0F*book_info.get_total_moves();
         }
         opening_book->update(book_info,learn_factor);
         first = 0;
      }
   }
}

void learn_result(ColorType computerSide, int result, int ratingDiff)
{
   int last_book_move = -1;
   int i;
   for (i = theLog->current()-1; i>=0; i--)
   {
      const Log_Entry &entry = (*theLog)[i];
      const Book_Info &book_info = entry.get_book_info();
      if (book_info.is_valid()) // it was a book move
      {
         last_book_move = i;
         break;
      }
   }
   if (last_book_move < 0) return;
   int diff = 0;
   // don't learn from low-rated opponents
   if (ratingDiff < -250) return;
   if (result == 0) {
     if (ratingDiff>100) 
       diff = 1; // draw from hi-rated opponent
     else
       return; // regular draw
   }
   else if (result == 1) {
     diff = 1;
   }
   else {
     diff = -1;
   }
   if (computerSide == Black) diff = -diff;
   Book_Location loc;
   while (i>0) {
     const Log_Entry &entry = (*theLog)[i];
     Book_Info &book_info = (Book_Info&)entry.get_book_info();
     book_info.get_location(loc);
     int before = book_info.get_learned_result();
     book_info.update_learned_result(diff);      
     int after = book_info.get_learned_result();
     char msg[100];
     opening_book->update(&book_info);

     sprintf(msg,"Learning result .. page = %d index = %d current value = %d, new value = %d\n",(int)loc.page, (int)loc.index, before, after);
     theLog->write(msg);

     // see if there's an alternative book move at this position
     if (opening_book->reader->book_move_count(book_info.hash_code())> 1)
       return; // all done
     i-=2;
   }
}

