// Copyright 1999-2005 Jon Dart. All Rights Reserved.

#include "hash.h"
#include "util.h"
#include "debug.h"
#include "constant.h"
#include "globals.h"
#include <malloc.h>
#include <memory.h>
#include <stddef.h>

static Dual_Hash_Entry *hashTable = NULL;
static void *hashTable_allocated = NULL;
int hashSize;
static const int MaxRehash = 3;
static int refCount = 0;
static int hash_init_done = 0;

void initHash(int size)
{
    if (!hash_init_done) {
       hashSize = size/2 + (size/2 % 2);             // assure it's even
       hashTable_allocated = malloc(sizeof(Dual_Hash_Entry)*hashSize+256);
       hashTable = (Dual_Hash_Entry*)ALIGN_POINTER(hashTable_allocated,128);
       clearHash();
       hash_init_done++;
       ++refCount;
    }
}


void freeHash()
{
    if (--refCount == 0) {
        free(hashTable_allocated);
        hash_init_done = 0;
    }
}


void clearHash()
{
    memset(hashTable,'\0',sizeof(Dual_Hash_Entry)*hashSize);
}


Position_Info::ValueType searchHash(const Board& b,hash_t hashCode,
int alpha, int beta, int ply,
int depth,
Position_Info &entry
)
{
    int i;
    int probe = quickmod(hashCode,hashSize);
    Dual_Hash_Entry *p = (Dual_Hash_Entry*)&(hashTable[probe]);
    Hash_Entry *hit = NULL;
    if (p->replace.hash_code() == hashCode) {
        entry.copy(p->replace);
        hit = &p->replace;
        if (p->replace.depth() >= depth) {
            // usable depth
            goto hash_hit;
        }
        // got a hit but depth not sufficient: keep the
        // data, but try the depth table too
    }
    for (i = MaxRehash; i != 0; --i) {
        if (p->depth.hash_code() == hashCode) {
            entry.copy(p->depth);
            hit = &p->depth;
            if (p->depth.depth() >= depth) {
                // usable depth
                goto hash_hit;
            }
            else {
                break;
            }
        }
        probe++; p++;
        if (probe >= hashSize) {
            probe = 0; p = hashTable;
        }
    }
    // If we got a hash hit but insufficient depth return here:
    if (hit) return Position_Info::Invalid;
    // No regular hash hit. Try the permanent hash table:
    if (position_book && options.learning.position_learning &&
        position_book->lookup(b,hashCode,entry)) {
        goto hash_hit;
    }
    return Position_Info::NoHit;
hash_hit:
    //
    // Whether we can use the hash value or not depends on
    // the flags:
    //
    int value = entry.value();
    switch (entry.type()) {
        case Position_Info::Valid:
            // If this is a mate score, adjust it to reflect the
            // current ply depth.
            //
            if (value >= Constants::BIG-200) {
                value -= ply;
                entry.setValue(value);
            }
            else if (value <= -Constants::BIG+200) {
                value += ply;
                entry.setValue(value);
            }
            return Position_Info::Valid;
        case Position_Info::LowerBound:
            if (value >= beta)
                return Position_Info::LowerBound;
            break;
        case Position_Info::UpperBound:
            if (value <= alpha)
                return Position_Info::UpperBound;
            break;
        default:
            break;
    }                                             // end switch
    return Position_Info::Invalid;
}


void storeHash(const Hash_Entry &newPos,Statistics &stats) {
    int probe = quickmod(newPos.hash_code(),hashSize);
    Dual_Hash_Entry *p = (Dual_Hash_Entry*)&(hashTable[probe]);
    p->replace = newPos;

    Hash_Entry *best = NULL;
    int minDepth = 10000;
    ASSERT(Util::Abs(newPos.value()) <= Constants::BIG);
    // Of the positions that hash to the same locations
    // as this one, find the best one to replace.
    for (int i = MaxRehash; i != 0; --i) {
        Hash_Entry *q = &p->depth;

        if (q->hash_code() == (hash_t)0 || q->age() != newPos.age()) {
            *q = newPos; 
            stats.hash_inserts++;
            stats.new_hash_inserts++;
            return;  // stored
        }
        if (q->hash_code() == newPos.hash_code()) {
            // same position
            q->update_best_move(newPos);
            if (q->depth() <= newPos.depth()) {
               best = q;
               break;
           }
        }
        if (q->depth() < minDepth) {
            minDepth = q->depth();
            if (newPos.depth() >= minDepth) best = q;
        }
        probe++; p++;
        if (probe >= hashSize) {
            probe = 0; p = hashTable;
        }
    }
    if (best != NULL) {
        if (best->age() != newPos.age())
	  stats.new_hash_inserts++;
        *best = newPos;
        stats.hash_replaces++;
        return;
    }
    else {
      stats.hash_inserts_failed++;
    }
}
