// Copyright 1996-2008 by Jon Dart.  All Rights Reserved.

#include "chess.h"
#include "globals.h"
#include "bearing.h"
#include "movegen.h"
#include "scoring.h"
#include "pinfo.h"
#include "config.h"
#include "bitprobe.h"
#include "tbprobe.h"
#include <stdio.h>

char cmd_buf[CMD_BUF_SIZE];
int cmd_buf_count;

Book *opening_book = NULL;
PositionBook *position_book = NULL;
Log *theLog = NULL;
MoveArray *game_moves;
static string programPath;
Options options;
int EGTBMenCount;
int ncpus = 1;
LockDefine(hash_lock);

#if defined(_MSC_VER) && (_MSC_VER <= 1300)
// workaround for lack of << operator on uint64
// Microsoft support article 168440
extern std::ostream& operator<<(std::ostream& os, uint64 i ) {
    char buf[20];
    sprintf(buf,"%I64d", i );
    os << buf;
    return os;
}
#endif

string derivePath(const char *fileName) {
   return derivePath(programPath.c_str(),fileName);
}

string derivePath(const char *base, const char *fileName) {
    string result(base);
    size_t pos;
#ifdef _WIN32
    pos = result.rfind('\\',string::npos);
#else
    pos = result.rfind('/',string::npos);
#endif
    if (pos == string::npos) {
      return fileName;
    }
    else {
      return result.substr(0,pos+1) + fileName;
    }
}

void init_options(const char *programPath) {
    string rcPath(derivePath(programPath,"arasan.rc"));
    // try to read arasan.rc file
    options.init(rcPath.c_str());
#ifndef _WIN32
    // Also read .rc from the user's HOME,
    // if there is one.
    if (getenv("HOME")) {
      rcPath = getenv("HOME");
      rcPath += '/';
      rcPath += "arasan.rc";
      options.init(rcPath.c_str());       
    }
#endif
}

int init_globals(const char *pathName, int open_trace,
int init_bitbases)
{
    // must do this first, many things depend on it:
    Bitmap::init();
    // set up attack bitmaps, etc.
    Bearing::init();
    LockInit(hash_lock);
    programPath = pathName;
    string tracePath;
    cmd_buf_count = 0;
    *cmd_buf = '\0';
    game_moves = new MoveArray();
    theLog = new Log();
    theLog->clear();
    theLog->write_header();
    Scoring::init();
    if (init_bitbases && !initBitbase()) {
#ifdef _WIN32
       ::MessageBox(NULL,"Error: can't start chess engine, missing bitbase files","",MB_OK | MB_ICONINFORMATION);
#else
       cerr << "error: missing bitbase files" << endl;
#endif
       return 0;
    }
    if (open_trace) 
    {
       tracePath = derivePath("trace.log");
       freopen(tracePath.c_str(),"w",stdout);
    }
    return 1;
}

void delayed_init() {
    // If no tablebase directory was specified in the .rc file,
    // try to find the table bases under the arasan install dir
    string tbPath;
    if (options.search.use_tablebases) {
       if (options.search.tablebase_location == "") {
	 options.search.tablebase_location = derivePath("TB");
       }
    }
    if (options.search.use_tablebases) {
       EGTBMenCount = initTB((char*)options.search.tablebase_location.c_str(),
         options.search.tb_cache_size);
    }
    // also initialize the book here
    if (options.book.book_enabled) {
       opening_book = new Book();
    }
    position_book = new PositionBook();
}

// Make this a __cdecl function so we can pass it to atexit()
void CDECL cleanup_globals()
{
    delete opening_book;
    delete position_book;
    delete game_moves;
    if (theLog) delete theLog;
    Scoring::cleanup();
    LockDestroy(hash_lock);
    Bitmap::cleanup();
}
