// Copyright 1994-2008 by Jon Dart.  All Rights Reserved.

#ifndef _CHESS_H
#define _CHESS_H

#include "types.h"
#include "constant.h"
#include "arasset.h"
#include <iostream>
using namespace std;

// basic types

typedef int ColorType;
#define Black 0
#define White 1

inline ColorType OppositeColor( const ColorType color )
{
    return 1-color;
}

extern const char *ColorImage( const ColorType side );

extern const CACHE_ALIGN int Files[64];
extern const int Ranks[64];
extern const int Edge[64];
extern const int Colors[64];
extern const int Center[64];
extern const int Corners[64];
extern const int Rank7[64];
extern const int Flip[64];

#define Square( file, rank, side ) \
  ((side == Black) ? ((rank-1)*8 + file - 1) : (55 - (rank-1)*8 + file))
    
// returns "rank" from the perspective of "side".  1 = 1st
// rank for side, 8 = last rank.	    
#define Rank(square, side ) ((side == Black) ? Ranks[square] : 9-Ranks[square])
#define WhiteRank(square) 9-Ranks[square]
#define BlackRank(square) Ranks[square]

// returns file.  Queen rook file = 1, King rook file = 8.
#define File(square) Files[square]
	    
#define OnBoard(square) ((square & ~0x3f) == 0)

#define OnEdge(square) Edge[square]

#define InCenter(square) Center[square]

#define InCorner(square) Corners[square]
    
#define IsInvalid(square) (square == InvalidSquare)
    
#define SquareColor(square) ((ColorType)Colors[square])
    
// Parse a square value in algebraic notation (e.g. "g7") and
// return a corresponding Square type.
extern Square SquareValue(const char *p);
    
extern char FileImage(Square sq);

extern char RankImage(Square sq);

extern char *SquareImage(Square sq); 

// algebraic notation to square mapping.

#define A8 0
#define B8 1
#define C8 2
#define D8 3
#define E8 4
#define F8 5
#define G8 6
#define H8 7
#define A7 8
#define B7 9
#define C7 10
#define D7 11
#define E7 12
#define F7 13
#define G7 14
#define H7 15
#define A6 16
#define B6 17
#define C6 18
#define D6 19
#define E6 20
#define F6 21
#define G6 22
#define H6 23
#define A5 24
#define B5 25
#define C5 26
#define D5 27
#define E5 28
#define F5 29
#define G5 30
#define H5 31
#define A4 32
#define B4 33
#define C4 34
#define D4 35
#define E4 36
#define F4 37
#define G4 38
#define H4 39
#define A3 40
#define B3 41
#define C3 42
#define D3 43
#define E3 44
#define F3 45
#define G3 46
#define H3 47
#define A2 48
#define B2 49
#define C2 50
#define D2 51
#define E2 52
#define F2 53
#define G2 54
#define H2 55
#define A1 56
#define B1 57
#define C1 58
#define D1 59
#define E1 60
#define F1 61
#define G1 62
#define H1 63

#define AFILE 1
#define BFILE 2
#define CFILE 3
#define DFILE 4
#define EFILE 5
#define FFILE 6
#define GFILE 7
#define HFILE 8

typedef int Piece;

typedef int PieceType;

#define EmptyPiece 0
#define Pawn 1
#define Knight 2
#define Bishop 3
#define Rook 4
#define Queen 5
#define King 6
#define InvalidPiece 7

#define PAWN_VALUE 64
#define BISHOP_VALUE 3*PAWN_VALUE
#define KNIGHT_VALUE 3*PAWN_VALUE
#define BISHOP_BONUS (BISHOP_VALUE-KNIGHT_VALUE)
#define ROOK_VALUE 320
#define QUEEN_VALUE 576
#define KING_VALUE 2000

extern const unsigned PieceValues[8];
extern const int _sliders[16];

inline int sliding(Piece p) 
{
   return _sliders[p];
}

#define MakePiece( type, color ) _pieces[type][color]
#define MakeWhitePiece( type ) (((Piece)type) + 8)
#define MakeBlackPiece( type ) ((Piece)type)
    
#define TypeOfPiece(piece) ((PieceType)(piece & 7))
   
#define PieceColor(piece) ((ColorType)(piece > 8))
#define ColorOfPiece(piece) ((ColorType)(piece > 8))

#define IsEmptyPiece(piece) (TypeOfPiece(piece) == EmptyPiece)
   
#define PieceValue(piece) PieceValues[TypeOfPiece(piece)]   

#define IsValid(piece) (TypeOfPiece(piece) != InvalidPiece)

extern PieceType PieceCharValue( const char );

// 1-character representation of piece
extern char PieceImage(const PieceType p);

extern const Piece _pieces[8][2];	   

#define whitePawn 9
#define whiteKnight 10
#define whiteBishop 11
#define whiteRook 12
#define whiteQueen 13
#define whiteKing 14
#define blackPawn 1
#define blackKnight 2
#define blackBishop 3
#define blackRook 4
#define blackQueen 5
#define blackKing 6

#ifndef _BOARD_H
class Board;
#endif

typedef uint64 Move;

enum MoveFlags {NewMove = 0, Used = 1, Mate = 2, HPrune = 4, HReduce = 8, Forced = 16};

union MoveUnion
{
   struct
   {
        byte my_start;
	byte my_dest;
        byte pieceMoved;
        byte my_promotion;
        byte capture;
        byte type;
        byte flag;
        byte junk;
   } contents;

   Move m;
   struct
   {
      uint32 lopart;
      uint32 hipart;
   } split;
   
   MoveUnion(const Move move)
   : m(move)
   {
   }

   MoveUnion()
   {
   }
};

enum SpecialType { Normal, KCastle, QCastle, EnPassant, Promotion };
	
#define NullMove (Move)MAKELONGLONG(0x00000000,0x07077f7f)

extern Move CreateMove(const Board &b,Square start, Square dest, PieceType promotion );

FORCEINLINE Move CreateMove(Square start, Square dest, PieceType pieceMoved,
  PieceType capture=EmptyPiece, PieceType promotion=InvalidPiece, 
  SpecialType type=Normal) {
  return (Move)(uint64(start | (dest<<8) | (pieceMoved << 16) |
		       ((uint64)promotion << 24) |
		       ((uint64)capture << 32) |
		       ((uint64)type << 40)));
}	

#define IsUsed(move) (((union MoveUnion*)&(move))->contents.flag & Used)

#define SetUsed(move) ((union MoveUnion*)&(move))->contents.flag |= Used

#define ClearUsed(move) ((union MoveUnion*)&(move))->contents.flag &= ~Used;

#define IsMate(move) (((union MoveUnion*)&(move))->contents.flag & Mate)

#define SetMate(move) ((union MoveUnion*)&(move))->contents.flag |= Mate

#define SetHPrune(move) ((union MoveUnion*)&(move))->contents.flag |= HPrune

#define ClearHPrune(move) ((union MoveUnion*)&(move))->contents.flag &= ~HPrune

#define CanHPrune(move)  ((union MoveUnion*)&(move))->contents.flag & HPrune

#define SetHReduce(move) ((union MoveUnion*)&(move))->contents.flag |= HReduce

#define ClearHReduce(move) ((union MoveUnion*)&(move))->contents.flag &= ~HReduce

#define CanHReduce(move)  ((union MoveUnion*)&(move))->contents.flag & HReduce

#define IsForced(move) (((union MoveUnion*)&(move))->contents.flag & Forced)

#define SetForced(move) ((union MoveUnion*)&(move))->contents.flag |= Forced

#define MovesEqual(move1,move2) (((union MoveUnion)move1).split.lopart  == ((union MoveUnion)move2).split.lopart)

#define EqualMove(move1,move2) MovesEqual(move1,move2)
                                 
#define StartSquare(move) (Square)(((union MoveUnion)move).contents.my_start)

#define DestSquare(move) (Square)(((union MoveUnion)move).contents.my_dest)

#define PromoteTo(move) ((PieceType)((union MoveUnion)move).contents.my_promotion)

#define Capture(move) ((PieceType)((union MoveUnion)move).contents.capture)

#define PieceMoved(move) ((PieceType)((union MoveUnion)move).contents.pieceMoved)

#define TypeOfMove(move) ((SpecialType)((union MoveUnion)move).contents.type)

#define IsPromotion(move) (TypeOfMove(move) == Promotion)

#define IsNull(move) (StartSquare(move) == InvalidSquare)
	
#define Gain(move) ((TypeOfMove(move) == Promotion) ? \
PieceValue(Capture(move)) + PieceValue(PromoteTo(move)) - PAWN_VALUE \
 : PieceValue(Capture(move)))

#define SetCapture(move,p) ((union MoveUnion*)&(move))->contents.capture = p

#define SetType(move,t) ((union MoveUnion*)&(move))->contents.type = t

#define SetPromotion(move,p) ((union MoveUnion*)&(move))->contents.my_promotion = p

#define IllegalMove (Move)0xffffffff

extern void MoveImage(Move m,ostream &out);


#endif
