// Copyright 1994-2006 by Jon Dart. All Rights Reserved.

#include "chess.h"
#include "constant.h"
#include "util.h"
#include "debug.h"
#include "board.h"
#include <iostream>
#include <iomanip>

using namespace std;

extern "C"
{
#include <ctype.h>
#include <string.h>
};

const int CACHE_ALIGN Files[64] =
{
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8,
   1, 2, 3, 4, 5, 6, 7, 8
};

const int Ranks[64] =
{
   1, 1, 1, 1, 1, 1, 1, 1,
   2, 2, 2, 2, 2, 2, 2, 2,
   3, 3, 3, 3, 3, 3, 3, 3,
   4, 4, 4, 4, 4, 4, 4, 4,
   5, 5, 5, 5, 5, 5, 5, 5,
   6, 6, 6, 6, 6, 6, 6, 6,
   7, 7, 7, 7, 7, 7, 7, 7,
   8, 8, 8, 8, 8, 8, 8, 8
};

const int Edge[64] =
{
   1, 1, 1, 1, 1, 1, 1, 1,
   1, 0, 0, 0, 0, 0, 0, 1,
   1, 0, 0, 0, 0, 0, 0, 1,
   1, 0, 0, 0, 0, 0, 0, 1,
   1, 0, 0, 0, 0, 0, 0, 1,
   1, 0, 0, 0, 0, 0, 0, 1,
   1, 0, 0, 0, 0, 0, 0, 1,
   1, 1, 1, 1, 1, 1, 1, 1
};

const int Colors[64] =
{
   1, 0, 1, 0, 1, 0, 1, 0,
   0, 1, 0, 1, 0, 1, 0, 1,
   1, 0, 1, 0, 1, 0, 1, 0,
   0, 1, 0, 1, 0, 1, 0, 1,
   1, 0, 1, 0, 1, 0, 1, 0,
   0, 1, 0, 1, 0, 1, 0, 1,
   1, 0, 1, 0, 1, 0, 1, 0,
   0, 1, 0, 1, 0, 1, 0, 1
};

const int Center[64] =
{
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 1, 1, 0, 0, 0,
   0, 0, 0, 1, 1, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0
};

const int Corners[64] =
{
   1, 0, 0, 0, 0, 0, 0, 1,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   1, 0, 0, 0, 0, 0, 0, 1
};

const int Rank7[64] =
{
   0, 0, 0, 0, 0, 0, 0, 0,
   1, 1, 1, 1, 1, 1, 1, 1,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   1, 1, 1, 1, 1, 1, 1, 1,
   0, 0, 0, 0, 0, 0, 0, 0
};

const int Flip[64] =
{
        A1, B1, C1, D1, E1, F1, G1, H1,
        A2, B2, C2, D2, E2, F2, G2, H2,
        A3, B3, C3, D3, E3, F3, G3, H3,
        A4, B4, C4, D4, E4, F4, G4, H4,
        A5, B5, C5, D5, E5, F5, G5, H5,
        A6, B6, C6, D6, E6, F6, G6, H6,
        A7, B7, C7, D7, E7, F7, G7, H7,
        A8, B8, C8, D8, E8, F8, G8, H8
};

const char *ColorImage( const ColorType side )
{
    static const char *whiteImage = "White";
    static const char *blackImage = "Black";
    if (side == White)
       return whiteImage;
    else
       return blackImage;
}

Square SquareValue(const char *p )
{
    int rank, file;

    if ((*p >= 'a') && (*p <= 'h'))
       file = *p - 'a' + 1;
    else
       return InvalidSquare;
    ++p;
    if ((*p >= '1') && (*p <= '8'))
       rank = *p - '1' + 1;
    else
       return InvalidSquare;
    return (8-rank)*8 + file - 1;
}

char FileImage(Square sq)
{
    return 'a' + File(sq) - 1;
}

char RankImage(Square sq)
{
    return '1' + 8 - Ranks[sq];
}

char *SquareImage(Square sq) {
    static char chars[3];
    chars[0] = FileImage(sq);
    chars[1] = RankImage(sq);
    chars[2] = '\0';
    return chars;
}

const unsigned PieceValues[8] = 
{
   0,
   PAWN_VALUE,
   KNIGHT_VALUE,
   BISHOP_VALUE,
   ROOK_VALUE,
   QUEEN_VALUE,
   KING_VALUE,
   0
};

const Piece _pieces[8][2] =
{ {0,0},
{ 1, 9 },
{ 2, 10 },
{ 3, 11 },
{ 4, 12 },
{ 5, 13 },
{ 6, 14 },
{ 7, 15 }};

const int _sliders[16] =
{ 0, 0, 0, 1, 1, 1, 0, 0,
  0, 0, 0, 1, 1, 1, 0, 0 };

static const char Images[] = "?PNBRQK?";

PieceType PieceCharValue( const char c )
{
    const char *p = Images;
    p = strchr(p,c);
    if (p)
       return (PieceType)(p-Images);
    else
       return InvalidPiece;
}

char PieceImage(const PieceType p)
{
    return Images[p];
}

static SpecialType get_move_type(const Board &board, Move m)
{
    // set special field
    SpecialType my_special = Normal;
    switch (PieceMoved(m))
    {
    case Pawn:
       if (Rank(DestSquare(m),board.Side()) == 8)
       {
          my_special = Promotion;
       }
       else if (IsEmptyPiece(board[DestSquare(m)]))
       {
          if (File(DestSquare(m)) != File(StartSquare(m)))
	  {
	     my_special = EnPassant;
	  }
       }
       break;
    case King:
       if (Util::Abs((int)StartSquare(m) - (int)DestSquare(m)) == 2)
       {
          if (StartSquare(m) > DestSquare(m))
	         my_special = QCastle;
          else
	         my_special = KCastle;
       }
       break;
    default:
       break;
    }
    return my_special;
}

Move CreateMove(const Board &board, Square start, Square dest, PieceType promotion )
{
   MoveUnion mu;
   mu.contents.my_start = start;
   mu.contents.my_dest = dest;
   mu.contents.my_promotion = (byte)promotion;
   mu.contents.flag = NewMove;
   if (IsInvalid(start))
   {
        return NullMove;
   }
   else
   {
      mu.contents.pieceMoved = TypeOfPiece(board[start]);
      ASSERT(mu.contents.pieceMoved != EmptyPiece);
   }
   mu.contents.capture = TypeOfPiece(board[dest]);
   mu.contents.type = get_move_type(board,mu.m);
   if (mu.contents.type == EnPassant)
      mu.contents.capture = Pawn;
   mu.contents.flag = NewMove;
   return mu.m;
}	

void MoveImage(Move m,ostream &out) {
    char image[10];
    if (IsNull(m))
    {
       cout << "(null)";
       return;
    }
    image[0] = FileImage(StartSquare(m));
    image[1] = RankImage(StartSquare(m));
    if (Capture(m) != EmptyPiece)
      image[2] = 'x';
    else
      image[2] = '-';
    image[3] = FileImage(DestSquare(m));
    image[4] = RankImage(DestSquare(m));
    int i = 5;
    if (PromoteTo(m) != EmptyPiece && PromoteTo(m) != InvalidPiece)
    {
       image[i++] = '=';
       image[i++] = PieceImage(PromoteTo(m));
    }
    image[i] = '\0';
    out << image;
}

