// Copyright 1997 by Jon Dart.  All Rights Reserved.

#include "boardio.h"
#include "bearing.h"
#include "bhash.h"
#include <ctype.h>

int BoardIO::read_fen(Board &board, char *buf)
{
   board.Reset();
   for (int i = 0; i < 64; i++)
   {
      board.my_Contents[i] = EmptyPiece;
   }

   char *bp = buf;
   char c;
   while (isspace(*bp)) bp++;
   for (int line = 0; line < 8; line++)
   {
      int sqval = line*8;
      while ((c = *bp) != ' ' && c != '/')
      {
         Piece piece;
         if (isdigit(c))
         {
            sqval += (*bp - '0');
            bp++;
         }
         else if (isalnum(c))
         {
            if (!OnBoard(sqval))
            {
	      return 0;
            }
            switch (c)
            {
            case 'p': piece = MakePiece(Pawn,Black); 
               break;
            case 'n': piece = MakePiece(Knight,Black);
               break;
            case 'b': piece = MakePiece(Bishop,Black);
               break;
            case 'r': piece = MakePiece(Rook,Black);
               break;
            case 'q': piece = MakePiece(Queen,Black); 
               break;
            case 'k': piece = MakePiece(King,Black);
               break;
            case 'P': piece = MakePiece(Pawn,White);
               break;
            case 'N': piece = MakePiece(Knight,White);
               break;
            case 'B': piece = MakePiece(Bishop,White);
               break;
            case 'R': piece = MakePiece(Rook,White);
               break;
            case 'Q': piece = MakePiece(Queen,White);
               break;
            case 'K': piece = MakePiece(King,White);
               break;
            default:
	      return 0;
            }
            board.my_Contents[sqval] = piece;
            sqval++;
            bp++;
         }
         else // not a letter or a digit
         {
	   return 0;
         }
      }
      if (c=='/') ++bp;  // skip delimiter
   }
   while (isspace(*bp)) bp++;
   if (toupper(*bp) == 'W')
      board.side = White;
   else if (toupper(*bp) == 'B')
      board.side = Black;
   else
   {
     return 0;
   }
   bp++;
   while (isspace(*bp)) bp++;
   c = *bp;
   if (c == '-')
   {
      board.state.castleStatus[White] = 
      board.state.castleStatus[Black] = CantCastleEitherSide;
      bp++;
   }
   else
   {
      int k = 0;
      for (; !isspace(*bp); bp++)
      {
         if (*bp == 'K')
            k += 1;
         else if (*bp == 'Q')
            k += 2;
         else if (*bp == 'k')
            k += 4;
         else if (*bp == 'q')
            k += 8;
         else
         {
	   return 0;
         }
      }
      static CastleType vals[4] = 
      { CantCastleEitherSide, CanCastleKSide,
         CanCastleQSide, CanCastleEitherSide};
      board.state.castleStatus[White] = vals[k % 4];
      board.state.castleStatus[Black] = vals[k / 4];
   }
   board.set_secondary_vars();
   while (isspace(*bp)) bp++;
   c = *bp;
   if (c == '-')
   {
     board.state.enPassantSq = InvalidSquare;
   }
   else if (isalpha(c))
   {
      char sqbuf[2];
      sqbuf[0] = *bp++;
      sqbuf[1] = *bp++;
      Square epsq = SquareValue(sqbuf);
      if (epsq == InvalidSquare)
      {
         return 0;
      }
      board.state.enPassantSq = InvalidSquare;
      Square ep_candidate =  
        SquareValue(sqbuf) - (8 * Direction[board.Side()]);
      // only actually set the e.p. square on the board if an en-passant capture
      // is truly possible:
      if (TEST_MASK(Bearing::ep_mask[File(ep_candidate)-1][(int)board.OppositeSide()],
		  board.pawn_bits[board.Side()])) {
        board.state.enPassantSq = ep_candidate;
	    // re-calc hash code since ep has changed
	    board.state.hashCode = BoardHash::hashCode(board);
	  }
   }
   else
   {
     return 0;
   }
   *board.rep_list_head++ = board.HashCode();

   return 1;
}

void BoardIO::write_fen( const Board &board, char *buf, int addMoveInfo)
{
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
   string ostr(buf);
   std::stringstream o(ostr,ios::out);
#else
   strstream o(buf,128,ios::out);
#endif
   // write out the board in Forsythe-Edwards (FEN) notation.
   for (int i=1;i<=8;i++)
   {
      int j = 1;
      while (j <= 8)
      {
         int n = 0;
         Square sq;
         Piece p;
         do
         {
            sq = Square(j,i,Black);
            p = board.my_Contents[sq];
            if (p != EmptyPiece)
               break;
            ++j; ++n;
         } while (j <= 8);
         if (n)
            o << (char)(n + '0');
         if (p != EmptyPiece)
         {
            char img = PieceImage(TypeOfPiece(p));
            if (PieceColor(p) == Black) img = tolower(img);
            o << img;
            j++;
         }
      }
      if (i != 8) o << '/';
   }
   if (board.Side() == White)
      o << " w";
   else
      o << " b";

   // used in I/O of castling data:
   const bool kcastle[6] = { true, true, false, false, false, false};
   const bool qcastle[6] = { true, false, true, false, false, false};

   // note : unfortunately FEN doesn't allow recording if castling
   // has taken place, only whether or not it is possible.
   CastleType wcs = board.CastleStatus(White);
   CastleType bcs = board.CastleStatus(Black);
   o << ' ';
   if (!kcastle[(int)wcs] && !qcastle[(int)bcs])
      o << '-';
   else
   {
      if (kcastle[(int)wcs])
         o << 'K';
      if (qcastle[(int)wcs])
         o << 'Q';
      if (kcastle[(int)bcs])
         o << 'k';
      if (qcastle[(int)bcs])
         o << 'q';
   }
   o << ' ';
   Square epsq = board.EnPassantSq();
   if (epsq == InvalidSquare)
      o << '-';
   else
   {
      // FEN stores the destination square for an en passant capture;
      // we store the location of the capturable pawn.
      Square target = (epsq + (RankIncr * Direction[board.Side()]));
      o << FileImage(target) << RankImage(target);
   }
   if (addMoveInfo)
   {
      // FEN is supposed to include the halfmove and fullmove numbers,
      // but these are attributes of the game - they are not stored in
      // the board.
      o << " 0 1";
   }
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
   strcpy(buf,ostr.c_str());
#endif
   o << '\0';
}
