// Copyright 1994-2000, 2004, 2005 by Jon Dart.  All Rights Reserved.

#include "types.h"
#include "chess.h"

#ifndef _BOARD_H
#define _BOARD_H

#include "bitmap.h"
#include "material.h"
#include <iostream>

class Board;

extern const hash_t rep_codes[3];

enum CastleType { CanCastleEitherSide,
                  CanCastleKSide,
                  CanCastleQSide,
                  CastledKSide,
                  CastledQSide,
                  CantCastleEitherSide};
          
enum CheckStatusType { NotInCheck, InCheck, CheckUnknown };

struct Board_State {
   hash_t hashCode; //, pawnHashCodeW, pawnHashCodeB;
   Square enPassantSq;
   int moveCount;
   CheckStatusType checkStatus;
   CastleType castleStatus[2];
};

class Board
{
   // This class encapsulates the chess board, and provides functions
   // for making and undoing moves.          

public: 

   friend class BoardIO;
   friend class Scoring;
   
   // creates a board set up in its initial position, White to move.
   Board();
           
  ~Board();
  
   // Copy constructor and assignment operator  
   Board(const Board &);
   Board &operator = (const Board &);

   // resets board to initial position
   void Reset();

   // remove all pieces
   void makeEmpty();
           
#ifdef _DEBUG
   const Piece &operator[]( const Square sq ) const;
#else
   const Piece &operator[]( const Square sq ) const
   {
       return my_Contents[sq];
   }
#endif

   void setContents( const Piece &p, const Square sq )
   {
       my_Contents[sq] = p;
   }
   
   void set_secondary_vars();
   // update all fields of the board, based on the piece positions.

   CastleType CastleStatus( const ColorType side ) const
   {
       return state.castleStatus[side];        
   }
   
   void setCastleStatus( const CastleType t, const ColorType side )
   {
       state.castleStatus[side] = t;
   }

   int castled(const ColorType side) const
   {
      return state.castleStatus[side] == CastledKSide ||
             state.castleStatus[side] == CastledQSide;
   }
  
   ColorType Side() const
   // side to move         
   {
       return side;       
   }
   
   ColorType OppositeSide() const
   {
       return OppositeColor(side);
   }
   
   void setSide( const ColorType side )
   {
       this->side = side;
   }
   
   const Material &getMaterial( const ColorType side ) const
   {
       return my_Material[side];
   }
   
   hash_t pawn_hash() const {
      return pawnHashCodeW ^ pawnHashCodeB;
   }

   const Square EnPassantSq()  const
   {
      return state.enPassantSq;
   }
           
   CheckStatusType CheckStatus( ) const;
   
   CheckStatusType CheckStatus(Move lastMove) const;

   // returns InCheck if the move could cause check to the opponent
   // after being made (call before the move is made).
   CheckStatusType wouldCheck(Move lastMove) const;
   
   void set_check_status(CheckStatusType s)
   {
       state.checkStatus = s;
   }

   const Square &KingPos( const ColorType side ) const
   {
        return my_KingPos[side];           
   }
           
   hash_t HashCode() const
   {
       return state.hashCode;
   }
   
   // returns a hash code factoring in the position repetition count
   hash_t HashCode(int rep_count) const
   {
       return state.hashCode ^ rep_codes[rep_count];
   }
   
   // returns what hash code will be after move
   hash_t HashCode( const Move &m ) const;

   int operator == ( const Board &ABoard )
   {
       return state.hashCode == ABoard.HashCode();
   }
   
   // set a bitmap of squares that lie in a straight line between
   // sq1 and sq2
   void between( Square sq1, Square sq2, Bitmap &) const;
   
   // returns "true" if there is a clear path between sq1 and sq2
   int Clear( Square sq1, Square sq2 ) const;

   // makes a move
   void DoMove(Move m);

   // makes a null move
   void DoNull();

   // undoes a previous move.
   void UndoMove( const Move rmove, const Board_State &stat );
   
   void UndoNull(const Board_State &oldState) {
      state = oldState;
       --rep_list_head;
      side = OppositeColor(side);
   }

   // Return "TRUE" if any piece of color "side" attacks "sq".
   int any_attacks(const Square sq, const ColorType side) const;

   // Return "TRUE" if any piece of color "side" attacks "sq".
   int any_attacks(const Square sq, const ColorType side, Bitmap &source) const;

   // Return a bit vector of all pieces of color "side" that attack "sq".
   Bitmap calc_attacks(const Square sq, const ColorType side) const;
   
   // Return a bit vector of all pieces of color "side" that can move to
   // unoccupied "sq".
   Bitmap calc_blocks(const Square sq, const ColorType side) const;
   
   // Return the least-valued piece of color "side" out of the bitmap.
   Bitmap min_attacker(const Bitmap &, const ColorType side) const;
   
   // Return all squares attacked by pawns of color "side".
   Bitmap all_pawn_attacks(const ColorType side) const;
   
   // Return true if previous move was legal (did not involve a move into check)
   int wasLegal(Move lastMove) const;

   int rep_count() const;

   // flip the board position up/down and change side to move
   void flip();

   // flip board position right/left
   void flip2();

   friend istream & operator >> (istream &i, Board &board);
   friend ostream & operator << (ostream &o, const Board &board);

#if !defined(_WIN32) && defined(DEBUG_ATTACKS)
   // for debugging :
   void dump_attacks()
   {
       my_attacks.dump_attacks();           
   }
#endif   
   
   private:
           
   ALIGN_VAR(16) Piece my_Contents[64];
   Square my_KingPos[2];
   Material my_Material[2];

public:
   ColorType side;
   Board_State state;
   hash_t pawnHashCodeW, pawnHashCodeB;

   Bitmap pawn_bits[2];
   Bitmap knight_bits[2];
   Bitmap bishop_bits[2];
   Bitmap rook_bits[2];
   Bitmap rooksqueens[2];
   Bitmap bishopsqueens[2];
   Bitmap queen_bits[2];
   Bitmap occupied[2];
   Bitmap all_occupied;
   Bitmap occupiedR90;
   Bitmap occupiedDa1;
   Bitmap occupiedDa8;

   hash_t *rep_list; //  move history for repetition detection
   hash_t *rep_list_head; // head of history list
private:

   CheckStatusType checkDiscovered(
       Square startSquare, Square kp, int direction) const;

   void undo_castling(const Square &kp,const Square &oldkingsq,
           const Square &newrooksq, const Square &oldrooksq);
           
   void setAll(ColorType side,int n)
   {
      register const masks *masks = &all_masks[n];
      occupied[side].data |= masks->mask;
      occupiedR90.data |= masks->maskR90;
      occupiedDa1.data |= masks->maskDa1;
      occupiedDa8.data |= masks->maskDa8;
   }
   void clearAll(ColorType side,int n)
   {
      register const masks *masks = &all_masks[n];
      occupied[side].data &= masks->not_mask;
      occupiedR90.data &= masks->not_maskR90;
      occupiedDa1.data &= masks->not_maskDa1;
      occupiedDa8.data &= masks->not_maskDa8;
   }
};

#endif

