// Copyright 1994-2006 by Jon Dart.  All Rights Reserved.

#include "constant.h"
#include "chess.h"
#include "board.h"
#include "bhash.h"
#include "util.h"
#include "bearing.h" 
#include "debug.h"
#include "boardio.h"
#include <ctype.h>
#include <memory.h>
#include <assert.h>
#include <iostream>

using namespace std;

enum
{
   BoardSize = 64
};
enum
{
   RepListSize = 1024
};

const hash_t rep_codes[3] =
{
    MAKELONGLONG(0x194ca2c4,0x5c8e7baa),
    MAKELONGLONG(0x804e48e8,0xe8f5544f),
    MAKELONGLONG(0xd4767986,0xf0ab49a7)
};

void Board::set_secondary_vars()
{
   int i;

   my_Material[White].clear();
   my_Material[Black].clear();
   pawn_bits[White].clear();
   pawn_bits[Black].clear();
   knight_bits[White].clear();
   knight_bits[Black].clear();
   bishop_bits[White].clear();
   bishop_bits[Black].clear();
   rook_bits[White].clear();
   rook_bits[Black].clear();
   rooksqueens[White].clear();
   rooksqueens[Black].clear();
   bishopsqueens[White].clear();
   bishopsqueens[Black].clear();
   queen_bits[White].clear();
   queen_bits[Black].clear();
   occupied[White].clear();
   occupied[Black].clear();
   all_occupied.clear();
   occupiedR90.clear();
   occupiedDa1.clear();
   occupiedDa8.clear();
   for (i=0;i<BoardSize;i++)
   {
      Square sq(i);
      if (my_Contents[sq] != EmptyPiece)
      {
         const Piece piece = my_Contents[sq];
         const ColorType color = PieceColor(piece);
         occupied[color].set(sq);
         all_occupied.set(sq);
         occupiedR90.setR90(sq);
         occupiedDa1.setDa1(sq);
         occupiedDa8.setDa8(sq);
         my_Material[color].add_piece(TypeOfPiece(piece));
         switch (TypeOfPiece(piece))
         {
         case King:
            my_KingPos[color] = sq;
            break;
         case Pawn:
            pawn_bits[color].set(sq);
            break;
         case Knight:
            knight_bits[color].set(sq);
            break;
         case Bishop:
            bishop_bits[color].set(sq);
            bishopsqueens[color].set(sq);
            break;
         case Rook:
            rook_bits[color].set(sq);
            rooksqueens[color].set(sq);
            break;
         case Queen:
            queen_bits[color].set(sq);
            rooksqueens[color].set(sq);
            bishopsqueens[color].set(sq);
            break;
         default:
            break;
         }
      }
   }
   state.hashCode = BoardHash::hashCode(*this);
   pawnHashCodeW = BoardHash::pawnHash(*this,White);
   pawnHashCodeB = BoardHash::pawnHash(*this,Black);
}

void Board::Reset()
{
   static PieceType pieces[] =
   {
      Rook,
      Knight,
      Bishop,
      Queen,
      King,
      Bishop,
      Knight,
      Rook
   };

   side = White;
   state.checkStatus = CheckUnknown;
   for (int i=0;i<BoardSize;i++)
   {
      const Square sq(i);
      if (Rank(sq,White) == 1)
         my_Contents[sq] = MakeWhitePiece( pieces[File(sq)-1]);
      else if (Rank(sq,Black) == 1)
         my_Contents[sq] = MakeBlackPiece( pieces[File(sq)-1]);
      else if (Rank(sq,White) == 2)
         my_Contents[sq] = whitePawn;
      else if (Rank(sq,Black) == 2)
         my_Contents[sq] = blackPawn;
      else
         my_Contents[sq] = EmptyPiece;
   }
   state.enPassantSq = InvalidSquare;
   state.castleStatus[White] = state.castleStatus[Black] = CanCastleEitherSide;
   if (rep_list)
      delete [] rep_list;
   rep_list = rep_list_head = new hash_t[RepListSize];
   state.moveCount = 0;
   set_secondary_vars();
   *rep_list_head++ = HashCode();
}

void Board::makeEmpty() {
   for (Square sq = 0; sq < 64; sq++) my_Contents[sq] = EmptyPiece;
   state.castleStatus[White] = state.castleStatus[Black] = CantCastleEitherSide;
}

Board::Board()
: rep_list(NULL)
{
   Reset();
}

Board::Board(const Board &b)
{
   // Copy all contents except the repetition list
   memcpy(&my_Contents,&b.my_Contents,(byte*)&rep_list-(byte*)&my_Contents);
   // Copy the repetition table
   rep_list = new hash_t[RepListSize];
   int rep_entries = (int)(b.rep_list_head - b.rep_list);
   ASSERT(rep_entries>=0 && rep_entries<RepListSize);
   if (rep_entries) {
     memcpy(rep_list,b.rep_list,sizeof(hash_t)*rep_entries);
   }
   rep_list_head = rep_list + rep_entries;
}

Board &Board::operator =(const Board &b)
{
   if (&b != this)
   {
      // Copy all contents except the repetition list
      memcpy(&my_Contents,&b.my_Contents,(byte*)&rep_list-(byte*)&my_Contents);
      // Copy the repetition table
      state.moveCount = b.state.moveCount;
      delete [] rep_list;
      rep_list = new hash_t[RepListSize];
      int rep_entries = (int)(b.rep_list_head - b.rep_list);
      for (int i = 0; i < rep_entries; i++)
         rep_list[i] = b.rep_list[i];
      rep_list_head = rep_list + rep_entries;
   }
   return *this;
}

Board::~Board()
{
   delete [] rep_list;
}

#ifdef _DEBUG
const Piece &Board::operator[]( const Square sq ) const
{
   ASSERT(OnBoard(sq));
   return my_Contents[sq];
}
#endif

static inline CastleType UpdateCastleStatusW( CastleType cs, Square sq )
// after a move of or capture of the rook on 'sq', update castle status
// for 'side'
{
   ASSERT(cs<3);
   if (sq == A1) // Queen Rook moved or captured
   {
      if (cs == CanCastleEitherSide)
         return CanCastleKSide;
      else if (cs == CanCastleQSide)
         return CantCastleEitherSide;
   }
   else if (sq == H1) // King Rook moved or captured
   {
      if (cs == CanCastleEitherSide)
         return CanCastleQSide;
      else if (cs == CanCastleKSide)
         return CantCastleEitherSide;
   }
   return cs;
}

static inline CastleType UpdateCastleStatusB(CastleType cs, Square sq)
// after a move of or capture of the rook on 'sq', update castle status
// for 'side'
{
   ASSERT(cs<3);
   if (sq == A8) // Queen Rook moved or captured
   {
      if (cs == CanCastleEitherSide)
         return CanCastleKSide;
      else if (cs == CanCastleQSide)
         return CantCastleEitherSide;
   }
   else if (sq==H8) // King Rook moved or captured
   {
      if (cs == CanCastleEitherSide)
         return CanCastleQSide;
      else if (cs == CanCastleKSide)
         return CantCastleEitherSide;
   }
   return cs;
}

// set a bitmap of squares that lie in a straight line between
// sq1 and sq2. 
void Board::between( Square sq1, Square sq2, Bitmap &result) const {
   result.clear();
   int offset = (int)Bearing::Directions[sq1][sq2];
   if (offset == 0) return;
   Square sq = sq1; 
   for (;;)
   {
      sq += offset;
      if (sq == sq2)
         break;
      ASSERT(OnBoard(sq));
      result.set(sq);
   }
}

int Board::Clear( Square sq1, Square sq2 ) const
{
   int offset = (int)Bearing::Directions[sq1][sq2];
   if (offset == 0) return 0;
   Square sq = sq1;
   for (;;)
   {
      sq += offset;
      if (sq == sq2)
         break;
      ASSERT(OnBoard(sq));
      if (my_Contents[sq] != EmptyPiece)
         return 0;
   }
   return 1;
}

#define Xor(h,sq,piece) h ^= hash_codes[sq][(int)piece]

void Board::DoNull()
{
   state.checkStatus = CheckUnknown;
   state.moveCount++;
   if (state.enPassantSq != InvalidSquare)
   {
       state.hashCode ^= ep_codes[state.enPassantSq];
       state.hashCode ^= ep_codes[0];
   }
   state.enPassantSq = InvalidSquare;
   side = OppositeSide();
   if (Side() == Black)
       state.hashCode |= (hash_t)1;
   else
       state.hashCode &= (hash_t)~1;
   *rep_list_head++ = state.hashCode;
   ASSERT(rep_list_head-rep_list < (int)RepListSize);
   ASSERT(state.hashCode == BoardHash::hashCode(*this));
}

void Board::DoMove( Move move )
{
   state.checkStatus = CheckUnknown;
   state.moveCount++;
   if (state.enPassantSq != InvalidSquare)
   {
       state.hashCode ^= ep_codes[state.enPassantSq];
       state.hashCode ^= ep_codes[0];
   }
   int old_epsq = state.enPassantSq;
   state.enPassantSq = InvalidSquare;

   ASSERT(!IsEmptyPiece(PieceMoved(move)));

   const Square start = StartSquare(move);
   const Square dest = DestSquare(move);
   const SpecialType moveType = TypeOfMove(move);
   ASSERT(PieceMoved(move) == TypeOfPiece(my_Contents[start]));

   if (side == White)
   {
      if (moveType == KCastle)
      {
         state.moveCount = 0;

         // update the hash code
         const Square kp = KingPos(White);
         Xor(state.hashCode, kp+3, whiteRook);
         Xor(state.hashCode, kp, whiteKing);
         Xor(state.hashCode, kp+1, whiteRook);
         Xor(state.hashCode, kp+2, whiteKing);
         state.hashCode ^= w_castle_status[(int)state.castleStatus[White]];
         state.hashCode ^= w_castle_status[(int)CastledKSide];

         const int newkp = kp + 2;
         my_KingPos[White] = newkp;
         state.castleStatus[White] = CastledKSide;
         // find old square of rook
         Square oldrooksq = kp + 3;
         Square newrooksq = kp + 1;
         my_Contents[kp] = my_Contents[oldrooksq] = EmptyPiece;
         my_Contents[newrooksq] = whiteRook;
         my_Contents[newkp] = whiteKing;
         rook_bits[White].clear(oldrooksq);
         rooksqueens[White].clear(oldrooksq);
         rook_bits[White].set(newrooksq);
         rooksqueens[White].set(newrooksq);
         clearAll(White,kp);
         clearAll(White,oldrooksq);
         setAll(White,newkp);
         setAll(White,newrooksq);
      }
      else if (moveType == QCastle)
      {
         state.moveCount = 0;

         // update the hash code
         const Square kp = KingPos(White);
         Xor(state.hashCode, kp-4, whiteRook);
         Xor(state.hashCode, kp, whiteKing);
         Xor(state.hashCode, kp-1, whiteRook);
         Xor(state.hashCode, kp-2, whiteKing);
         state.hashCode ^= w_castle_status[(int)state.castleStatus[White]];
         state.hashCode ^= w_castle_status[(int)CastledQSide];

         const int newkp = kp - 2;
         my_KingPos[White] = newkp;
         state.castleStatus[White] = CastledQSide;
         // find old square of rook
         Square oldrooksq = kp - 4;
         Square newrooksq = kp - 1;
         my_Contents[kp] = my_Contents[oldrooksq] = Piece();
         my_Contents[newrooksq] = whiteRook;
         my_Contents[kp-2] = whiteKing;
         rook_bits[White].clear(oldrooksq);
         rooksqueens[White].clear(oldrooksq);
         rook_bits[White].set(newrooksq);
         rooksqueens[White].set(newrooksq);
         clearAll(White,kp);
         clearAll(White,oldrooksq);
         setAll(White,newkp);
         setAll(White,newrooksq);
      }
      else // not castling
      {
         ASSERT(my_Contents[start] != EmptyPiece);
         const Bitmap bits(Bitmap::mask[start] |
                           Bitmap::mask[dest]);
         Square target = dest; // where we captured
#ifdef _DEBUG
		 target = (TypeOfMove(move) == EnPassant) ? old_epsq : dest;
         if (Capture(move) == EmptyPiece) ASSERT(my_Contents[target]==EmptyPiece);
#endif
         switch (PieceMoved(move))
         {
         case Pawn:
            state.moveCount = 0;
            switch (moveType)
            {
            case EnPassant:
               // update hash code
               Xor(state.hashCode, start, whitePawn);
               Xor(state.hashCode, dest, whitePawn);
               Xor(pawnHashCodeW, start, whitePawn);
               Xor(pawnHashCodeW, dest, whitePawn);
               target = old_epsq;
               my_Contents[dest] = whitePawn;
               pawn_bits[White].set(dest);
               break;
            case Promotion:
               // update hash code
               Xor(state.hashCode, start, whitePawn);
               Xor(state.hashCode, dest, MakeWhitePiece(PromoteTo(move)));
               Xor(pawnHashCodeW, start, whitePawn);
               my_Contents[dest] = MakeWhitePiece(PromoteTo(move));
               my_Material[White].remove_pawn();
               my_Material[White].add_piece(PromoteTo(move));
               switch (PromoteTo(move))
               {
               case Knight:
                  knight_bits[White].set(dest);
                  break;
               case Bishop:
                  bishop_bits[White].set(dest);
                  bishopsqueens[White].set(dest);
                  break;
               case Rook:
                  rook_bits[White].set(dest);
                  rooksqueens[White].set(dest);
                  break;
               case Queen:
                  queen_bits[White].set(dest);
                  bishopsqueens[White].set(dest);
                  rooksqueens[White].set(dest);
                  break;
               default:
                  break;
               }
               break;
            default:
               Xor(state.hashCode, start, whitePawn );
               Xor(state.hashCode, dest, whitePawn );
               Xor(pawnHashCodeW, start, whitePawn);
               Xor(pawnHashCodeW, dest, whitePawn);
               my_Contents[dest] = whitePawn;
               if (start - dest == 16) // 2-square pawn advance
               {
                  if (TEST_MASK(Bearing::ep_mask[File(dest)-1][(int)White],pawn_bits[Black])) {
                    state.enPassantSq = dest;
                    state.hashCode ^= ep_codes[0];
                    state.hashCode ^= ep_codes[dest];
                  }
               }
               pawn_bits[White].set(dest);
               break;
            }
            pawn_bits[White].clear(start);
            break;
         case Knight:
            Xor(state.hashCode, start, whiteKnight);
            Xor(state.hashCode, dest, whiteKnight);
            my_Contents[dest] = whiteKnight;
            knight_bits[White].setClear(bits);
            break;
         case Bishop:
            Xor(state.hashCode, start, whiteBishop);
            Xor(state.hashCode, dest, whiteBishop);
            my_Contents[dest] = whiteBishop;
            bishop_bits[White].setClear(bits);
            bishopsqueens[White].setClear(bits);
            break;
         case Rook:
            Xor(state.hashCode, start, whiteRook );
            Xor(state.hashCode, dest, whiteRook );
            my_Contents[dest] = whiteRook;
            rook_bits[White].setClear(bits);
            rooksqueens[White].setClear(bits);
            if ((int)state.castleStatus[White]<3) {
               state.hashCode ^= w_castle_status[(int)state.castleStatus[White]];
               state.castleStatus[White] = UpdateCastleStatusW(state.castleStatus[White],start);
               state.hashCode ^= w_castle_status[(int)state.castleStatus[White]];
            }
            break;
         case Queen:
            Xor(state.hashCode, start, whiteQueen);
            Xor(state.hashCode, dest, whiteQueen);
            my_Contents[dest] = whiteQueen;
            queen_bits[White].setClear(bits);
            rooksqueens[White].setClear(bits);
            bishopsqueens[White].setClear(bits);
            break;
         case King:
            Xor(state.hashCode, start, whiteKing );
            Xor(state.hashCode, dest, whiteKing );
            my_Contents[dest] = whiteKing;
            my_KingPos[White] = dest;
            if ((CastleStatus(White) != CastledQSide) &&
                (CastleStatus(White) != CastledKSide))
            {
               state.hashCode ^= w_castle_status[(int)CastleStatus(White)];
               state.hashCode ^= w_castle_status[(int)CantCastleEitherSide];
               state.castleStatus[White] = CantCastleEitherSide;
            }
            break;
         }
         my_Contents[start] = EmptyPiece;
         if (Capture(move) != EmptyPiece) 
         {
            state.moveCount = 0;
            ASSERT(target != InvalidSquare);
            occupied[Black].clear(target);
            Piece cap = MakeBlackPiece(Capture(move));
            Xor(state.hashCode, target, cap);
            switch (Capture(move))
            {
            case Pawn:
               pawn_bits[Black].clear(target);
               Xor(pawnHashCodeB, target, cap);
               if (moveType == EnPassant)
               {
                  my_Contents[target] = EmptyPiece;
                  clearAll(Black,target);
               }
               my_Material[Black].remove_pawn();
               break;
            case Rook:
               rook_bits[Black].clear(target);
               rooksqueens[Black].clear(target);
               my_Material[Black].remove_piece(Rook);
               if ((int)state.castleStatus[Black]<3) {
                  state.hashCode ^= b_castle_status[(int)state.castleStatus[Black]];
                  state.castleStatus[Black] = UpdateCastleStatusB(state.castleStatus[Black],dest);
                  state.hashCode ^= b_castle_status[(int)state.castleStatus[Black]];
               }
               break;
            case Knight:
               knight_bits[Black].clear(target);
               my_Material[Black].remove_piece(Knight);
               break;
            case Bishop:
               bishop_bits[Black].clear(target);
               bishopsqueens[Black].clear(target);
               my_Material[Black].remove_piece(Bishop);
               break;
            case Queen:
               queen_bits[Black].clear(target);
               rooksqueens[Black].clear(target);
               bishopsqueens[Black].clear(target);
               my_Material[Black].remove_piece(Queen);
               break;
            case King:
               ASSERT(0);
               my_KingPos[Black] = InvalidSquare;
               state.castleStatus[Black] = CantCastleEitherSide;
               my_Material[Black].remove_piece(King);
               break;
            default: 
               break;
            }
         }
       }
       setAll(White,dest);
       clearAll(White,start);
   }
   else // side == Black
   {
      if (moveType == KCastle)
      {
         state.moveCount = 0;
         const Square kp = KingPos(Black);

         // update the hash code
         Xor(state.hashCode, kp+3, blackRook);
         Xor(state.hashCode, kp, blackKing);
         Xor(state.hashCode, kp+1, blackRook);
         Xor(state.hashCode, kp+2, blackKing);
         state.hashCode ^= b_castle_status[(int)state.castleStatus[Black]];
         state.hashCode ^= b_castle_status[(int)CastledKSide];

         const int newkp = kp + 2;
         my_KingPos[Black] = newkp;
         state.castleStatus[Black] = CastledKSide;
         // find old square of rook
         Square oldrooksq = kp + 3;
         Square newrooksq = kp + 1;
         my_Contents[kp] = my_Contents[oldrooksq] = EmptyPiece;
         my_Contents[newrooksq] = blackRook;
         my_Contents[kp+2] = blackKing;
         rook_bits[Black].clear(oldrooksq);
         rooksqueens[Black].clear(oldrooksq);
         rook_bits[Black].set(newrooksq);
         rooksqueens[Black].set(newrooksq);
         clearAll(Black,kp);
         clearAll(Black,oldrooksq);
         setAll(Black,newkp);
         setAll(Black,newrooksq);
      }
      else if (moveType == QCastle)
      {
         state.moveCount = 0;
         const Square kp = KingPos(Black);

         // update the hash code
         Xor(state.hashCode, kp-4, blackRook);
         Xor(state.hashCode, kp, blackKing);
         Xor(state.hashCode, kp-1, blackRook);
         Xor(state.hashCode, kp-2, blackKing);
         state.hashCode ^= b_castle_status[(int)state.castleStatus[Black]];
         state.hashCode ^= b_castle_status[(int)CastledQSide];

         const int newkp = kp - 2;
         my_KingPos[Black] = newkp;
         state.castleStatus[Black] = CastledQSide;
         // find old square of rook
         Square oldrooksq = kp - 4;
         Square newrooksq = kp - 1;
         my_Contents[kp] = my_Contents[oldrooksq] = EmptyPiece;
         my_Contents[newrooksq] = blackRook;
         my_Contents[kp-2] = blackKing;
         rook_bits[Black].clear(oldrooksq);
         rooksqueens[Black].clear(oldrooksq);
         rook_bits[Black].set(newrooksq);
         rooksqueens[Black].set(newrooksq);
         clearAll(Black,kp);
         clearAll(Black,oldrooksq);
         setAll(Black,newkp);
         setAll(Black,newrooksq);
      }
      else // not castling
      {
    	 ASSERT(my_Contents[start] != EmptyPiece);
         const Bitmap bits(Bitmap::mask[start] |
                           Bitmap::mask[dest]);
         Square target = dest; // where we captured
#ifdef _DEBUG
		 target = (TypeOfMove(move) == EnPassant) ? old_epsq : dest;
         if (Capture(move) == EmptyPiece) ASSERT(my_Contents[target]==EmptyPiece);
#endif
         switch (PieceMoved(move))
         {
         case Pawn:
            state.moveCount = 0;
            switch (moveType)
            {
            case EnPassant:
               // update hash code
               Xor(state.hashCode, start, blackPawn);
               Xor(state.hashCode, dest, blackPawn);
               Xor(pawnHashCodeB, start, blackPawn);
               Xor(pawnHashCodeB, dest, blackPawn);
               target = old_epsq;
               my_Contents[dest] = blackPawn;
               pawn_bits[Black].set(dest);
               break;
            case Promotion:
               // update hash code
               Xor(state.hashCode, start, blackPawn);
               Xor(state.hashCode, dest, MakeBlackPiece(PromoteTo(move)));
               Xor(pawnHashCodeB, start, blackPawn);
               my_Contents[dest] = MakeBlackPiece(PromoteTo(move));
               my_Material[Black].remove_pawn();
               my_Material[Black].add_piece(PromoteTo(move));
               switch (PromoteTo(move))
               {
               case Knight:
                  knight_bits[Black].set(dest);
                  break;
               case Bishop:
                  bishop_bits[Black].set(dest);
                  bishopsqueens[Black].set(dest);
                  break;
               case Rook:
                  rook_bits[Black].set(dest);
                  rooksqueens[Black].set(dest);
                  break;
               case Queen:
                  queen_bits[Black].set(dest);
                  bishopsqueens[Black].set(dest);
                  rooksqueens[Black].set(dest);
                  break;
               default:
                  break;
               }
               break;
            default:
               Xor(state.hashCode, start, blackPawn );
               Xor(state.hashCode, dest, blackPawn );
               Xor(pawnHashCodeB, start, blackPawn);
               Xor(pawnHashCodeB, dest, blackPawn);
               my_Contents[dest] = blackPawn;
               if (dest - start == 16) // 2-square pawn advance
               { 
                  if (TEST_MASK(Bearing::ep_mask[File(dest)-1][(int)Black],pawn_bits[White])) {
                    state.enPassantSq = dest;
                    state.hashCode ^= ep_codes[0];
                    state.hashCode ^= ep_codes[dest];
                  }
               }
               pawn_bits[Black].set(dest);
               break;
            }
            pawn_bits[Black].clear(start);
            break;
         case Knight:
            Xor(state.hashCode, start, blackKnight);
            Xor(state.hashCode, dest, blackKnight);
            my_Contents[dest] = blackKnight;
            knight_bits[Black].setClear(bits);
            break;
         case Bishop:
            Xor(state.hashCode, start, blackBishop);
            Xor(state.hashCode, dest, blackBishop);
            my_Contents[dest] = blackBishop;
            bishop_bits[Black].setClear(bits);
            bishopsqueens[Black].setClear(bits);
            break;
         case Rook:
            Xor(state.hashCode, start, blackRook );
            Xor(state.hashCode, dest, blackRook );
            my_Contents[dest] = blackRook;
            rook_bits[Black].setClear(bits);
            rooksqueens[Black].setClear(bits);
            if ((int)state.castleStatus[Black]<3) {
                state.hashCode ^= b_castle_status[(int)state.castleStatus[Black]];
                state.castleStatus[Black] = UpdateCastleStatusB(state.castleStatus[Black],start);
                state.hashCode ^= b_castle_status[(int)state.castleStatus[Black]];
            }
            break;
         case Queen:
            Xor(state.hashCode, start, blackQueen);
            Xor(state.hashCode, dest, blackQueen);
            my_Contents[dest] = blackQueen;
            queen_bits[Black].setClear(bits);
            rooksqueens[Black].setClear(bits);
            bishopsqueens[Black].setClear(bits);
            break;
         case King:
            Xor(state.hashCode, start, blackKing );
            Xor(state.hashCode, dest, blackKing );
            my_Contents[dest] = blackKing;
            my_KingPos[Black] = dest;
            if ((CastleStatus(Black) != CastledQSide) &&
                (CastleStatus(Black) != CastledKSide))
            {
               state.hashCode ^= b_castle_status[(int)CastleStatus(Black)];
               state.hashCode ^= b_castle_status[(int)CantCastleEitherSide];
               state.castleStatus[Black] = CantCastleEitherSide;
            }
            break;
         }
         my_Contents[start] = EmptyPiece;
         if (Capture(move) != EmptyPiece)
         {
            state.moveCount = 0;
            ASSERT(target != InvalidSquare);
            occupied[White].clear(target);
            Piece cap = MakeWhitePiece(Capture(move));
            Xor(state.hashCode, target, cap);
            switch (Capture(move)) {
            case Pawn:
               pawn_bits[White].clear(target);
               Xor(pawnHashCodeW, target, cap);
               if (moveType == EnPassant)
               {
                  my_Contents[target] = EmptyPiece;
                  clearAll(White,target);
               }
               my_Material[White].remove_pawn();
               break;
            case Rook:
               rook_bits[White].clear(target);
               rooksqueens[White].clear(target);
               my_Material[White].remove_piece(Rook);
               if ((int)state.castleStatus[White]<3) {
                  state.hashCode ^= w_castle_status[(int)state.castleStatus[White]];
                  state.castleStatus[White] = UpdateCastleStatusW(state.castleStatus[White],dest);
                  state.hashCode ^= w_castle_status[(int)state.castleStatus[White]];
               }
               break;
            case Knight:
               knight_bits[White].clear(target);
               my_Material[White].remove_piece(Knight);
               break;
            case Bishop:
               bishop_bits[White].clear(target);
               bishopsqueens[White].clear(target);
               my_Material[White].remove_piece(Bishop);
               break;
            case Queen:
               queen_bits[White].clear(target);
               rooksqueens[White].clear(target);
               bishopsqueens[White].clear(target);
               my_Material[White].remove_piece(Queen);
               break;
            case King:
               ASSERT(0);
               my_KingPos[White] = InvalidSquare;
               state.castleStatus[White] = CantCastleEitherSide;
               my_Material[White].remove_piece(King);
               break;
            default: 
               break;
            }
         }
         setAll(Black,dest);
         clearAll(Black,start);
      }
   }

   if (Side() == White)
      state.hashCode |= (hash_t)1;
   else
      state.hashCode &= (hash_t)~1;
   *rep_list_head++ = state.hashCode;
   ASSERT(pawnHashCodeW == BoardHash::pawnHash(*this,White));
   ASSERT(pawnHashCodeB == BoardHash::pawnHash(*this,Black));
   side = OppositeSide();
   ASSERT(state.hashCode == BoardHash::hashCode(*this));
   all_occupied = occupied[White];
   all_occupied.Or(occupied[Black]);
#if defined(_DEBUG) && defined(FULL_DEBUG)
   // verify correct updating of bitmaps:
   Board copy(*this);
   copy.set_secondary_vars();
   ASSERT(pawn_bits[White] == copy.pawn_bits[White]);
   ASSERT(knight_bits[White] == copy.knight_bits[White]);
   ASSERT(bishop_bits[White] == copy.bishop_bits[White]);
   ASSERT(rook_bits[White] == copy.rook_bits[White]);
   ASSERT(queen_bits[White] == copy.queen_bits[White]);
   ASSERT(bishopsqueens[White] == copy.bishopsqueens[White]);
   ASSERT(rooksqueens[White] == copy.rooksqueens[White]);
   ASSERT(occupied[White] == copy.occupied[White]);
   ASSERT(occupiedR90 == copy.occupiedR90);
   ASSERT(occupiedDa1 == copy.occupiedDa1);
   ASSERT(occupiedDa8 == copy.occupiedDa8);

   ASSERT(pawn_bits[Black] == copy.pawn_bits[Black]);
   ASSERT(knight_bits[Black] == copy.knight_bits[Black]);
   ASSERT(bishop_bits[Black] == copy.bishop_bits[Black]);
   ASSERT(rook_bits[Black] == copy.rook_bits[Black]);
   ASSERT(queen_bits[Black] == copy.queen_bits[Black]);
   ASSERT(bishopsqueens[Black] == copy.bishopsqueens[Black]);
   ASSERT(rooksqueens[Black] == copy.rooksqueens[Black]);
   ASSERT(occupied[Black] == copy.occupied[Black]);
   ASSERT(occupiedR90 == copy.occupiedR90);
   ASSERT(my_Contents[my_KingPos[White]]==whiteKing);
   ASSERT(my_Contents[my_KingPos[Black]]==blackKing);
#endif     
}

hash_t Board::HashCode( const Move &move ) const
{
   hash_t newHash = state.hashCode;
   if (state.enPassantSq != InvalidSquare)
   {
       newHash ^= ep_codes[state.enPassantSq];
       newHash ^= ep_codes[0];
   }
   const Square start = StartSquare(move);
   const Square dest = DestSquare(move);
   const SpecialType moveType = TypeOfMove(move);
   if (side == White)
   {
      if (moveType == KCastle)
      {
         const Square kp = KingPos(White);
         Xor(newHash, kp+3, whiteRook);
         Xor(newHash, kp, whiteKing);
         Xor(newHash, kp+1, whiteRook);
         Xor(newHash, kp+2, whiteKing);
         newHash ^= w_castle_status[(int)state.castleStatus[White]];
         newHash ^= w_castle_status[(int)CastledKSide];
      }
      else if (moveType == QCastle)
      {
         const Square kp = KingPos(White);
         Xor(newHash, kp-4, whiteRook);
         Xor(newHash, kp, whiteKing);
         Xor(newHash, kp-1, whiteRook);
         Xor(newHash, kp-2, whiteKing);
         newHash ^= w_castle_status[(int)state.castleStatus[White]];
         newHash ^= w_castle_status[(int)CastledQSide];
      }
      else // not castling
      {
         Square target = dest; // where we captured
         switch (PieceMoved(move))
         {
         case Pawn:
            switch (moveType)
            {
            case EnPassant:
               // update hash code
               Xor(newHash, start, whitePawn);
               Xor(newHash, dest, whitePawn);
               target = state.enPassantSq;
               break;
            case Promotion:
               // update hash code
               Xor(newHash, start, whitePawn);
               Xor(newHash, dest, MakeWhitePiece(PromoteTo(move)));
               break;
            default:
               Xor(newHash, start, whitePawn );
               Xor(newHash, dest, whitePawn );
               if (start - dest == 16) // 2-square pawn advance
               {
                  if (TEST_MASK(Bearing::ep_mask[File(dest)-1][(int)White],pawn_bits[Black])) {
                    newHash ^= ep_codes[0];
                    newHash ^= ep_codes[dest];
                  }
               }
               break;
            }
            break;
         case Knight:
            Xor(newHash, start, whiteKnight);
            Xor(newHash, dest, whiteKnight);
            break;
         case Bishop:
            Xor(newHash, start, whiteBishop);
            Xor(newHash, dest, whiteBishop);
            break;
         case Rook:
            Xor(newHash, start, whiteRook );
            Xor(newHash, dest, whiteRook );
            if ((int)state.castleStatus[White]<3) {
               newHash ^= w_castle_status[(int)state.castleStatus[White]];
               newHash ^= w_castle_status[(int)UpdateCastleStatusW(state.castleStatus[White],start)];
            }
            break;
         case Queen:
            Xor(newHash, start, whiteQueen);
            Xor(newHash, dest, whiteQueen);
            break;
         case King:
            Xor(newHash, start, whiteKing );
            Xor(newHash, dest, whiteKing );
            if ((CastleStatus(White) != CastledQSide) &&
                (CastleStatus(White) != CastledKSide))
            {
               newHash ^= w_castle_status[(int)CastleStatus(White)];
               newHash ^= w_castle_status[(int)CantCastleEitherSide];
            }
            break;
         }
         if (Capture(move) != EmptyPiece) 
         {
            Piece cap = MakeBlackPiece(Capture(move));
            Xor(newHash, target, cap);
            if (Capture(move) == Rook) {
               if ((int)state.castleStatus[Black]<3) {
                  newHash ^= b_castle_status[(int)state.castleStatus[Black]];
                  newHash ^= b_castle_status[(int)UpdateCastleStatusB(state.castleStatus[Black],dest)];
               }
            }
         }
      }

   }
   else // side == Black
   {
      if (moveType == KCastle)
      {
         const Square kp = KingPos(Black);
         Xor(newHash, kp+3, blackRook);
         Xor(newHash, kp, blackKing);
         Xor(newHash, kp+1, blackRook);
         Xor(newHash, kp+2, blackKing);
         newHash ^= b_castle_status[(int)state.castleStatus[Black]];
         newHash ^= b_castle_status[(int)CastledKSide];
      }
      else if (moveType == QCastle)
      {
         const Square kp = KingPos(Black);
         Xor(newHash, kp-4, blackRook);
         Xor(newHash, kp, blackKing);
         Xor(newHash, kp-1, blackRook);
         Xor(newHash, kp-2, blackKing);
         newHash ^= b_castle_status[(int)state.castleStatus[Black]];
         newHash ^= b_castle_status[(int)CastledQSide];
      }
      else // not castling
      {
         Square target = dest; // where we captured
         switch (PieceMoved(move))
         {
         case Pawn:
            switch (moveType)
            {
            case EnPassant:
               // update hash code
               Xor(newHash, start, blackPawn);
               Xor(newHash, dest, blackPawn);
               target = state.enPassantSq;
               break;
            case Promotion:
               // update hash code
               Xor(newHash, start, blackPawn);
               Xor(newHash, dest, MakeBlackPiece(PromoteTo(move)));
               break;
            default:
               Xor(newHash, start, blackPawn );
               Xor(newHash, dest, blackPawn );
               if (dest - start == 16) // 2-square pawn advance
               { 
                  if (TEST_MASK(Bearing::ep_mask[File(dest)-1][(int)Black],pawn_bits[White])) {
                    newHash ^= ep_codes[0];
                    newHash ^= ep_codes[dest];
                  }
               }
               break;
            }
            break;
         case Knight:
            Xor(newHash, start, blackKnight);
            Xor(newHash, dest, blackKnight);
            break;
         case Bishop:
            Xor(newHash, start, blackBishop);
            Xor(newHash, dest, blackBishop);
            break;
         case Rook:
            Xor(newHash, start, blackRook );
            Xor(newHash, dest, blackRook );
            if ((int)state.castleStatus[Black]<3) {
               newHash ^= b_castle_status[(int)state.castleStatus[Black]];
               newHash ^= b_castle_status[(int)UpdateCastleStatusB(state.castleStatus[Black],dest)];
            }
            break;
         case Queen:
            Xor(newHash, start, blackQueen);
            Xor(newHash, dest, blackQueen);
            break;
         case King:
            Xor(newHash, start, blackKing );
            Xor(newHash, dest, blackKing );
            if ((CastleStatus(Black) != CastledQSide) &&
                (CastleStatus(Black) != CastledKSide))
            {
               newHash ^= b_castle_status[(int)CastleStatus(Black)];
               newHash ^= b_castle_status[(int)CantCastleEitherSide];
            }
            break;
         }
         if (Capture(move) != EmptyPiece)
         {
            Piece cap = MakeWhitePiece(Capture(move));
            Xor(newHash, target, cap);
            if (Capture(move) == Rook) {
               if ((int)state.castleStatus[White]<3) {
                  newHash ^= w_castle_status[(int)state.castleStatus[White]];
                  newHash ^= w_castle_status[(int)UpdateCastleStatusW(state.castleStatus[White],dest)];
               }
            }
         }

      }
   }

   if (Side() == White)
      newHash |= (hash_t)1;
   else
      newHash &= (hash_t)~1;
   return newHash;
}

void Board::undo_castling(const Square &kp,
                          const Square &oldkingsq, const Square &newrooksq,
                          const Square &oldrooksq)
{
   my_Contents[kp] = EmptyPiece;
   my_Contents[oldrooksq] = MakePiece(Rook,side);
   my_Contents[newrooksq] = EmptyPiece;
   my_Contents[oldkingsq] = MakePiece(King,side);
   my_KingPos[side] = oldkingsq;
   rook_bits[side].set(oldrooksq);
   rook_bits[side].clear(newrooksq);
   rooksqueens[side].set(oldrooksq);
   rooksqueens[side].clear(newrooksq);

   setAll(side,oldrooksq);
   setAll(side,oldkingsq);
   clearAll(side,kp);
   clearAll(side,newrooksq);
}

void Board::UndoMove( const Move move, const Board_State &old_state )
{
   side = OppositeColor(side);
   ASSERT(!IsNull(move));
   const SpecialType moveType = TypeOfMove(move);
   const Square start = StartSquare(move);
   const Square dest = DestSquare(move);
   if (moveType == KCastle)
   {
      Square kp = KingPos(side);
      Square oldrooksq = kp+1;
      Square newrooksq = kp-1;
      Square oldkingsq = kp-2;
      undo_castling(kp,oldkingsq,newrooksq,oldrooksq);
   }
   else if (moveType == QCastle)
   {
      Square kp = KingPos(side);
      Square oldrooksq = kp-2;
      Square newrooksq = kp+1;
      Square oldkingsq = kp+2;
      undo_castling(kp,oldkingsq,newrooksq,oldrooksq);
   }
   else if (side == White)
   {
      const Bitmap bits(Bitmap::mask[start] |
                        Bitmap::mask[dest]);
      // not castling
      Square target;
      target = DestSquare(move);
      // fix up start square:
      if (moveType == Promotion)
      {
         my_Contents[start] = whitePawn;
      }
      else
      {
         my_Contents[start] = MakeWhitePiece(PieceMoved(move));
      }
      setAll(White,start);
      switch (PieceMoved(move))
      {
      case Pawn:
         Xor(pawnHashCodeW,start,whitePawn);
         switch (moveType) {
         case Promotion:
            my_Material[White].add_pawn();
            my_Material[White].remove_piece(PromoteTo(move));
            switch (PromoteTo(move))
            {
            case Knight:
               knight_bits[White].clear(dest);
               break;
            case Bishop:
               bishop_bits[White].clear(dest);
               bishopsqueens[White].clear(dest);
               break;
            case Rook:
               rook_bits[White].clear(dest);
               rooksqueens[White].clear(dest);
               break;
            case Queen:
               queen_bits[White].clear(dest);
               bishopsqueens[White].clear(dest);
               rooksqueens[White].clear(dest);
               break;
            default:
               break;
            }
            break;
         case EnPassant:
           target = DestSquare(move) + RankIncr;
         case Normal:
           pawn_bits[White].clear(dest);
           Xor(pawnHashCodeW,dest,whitePawn);
         default: break;
         }
         pawn_bits[White].set(start);
         break;
      case Knight:
         knight_bits[White].setClear(bits);
         break;
      case Bishop:
         bishop_bits[White].setClear(bits);
         bishopsqueens[White].setClear(bits);
         break;
      case Rook:
         rook_bits[White].setClear(bits);
         rooksqueens[White].setClear(bits);
         break;
      case Queen:
         queen_bits[White].setClear(bits);
         rooksqueens[White].setClear(bits);
         bishopsqueens[White].setClear(bits);
         break;
      case King:
         my_KingPos[White] = start;
         break;
      default:
         break;
      }
      // fix up dest square
      clearAll(White,dest);
      my_Contents[dest] = EmptyPiece;
      my_Contents[target] = MakePiece(Capture(move),Black);
      state.enPassantSq = InvalidSquare;
      if (Capture(move) != EmptyPiece)
      {
         switch (Capture(move))
         {
         case Pawn:
            pawn_bits[Black].set(target);
            Xor(pawnHashCodeB,target,blackPawn);
            my_Material[Black].add_pawn();
            break;
         case Knight:
            knight_bits[Black].set(target);
            my_Material[Black].add_piece(Knight);
            break;
         case Bishop:
            bishop_bits[Black].set(target);
            bishopsqueens[Black].set(target);
            my_Material[Black].add_piece(Bishop);
            break;
         case Rook:
            rook_bits[Black].set(target);
            rooksqueens[Black].set(target);
            my_Material[Black].add_piece(Rook);
            break;
         case Queen:
            queen_bits[Black].set(target);
            bishopsqueens[Black].set(target);
            rooksqueens[Black].set(target);
            my_Material[Black].add_piece(Queen);
            break;
         case King:
            my_KingPos[Black] = target;
            my_Material[Black].add_piece(King);
            break;
         default:
            break;
         }
         setAll(Black,target);
      }
   }
   else // side == Black
   {
      const Bitmap bits(Bitmap::mask[start] |
                        Bitmap::mask[dest]);
      // not castling
      Square target;
      if (moveType == EnPassant)
         target = dest - RankIncr;
      else
         target = dest;
      // fix up start square:
      if (moveType == Promotion)
      {
         my_Contents[start] = blackPawn;
      }
      else
      {
         my_Contents[start] = MakeBlackPiece(PieceMoved(move));
      }
      setAll(Black,start);
      switch (PieceMoved(move))
      {
      case Pawn:
         Xor(pawnHashCodeB,start,blackPawn);
         if (moveType == Promotion)
         {
            my_Material[Black].add_pawn();
            my_Material[Black].remove_piece(PromoteTo(move));
            switch (PromoteTo(move))
            {
            case Knight:
               knight_bits[Black].clear(dest);
               break;
            case Bishop:
               bishop_bits[Black].clear(dest);
               bishopsqueens[Black].clear(dest);
               break;
            case Rook:
               rook_bits[Black].clear(dest);
               rooksqueens[Black].clear(dest);
               break;
            case Queen:
               queen_bits[Black].clear(dest);
               bishopsqueens[Black].clear(dest);
               rooksqueens[Black].clear(dest);
               break;
            default:
               break;
            }
         }
         else
         {
            pawn_bits[Black].clear(dest);
            Xor(pawnHashCodeB,dest,blackPawn);
         }
         pawn_bits[Black].set(start);
         break;
      case Knight:
         knight_bits[Black].setClear(bits);
         break;
      case Bishop:
         bishop_bits[Black].setClear(bits);
         bishopsqueens[Black].setClear(bits);
         break;
      case Rook:
         rook_bits[Black].setClear(bits);
         rooksqueens[Black].setClear(bits);
         break;
      case Queen:
         queen_bits[Black].setClear(bits);
         rooksqueens[Black].setClear(bits);
         bishopsqueens[Black].setClear(bits);
         break;
      case King:
         my_KingPos[Black] = start;
         break;
      default:
         break;
      }
      // fix up dest square
      clearAll(Black,dest);
      my_Contents[dest] = EmptyPiece;
      my_Contents[target] = MakePiece(Capture(move),White);
      state.enPassantSq = InvalidSquare;
      if (Capture(move) != EmptyPiece)
      {
         switch (Capture(move))
         {
         case Pawn:
            pawn_bits[White].set(target);
            Xor(pawnHashCodeW,target,whitePawn);
            my_Material[White].add_pawn();
            break;
         case Knight:
            knight_bits[White].set(target);
            my_Material[White].add_piece(Knight);
            break;
         case Bishop:
            bishop_bits[White].set(target);
            bishopsqueens[White].set(target);
            my_Material[White].add_piece(Bishop);
            break;
         case Rook:
            rook_bits[White].set(target);
            rooksqueens[White].set(target);
            my_Material[White].add_piece(Rook);
            break;
         case Queen:
            queen_bits[White].set(target);
            bishopsqueens[White].set(target);
            rooksqueens[White].set(target);
            my_Material[White].add_piece(Queen);
            break;
         case King:
            my_KingPos[White] = target;
            my_Material[White].add_piece(King);
            break;
         default:
            break;
         }
         setAll(White,target);
      }
   }
   state = old_state;
   
   --rep_list_head;
   all_occupied = occupied[White];
   all_occupied.Or(occupied[Black]);
#if defined(_DEBUG) && defined(FULL_DEBUG)
   // verify correct updating of bitmaps:
   Board copy = *this;
   ASSERT(pawn_bits[White] == copy.pawn_bits[White]);
   ASSERT(knight_bits[White] == copy.knight_bits[White]);
   ASSERT(bishop_bits[White] == copy.bishop_bits[White]);
   ASSERT(rook_bits[White] == copy.rook_bits[White]);
   ASSERT(queen_bits[White] == copy.queen_bits[White]);
   ASSERT(bishopsqueens[White] == copy.bishopsqueens[White]);
   ASSERT(rooksqueens[White] == copy.rooksqueens[White]);
   ASSERT(occupied[White] == copy.occupied[White]);
   ASSERT(occupiedR90 == copy.occupiedR90);
   ASSERT(occupiedDa1 == copy.occupiedDa1);
   ASSERT(occupiedDa8 == copy.occupiedDa8);

   ASSERT(pawn_bits[Black] == copy.pawn_bits[Black]);
   ASSERT(knight_bits[Black] == copy.knight_bits[Black]);
   ASSERT(bishop_bits[Black] == copy.bishop_bits[Black]);
   ASSERT(rook_bits[Black] == copy.rook_bits[Black]);
   ASSERT(queen_bits[Black] == copy.queen_bits[Black]);
   ASSERT(bishopsqueens[Black] == copy.bishopsqueens[Black]);
   ASSERT(rooksqueens[Black] == copy.rooksqueens[Black]);
   ASSERT(occupied[Black] == copy.occupied[Black]);
   ASSERT(occupiedR90 == copy.occupiedR90);
   ASSERT(my_Contents[my_KingPos[White]]==whiteKing);
   ASSERT(my_Contents[my_KingPos[Black]]==blackKing);

#endif     
}

int Board::any_attacks(const Square sq, const ColorType side) const
{
   if (sq == InvalidSquare)
      return 0;

   if (side == White)
   {
      if (TEST_MASK(Bearing::pawn_attacks[sq][White],pawn_bits[White]))
         return 1;

      if (TEST_MASK(Bearing::knight_attacks[sq],knight_bits[White]))
         return 1;

      if (Bearing::king_attacks[sq].is_set(KingPos(White)))
         return 1;
   }
   else
   {
      if (TEST_MASK(Bearing::pawn_attacks[sq][Black],pawn_bits[Black]))
         return 1;

      if (TEST_MASK(Bearing::knight_attacks[sq],knight_bits[Black]))
         return 1;

      if (Bearing::king_attacks[sq].is_set(KingPos(Black)))
         return 1;
   }

   // Find attacks on the rank
   const int r = ((int)sq)/8;
   const int f = File(sq)-1;
   if
       (TEST_MASK(rooksqueens[side],Bearing::rank_mask[r]))
   {
      // See if any rooks or queens are there
      if (TEST_MASK(RankAttacks(*this,sq),rooksqueens[side]))
         return 1;
   }
   if (TEST_MASK(rooksqueens[side],Bearing::file_mask[f]))
   {
      if (TEST_MASK(FileAttacks(*this,sq),rooksqueens[side]))
         return 1;
   }

   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a1_mask[sq]))
   {
      // Now find attacks on the a1-h8 diagonal
      if (TEST_MASK(DiagAtcksA1(*this,sq),bishopsqueens[side]))
         return 1;
   }

   // Now find attacks on the a8-h1 diagonal
   // See if any bishops or queens are there
   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a8_mask[sq]))
   {
      if (TEST_MASK(DiagAtcksA8(*this,sq),bishopsqueens[side]))
         return 1;
   }
   return 0;
}

int Board::any_attacks(const Square sq, const ColorType side,
                        Bitmap &source) const
{
   if (sq == InvalidSquare)
      return 0;

   source = Bearing::pawn_attacks[sq][side];
   source.And(pawn_bits[side]);
   if (!source.is_clear())
      return 1;

   source = Bearing::knight_attacks[sq];
   source.And(knight_bits[side]);
   if (!source.is_clear())
      return 1;

   source = Bearing::king_attacks[sq];
   if (source.is_set(KingPos(side)))
      return 1;

   // Find attacks on the rank
   const int r = ((int)sq)/8;
   const int f = File(sq)-1;
   if
       (TEST_MASK(rooksqueens[side],Bearing::rank_mask[r]))
   {
      source = Bitmap::And(RankAttacks(*this,sq),rooksqueens[side]);
      if (!source.is_clear())
         return 1;
   }

   if (TEST_MASK(rooksqueens[side],Bearing::file_mask[f]))
   {
      if (!Bitmap::And(FileAttacks(*this,sq),rooksqueens[side]).is_clear())
         return 1;
   }

   // Now find attacks on the a1-h8 diagonal
   // find the pieces nearest to sq on the diagonal, then unrotate the bits
   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a1_mask[sq]))
   {
      source = DiagAtcksA1(*this,sq);
      // See if any bishops or queens are there
      if (!source.is_clear())
         return 1;
   }

   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a8_mask[sq]))
   {
      // Now find attacks on the a8-h1 diagonal
      source = DiagAtcksA8(*this,sq);
      if (!source.is_clear())
         return 1;
   }
   return 0;
}

Bitmap Board::calc_attacks(const Square sq, const ColorType side) const
{
   Bitmap retval;

   Bitmap patcks(Bearing::pawn_attacks[sq][side]);
   patcks.And(pawn_bits[side]);
   retval.Or(patcks);

   Bitmap natcks(Bearing::knight_attacks[sq]);
   natcks.And(knight_bits[side]);
   retval.Or(natcks);

   Bitmap katcks(Bearing::king_attacks[sq]);
   Bitmap mask;
   mask.set(KingPos(side));
   katcks.And(mask);
   retval.Or(katcks);

   const int r = ((int)sq)/8;
   const int f = File(sq)-1;
   if
       (TEST_MASK(rooksqueens[side],Bearing::rank_mask[r]))
   {
      retval.data |= Bitmap::And(RankAttacks(*this,sq),rooksqueens[side]).data;
   }

   if (TEST_MASK(rooksqueens[side],Bearing::file_mask[f]))
   {
      // To find attacks on the file, use the 90-degree rotated bitmaps.
      // Same procedure as for ranks, above.
      // shift the occupied bits so that the file that sq occupies are
      // in bits 0-7.
      // find the pieces nearest to sq on the file, then
      // "unrotate" the result and shift it back up.
      // See if any rooks or queens are there
      retval.data |= Bitmap::And(FileAttacks(*this,sq),rooksqueens[side]).data;
   }

   // Now find attacks on the a1-h8 diagonal
   // find the pieces nearest to sq on the diagonal, then unrotate the bits
   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a1_mask[sq]))
   {
      retval.data |= Bitmap::And(DiagAtcksA1(*this,sq),
                            bishopsqueens[side]).data;
   }
   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a8_mask[sq]))
   {
      // Now find attacks on the a8-h1 diagonal
      retval.data |= Bitmap::And(DiagAtcksA8(*this,sq),
                            bishopsqueens[side]).data;
   }
   return retval;
}

Bitmap Board::calc_blocks(const Square sq, const ColorType side) const
{
   Bitmap retval;

   if (side == Black)
   {
       Square origin = sq-RankIncr;
       if (OnBoard(origin) && my_Contents[origin] == blackPawn)
          retval.set(origin);
       if (Rank(sq,Black) == 4 && my_Contents[origin] == EmptyPiece &&
           my_Contents[origin - RankIncr] == blackPawn)
          retval.set(origin - RankIncr);
   }
   else
   {
       Square origin = sq+RankIncr;
       if (OnBoard(origin) && my_Contents[origin] == whitePawn)
          retval.set(origin);
       if (Rank(sq,White) == 4 && my_Contents[origin] == EmptyPiece &&
          my_Contents[origin + RankIncr] == whitePawn)
          retval.set(origin + RankIncr);
   }
   
   Bitmap natcks(Bearing::knight_attacks[sq]);
   natcks.And(knight_bits[side]);
   retval.Or(natcks);

//   if (Bearing::king_attacks[sq].is_set(KingPos(side)))
//      retval.set(KingPos(side));

   const int r = ((int)sq)/8;
   const int f = File(sq)-1;
   if
       (TEST_MASK(rooksqueens[side],Bearing::rank_mask[r]))
   {
      retval.data |= Bitmap::And(RankAttacks(*this,sq),rooksqueens[side]).data;
   }

   if (TEST_MASK(rooksqueens[side],Bearing::file_mask[f]))
   {
      // To find attacks on the file, use the 90-degree rotated bitmaps.
      // Same procedure as for ranks, above.
      // shift the occupied bits so that the file that sq occupies are
      // in bits 0-7.
      // find the pieces nearest to sq on the file, then
      // "unrotate" the result and shift it back up.
      // See if any rooks or queens are there
      retval.data |= Bitmap::And(FileAttacks(*this,sq),rooksqueens[side]).data;
   }

   // Now find attacks on the a1-h8 diagonal
   // find the pieces nearest to sq on the diagonal, then unrotate the bits
   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a1_mask[sq]))
   {
      retval.data |= Bitmap::And(DiagAtcksA1(*this,sq),
                            bishopsqueens[side]).data;
   }
   if (TEST_MASK(bishopsqueens[side],Bearing::diag_a8_mask[sq]))
   {
      // Now find attacks on the a8-h1 diagonal
      retval.data |= Bitmap::And(DiagAtcksA8(*this,sq),
                            bishopsqueens[side]).data;
   }
   return retval;
}


Bitmap Board::min_attacker(const Bitmap &atcks, const ColorType side)
const
{
   Bitmap retval;
   if (side == White)
   {
      retval = Bitmap::And(atcks,pawn_bits[White]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,knight_bits[White]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,bishop_bits[White]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,rook_bits[White]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,queen_bits[White]);
      if (!retval.is_clear())
         return retval;
      if (atcks.is_set(KingPos(White)))
	return atcks;
      else
        return Bitmap(0);
   }
   else
   {
      retval = Bitmap::And(atcks,pawn_bits[Black]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,knight_bits[Black]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,bishop_bits[Black]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,rook_bits[Black]);
      if (!retval.is_clear())
         return retval;
      retval = Bitmap::And(atcks,queen_bits[Black]);
      if (!retval.is_clear())
         return retval;
      if (atcks.is_set(KingPos(Black)))
	return atcks;
      else
        return Bitmap(0);
   }    
}

Bitmap Board::all_pawn_attacks( const ColorType side) const
{
   if (side == White)
   {
      Bitmap pawns1(pawn_bits[White]);
      Bitmap pawns2(pawns1);
      pawns1.shr(7);
      pawns1.andNot(Bitmap(MAKELONGLONG(0x01010101,0x01010101)));
      pawns2.shr(9);
      pawns2.andNot(Bitmap(MAKELONGLONG(0x80808080,0x80808080)));
      return Bitmap::Or(pawns1,pawns2);
   }
   else
   {
      Bitmap pawns1(pawn_bits[Black]);
      Bitmap pawns2(pawns1);
      pawns1.shl(7);
      pawns1.andNot(Bitmap(MAKELONGLONG(0x80808080,0x80808080)));
      pawns2.shl(9);
      pawns2.andNot(Bitmap(MAKELONGLONG(0x01010101,0x01010101)));
      return Bitmap::Or(pawns1,pawns2);
   }
}

CheckStatusType Board::CheckStatus() const
{
   if (state.checkStatus != CheckUnknown)
   {
      return state.checkStatus;
   }
   else if (any_attacks(KingPos(Side()),OppositeSide()))
   {
      // This is a const function, but we cache its result
      Board &b = (Board&)*this;
      b.state.checkStatus = InCheck;
   }
   else
   {
      // This is a const function, but we cache its result
      Board &b = (Board&)*this;
      b.state.checkStatus = NotInCheck;
   }
   return state.checkStatus;
}

// This variant of CheckStatus sees if the last move made
// delivered check. It is generally faster than CheckStatus
// with no param, because we can use the last move information
// to avoid calling any_attacks, in many cases.
CheckStatusType Board::CheckStatus(Move lastMove) const
{
   if (state.checkStatus != CheckUnknown)
   {
      return state.checkStatus;
   }
   if (IsNull(lastMove))
      return CheckStatus();
   Square kp = my_KingPos[Side()];
   Square checker = DestSquare(lastMove);
   int d = Bearing::Directions[checker][kp];
   Board &b = (Board&)*this;
   switch(PieceMoved(lastMove)) {
      case Pawn: {
          if (TypeOfMove(lastMove) != Normal)
             return CheckStatus();
          if (Bearing::pawn_attacks[kp][OppositeSide()].is_set(checker)) {
              b.state.checkStatus = InCheck;
          }
          else if (Bearing::Directions[kp][StartSquare(lastMove)] == 0) {
              b.state.checkStatus = NotInCheck;
          }
          if (state.checkStatus == CheckUnknown)
              return CheckStatus();
          else
              return state.checkStatus;
       }
       case Rook:
        switch(d) {
          case 1:
              if (RankAttacksRight(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -1:
              if (RankAttacksLeft(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case 8:
              if (FileAttacksDown(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -8:
              if (FileAttacksUp(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
          default:
            break;
        }
        if (state.checkStatus == CheckUnknown) {
          // see if move could generate a discovered check
            d = Util::Abs(Bearing::Directions[kp][StartSquare(lastMove)]);
            switch(d) {
	    case 0:
            case 1:
            case 8:
                 b.state.checkStatus = NotInCheck;
                 return NotInCheck;
            case 7: {
              Bitmap attacks = DiagAtcksA1(*this,StartSquare(lastMove));
              if (attacks.is_set(kp) && !Bitmap::And(attacks,bishopsqueens[OppositeSide()]).is_clear())
		  b.state.checkStatus = InCheck;
              else
		  b.state.checkStatus = NotInCheck;
              
	    }
              break;
			case 9: {
              Bitmap attacks = DiagAtcksA8(*this,StartSquare(lastMove));
              if (attacks.is_set(kp) && !Bitmap::And(attacks,bishopsqueens[OppositeSide()]).is_clear())
		  b.state.checkStatus = InCheck;
              else
		  b.state.checkStatus = NotInCheck;
              
	    }
              break;
            default:
	      break;
			} // end switch
	    }
        else
            return state.checkStatus;
   case Bishop: 
        switch(d) {
          case 7:
              if (DiagAtcksA1Lower(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -7:
              if (DiagAtcksA1Upper(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case 9:
              if (DiagAtcksA8Lower(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -9:
              if (DiagAtcksA8Upper(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
          default:
            break;
        }
        if (state.checkStatus == CheckUnknown) {
          // see if move could generate a discovered check
            d = Util::Abs(Bearing::Directions[kp][StartSquare(lastMove)]);
            switch(d) {
            case 0:
            case 7:
            case 9: {
                 b.state.checkStatus = NotInCheck;
                 return NotInCheck;
            }
	      break;
            case 8:{
              Bitmap attacks = FileAttacks(*this,StartSquare(lastMove));
              if (attacks.is_set(kp) && !Bitmap::And(attacks,rooksqueens[OppositeSide()]).is_clear())
		  b.state.checkStatus = InCheck;
              else
		  b.state.checkStatus = NotInCheck;
              
	    }
              break;
			case 1: {
              Bitmap attacks = RankAttacks(*this,StartSquare(lastMove));
              if (attacks.is_set(kp) && !Bitmap::And(attacks,rooksqueens[OppositeSide()]).is_clear())
		  b.state.checkStatus = InCheck;
              else
		  b.state.checkStatus = NotInCheck;
              
	    }
              break;
            default:
                return CheckStatus();
	    } // end switch
        }
        else
            return state.checkStatus;
        break;
      case Knight:
        if (Bearing::knight_attacks[checker].is_set(kp))
        {
           b.state.checkStatus = InCheck;
        }
        else {
            d = Util::Abs(Bearing::Directions[kp][StartSquare(lastMove)]);
            if (d == 0)
            {
               b.state.checkStatus = NotInCheck;
            }
            else
               return CheckStatus();
        }
        break;
        case Queen: {
	  b.state.checkStatus = NotInCheck;     
         switch(d) {
	 case 0: 
	    break;
          case 1:
              if (RankAttacksRight(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -1:
              if (RankAttacksLeft(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case 8:
              if (FileAttacksDown(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -8:
              if (FileAttacksUp(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
          case 7:
              if (DiagAtcksA1Lower(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -7:
              if (DiagAtcksA1Upper(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case 9:
              if (DiagAtcksA8Lower(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
              break;
          case -9:
              if (DiagAtcksA8Upper(*this,checker).is_set(kp)) {
                 b.state.checkStatus = InCheck;
              }
          default:
            break;
          }
	}
        break;
      case King: 
          if (TypeOfMove(lastMove) != Normal) /* castling */
              return CheckStatus();
          if (Bearing::king_attacks[checker].is_set(kp)) {
            b.state.checkStatus = InCheck;
	    return InCheck;
          }
          else if (Bearing::Directions[StartSquare(lastMove)][kp] == 0) {
            b.state.checkStatus = NotInCheck;
	    return NotInCheck;
          }
          else
            return CheckStatus();
      default:
          break;
   }
   return CheckStatus();
}

CheckStatusType Board::wouldCheck(Move lastMove) const {
   Square kp = my_KingPos[OppositeSide()];
   Square checker = DestSquare(lastMove);
   int d = Bearing::Directions[checker][kp];
   switch(PieceMoved(lastMove)) {
      case Pawn: {
          switch (TypeOfMove(lastMove)) {
          case EnPassant:
             if (Bearing::pawn_attacks[kp][Side()].is_set(checker)) {
                 return InCheck;
             }
             return CheckUnknown;             
          case Promotion:
             return CheckUnknown;
          case Normal: {
             if (Bearing::pawn_attacks[kp][Side()].is_set(checker)) {
                 return InCheck;
             }
             else {
               int d2 = Bearing::Directions[kp][StartSquare(lastMove)];
               if (d2 == 0) {
                 return NotInCheck;
               }
               else
                 return Bearing::is_pinned(*this,OppositeSide(),lastMove)
                    ? InCheck : NotInCheck;
             }
          }
          case KCastle:
          case QCastle:
          break;
          }
       }
       case Knight:
        if (Bearing::knight_attacks[checker].is_set(kp))
        {
           return InCheck;
        }
        else {
            int d2 = Bearing::Directions[kp][StartSquare(lastMove)];
            if (d2 == 0)
            {
               return NotInCheck;
            }
            else
               return Bearing::is_pinned(*this,OppositeSide(),lastMove)
                      ? InCheck : NotInCheck;
        }
      case King: {
          int d;
          if (TypeOfMove(lastMove) != Normal){ /* castling */
              return CheckUnknown;
          }
          if (Bearing::king_attacks[DestSquare(lastMove)].is_set(kp)) {
              return InCheck;
          }
          else if ((d=Bearing::Directions[StartSquare(lastMove)][kp]) == 0) {
            return NotInCheck;
          }
          else {
             return Bearing::is_pinned(*this,OppositeSide(),lastMove)
                    ? InCheck : NotInCheck;
          }
         }
       case Bishop:
          switch(d) {
            case 7:
                if (DiagAtcksA1Lower(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -7:
                if (DiagAtcksA1Upper(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case 9:
                if (DiagAtcksA8Lower(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -9:
                if (DiagAtcksA8Upper(*this,checker).is_set(kp)) {
                   return InCheck;
                }
            default:
              break;
          }
          // see if move could generate a discovered check
          {
          int d2 = Bearing::Directions[kp][StartSquare(lastMove)];
          d = Util::Abs(d2);
          if (d == 0 || d == 7 || d == 9) {
               return NotInCheck;
          }
          else
             return Bearing::is_pinned(*this,OppositeSide(),lastMove)
                    ? InCheck : NotInCheck;
          break;
          }
       case Rook:
          switch(d) {
            case 1:
                if (RankAttacksRight(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -1:
                if (RankAttacksLeft(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case 8:
                if (FileAttacksDown(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -8:
                if (FileAttacksUp(*this,checker).is_set(kp)) {
                   return InCheck;
                }
            default:
              break;
          }
          // see if move could generate a discovered check
          {
          int d2 = Bearing::Directions[kp][StartSquare(lastMove)];
          d = Util::Abs(d2);
          if (d == 0 || d == 8 || d == 1) {
                return NotInCheck;
          }
          else
             return Bearing::is_pinned(*this,OppositeSide(),lastMove)
                    ? InCheck : NotInCheck;
          break;
           }
        case Queen: 
          switch(d) {
            case 7:
                if (DiagAtcksA1Lower(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -7:
                if (DiagAtcksA1Upper(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case 9:
                if (DiagAtcksA8Lower(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -9:
                if (DiagAtcksA8Upper(*this,checker).is_set(kp)) {
                   return InCheck;
                }
            case 1:
                if (RankAttacksRight(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -1:
                if (RankAttacksLeft(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case 8:
                if (FileAttacksDown(*this,checker).is_set(kp)) {
                   return InCheck;
                }
                break;
            case -8:
                if (FileAttacksUp(*this,checker).is_set(kp)) {
                   return InCheck;
                }
            default:
              break;
          }
          // see if move could generate a discovered check
          {
          int d2 = Bearing::Directions[kp][StartSquare(lastMove)];
          if (d == 0 && d2==0) {
                return NotInCheck;
          }
          else
             return Bearing::is_pinned(*this,OppositeSide(),lastMove)
                    ? InCheck : NotInCheck;
          }
        default:
            break;
   }
   return CheckUnknown;
}
   
int Board::wasLegal(Move lastMove) const {
	if (IsNull(lastMove)) return 1;
    Square kp = KingPos(OppositeSide());
    switch (TypeOfMove(lastMove)) {
	   case QCastle:
	   case KCastle:
	     return 1; // checked for legality in move generator
       case EnPassant:
   	     return !any_attacks(kp,Side());
       default:
         break;
    }
	if (PieceMoved(lastMove)==King) {
	   return !any_attacks(kp,Side());
    }
	else {
       Square start = StartSquare(lastMove);
       int dir = Bearing::Directions[start][kp];
	   switch (dir) {
          case 1:
              return Bitmap::And(rooksqueens[Side()],RankAttacksLeft(*this,kp)).is_clear();
	      break;
          case -1:
              return Bitmap::And(rooksqueens[Side()],RankAttacksRight(*this,kp)).is_clear();
	      break;
          case 8:
              return Bitmap::And(rooksqueens[Side()],FileAttacksUp(*this,kp)).is_clear();
	      break;
          case -8:
              return Bitmap::And(rooksqueens[Side()],FileAttacksDown(*this,kp)).is_clear();
	      break;
	      case -7:
              return Bitmap::And(bishopsqueens[Side()],DiagAtcksA1Lower(*this,kp)).is_clear();
	      break;
	      case 7:
              return Bitmap::And(bishopsqueens[Side()],DiagAtcksA1Upper(*this,kp)).is_clear();
	      break;
          case -9:
              return Bitmap::And(bishopsqueens[Side()],DiagAtcksA8Lower(*this,kp)).is_clear();
	      break;
	      case 9:
              return Bitmap::And(bishopsqueens[Side()],DiagAtcksA8Upper(*this,kp)).is_clear();
	      break;
	      default:
	          return 1;
        }
    }
}

static void set_bad( istream &i )
{
   i.clear( ios::badbit | i.rdstate() );
}

int Board::rep_count() const
{
   int entries = state.moveCount-2;
   if (entries <= 0) return 0;
   hash_t to_match = HashCode();
   int count = 0;
   for (hash_t *rep_list=rep_list_head-3;
       entries>=0;
       rep_list-=2,entries-=2)
   {
      if (*rep_list == to_match)
      {
         count++;
         if (count >= 2)
         {
            return count;
         }
      }
   }
   return count;
}

void Board::flip() {
   for (int i=0;i<4;i++) {
     for (int j=0;j<8;j++) {
        Piece tmp = my_Contents[i*8+j];
        tmp = MakePiece(TypeOfPiece(tmp),OppositeColor(PieceColor(tmp)));
        Piece tmp2 = my_Contents[(7-i)*8+j];
        tmp2 = MakePiece(TypeOfPiece(tmp2),OppositeColor(PieceColor(tmp2)));
        my_Contents[i*8+j] = tmp2;
        my_Contents[(7-i)*8+j] = tmp;
     }
   }
   side = OppositeColor(side);
   set_secondary_vars();
}

void Board::flip2() {
   for (int i=1;i<=4;i++) {
     for (int j=1;j<=8;j++) {
       Square sq = Square(i,j,White);
       Square sq2 = Square(9-i,j,White);
       Piece tmp = my_Contents[sq];
       my_Contents[sq] = my_Contents[sq2];
       my_Contents[sq2] = tmp;
     }
   }
   set_secondary_vars();
}

istream & operator >> (istream &i, Board &board)
{
   // read in a board position in Forsythe-Edwards (FEN) notation.
   static char buf[128];

   char *bp = buf;
   int c;
   int fields = 0; int count = 0;
   while (i.good() && fields < 4 && (c = i.get()) != '\n' && 
          c != -1 && 
          ++count < 128)
   {
      *bp++ = c;
      if (isspace(c))
         fields++;
   }
   *bp = '\0';
   if (!i)
      return i;
   if (!BoardIO::read_fen(board, buf))
      set_bad(i);
   return i;
}

ostream & operator << (ostream &o, const Board &board)
{
   char buf[128];
   memset(buf,'\0',128);
   BoardIO::write_fen(board,buf,1);
   o << buf;
   return o;
}

