// Copyright 1992, 1997-2000 by Jon Dart.  All Rights Reserved.

#ifndef _BEARING_H
#define _BEARING_H

#include "board.h"

extern const int Direction[2];

extern const int KnightSquares[64][8];

extern const int KingSquares[64][8];

extern const int BishopSquares[64][32];

extern const int RookSquares[64][32];

extern const int RankIncr;	// add this to move 1 rank

#define DIAG_ATTACKS_LEN 220

class Bearing
{
     // Finds pseudo-legal moves for individual pieces, plus
     // related functions.

     public:
     
     enum { MaxBearSq = 28 };	     
     
     static CACHE_ALIGN Bitmap knight_attacks[64];
     static CACHE_ALIGN Bitmap pawn_attacks[64][2];
     static CACHE_ALIGN Bitmap king_attacks[64];
     
     // support for "compact attacks"
     static CACHE_ALIGN byte whichAttack[256][8];
     static CACHE_ALIGN Bitmap rankAttacks[64];
     static CACHE_ALIGN Bitmap rankAttacksUp[64];
     static CACHE_ALIGN Bitmap rankAttacksDown[64];
     static CACHE_ALIGN byte fileAttacks[64];
     static CACHE_ALIGN byte fileAttacksUp[64];
     static CACHE_ALIGN byte fileAttacksDown[64];
     static CACHE_ALIGN Bitmap diagA1Attacks[DIAG_ATTACKS_LEN];
     static CACHE_ALIGN Bitmap diagA8Attacks[DIAG_ATTACKS_LEN];

     static CACHE_ALIGN signed char Directions[64][64];
     static Bitmap file_mask[8];
     static Bitmap rank_mask[8];
     static Bitmap file_mask_down[64];
     static Bitmap file_mask_up[64];
     static Bitmap rank_mask_right[64],rank_mask_left[64];
     static Bitmap diag_a1_mask[64];
     static Bitmap diag_a8_mask[64];
     static Bitmap white_squares;
     static Bitmap black_squares;
     // squares that could attack a given square on rank or file
     static Bitmap rank_file_attack[64];
     // squares that could attack a given square on the diagonal
     static Bitmap diag_attack[64];
     static const int FilesR[64];

     // mask for e.p. checking
     static Bitmap ep_mask[8][2];
     
     // Initialize the bitmaps.  Call before using this class.
     static void init();

     static int is_pinned(const Board &board, ColorType kingColor,
			  Move move) {
       return is_pinned(board,kingColor,board[StartSquare(move)],
			StartSquare(move),DestSquare(move));
     }

     // Return TRUE if moving a pices from source to dest would cause
     // the designated king (of color kingColor) to be in check.
     // Can also be used to find discovered checking moves.
     static int is_pinned(const Board & board, ColorType kingColor, 
                          Piece p, Square source, Square dest);
};

#define OnFile(bitmap,file) (!Bitmap::And(bitmap,Bearing::file_mask[file-1]).is_clear())

FORCEINLINE Bitmap FileAttacks(const Board &board,Square sq)
{
    byte atck = Bearing::whichAttack[board.occupiedR90.rank_bits(Bearing::FilesR[sq])][((int)sq)/8];
    return Bitmap(Bearing::rankAttacks[(int)atck]).shl_all(Bearing::FilesR[sq]);
}

FORCEINLINE Bitmap FileAttacksDown(const Board &board,Square sq) {
  //    byte atck = Bearing::whichAttack[board.occupiedR90.rank_bits(Bearing::FilesR[sq])][((int)sq)/8];
  //    return Bitmap(Bearing::rankAttacksDown[(int)atck]).shl_all(Bearing::FilesR[sq]);
  Bitmap b(Bearing::file_mask_down[sq]);
  b.And(board.all_occupied);
  if (b.is_clear())
    return Bitmap(0);
  else
    return Bitmap::mask[b.first_one()];
}

FORCEINLINE Bitmap FileAttacksUp(const Board &board,Square sq) {
    byte atck = Bearing::whichAttack[board.occupiedR90.rank_bits(Bearing::FilesR[sq])][((int)sq)/8];
    return Bitmap(Bearing::rankAttacksUp[(int)atck]).shl_all(Bearing::FilesR[sq]);
}

FORCEINLINE Bitmap RankAttacks(const Board &board,Square sq) {
      Bitmap value;
      ((char*)&value)[sq/8] = 
        Bearing::fileAttacks[(int)
        Bearing::whichAttack[board.all_occupied.rank_bits((int)sq/8)][File(sq)-1]];
      return value;
}

FORCEINLINE Bitmap RankAttacksRight(const Board &board,Square sq) {
      Bitmap value;
      ((char*)&value)[sq/8] = Bearing::fileAttacksDown[(int)
        Bearing::whichAttack[board.all_occupied.rank_bits((int)sq/8)][File(sq)-1]];
      return value;
}

FORCEINLINE Bitmap RankAttacksLeft(const Board &board,Square sq) {
      Bitmap value;
      ((char*)&value)[sq/8] = 
        Bearing::fileAttacksUp[(int)
        Bearing::whichAttack[board.all_occupied.rank_bits((int)sq/8)][File(sq)-1]];
      return value;
}

FORCEINLINE Bitmap DiagAtcksA1(const Board &board,Square sq) {
   register const DiagInfo2 *info2 = &DiagInfoA1[sq];
   return info2->attacks[(int)
     Bearing::whichAttack[(int)((((byte*)(&board.occupiedDa1))[info2->byte_shift])
       >> info2->bit_shift) & info2->mask][info2->bit]];
}

FORCEINLINE Bitmap DiagAtcksA8(const Board &board,Square sq) {
   register const DiagInfo2 *info2 = &DiagInfoA8[sq];
   return info2->attacks[(int)
     Bearing::whichAttack[(int)((((byte*)(&board.occupiedDa8))[info2->byte_shift])
        >> info2->bit_shift) & info2->mask][info2->bit]];
}

FORCEINLINE Bitmap DiagAtcksA1Upper(const Board &board,Square sq) {
   register const DiagInfo2 *info2 = &DiagInfoA1[sq];
   return Bitmap(info2->attacks[(int)
     Bearing::whichAttack[(int)((((byte*)(&board.occupiedDa1))[info2->byte_shift])
       >> info2->bit_shift) & info2->mask][info2->bit]].data & info2->upperMask.data);
}

FORCEINLINE Bitmap DiagAtcksA1Lower(const Board &board,Square sq) {
   register const DiagInfo2 *info2 = &DiagInfoA1[sq];
   return Bitmap(info2->attacks[(int)
     Bearing::whichAttack[(int)((((byte*)(&board.occupiedDa1))[info2->byte_shift])
       >> info2->bit_shift) & info2->mask][info2->bit]].data & info2->lowerMask.data);
}

FORCEINLINE Bitmap DiagAtcksA8Upper(const Board &board,Square sq) {
   register const DiagInfo2 *info2 = &DiagInfoA8[sq];
   return Bitmap(info2->attacks[(int)
     Bearing::whichAttack[(int)((((byte*)(&board.occupiedDa8))[info2->byte_shift])
        >> info2->bit_shift) & info2->mask][info2->bit]].data & info2->upperMask.data);
}

FORCEINLINE Bitmap DiagAtcksA8Lower(const Board &board,Square sq) {
   register const DiagInfo2 *info2 = &DiagInfoA8[sq];
   return Bitmap(info2->attacks[(int)
     Bearing::whichAttack[(int)((((byte*)(&board.occupiedDa8))[info2->byte_shift])
        >> info2->bit_shift) & info2->mask][info2->bit]].data & info2->lowerMask.data);
}

#endif
