// Main driver module for Arasan chess engine.
// Handles Winboard/xboard/UCI protocol.
// Copyright 1997-2007 by Jon Dart. All Rights Reserved.
//
#include "board.h"
#include "movegen.h"
#include "search.h"
#include "movearr.h"
#include "notation.h"
#include "scoring.h"
#include "chessio.h"
#include "bitmap.h"
#include "bearing.h"
#include "globals.h"
#include "log.h"
#include "util.h"
#include "calctime.h"
#include "eco.h"
#include "strpair.h"
#include "config.h"
#include "learn.h"
#include "boardio.h"
#include "legal.h"
extern "C"
{
#include <string.h>
#include <ctype.h>
}


#include "book.h"
#include "tbprobe.h"
#include <fstream>
#include <iostream>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <sys/timeb.h>

#include <string>
using namespace std;

static int verbose = 0, post = 0;
static SearchController *searcher = NULL;

static Move last_move;
static char last_move_image[80];
static string game_pathname;
static int32 time_left = 0;
static int32 opp_time = 0;
static int minutes = 5;
static int incr = 0;
static int computer = 0;
static bool computer_plays_white = false;
static char *opponent_name = NULL;
static bool ics = false;
static int forceMode = 0, analyzeMode = 0;
static int moves = 40;                            /* moves in a time control bloc */
static Board *ponder_board = NULL, *main_board = NULL;
static bool ponder_move_ok = false;
static Move predicted_move;
static int pondering = 0;
static Move ponder_move;
static Statistics stats, last_stats, ponder_stats;
static int time_target = 0;
static int num_pending = 0;
static int computer_rating = 0;
static int opponent_rating = 0;
#define MAX_PENDING 20
// stack of pending commands
static string pending[MAX_PENDING];
static bool doTrace = false;
static bool easy = false;
static int game_end = 0;
static int32 last_score = Constants::BIG;
static Book_Info book_inf;
static ECO *ecoCoder = NULL;
static string hostname;
static int xboard42 = 0;
static Search_Type srctype = Time_Limit;
static int time_limit;
static int ply_limit;
static char start_fen[128];
static int tb_init = 0;
static int uci = 0;                               // non-zero for UCI mode
static int movestogo = -1;
static int ponderhit = 0, stopped = 0;
static int testing = 0;
static string test_file;

// profile
uint64 total_nodes = 0L;
int total_correct = 0L;
int total_tests = 0L;
time_t total_time = 0L;
int early_exit_plies = Constants::MaxPly;
int early_exit = 0;
int solution_move_count = 0;
time_t solution_time = 0;
int last_iteration_depth = -1;
int iterations_correct = 0;
int moves_to_search = 1;
int flip = 0;
Move solution_moves[10];
bool avoid = false;
ArasanVector<int> solution_times;
static uint64 nodes_to_find_total;
static int depth_to_find_total;
static uint64 time_to_find_total;
static int max_depth;
static struct {
  Move move;   
  int value;
  time_t time;
  int depth;
  uint64 num_nodes;
} search_progress[256];

// forward declaration
static bool do_command(const char *buf, Board &board);

static bool is_move(const Board &b,const char *buf)
{
    //   if (doTrace) cout << "(debug) is_move: checking " << buf << endl;
    if (isalpha(*buf) && isdigit(buf[1]) &&
        isalpha(buf[2]) && isdigit(buf[3]))
        return true;
    else if (strncmp(buf,"o-o",3)==0 || strncmp(buf,"O-O",3)==0 ||
        strncmp(buf,"o-o-o",5)==0 || strncmp(buf,"O-O-O",5) ==0)
        return true;
    else
        return Notation::Value(b,b.Side(),buf) != NullMove;
}


static Move text_to_move(const Board &board,const char *input)
{
    Move rmove;

    // Winboard/xboard 4.2 and above support SAN. First try to
    // decode the move as a SAN string.
    //
    rmove = Notation::Value(board,board.Side(),input);
    if (!IsNull(rmove)) return rmove;             // success

    // Now try "old" coordinate-style notation, used in Winboard/
    // xboard before 4.2
    //
    char buf[80];
    strcpy(buf,input);
    if (strcmp(buf,"o-o") == 0 ||
        strcmp(buf,"o-o-o") == 0 ||
        strcmp(buf,"O-O") == 0 ||
        strcmp(buf,"O-O-O") == 0) {
        char *p =  buf;
        while (*p) {
            *p = toupper(*p); p++;
        }
        rmove = Notation::MoveValue(board,buf,board.Side());
    }
    else {
        Square start = SquareValue(buf);
        Square dest = SquareValue(buf+2);
        if (buf[4]) {
            PieceType promotion =
                PieceCharValue(toupper(buf[4]));
            rmove = CreateMove(board,start,dest,
                promotion);
        }
        else
            rmove = CreateMove(board,start,dest,InvalidPiece);
    }
    return rmove;
}


static bool accept_draw(Board &board)
{
    if (doTrace)
        cout << "(debug) in accept_draw" << endl;
    // Code to handle draw offers.  This is somewhat experimental.
    int rating_diff = opponent_rating - computer_rating;
    // ignore draw if we have just started searching
    if (last_score == Constants::BIG)
        return false;
    // If it's a 0-increment game and the opponent has < 1 minute,
    // and we have more time, decline
    if (incr == 0 && opp_time < 6000 && time_left > opp_time)
        return false;
    // See if we do not have enough material to mate
    const ColorType side = computer_plays_white ? White : Black;
    const Material &ourmat = board.getMaterial(side);
    const Material &oppmat = board.getMaterial(OppositeColor(side));
    if (ourmat.no_pawns() && ourmat.value() <= KING_VALUE + BISHOP_VALUE) {
        // We don't have mating material
        if (doTrace)
            cout << "(debug) no mating material, accept draw" << endl;
        return true;
    }
    // accept a draw in pawnless endings with even material, unless
    // our score is way positive
    if (ourmat.no_pawns() && oppmat.no_pawns() &&
        (Util::Abs(ourmat.value() - oppmat.value()) < (int)PAWN_VALUE/2) &&
         last_score < (int)PAWN_VALUE) {
        if (doTrace)
            cout << "(debug) pawnless ending, accept draw" << endl;
        return true;
    }
    const Material &wMat = board.getMaterial(White);
    const Material &bMat = board.getMaterial(Black);
    if(options.search.use_tablebases &&
       wMat.men() + bMat.men() <= EGTBMenCount) {
        if (doTrace)
            cout << "(debug) checking tablebases .." << endl;
        // accept a draw when the tablebases say it's a draw
        int tbscore;
        if (probe_tb(board,tbscore,0) && tbscore == 0) {
            if (doTrace)
                cout << "(debug) tablebase score says draw" << endl;
            return true;
        }
    }
    if (opponent_rating == 0)
        return false;
    // accept a draw if our score is negative .. how much negative
    // depends on opponent rating.
    if (doTrace)
        cout << "(debug) checking draw score .." << endl;
    ColorType tmp = board.Side();
    board.setSide(side);
    int draw_score = searcher->draw_score(board,
        side,
        board.getMaterial(White).value() + board.getMaterial(Black).value());
    board.setSide(tmp);
    int threshold = (3*PAWN_VALUE)/4 -(PAWN_VALUE*rating_diff)/200;
    if (rating_diff > 0)
        threshold = Util::Max(-PAWN_VALUE,threshold);
    else
        threshold = Util::Min(3*PAWN_VALUE,threshold);
    if (doTrace) {
        cout << "(debug) rating_diff = " << rating_diff << endl;
        cout << "(debug) draw_score = " << draw_score << endl;
        cout << "(debug) last_score = " << last_score << endl;
        cout << "(debug) threshold = " << threshold << endl;
    }
    return draw_score-last_score > threshold;
}


static void do_help()
{

#ifndef _WIN32
    if (!isatty(1)) return;
#endif
    cout << "analyze:         enter Winboard analyze mode" << endl;
    cout << "black:           set computer to play Black" << endl;
    cout << "bk:              show book moves" << endl;
    cout << "computer:        used to indicate the opponent is a computer" << endl;
    cout << "draw:            offer a draw" << endl;
    cout << "easy:            disable pondering" << endl;
    cout << "edit:            enter Winboard edit mode" << endl;
    cout << "force:           disable computer moving" << endl;
    cout << "go:              start searching" << endl;
    cout << "hard:            enable pondering" << endl;
    cout << "hint:            compute a hint for the current position" << endl;
    cout << "ics <hostname>:  set the name of the ICS host" << endl;
    cout << "level <a b c>:   set the time control:" << endl;
    cout << "  a -> moves to time control" << endl;
    cout << "  b -> minutes per game" << endl;
    cout << "  c -> increment in seconds" << endl;
    cout << "name <string>:   set the name of the opponent" << endl;
    cout << "new:             start a new game" << endl;
    cout << "nopost:          disable output during search" << endl;
    cout << "otim <int>:      set opponent time remaining (in centiseconds)" << endl;
    cout << "post:            show output during search" << endl;
    cout << "quit:            terminate the program" << endl;
    cout << "remove:          back up a full move" << endl;
    cout << "resign:          resign the current game" << endl;
    cout << "result <string>: set the game result (0-1, 1/2-1/2 or 1-0)" << endl;
    cout << "sd <x>:          limit thinking to depth x" << endl;
    cout << "setboard <FEN>:  set board to a specified FEN string" << endl;
    cout << "st <x>:          limit thinking to x centiseconds" << endl;
    cout << "time <int>:      set computer time remaining (in centiseconds)" << endl;
    cout << "undo:            back up a half move" << endl;
    cout << "white:           set computer to play White" << endl;
    cout << "test <file> <seconds> <early exit> <# moves> <-v> <outfile>: "<< endl;
    cout << "   - run an EPD testsuite" << endl;
    cout << "eval <file>:     evaluate a FEN position." << endl;
}


static void save_game()
{
    if (uci) return;                              // not supported
    if (doTrace) cout << "(debug) in save_game" << endl;
    if (game_moves->num_moves() == 0 || !options.store_games) {
        if (doTrace) cout << "(debug) out of save_game" << endl;
        return;
    }
    if (options.learning.result_learning) {
        int result = 0;
        const char *gameResult = theLog->getResult();
        if (strncmp(gameResult,"1/2-1/2",7)==0)
            result = 0;
        else if (strncmp(gameResult,"1-0",3)==0)
            result = 1;
        else if (strncmp(gameResult,"0-1",3)==0)
            result = -1;
        else
            result = -2;                          /* incomplete */
        if (result != -2)
            learn_result(computer_plays_white ? White: Black,
                         result,opponent_rating-computer_rating);
    }
    ofstream ofs(game_pathname.c_str(),ios::out | ios::app);
    if (ofs.good()) {
        ArasanVector<StringPair> headers;
        string opening_name, eco;
        if (ecoCoder) {
            if (doTrace) cout << "(debug) calling classify" << endl;
            ecoCoder->classify(*game_moves,eco,opening_name);
            headers.append(StringPair("ECO",eco.c_str()));
        }
        static char timec[50];
        static char crating[15];
        static char orating[15];
        sprintf(crating,"%d",computer_rating);
        sprintf(orating,"%d",opponent_rating);

        if (hostname.length() > 0) {
            headers.append(StringPair("Site",hostname.c_str()));
        }

        if (computer_plays_white) {
            headers.append(StringPair("Black",
                opponent_name ? opponent_name : "?"));
            if (computer_rating)
                headers.append(StringPair("WhiteElo",crating));
            if (opponent_rating)
                headers.append(StringPair("BlackElo",orating));
        }
        else {
            headers.append(StringPair("White",
                opponent_name ? opponent_name : "?"));
            if (opponent_rating)
                headers.append(StringPair("WhiteElo",orating));
            if (computer_rating)
                headers.append(StringPair("BlackElo",crating));
        }
        if (*start_fen) {
            // we had a non-standard starting position for the game
            headers.append(StringPair("FEN",start_fen));
        }
        sprintf(timec,"%d+%d",minutes*60,incr);
        headers.append(StringPair("TimeControl",timec));
        ChessIO::store_pgn(ofs, *game_moves,
            computer_plays_white ? White : Black,
            theLog->getResult(),
            headers);
        // free headers
        headers.removeAll();
        ofs.close();
    }
    else {
        perror("error opening game file.");
    }
    if (doTrace) cout << "(debug) out of save_game" << endl;
}


static int calc_extra_time()
{
    // Determine if we are allowed to use any time beyond
    // the normal time target.
    if (srctype == Fixed_Time || srctype == Fixed_Depth)
        return 0;
    if (time_left < 1500)
        return 0;                                 // no extra time
    else if (time_target <= 100)
        return 100;                               // lightning game
    else if ((incr == 0 && time_left > time_target*10) ||
        (incr > 0 && time_left > time_target*6))
        return time_target;
    else
        return 0;
}


// handle ^C interrupts (sent by winboard)
static void CDECL sigHandler(int)
{
    cout << "got ^C" << endl;
    if (searcher) {
        searcher->getRootSearch()->terminate_now();
    }
    signal(SIGINT,sigHandler);
}


static void move_image(const Board &board, Move m, char *buf, int uci)
{
    // Produce a text string representing the move, in a form
    // the GUI understands
    if (uci) {
        Notation::UCIMoveImage(m,buf);
    }
    else {
        if (TypeOfMove(m) == KCastle) {
            strcpy(buf,"o-o");
        }
        else if (TypeOfMove(m) == QCastle) {
            strcpy(buf,"o-o-o");
        }
        else {
            *buf++ = FileImage(StartSquare(m));
            *buf++ = RankImage(StartSquare(m));
            *buf++ = FileImage(DestSquare(m));
            *buf++ = RankImage(DestSquare(m));
            if (TypeOfMove(m) == Promotion) {
                // N.b. ICS requires lower case.
                *buf++ = tolower(PieceImage(PromoteTo(m)));
            }
            *buf = '\0';
        }
    }
}


struct MultiPVEntry
{
    int depth;
    int score;
    time_t time;
    uint64 nodes;
    unsigned long tb_hits;
    uint64 hash_inserts;
    string best_line_image;
    MultiPVEntry() {}
    MultiPVEntry(const Statistics &stats)
        : depth(stats.depth),score(stats.value),
          time(stats.elapsed_time),nodes(stats.num_nodes),
          tb_hits(stats.tb_hits),hash_inserts(stats.hash_inserts) {
        best_line_image = stats.best_line_image;
    }
} multi_pvs[MAX_PV];

static void uciOut(int depth, int score, time_t time,
uint64 nodes, uint64 tb_hits, uint64 hash_inserts, const char *best_line_image, int multipv)
{
    char buf[512];
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
    std::ostringstream s;
#else
    strstream s(buf,512,ios::out);
#endif
    char score_buf[20];
    Scoring::print_score_uci(score,score_buf);
    s << "info";
    if (multipv !=0) {
        s << " multipv " << multipv;
    }
    s << " depth " << depth << " score " << score_buf <<
        " time " << time*10L << " nodes " << nodes;
    if (time > 10) s << " nps " << (long)((100L*nodes)/time);
    if (tb_hits) {
        s << " tbhits " << tb_hits;
    }
    extern int hashSize;
    s << " hashfull " <<
        (1000*hash_inserts)/hashSize;
    if (*best_line_image) {
        s << " pv ";
        s << best_line_image;
    }
    s << '\0';
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
    strcpy(buf,s.str().c_str());
#endif
    cout << buf << endl;
    if (doTrace) {
        theLog->write(buf); theLog->write_eol();
    }
}


static void uciOut(const Statistics &stats)
{
    uciOut(stats.depth,stats.value,stats.elapsed_time,
        stats.num_nodes,stats.tb_hits,
        stats.new_hash_inserts,stats.best_line_image,0);
}


static void post_output(const Statistics &stats, int force)
{
    last_score = stats.value;
    int score = stats.display_value;
    if (verbose) {
        if (uci) {
            if (stats.elapsed_time) {
                if (options.search.multipv > 1) {
                    // accumulate multiple pvs until we are ready to output them
                    ASSERT(stats.multipv_count<MAX_PV);
                    multi_pvs[stats.multipv_count] = MultiPVEntry(stats);
                    if (stats.multipv_count+1 == options.search.multipv) {
                        for (int i = 0; i < options.search.multipv; i++) {
                            uciOut(multi_pvs[i].depth,multi_pvs[i].score,
                                multi_pvs[i].time,multi_pvs[i].nodes,
                                multi_pvs[i].tb_hits,
                                multi_pvs[i].hash_inserts,
                                multi_pvs[i].best_line_image.c_str(),
                                i+1);
                        }
                    }
                }
                else {
                    uciOut(stats);
                }
            }
        }
    }
    else if (post) {
        if (score > -Constants::BIG+200 &&
            score < Constants::BIG-200)
            score = (100*score)/PAWN_VALUE;
        // "post" output for Winboard
#ifdef _WIN32
        printf("%2u%c %6d %6ld %8I64u %s\n",
#else
        printf("%2u%c %6d %6ld %8llu %s\n",
#endif
            stats.depth,' ',
            score,
            (long)stats.elapsed_time,
            stats.num_nodes,
            stats.best_line_image);
            fflush(stdout);
    }
}


static void post_output(const Statistics &stats)
{
    post_output(stats,0);
}

static int solution_match(Move result) {
    for (int i = 0; i < solution_move_count; ++i) {
      if (MovesEqual(solution_moves[i],result)) {
          return 1;
      }
    }
   return 0;
} 


static void post_test(const Statistics &stats)
{
    if (!IsNull(stats.best_line[0]) &&	max_depth < 256) {
       int ply = stats.depth-1;
       search_progress[max_depth].move = stats.best_line[0];
       search_progress[max_depth].value = stats.value;
       search_progress[max_depth].time = stats.elapsed_time;
       search_progress[max_depth].depth = ply;
       search_progress[max_depth].num_nodes = stats.num_nodes;
       max_depth++;
    }
    if (avoid) {
        // note: doesn't handle multiple "am" moves
        int ok = !MovesEqual(solution_moves[0],stats.best_line[0]);
        if (ok) {
            if (solution_time == -1) solution_time = stats.elapsed_time;
        }
        else 
            solution_time = -1;
        return;
    }
    else {
      if (solution_match(stats.best_line[0])) {
        if ((int)stats.depth > last_iteration_depth) {
           // Wait 2 sec before counting iterations correct, unless
           // we found a mate
           if (stats.elapsed_time >= 200 ||
               stats.value > Constants::BIG-200)
               ++iterations_correct;
               last_iteration_depth = stats.depth;
        }
        if (iterations_correct >= early_exit_plies)
          early_exit = 1;
        if (solution_time == -1) solution_time = stats.elapsed_time;
          return;
      }
      else
	solution_time = -1; // not solved yet, or has moved off solution
    }
    iterations_correct = 0;
    solution_time = -1;
}


static int terminate(const Statistics &stats)
{
    post_test(stats);
    return early_exit;
}


// handle editing the chessboard
static void edit_board(istream &fin, Board &board)
{
    for (int i = 0; i < 64; i++)
        board.setContents(EmptyPiece,i);
        static char buf[80];
        static const char pieces[]="PNBRQKpnbrqk";
        ColorType side = White;
    while (fin.getline(buf,80)) {
        if (strcmp(buf,"white") == 0)
            side = White;
            else if (strcmp(buf,"black")==0)
            side = Black;
        else if (*buf == '#') {
            for (int i = 0; i < 64; i++)
                board.setContents(EmptyPiece,i);
        }
        else if (strcmp(buf,"c")==0)
            side = OppositeColor(side);
            else if (strcmp(buf,".")==0)
            break;
        else {
            if (strchr(pieces,buf[0])) {
                char c = tolower(buf[0]);
                    Piece p = InvalidPiece;
                switch (c) {
                    case 'p':
                        p = MakePiece(Pawn,side);
                            break;
                        case 'n':
                        p = MakePiece(Knight,side);
                            break;
                        case 'b':
                        p = MakePiece(Bishop,side);
                            break;
                        case 'r':
                        p = MakePiece(Rook,side);
                            break;
                        case 'q':
                        p = MakePiece(Queen,side);
                            break;
                        case 'k':
                        p = MakePiece(King,side);
                            break;
                        default:
                        break;
                }

                if (p != InvalidPiece) {
                    Square sq = SquareValue(buf+1);
                        if (!IsInvalid(sq))
                        board.setContents(p,sq);
                }
            }
        }
    }
    board.set_secondary_vars();
    // edit doesn't set the castle status, so try to deduce it
    // from the piece positions
    if (board.KingPos(White) == E1) {
        if (board[A1] == whiteRook &&
            board[H1] == whiteRook)
            board.setCastleStatus(CanCastleEitherSide,White);
        else if (board[A1] == whiteRook)
            board.setCastleStatus(CanCastleQSide,White);
        else if (board[H1] == whiteRook)
            board.setCastleStatus(CanCastleKSide,White);
    }
    else
        board.setCastleStatus(CantCastleEitherSide,White);
    if (board.KingPos(Black) == E8) {
        if (board[A8] == blackRook &&
            board[H8] == blackRook)
            board.setCastleStatus(CanCastleEitherSide,Black);
        else if (board[A8] == blackRook)
            board.setCastleStatus(CanCastleQSide,Black);
        else if (board[H8] == blackRook)
            board.setCastleStatus(CanCastleKSide,Black);
    }
    else
        board.setCastleStatus(CantCastleEitherSide,Black);
}


static void ponder(Board &board, Move move, Move predicted_reply,int uci)
{
    ponder_move_ok = false;
    ponder_move = NullMove;
    ponder_stats.clear();
    if (doTrace) {
        cout << "(debug) predicted reply = ";
        MoveImage(predicted_reply,cout);
        cout << endl;
    }
    if (uci || (!IsNull(move) && !IsNull(predicted_reply))) {
        if (!uci) {
            predicted_move = predicted_reply;
            // We have already set up the ponder board with the board
            // position after our last move. Now make the move we predicted.
            //
            Board_State previous_state = ponder_board->state;
            ASSERT(legal_move(*ponder_board,predicted_reply));
            ponder_board->DoMove(predicted_reply);
            //
            // We must also add this move to the global move list,
            // otherwise repetition detection will be broken.
            //
            game_moves->add_move(board,previous_state,predicted_reply,"");
            // Start a search for the reply
        }
        book_inf.clear();
        if (options.book.book_enabled && opening_book)
            ponder_move = opening_book->book_move(board,book_inf);
        else
            ponder_move = NullMove;
        if (IsNull(ponder_move)) {
            pondering++;
            if (doTrace)
                cout << "(debug) starting to ponder" << endl;
            if (srctype == Fixed_Depth)
                ponder_move = searcher->find_best_move(
                                uci ? *main_board : *ponder_board,
                                srctype,
                                0,
                                0,
                                ply_limit, true, uci,
                                ponder_stats,
                                (doTrace) ? Trace : Silent);
            else
                ponder_move = searcher->find_best_move(
                                uci ? *main_board : *ponder_board,
                                Fixed_Time,
                                99999 /* time limit (999 seconds) */,
                                0,  /* extra time allowed */
                                99 /* ply limit */,
                                true /* background */, uci,
                                ponder_stats,
                                (doTrace) ? Trace : Silent);
            pondering--;
            if (doTrace) {
                cout << "(debug) done pondering" << endl;
                    cout << "(debug) ponder move = ";
                    MoveImage(ponder_move,cout);
                    cout << endl;
            }
        }
        else                                      // we got a book move
            ponder_move = NullMove;
        // Clean up the global move array, if we got no ponder hit.
        if (!uci && game_moves->length() > 0 && strlen(game_moves->last().image())==0)
            game_moves->remove_move();
    }
    if (doTrace) cout << "(debug) out of ponder()" << endl;
}


// Search using the current board position.
//
static Move search(SearchController *searcher, Board &board, Statistics &stats)
{
    last_stats.clear();
    last_score = Constants::BIG;
    book_inf.clear();
    ponder_move = NullMove;
    if (doTrace) cout << "(debug) in search()" << endl;

    Move move = NullMove;
    if (options.book.book_enabled && !testing && opening_book) {
        move = opening_book->book_move(board,book_inf);
    }

    if (IsNull(move)) {
        // no book move
        stats.clear();
        Talk_Level level = Silent;
        if (verbose && !uci) level = Debug;
        else if (doTrace) level = Trace;
        else level = Silent;
        if (srctype == Fixed_Depth) {
            move = searcher->find_best_move(board,
                srctype,
                0,
                0,
                ply_limit, false, uci,
                stats,
                level);
        }
        else {
            time_target =
                (srctype == Fixed_Time) ? time_limit :
            (uci ? calc_time_limit(movestogo, incr, time_left, opp_time, !easy, doTrace)
                : calc_time_limit(moves, minutes, incr, time_left, opp_time, !easy, doTrace));
            if (doTrace) {
                cout << "(debug) entering search, time_target = " << time_target << endl;
                    cout << "(debug) xtra time = " << calc_extra_time() << endl;
            }
            move = searcher->find_best_move(board,
                srctype,
                time_target,
                calc_extra_time(),
                9999, false, uci,
                stats,
                level);
        }
        if (doTrace) {
            cout << "(debug) search done : move = ";
                MoveImage(move,cout);
                cout << endl;
        }
        last_stats = stats;
        if (!forceMode) {
            if (testing)
                post_test(stats);
            else
                post_output(stats,1);
        }
    }
    else {
        if (ics || uci) {
            Move choices[20];
            int scores[20];
            int moveCount = 0;
            if (options.book.book_enabled && opening_book)
                moveCount = opening_book->book_moves(board,choices,scores,20);
            char buf[256];
            if (uci)
                strcpy(buf,"info string book moves: (");
            else if (computer)
                strcpy(buf,"kibitz book moves: (");
            else
                strcpy(buf,"whisper book moves: (");
            for (int i=0;i<moveCount;i++) {
                Notation::Image(board,choices[i],buf+strlen(buf));
                char score_image[12];
                sprintf(score_image,"%d",scores[i]);
                strcat(buf," ");
                strcat(buf,score_image);
                if (i < moveCount-1)
                    strcat(buf,", ");
            }
            strcat(buf,"), choosing ");
            Notation::Image(board,move,buf+strlen(buf));
            cout << buf << endl;
        }
        stats.clear();
    }
    return move;
}


static void send_move(Board &board, Move &move, Statistics
&stats)
{
    last_move = move;
    last_stats = stats;
    ColorType sideToMove = board.Side();
    if (!IsNull(move)) {
        int rep_count = board.rep_count();
        if (!uci && rep_count >= 3) {
            // It is possible the opponent has not claimed the
            // draw by repetition. If it's a draw before we even
            // move, don't make a move, just claim the draw
            cout << "1/2-1/2 {Repetition}" << endl;
            // Wait for Winboard to send a "result" command before
            // actually concluding it's a draw.
            return;
        }
        Notation::Image(board,last_move,last_move_image);
        theLog->add_move(board,last_move,last_move_image,&last_stats,&book_inf,true);
        // Perform book and/or position learning (if enabled):
        learn(board,rep_count);
        char movebuf[20];
        move_image(board,last_move,movebuf,uci);
        Board_State previous_state = board.state;
        board.DoMove(last_move);
        game_moves->add_move(board,previous_state,last_move,last_move_image);

        delete ponder_board;
        ponder_board = new Board(board);

        if (uci) {
            cout << "bestmove " << movebuf;
            if (!IsNull(stats.best_line[1])) {
                move_image(board,stats.best_line[1],movebuf,uci);
                    cout << " ponder " << movebuf;
            }
            cout << endl;
        }
        else if (xboard42) {
            cout << "move " << movebuf << endl;
        }
        else {
            cout << game_moves->num_moves()/2 << ". ... ";
                cout << movebuf << endl;
        }
        cout << (flush);
    }
    else
        cout << "(debug) warning : move is null" << endl;
    if (ics && time_target >= 300 && stats.num_moves > 0) {
        char statbuf[128];
        char score_buf[20];
        Scoring::print_score(last_stats.display_value,score_buf);
        sprintf(statbuf,"%ld.%02lds %s d:%u",
                (long)last_stats.elapsed_time/100,
                (long)last_stats.elapsed_time % 100,
                score_buf,
                last_stats.depth);
        if (last_stats.elapsed_time > 0) {
            strcat(statbuf," nps:");
                last_stats.print_nps(statbuf+strlen(statbuf));
        }
        if (last_stats.tb_hits) {
            char tbstr[100];
            sprintf(tbstr," egtb:%ld/%ld",last_stats.tb_hits,last_stats.tb_probes);
            strcat(statbuf,tbstr);
        }
#if defined(SMP) && defined(SMP_STATS)
        if (last_stats.samples && options.search.ncpus>1) {
            sprintf(statbuf+strlen(statbuf)," cpu:%3.0f%%",(100.0*last_stats.threads)/(float)last_stats.samples);
        }
#endif
        if (ics) {
            if (computer) cout << "kibitz ";
            else cout << "whisper ";
            cout << statbuf << endl;
            if (last_stats.best_line_image[0] != '\0' && !game_end) {
                char *p = strchr(last_stats.best_line_image,' ');
                if (p && *(p+1) != '\0') {
                    if (computer) cout << "kibitz ";
                    else cout << "whisper ";
                    cout << "predicting: " << p+1 << endl;
                }
            }
        }
    }
    if (uci) return;                              // GUI is in charge of game end detection
    // check for checkmate first, before draw (avoids problem
    // when last move was mate but also hits 50 move counter).
    if (last_stats.value >= Constants::BIG-1) {
        if (doTrace) cout << "(debug) last_score = mate" << endl;
        if (sideToMove == White) {
            theLog->setResult("1-0");
            cout << "1-0 {White mates}" << endl;
        }
        else {
            theLog->setResult("0-1");
            cout << "0-1 {Black mates}" << endl;
        }
        game_end = 1;
    }
    else if (last_stats.state == Checkmate) {
        if (doTrace) cout << "(debug) state = Checkmate" << endl;
        if (sideToMove == White) {
            theLog->setResult("0-1");
            cout << "0-1 {Black mates}" << endl;
        }
        else {
            theLog->setResult("1-0");
            cout << "1-0 {White mates}" << endl;
        }
        game_end = 1;
    }
    else if (last_stats.state == Stalemate) {
        cout << "1/2-1/2 {Stalemate}" << endl;
        theLog->setResult("1/2-1/2 {stalemate}");
        game_end = 1;
    }
    else if (board.state.moveCount >= 100) {
        cout << "1/2-1/2 {50 move rule}" << endl;
        theLog->setResult("1/2-1/2 {50 move draw}");
        if (doTrace) cout << "(debug) 50 move draw" << endl;
        game_end = 1;
    }
    else if (Scoring::material_draw(board)) {
        cout << "1/2-1/2 {Insufficient material}" << endl;
        theLog->setResult("1/2-1/2 {insufficient material}");
        if (doTrace) cout << "(debug) material draw" << endl;
        game_end = 1;
    }
    else if (Scoring::repetition_draw(board)) {
        cout << "1/2-1/2 {Repetition}" << endl;
        theLog->setResult("1/2-1/2 {repetition}");
        game_end = 1;
    }
    else if (stats.state == Draw) {
        // We can get here if our opponent does not claim the
        // repetition draw.
        if (doTrace) cout << "(debug) opponent didn't claim repetition?" << endl;
        cout << "1/2-1/2 {Repetition}" << endl;
    }
    else if (last_stats.state == Resigns) {
        // Don't resign a zero-increment game if the opponent is short
        // on time
        if (!(incr == 0 && opp_time < 2000)) {
            // Winboard passes the resign command to ICS, but ignores it
            // itself.
            if (computer_plays_white) {
                theLog->setResult("0-1 {White resigns}");
                cout << "0-1 {White resigns}" << endl;
            }
            else {
                theLog->setResult("1-0 {Black resigns}");
                cout << "1-0 {Black resigns}" << endl;
            }
            game_end = 1;
        }
        else {   // reset flag - we're not resigning
            stats.end_of_game = 0;
            game_end = 0;
        }
    }
}


// Find best move using the current board position.
//
static Move analyze(SearchController &searcher, Board &board, Statistics &stats)
{
    last_stats.clear();
    last_score = Constants::BIG;

    stats.clear();
    if (doTrace)
        cout << "(debug) entering analysis search" << endl;
    Move move = searcher.find_best_move(board,
        Fixed_Time,
        999999,                                   //(100*60*minutes)/moves,
        0,
        9999, false, uci,
        stats,
        Whisper);
    if (doTrace) {
        cout << "(debug) search done : move = ";
        MoveImage(move,cout);
        cout << endl;
    }

    last_stats = stats;
    post_output(stats);

    return move;
}


static void do_options(const char *buf)
{
    while (*buf) {
        char cmd[64],val[64];
        char *p = cmd;
        while (*buf != '=')
            *p++ = *buf++;
        buf++;
        p = val;
        while (*buf && !isspace(*buf))
        *p++ = *buf++;
        if (strcmp(cmd,"search.can_resign")==0) {
            options.search.can_resign =
                (strcmp(val,"true")==0);
        }
        else if (strcmp(cmd,"search.hash_table_size")==0) {
            sscanf(val,"%d",&options.search.hash_table_size);
        }
    }
}


static void doHint()
{
    // try book move first
    Move moves[20];
    int scores[20];
    unsigned count = 0;
    if (options.book.book_enabled && opening_book)
        count = opening_book->book_moves(*main_board,moves,scores,20);
    if (count > 0) {
        char result[64];
        if (count == 1)
            cout << "Book move: " ;
        else
            cout << "Book moves: ";
        for (unsigned i = 0; i<count; i++) {
            Notation::Image(*main_board,moves[i],result);
            cout << result;
            if (i<count-1) cout << ' ';
        }
        cout << endl;
            return;
    }
    else {
        // no book move, see if we have a ponder move
        char *p = last_stats.best_line_image;
        if (*p != '\0') {
            char move[20];
            char *q = move;
            while (*p && !isspace(*p)) p++;
            if (*p) {
                p++;
                while (*p && !isspace(*p)) *q++ = *p++;
            }
            *q='\0';
            if (*move) {
                cout << "Hint: " << move << endl;
                return;
            }
        }
    }
    // no ponder move or book move. If we are already pondering but
    // have no ponder move we could wait a while for a ponder result,
    // but we just return for now.
    if (pondering) return;
    if (doTrace) cout << "(debug) computing hint" << endl;
    Statistics tmp;
    Move move = searcher->find_best_move(*main_board,
                   Fixed_Depth,
                        0,
                        0,
                        4, false, uci,
                        tmp,
                        (doTrace) ? Trace : Silent);
    if (!IsNull(move)) {
        char image[20];
        Notation::Image(*main_board,move,image);
        cout << "Hint: " << image << endl;
    }
}


void add_pending(const char *buf)
{
    ASSERT(num_pending < MAX_PENDING);
    if (doTrace) cout << "(debug) adding to pending list " << buf << ", list size=" << num_pending << endl;
    pending[num_pending] = buf;
    ++num_pending;
}


static bool do_pending(Board &board)
{
    int limit = num_pending;
    if (doTrace) cout << "(debug) in do_pending, list size=" << limit << endl;
    for (int i = 0; i < limit; i++) {
        --num_pending;
        bool terminate;
        if (doTrace)
            cout << "(debug) pending command: " << pending[i] << endl;
        terminate = !do_command(pending[i].c_str(),board);
        if (terminate) {
            int k = 0;
            for (int j = i+1; j<limit;j++) {
                pending[k++] = pending[j];
            }
            if (doTrace) {
                cout << "(debug) still pending:" << endl;
                for (int k=0;k<num_pending;k++)
                    cout << "(debug) " << pending[k].c_str() << endl;
            }
            if (doTrace) cout << "(debug) out of do_pending, list size=" << num_pending << endl;
                return false;
        }
    }
    if (doTrace) cout << "(debug) out of do_pending, list size=" << num_pending << endl;
        return true;
}


//
// Do all pending commands in stack. Return 1 if "new" command
// was executed, 2 if "quit" was seen.
//
static int do_all_pending(Board &board)
{
    int limit = num_pending;
        int retVal = 0;
        if (doTrace) cout << "(debug) in do_all_pending" << endl;
    for (int i = 0; i < limit; i++) {
        --num_pending;
        if (doTrace)
            cout << "(debug) pending command(a): " << pending[i] << endl;
        if (strncmp(pending[i].c_str(),"new",3) == 0) 
            retVal = 1;
        else if (strncmp(pending[i].c_str(),"quit",4) == 0) {
            retVal = 2;
            if (doTrace) cout << "(debug) out of do_all_pending, list size=" << num_pending << endl;
            break;
        }
        do_command(pending[i].c_str(),board);
    }
    if (doTrace) cout << "(debug) out of do_all_pending, list size=" << num_pending << endl;
        return retVal;
}


//
// Check for user move, resign or result in pending stack
//
static int check_pending(Board &board)
{
    int limit = num_pending;
    int i;
    for (i = 0; i < limit; i++) {
        const char *cmd = pending[i].c_str();
        if ((strncmp(cmd,"result",6) == 0) ||
                (strncmp(cmd,"new",3) == 0) ||
                (strncmp(cmd,"quit",4) == 0) ||
                (strncmp(cmd,"resign",6) == 0)) {
            if (doTrace) cout << "(debug) game end signal in pending stack" << endl;
            for (int j=0;j<i;j++) {
                pending[j]=pending[j+i];
                num_pending--;
                assert(num_pending>=0);
            }
            return 1;                             // game end
        }
        else if (strncmp(cmd,"usermove",8) == 0 || is_move(board,cmd)) {
            if (doTrace) cout << "(debug) move in pending stack" << endl;
            for (int j=0;j<i;j++) {
                pending[j]=pending[j+i];
                num_pending--;
                assert(num_pending>=0);
            }
            return 2;
        }
        else {                                    // might as well execute this
            do_command(cmd,board);
        }
    }
    for (int j=0;j<i;j++) {
        pending[j]=pending[j+i];
        num_pending--;
        assert(num_pending>=0);
    }
    return 0;
}


static void analyze(Board &board)
{
    while (analyzeMode) {
        Move reply = analyze(*searcher,board,stats);
        if (check_pending(board)) {
            // user move, do the command then come back to this loop
            if (doTrace) cout << "(debug) exiting analysis loop" << endl;
                break;
        }
        if (do_all_pending(board)==2) return;
        if (reply && !IsNull(reply) && !game_end && analyzeMode) {
            // we got a reply
            if (doTrace) {
                cout << "(debug) analyze result: ";
                MoveImage(reply,cout);
                cout << endl;
            }
            post_output(stats);
            if (Util::Abs(stats.value) >= Constants::BIG-200) {
                // We have a mate score. There is no point analyzing further.
                break;
            }
        }
        else {                                    // no analysis move
            if (doTrace) cout << "(debug) no move returned from analyze" << endl;
                break;
        }
    }
}


static void undo( Board &board)
{
    if (theLog->current() < 1) return;            // ignore "undo"

    board.UndoMove((*theLog)[theLog->current()-1].move(),
                   (*theLog)[theLog->current()-1].state());
    theLog->back_up();
    game_moves->remove_move();
    last_stats.clear();
    if (theLog->current()) {
        last_move = (*theLog)[theLog->current()-1].move();
        strcpy(last_move_image,(*theLog)[theLog->current()-1].image());
    }
    else {
        last_move = NullMove;
        *last_move_image = '\0';
    }
    // In case we have backed up from the end of the game, reset
    // the "game end" flag.
    game_end = 0;
}


// Execute a move made by the opponent or in "force" mode.
static void execute_move(Board &board,Move m)
{
    if (doTrace) {
        cout << "(debug) execute_move: ";
        MoveImage(m,cout);
        cout << endl;
    }
    last_move = m;
    Notation::Image(board,m,last_move_image);
    theLog->add_move(board,m,last_move_image,NULL,NULL,true);
    Board_State previous_state = board.state;
    board.DoMove(m);
    // If our last move added was the pondering move, replace it
    if (game_moves->length() > 0 && strlen(game_moves->last().image())==0)
        game_moves->remove_move();
    game_moves->add_move(board,previous_state,m,last_move_image);
}


// Handle a command received while searching.  terminate is set
// true if the search should stop.  Some commands are executed,
// some are placed on the pending stack to be executed after
// the search completes.
//
static void check_command(char *buf, int &terminate)
{
    if (doTrace)
        cout << "(debug) check_command: " << buf << endl;
    if (doTrace && uci) {
        theLog->write("check_command: "); theLog->write(buf); theLog->write_eol();
    }
    terminate = 0;
    if (uci && strcmp(buf,"stop")==0) {
        add_pending(buf);
        terminate++; stopped++;
        return;
    }
    else if (uci && strcmp(buf,"quit")==0) {
        add_pending(buf);
        terminate++;
        return;
    }
    else if (uci && strcmp(buf,"ponderhit")==0) {
        // We predicted the opponent's move, so we need to
        // continue doing the ponder search but adjust the time
        // limit.
        ++ponderhit;
        ponder_move_ok = true;
        if (pondering) {
            // continue the search in non-ponder mode
            if (srctype != Fixed_Depth) {
                // Compute how much longer we must search
                time_target =
                    (srctype == Fixed_Time) ? time_limit :
                (uci ? calc_time_limit(movestogo, incr, time_left, opp_time, !easy, doTrace)
                    : calc_time_limit(moves, minutes, incr, time_left, opp_time, !easy, doTrace));
                if (time_target <= 0) time_target = 30;
                if (doTrace) {
                    cout << "(debug) time_target = " << time_target << endl;
                }
                searcher->set_time_limit(time_target,(time_t)calc_extra_time());
            }
        }
        else if (!IsNull(ponder_move)) {
            // we have already completed pondering, send the move
            uciOut(ponder_stats);
            send_move(*main_board,ponder_move,ponder_stats);
        }
        searcher->set_talk_level(Whisper);
        searcher->set_background(false);
    }
    else if (uci) {
        add_pending(buf);
    }
    else if (strcmp(buf,"quit")==0 || strcmp(buf,"end")==0 || strncmp(buf,"test",4)==0) {
        add_pending(buf);
        analyzeMode = 0;
        terminate = 1;
    }
    else if (strcmp(buf,"new")==0) {
        add_pending(buf);
        terminate = 1;
    }
    else if (strcmp(buf,"computer")==0) {
        computer = 1;
    }
    else if (strcmp(buf,"?") == 0) {
        // Winboard 3.6 or higher sends this to terminate a search
        // in progress
        if (doTrace) cout << "(debug) terminating." << endl;
        terminate = true;
    }
    else if (strncmp(buf,"ping",4) == 0) {
        // new for Winboard 4.2
        if (pondering) {
            cout << "pong " << buf+5 << (flush) << endl;
        }
        else if (searcher->getRootSearch()->was_terminated()) {
            // we have been asked to terminate but are still
            // searching. Execute the "ping" command only after
            // the search completes.
            add_pending(buf);
        }
    }
    else if (strcmp(buf,"hint")==0) {
        doHint();
    }
    else if (strcmp(buf,"depth")==0) {
    }
    else if (strncmp(buf,"level",5)==0) {
        sscanf(buf+6,"%d %d %d",&moves,&minutes,&incr);
        srctype = Time_Limit;
    }
    else if (strncmp(buf,"st ",3)==0) {
        sscanf(buf+3,"%d",&time_limit);
        srctype = Fixed_Time;
    }
    else if (strncmp(buf,"sd ",3)==0) {
        sscanf(buf+3,"%d",&ply_limit);
        srctype = Fixed_Depth;
    }
    else if (strncmp(buf,"time",4)==0) {
        // my time left
        sscanf(buf+4,"%d",&time_left);
    }
    else if (strncmp(buf,"otime",4)==0) {
        // opponent's time left
        sscanf(buf+4,"%d",&opp_time);
    }
    else if (strcmp(buf,"post")==0) {
        post = 1;
    }
    else if (strcmp(buf,"nopost")==0) {
        post = 0;
    }
    else if (strcmp(buf,"savegame")==0) {
    }
    else if (strcmp(buf,"remove")==0 || strcmp(buf,"undo")==0) {
        add_pending(buf);
        terminate = true;
    }
    else if (strcmp(buf,"resign")==0 || strncmp(buf,"result",6)==0) {
        add_pending(buf);
        game_end = 1;
        terminate = true;
    }
    else if (strcmp(buf,"draw")==0) {
        // "draw" command. Requires winboard 3.6 or higher.
        if (accept_draw(*main_board)) {
            // Notify opponent. don't assume draw is concluded yet.
            cout << "offer draw" << endl;
        }
        else if (doTrace)
            cout << "(debug) draw declined" << endl;
    }
    else if (strcmp(buf,"edit")==0) {
        // TBD what to do in analyze mode
        add_pending(buf);
        terminate = true;
    }
    else if (strncmp(buf,"setboard",8) == 0 ||
    strcmp(buf,"analyze")==0) {
        add_pending(buf);
        terminate = true;
    }
    else if (strcmp(buf,".") == 0) {
        // for analysis mode
        cout << "stat01: " <<
        stats.elapsed_time << " " << stats.num_nodes << " " <<
        stats.depth << " " <<
        stats.mvleft << " " << stats.mvtot << endl;
    }
    else if (strcmp(buf,"go") == 0 || strcmp(buf,"exit") == 0) {
        add_pending(buf);
        terminate = true;
    }
    else if (strcmp(buf,"bk")==0) {
    }
    else if (strcmp(buf,"easy")==0) {
        easy = true;
    }
    else if (strcmp(buf,"hard")==0) {
        easy = false;
    }
    else if (strcmp(buf,"white")==0 || strcmp(buf,"black")==0 || strcmp(buf,"playother")==0) {
        add_pending(buf);
        terminate = true;
    }
    else if (strncmp(buf,"name",4)==0) {
        // We've received the name of our opponent.
        opponent_name = strdup(buf+5);
    }
    else if (strncmp(buf,"bogus",5)==0) {
    }
    else if (strncmp(buf,"force",5)==0) {
        forceMode = 1;
        terminate = true;
    }
    else if (strncmp(buf,"rating",6) == 0) {
        if (doTrace) cout << "(debug) rating = " << buf << endl;
        sscanf(buf+7,"%d %d",&computer_rating, &opponent_rating);
        if (searcher) searcher->setRatingDiff(opponent_rating-computer_rating);
    }
    else if (strncmp(buf,"accepted",8) == 0) {
    }
    else if (strncmp(buf,"computer",8) == 0) {
    }
    else if (strncmp(buf,"option",6) == 0) {
        // hack for Arasan 6.0 so GUI can send options to engine
        do_options(buf+7);
    }
    else {
        if (strncmp(buf,"usermove",8)==0) {
            // new for Winboard 4.2
            char *p = buf;
            char *q = buf+9;
            while (*q && !isspace(*q)) {
                *p++ = *q++;
            }
            *p = '\0';
        }
        // see if it could be a move
        if (isdigit(*buf)) {
            char *p = buf;
            while (*p && !isalpha(*p)) ++p;
            if (*p) {
                char *q = buf;
                while (*p) *q++=*p++;
            }
        }
        if (doTrace)
            cout << "(debug) move text = " << buf << endl;
        if (is_move(*main_board,buf)) {
            // make the move on the board
            Move rmove = text_to_move(*main_board,buf);
            last_move = rmove;

            if (doTrace) {
                cout << "(debug) predicted move = ";
                MoveImage(predicted_move,cout);
                cout << " last move = ";
                MoveImage(last_move,cout);
                cout << endl;
            }
            if (forceMode || analyzeMode || !pondering) {
                add_pending(buf);
                terminate = true;
            }
            else if (!IsNull(predicted_move) &&
                     MovesEqual(predicted_move,last_move)) {
                // make the move on the board
                if (doTrace) {
                    cout << "(debug) ponder ok" << endl;
                }
                execute_move(*main_board,last_move);

                // We predicted the opponent's move, so we need to
                // continue doing the ponder search but adjust the time
                // limit.
                ponder_move_ok = true;
                if (srctype != Fixed_Depth) {
                    // Compute how much longer we must search

                    time_target =
                        (srctype == Fixed_Time) ? time_limit :
                    (uci ? calc_time_limit(movestogo, incr, time_left, opp_time, !easy, doTrace)
                        : calc_time_limit(moves, minutes, incr, time_left, opp_time, !easy, doTrace));
                    if (time_target <= 0) time_target = 30;
                    if (doTrace) {
                        cout << "(debug) time_target = " << time_target << endl;
                        cout << "(debug) xtra time = " << calc_extra_time() << endl;
                    }
                    searcher->set_time_limit(time_target,(time_t)calc_extra_time());
                }
                searcher->set_talk_level(Whisper);
                searcher->set_background(false);
                terminate = false;
            }
            else {
                if (doTrace) cout << "(debug) ponder not ok" << endl;
                // We can't use the results of pondering because we
                // did not predict the opponent's move.  Stop the
                // search and then execute the move.
                ponder_move_ok = false;
                add_pending(buf);
                terminate = true;
            }
        }
    }
}


static int get_value(const char *&p,char *value)
{
    while (*p && isspace(*p)) p++;
        if (*p == '\0') return 0;
        char *q = value;
        while (*p && !isspace(*p)) *q++ = *p++;
        *q = '\0';
        return 1;
}


static const char *get_move(const char *buf,const Board &board, Move &m)
{
    const char *p = buf;
        m = NullMove;
    if (*p) {
        while (isspace(*p) && *p != '\0') ++p;
        if (*p == '\0')
           return NULL;
        char tmp[10];
        int i = 0;
        char *q = tmp;
        while (!isspace(*p) && *p != '+' && *p != '\0' && i < 10) {
            *q++ = *p++;
                ++i;
        }
        *q = '\0';
        m = Notation::Value(board,board.Side(),tmp);
        if (*p == '+') ++p;
    }
    return p;
}


// for support of the "test" command
static Move search(Board &board, int ply_limit,
int time_limit, Statistics &stats,
Move excludes [], int num_excludes)
{
    Move move = NullMove;
    solution_time = -1;

    move = searcher->find_best_move(board,
                                    srctype,
                                    time_limit, 0, ply_limit,
                                    0, 0, stats,
                                    verbose ? Debug : Silent,
                                    excludes, num_excludes);

    char move_text[20];
    Notation::Image(board, move, move_text);
    if (num_excludes)
       cout << "result(" << num_excludes+1 << "):";
    else
       cout << "result:";
    char score_buf[20];
    Scoring::print_score(stats.display_value,score_buf);
    cout << '\t' << move_text << "\t" << stats.elapsed_time/100 <<
            " seconds.\tscore:" << score_buf << "\t"  <<
            stats.num_nodes << " nodes.";
    total_time += stats.elapsed_time;
    total_nodes += stats.num_nodes;
    game_moves->removeAll();
    Scoring::cleanup();
    return move;
}


static void do_test(string test_file)
{
   if (!tb_init) {
     delayed_init(); tb_init++;
     if (EGTBMenCount)
        cerr << "found " << EGTBMenCount << "-man tablebases in directory " << options.search.tablebase_location.c_str() << endl;
    }
    int tmp = options.book.book_enabled;
    options.book.book_enabled = 0;
    total_nodes = (uint64)0;
    total_correct = total_tests = 0;
    total_time = (time_t)0;
    nodes_to_find_total = (uint64)0;
    depth_to_find_total = 0;
    time_to_find_total = (uint64)0;

    solution_times.removeAll();
    Board board;
#if defined (_WIN32) || (defined (__GNUC__) && (__GNUC__ >=3))
    ifstream pos_file( test_file.c_str(), ios::in);
#else
    ifstream pos_file( test_file.c_str(), ios::in | ios::nocreate);
#endif
    char buf[512];
    while (!pos_file.eof()) {
        pos_file.getline(buf,511);
        if (!pos_file) {
            cout << "File not found, or bad format." << endl;
            return;
        }
        // Try to parse this line as an EPD command.
#if defined(__GNUC__) && _GNUC_PREREQ(3,2)
        string istr(buf);
        istringstream stream(istr);
#else
        istrstream stream(buf,512);
#endif
        string id, comment;
        EPDRecord *epd_rec =
            ChessIO::readEPDRecord(stream,board);
        if (epd_rec->hasError()) {
            cerr << "error in EPD record ";
            if (id.length()>0) cerr << id;
            cerr << ": ";
            cerr << epd_rec->getError();
            cerr << endl;
        }
        else if (epd_rec != NULL) {
            int i;
            solution_move_count = 0;
            int illegal=0;
            id = "";
            for (i = 0; i < epd_rec->getSize(); i++) {
                string key, val;
                epd_rec->getData(i,key,val);
                if (key == "bm" || key == "am") {
                    Move m;
                    const char *p = val.c_str();
                    while (*p) {
                        p = get_move(p,board,m);
                        if (IsNull(m)) {
                            ++illegal;
                        }
                        else if (solution_move_count < 10) {
                            solution_moves[solution_move_count++] = m;
                        }
                        avoid = (key == "am");
                    }
                }
                else if (key == "id") {
                    id = val;
                }
		else if (key == "c0") {
		  comment = val;
		}
            }
            if (illegal) {
                cerr << "illegal or invalid solution move(s) for EPD record ";
                if (id.length()>0) cerr << id;
                cerr << endl;
                continue;
            }
            else if (!solution_move_count) {
                cerr << "no solution move(s) for EPD record ";
                if (id.length()>0) cerr << id;
                cerr << endl;
                continue;
            }
            last_iteration_depth = -1;
            iterations_correct = 0;
            early_exit = 0;
            solution_time = -1;
	    max_depth = 0;
            testing = 1;
            cout << id << ' ';
	    if (comment.length()) cout << comment << ' ';
            char move_string[100];
            if (avoid) {
                Notation::Image(board,solution_moves[0],move_string);
                    cout << "am " << move_string << endl;
            }
            else {
                cout << "bm";
                for (int i = 0; i < solution_move_count; i++) {
                    Notation::Image(board,solution_moves[i],move_string);
                        cout << ' ' << move_string;
                }
                cout << endl;
            }
            srctype = Fixed_Time;
            searcher->clearHashTables();
            Move excludes[Constants::MaxMoves];
            for (int index = 0; index < moves_to_search; index++) {
                Move result = search(board,ply_limit,time_limit,stats,
                                     excludes,index);
                if (IsNull(result)) break;
                excludes[index] = result;
                int correct = avoid ? 1 : 0;
                for (int i = 0; i < solution_move_count; ++i) {
                    if (MovesEqual(solution_moves[i],result)) {
                        correct = (avoid) ? 0 : 1;
                            break;
                    }
                }
                solution_times.append((int)solution_time);
                total_tests++;
		if (correct) {
                    cout << "\t++ correct" << endl;
                    total_correct++;
		}
		else {
                    cout << "\t** error" << endl;
		}
                cout << stats.best_line_image << endl;
                if (correct && max_depth>0) {
		  uint64 nodes_to_find = (uint64)0;
		  int depth_to_find = 0;
		  time_t time_to_find = 0;
           	    for (int i=max_depth-1;i>=0;i--) {
 	              if (avoid) {
                         if (MovesEqual(search_progress[i].move,result)) {
               	            nodes_to_find = search_progress[i+1].num_nodes;
			    time_to_find = search_progress[i+1].time;
            	            depth_to_find = search_progress[i+1].depth;
            	            break;
            	         }
		      }
		      else {
                         if (!solution_match(search_progress[i].move)) {
               	            nodes_to_find = search_progress[i+1].num_nodes;
			    time_to_find = search_progress[i+1].time;
            	            depth_to_find = search_progress[i+1].depth;
            	            break;
            	         }
		      }
            	    }
		    nodes_to_find_total += nodes_to_find;
		    depth_to_find_total += depth_to_find;
		    time_to_find_total += time_to_find;
                }
            }
        }

        char c;
        while (!pos_file.eof()) {
            c = pos_file.get();
            if (!isspace(c) && c != '\n') {
                if (!pos_file.eof())
                    pos_file.putback(c);
                    break;
            }
        }
    }
    pos_file.close();
    cout << endl << "solution times:" << endl;
    cout << "         ";
    int i = 0;
    for (i = 0; i < 10; i++)
        cout << i << "      ";
    cout << endl;
    double score = 0.0;
    for (i = 0; i < solution_times.length(); i++) {
        char digits[15];
        if (i == 0) {
            sprintf(digits,"% 4d |       ",i);
            cout << endl << digits;
        }
        else if ((i+1) % 10 == 0) {
            sprintf(digits,"% 4d |",(i+1)/10);
            cout << endl << digits;
        }
        if (solution_times[i] == -1) {
            cout << "  ***  ";
        }
        else {
            sprintf(digits,"%6.2f ",solution_times[i]/100.0);
            cout << digits;
            score += (float)time_limit/100.0 - solution_times[i]/100.0;
        }
    }
    cout << endl << endl << "correct : " << total_correct << '/' <<
        total_tests << endl;
    if (total_correct) {
       string avg = "";
       if (total_correct > 1) avg = "avg. ";
       cout << avg << "nodes to solution : ";
       uint64 avg_nodes = nodes_to_find_total/total_correct;
       if (avg_nodes > 1000000L) {
         cout << (float)(avg_nodes)/1000000.0 << "M" << endl;
       }
       else
         cout << (float)(avg_nodes)/1000.0 << "K" << endl;
       cout << avg << "depth to solution : " << (float)(depth_to_find_total)/total_correct << endl;
       cout << avg << "time to solution  : " << (float)(time_to_find_total)/(100.0*total_correct) << " sec." << endl;
    }
    options.book.book_enabled = tmp;
}


// Execute a command, return false if program should terminate.
static bool do_command(const char *buf, Board &board)
{
    if (doTrace)
        cout << "(debug) do_command: " << buf << endl;
    if (doTrace && uci) {
        theLog->write(buf); theLog->write_eol();
    }
    if (strcmp(buf,"uci")==0) {
        uci = 1; verbose = 1;                     // TBD: fixed for now
        // Learning is disabled because we don't have full game history w/ scores
        options.learning.score_learning = options.learning.position_learning = 0;
        cout << "id name " << "Arasan " << Arasan_Version;
#ifdef SMP
        cout << " (SMP)";
#endif
        cout << endl;
        cout << "id author Jon Dart" << endl;
        cout << "option name Hash type spin default 8 min 4 max 1000" << endl;
        cout << "option name Ponder type check default false" << endl;
        cout << "option name NalimovPath type string" << endl;
        cout << "option name NalimovCache type spin default 1 min 1 max 32" << endl;
        cout << "option name MultiPV type spin default 1 min 1 max " << MAX_PV << endl;
        cout << "option name OwnBook type check default true" << endl;
#ifdef SMP
        cout << "option name Threads type spin default " <<
        options.search.ncpus << " min 1 max " <<
        Constants::MaxCPUs << endl;
#endif
        cout << "uciok" << endl;
        return true;
    }
    else if (uci && strcmp(buf,"quit")==0) {
        return false;
    }
    else if (uci && strncmp(buf,"setoption",9)==0) {
        const char *p = buf+15;                   // after "setoption name "
        char name[255]; char *q = name;
        while (!isspace(*p) && q-name<=250) *q++ = *p++;
        *q = '\0';
        p += 7;
        char value[255];
        q = value;
        while (*p && !isspace(*p) && q-value<=250) *q++ = *p++;
        *q = '\0';
        if (strcmp(name,"Hash")==0) {
            options.search.hash_table_size = atol(value)*1000;
                searcher->init();
        }
        else if (strcmp(name,"Ponder")==0) {
            easy = (strcmp(value,"true")!=0);
        }
        else if (strcmp(name,"NalimovCache")==0) {
            options.search.tb_cache_size = atol(value)*1000;
        }
        else if (strcmp(name,"NalimovPath")==0) {
            options.search.tablebase_location = value;
        }
        else if (strcmp(name,"OwnBook")==0) {
            options.book.book_enabled = (strcmp(value,"true")==0);
        }
        else if (strcmp(name,"MultiPV")==0) {
            options.search.multipv = atoi(value);
            stats.multipv_count = 0;
        }
#ifdef SMP
        else if (strcmp(name,"Threads")==0) {
            options.search.ncpus = atoi(value);
	    searcher->setThreadCount(options.search.ncpus);
        }
#endif
    }
    else if (uci && strcmp(buf,"ucinewgame")==0) {
        do_command("new",board);
        return true;
    }
    else if (uci && strcmp(buf,"isready")==0) {
        if (!tb_init) {
            delayed_init(); tb_init++;
            if (EGTBMenCount)
                cerr << "found " << EGTBMenCount << "-man tablebases in directory " << options.search.tablebase_location.c_str() << endl;
        }
        cout << "readyok" << endl;
    }
    else if (uci && strncmp(buf,"position",8)==0) {
        ponder_move = NullMove;
        const char *p = buf+9;
        if (strncmp(p,"startpos",8)==0) {
            p += 9;
            board.Reset();
            game_moves->removeAll();
        }
        else if (strncmp(p,"fen",3)==0) {
            char fen[256]; char *q = fen; p+=4;
            while (*p && *p != '\013' && q-fen<255) {
                *q++=*p++;
            }
            *q = '\0';
            BoardIO::read_fen(board, fen);
            if (doTrace) cout << "(debug) read fen" << endl;
        }
        if (*p && strncmp(p,"moves",5)==0) {
            // "moves" clause
            p += 6;
            while (*p) {
                while (*p && isspace(*p)) ++p;
                char movebuf[64]; char *q = movebuf;
                while (*p && !isspace(*p)) *q++=*p++;
                *q = '\0';
                if (strlen(movebuf)==0) break;
                Square start = SquareValue(movebuf);
                Square dest = SquareValue(movebuf+2);
                PieceType promotion;
                switch (*(movebuf+4)) {
                    case 'q': promotion = Queen; break;
                    case 'n': promotion = Knight; break;
                    case 'b': promotion = Bishop; break;
                    case 'r': promotion = Rook; break;
                    default: promotion = InvalidPiece; break;
                }
                Move m = CreateMove(board,start,dest,promotion);
                Board_State previous_state = board.state;
                board.DoMove(m);
                game_moves->add_move(board,previous_state,m,"");
            }
        }
    }
    else if (strncmp(buf,"test",4)==0) {
        string filename;
        ofstream *out_file = NULL;
        streambuf *sbuf = cout.rdbuf();
        char *p = strtok((char*)(buf+5)," ");
        if (p) {
            filename = p;
            p = strtok(NULL," ");
            if (p) {
                time_limit = 100*atoi(p);
                p = strtok(NULL," ");
                early_exit_plies = Constants::MaxPly;
                moves_to_search = 1;
                verbose = 0;
                while (p) {
                    if (strcmp(p,"-v")==0) {
                        p = strtok(NULL," ");
                        ++verbose;
                        continue;
                    }
                    else if (strcmp(p,"-x")==0) {
                        p = strtok(NULL," ");
                        early_exit_plies = atoi(p);
                        p = strtok(NULL," ");
                    }
                    else if (strcmp(p,"-N")==0) {
                        p = strtok(NULL," ");
                        moves_to_search = atoi(p);
                        p = strtok(NULL," ");
                    }
                    else {
                        out_file = new ofstream( p, ios::out | ios::trunc);
                        // redirect stdout
                        cout.rdbuf(out_file->rdbuf());
                        break;
                    }
                }
                do_command("new",board);
                PostFunction old_post = searcher->register_post_function(post_test);
                TerminateFunction old_terminate = searcher->register_terminate_function(terminate);
                int tmp = options.book.book_enabled;
                options.book.book_enabled = 0;
                do_test(filename);
                if (out_file) {
                    out_file->close();
                    delete out_file;
                    cout.rdbuf(sbuf);         // restore console output
                }
                options.book.book_enabled = tmp;
                searcher->register_post_function(old_post);
                searcher->register_terminate_function(old_terminate);
                cout << "test complete" << endl;
            }
            else
                cout << "invalid command" << endl;
        }
        else
            cout << "invalid command" << endl;
    }
    else if (strncmp(buf,"eval",4)==0) {
        string filename;
            const char *p = buf + 4;
        if (p) {
            filename = p;
#if defined (_WIN32) || (defined (__GNUC__) && (__GNUC__ >=3))
            ifstream pos_file( p, ios::in);
#else
            ifstream pos_file( p, ios::in | ios::nocreate);
#endif
            pos_file >> board;
            if (!pos_file.good()) {
                cout << "File not found, or bad format." << endl;
            }
            else {
                int tbscore;
                char score_buf[20];
                if (options.search.use_tablebases && probe_tb(board,tbscore,0)) {
                    Scoring::print_score(tbscore,score_buf);
                    static char line[128];
                    sprintf(line,"score = %s (from tablebases)\n",score_buf);
                    cout << line << endl;
                    int score;
                    if ((score = Scoring::tryBitbase(board))!= Scoring::INVALID_SCORE)
                        cout << "bitbase score=" << score << endl;
                }
                Scoring::init();
                SearchController c;
                Search s(&c,NULL);
                if (Scoring::is_draw(board))
                    cout << "position evaluates to draw (statically)" << endl;
                Scoring::print_score(Scoring::evalu8(board,&s),score_buf);
                cout << "score = " << score_buf << endl;
            }
        }
    }
    else if (uci && strcmp(buf,"stop")==0) {
        // send ponder move (in case we did not do so earlier)
        if (!IsNull(ponder_move)) {
            uciOut(ponder_stats);
            send_move(board,ponder_move,ponder_stats);
        }
        return true;
    }
    else if (uci && strncmp(buf,"go",2)==0) {
        const char *p = buf+2;
        char option[64];
        char value[64];
        srctype = Time_Limit;
        int do_ponder = 0;
        movestogo = -1;
        while (*p) {
            if (!get_value(p,option)) break;
            if (strcmp(option,"wtime")==0) {
                if (!get_value(p,value)) break;
                long t = atol(value);
                if (board.Side() == White)
                    time_left = t/10;
                else
                    opp_time = t/10;
            }
            else if (strcmp(option,"btime")==0) {
                if (!get_value(p,value)) break;
                long t = atol(value);
                if (board.Side() == Black)
                    time_left = t/10;
                else
                    opp_time = t/10;
            }
            else if (strcmp(option,"infinite")==0) {
                srctype = Fixed_Time;
                time_limit = 999999;
            }
            else if (strcmp(option,"winc")==0) {
                if (!get_value(p,value)) break;
                incr = atoi(value)/1000;
            }
            else if (strcmp(option,"binc")==0) {
                if (!get_value(p,value)) break;
                incr = atoi(value)/1000;
            }
            else if (strcmp(option,"movestogo")==0) {
                if (!get_value(p,value)) break;
                movestogo = atoi(value);
            }
            else if (strcmp(option,"depth")==0) {
                if (!get_value(p,value)) break;
                srctype = Fixed_Depth;
                ply_limit = atoi(value);
            }
            else if (strcmp(option,"movetime")==0) {
                if (!get_value(p,value)) break;
                srctype = Fixed_Time;
                time_limit = atoi(value)/10;
            }
            else if (strcmp(option,"ponder")==0) {
                ++do_ponder;
                ponderhit = stopped = 0;
            }
        }
        forceMode = 0;
        if (do_ponder) {
            ponder(board,NullMove,NullMove,1);
            // We should not send the move unless we have received a
            // "ponderhit" command, in which case we were pondering the
            // right move. If pondering completes early, we must wait
            // for ponderhit before sending. But also send the move if
            // we were stopped - this is the "handshake" that tells the
            // UI we received the stop.
            if (ponderhit || stopped) {
                // ensure we send an "info" command - may not have been
                // sent if the previous move was forced or a tb hit.
                uciOut(ponder_stats);
                send_move(board,ponder_move,ponder_stats);
                ponder_move = NullMove;
            }
        }
        else {
            Move reply = search(searcher,board,stats);
            uciOut(stats);
            send_move(board,reply,stats);
        }
    }
    else if (uci && strcmp(buf,"ponderhit")==0) {
        ++ponderhit;
        if (!IsNull(ponder_move)) {
            uciOut(ponder_stats);
            send_move(board,ponder_move,ponder_stats);
        }
    }
    else if (strcmp(buf,"help")==0) {
        do_help();
    }
    else if (strcmp(buf,"quit")==0 || strcmp(buf,"end")==0) {
        return false;
    }
    else if (strcmp(buf,"new")==0) {
        save_game();
        board.Reset();
        theLog->clear();
        if (!uci) theLog->write_header();
        computer_plays_white = false;
        // Note: "new" does not reset analyze mode
        forceMode = 0;
        game_end = 0;
        testing = 0;
        computer = 0;
        stats.clear();
        ponder_stats.clear();
        last_stats.clear();
        last_move = NullMove;
        *last_move_image = '\0';
        game_moves->removeAll();
        predicted_move = NullMove;
        pondering = 0;
        ponder_move_ok = false;
        *start_fen = '\0';
        searcher->register_command_function(check_command);
        searcher->register_post_function(post_output);
        if (!tb_init) {
            // should be done already unless we are on Winboard protocol v. 1
            delayed_init(); tb_init++;
            if (EGTBMenCount)
                cerr << "found " << EGTBMenCount << "-man tablebases in directory " << options.search.tablebase_location.c_str() << endl;
        }
        if (ics)
            cout << "kib Hello from Arasan " << Arasan_Version << endl;
    }
    else if (strncmp(buf,"protover",8)==0) {
        // new in Winboard 4.2
        cout << "feature name=1 setboard=1 san=1 usermove=1 ping=1 ics=1 playother=0 analyze=1 myname=\"" << "Arasan " << Arasan_Version;
#ifdef SMP
        cout << " (" << options.search.ncpus;
        if (options.search.ncpus>1)
            cout << " cpus)";
        else
            cout << " cpu)";
#endif
        cout << "\"" << endl;
        ++xboard42;
        // set done = 0 because it may take some time to initialize tablebases.
        cout << "feature done=0" << endl;
        if (!tb_init) {
            delayed_init(); 
            tb_init++;
            if (EGTBMenCount)
                cerr << "found " << EGTBMenCount << "-man tablebases in directory " << options.search.tablebase_location.c_str() << endl;
        }
        cout << "feature done=1" << endl;
    }
    else if (strcmp(buf,"computer")==0) {
        computer = 1;
    }
    else if (strncmp(buf,"ping",4) == 0) {
        cout << "pong " << buf+5 << (flush) << endl;
    }
    else if (strncmp(buf,"ics",3) == 0) {
        hostname = "";
        for (const char *p=buf+4;*p;) {
            hostname += *p++;
        }
    }
    else if (strcmp(buf,"hint")==0) {
        doHint();
    }
    else if (strcmp(buf,"bk")==0) {
        Move moves[20];
        int scores[20];
        unsigned count = 0;
        if (!tb_init) {
            // allow "bk" before "new"
            delayed_init(); tb_init++;
            if (EGTBMenCount)
                cerr << "found " << EGTBMenCount << "-man tablebases in directory " << options.search.tablebase_location.c_str() << endl;
        }
        if (options.book.book_enabled && opening_book) {
            count = opening_book->book_moves(*main_board,moves,scores,20);
        }
        if (count == 0) {
            cout << '\t' << "No book moves for this position." << endl
                << endl;
        }
        else {
            cout << " book moves:" << endl;
            for (unsigned i = 0; i<count; i++) {
                char result[64];
                    Notation::Image(*main_board,moves[i],result);
                    cout << '\t' << result << endl;
            }
            cout << endl;
        }
    }
    else if (strcmp(buf,"depth")==0) {
    }
    else if (strncmp(buf,"level",5)==0) {
        sscanf(buf+6,"%d %d %d",&moves,&minutes,&incr);
            srctype = Time_Limit;
    }
    else if (strncmp(buf,"st ",3)==0) {
        sscanf(buf+3,"%d",&time_limit);
            srctype = Fixed_Time;
    }
    else if (strncmp(buf,"sd ",3)==0) {
        sscanf(buf+3,"%d",&ply_limit);
            srctype = Fixed_Depth;
    }
    else if (strncmp(buf,"time",4)==0) {
        // my time left
        sscanf(buf+4,"%d",&time_left);
    }
    else if (strncmp(buf,"otime",4)==0) {
        sscanf(buf+4,"%d",&opp_time);
    }
    else if (strcmp(buf,"post")==0) {
        post = 1;
    }
    else if (strcmp(buf,"nopost")==0) {
        post = 0;
    }
    else if (strncmp(buf,"result",6)==0) {
        // Game has ended
        theLog->setResult(buf+7);
        save_game();
        game_end = 1;
        game_moves->removeAll();
    }
    else if (strcmp(buf,"savegame")==0) {

    }
    else if (strcmp(buf,"remove")==0) {
        undo(board);
            undo(board);
    }
    else if (strcmp(buf,"undo")==0) {
        undo(board);
    }
    else if (strcmp(buf,"resign")==0) {
        // our opponent has resigned
        cout << "(debug) setting log result" << endl;
        if (computer_plays_white)
            theLog->setResult("0-1");
        else
            theLog->setResult("1-0");
        cout << "(debug) set log result" << endl;
    }
    else if (strcmp(buf,"draw")==0) {
        // "draw" command. Requires winboard 3.6 or higher.
        if (accept_draw(board)) {
            // Notify opponent. don't assume draw is concluded yet.
            cout << "offer draw" << endl;
        }
        else if (doTrace)
            cout << "(debug) draw declined" << endl;
    }
    else if (strncmp(buf,"setboard",8) == 0) {
        char *fen = (char*)buf+9;
            strcpy(start_fen,fen);
    #if defined(__GNUC__) && _GNUC_PREREQ(3,2)
            string istr(fen);
            istringstream s(istr);
    #else
            istrstream s(fen,(int)strlen(fen));
    #endif
            ChessIO::load_fen(s,board);
    }
    else if (strcmp(buf,"edit")==0) {
        edit_board(cin,board);
    }
    else if (strcmp(buf,"analyze")==0) {
        analyzeMode = 1;
            analyze(board);
    }
    else if (strcmp(buf,"exit")==0) {
        if (analyzeMode) analyzeMode = 0;
    }
    else if (strcmp(buf,".") == 0) {
        // for analysis mode: not supported presently
    }
    else if (strcmp(buf,"go")==0) {
        forceMode = 0;
        Move reply = search(searcher,board,stats);
        if (!forceMode) send_move(board,reply,stats);
    }
    else if (strcmp(buf,"bk")==0) {
    }
    else if (strcmp(buf,"easy")==0) {
        easy = true;
    }
    else if (strcmp(buf,"hard")==0) {
        easy = false;
    }
    else if (strcmp(buf,"white")==0) {
        computer_plays_white = true;
        //forceMode = 0;
    }
    else if (strcmp(buf,"black")==0) {
        computer_plays_white = false;
        //forceMode = 0;
    }
    /**   else if (strcmp(buf,"playother")==0) {
          computer_plays_white = !computer_plays_white;
          forceMode = 0;
          int tmp = opp_time;
          opp_time = time_left;
          time_left = tmp;
       }**/
    else if (strncmp(buf,"name",4)==0) {
        // We've received the name of our opponent.
        opponent_name = strdup(buf+5);
    }
    else if (strncmp(buf,"bogus",5)==0) {
    }
    else if (strncmp(buf,"force",5)==0) {
        forceMode = 1;
    }
    else if (strncmp(buf,"rating",6) == 0) {
        if (doTrace) cout << "(debug) rating = " << buf << endl;
        sscanf(buf+7,"%d %d",&computer_rating, &opponent_rating);
        if (searcher) searcher->setRatingDiff(opponent_rating-computer_rating);
    }
    else if (strncmp(buf,"accepted",8) == 0) {
    }
    else if (strncmp(buf,"computer",8) == 0) {
    }
    else if (strncmp(buf,"option",6) == 0) {
        // hack for Arasan 6.0 so GUI can send options to engine
        do_options(buf+7);
    }
    else {
        // see if it could be a move
        char movetext[80];
        if (strncmp(buf,"usermove",8) == 0) {
            // new for Winboard 4.2
            strcpy(movetext,buf+9);
        }
        else
            strcpy(movetext,buf);
        if (isdigit(*movetext)) {
            const char *p = movetext;
            while (*p && !isalpha(*p)) ++p;
            if (*p) {
                char *q = movetext;
                    while (*p && !isspace(*p)) *q++=*p++;
                    *q = '\0';
            }
        }
        if (doTrace)
            cout << "(debug) move text = " << buf << endl;
        Move rmove;
        process_move:
        if (is_move(board,movetext)) {
            if (game_end) {
                if (forceMode)
                    game_end = 0;
                else {
                    if (doTrace) cout << "(debug) ignoring move " << movetext << " received after game end" << endl;
                        return true;
                }
            }
            if (doTrace) {
                cout << "(debug) got move: " << movetext << endl;
            }
            // assume move
            cout << 1+game_moves->num_moves()/2 << ". " << movetext << endl;

            rmove = text_to_move(board,movetext);
            if (!IsNull(rmove)) {
                // make the move on the board
                execute_move(board,rmove);
                if (analyzeMode) {
                    // re-enter analysis loop
                    analyze(board);
                }
                Move reply;
                if (!forceMode && !analyzeMode) {
                    // determine what to do with the pondering result, if
                    // there is one.
                    if (MovesEqual(predicted_move,rmove) && !IsNull(ponder_move)) {
                        // We completed pondering already and we got a reply to
                        // this move (e.g. might be a forced reply).
                        if (doTrace) cout << "(debug) pondering complete already" << endl;
                        if (doTrace) {
                            cout << "(debug) sending ponder move ";
                                MoveImage(ponder_move,cout);
                                cout << endl << (flush);
                        }
                        reply = ponder_move;
                        stats = ponder_stats;
                        game_end |= stats.end_of_game;
                        if (doTrace) cout << "(debug) game_end = " << game_end << endl;
                        predicted_move = ponder_move = NullMove;
                    }
                    else {
                        predicted_move = ponder_move = NullMove;
                            reply = search(searcher,board,stats);
                        // Note: we may know the game has ended here before
                        // we get confirmation from Winboard. So be sure
                        // we set the global game_end flag here so that we won't
                        // start pondering after the game is over.
                            game_end |= stats.end_of_game;
                        if (doTrace) {
                            cout << "(debug) state = " << stats.state << endl;
                                cout << "(debug) game_end = " << game_end  << endl;
                        }
                    }
                    // Check for game end conditions like resign, draw acceptance, etc:
                    if (check_pending(board)==1)
                        game_end = true;
                    else if (!forceMode) {
                        // call send_move even if game_end = true because we
                        // handle resignation, etc. there.
                        send_move(board,reply,stats);
                    }
                }
                ponder:
                if (!forceMode && !analyzeMode && !game_end && !easy && time_target >= 100 /* 1 second */) {
                    // check pending commands again before pondering in case we have a
                    // resign or draw, or a move has come in (no good pondering if the
                    // opponent has replied already).
                    ponder_move_ok = false;
                    int result;
                    if ((result = check_pending(board)) == 0) {
                        ponder(board,reply,stats.best_line[1],uci);
                        if (check_pending(board)==1) {
                            return 1;             // game end signal seen
                        }
                    }
                    else if (result == 1)         // game end
                        return 1;
                    // We are done pondering. If we got a ponder hit (opponent made
                    // our predicted move) we are ready to move now
                    if (ponder_move_ok && !IsNull(ponder_move) && !game_end
                        && !forceMode && !analyzeMode) {
                        // we got a reply from pondering
                        if (doTrace)
                            cout << "(debug) sending ponder move" << endl;
                            stats = ponder_stats;
                            send_move(board,ponder_move,stats);
                            post_output(stats);
                            reply = ponder_move;
                            predicted_move = ponder_move = NullMove;
                            goto ponder;
                    }
                    else if (num_pending && check_pending(board)==2) {
                        if (doTrace) cout << "(debug) ponder failed, move in stack" << endl;
                            ponder_move = NullMove;
                            const char *cmd = pending[0].c_str();
                        if (strncmp(cmd,"usermove",8)==0) {
                            strcpy(movetext,cmd+9);
                        }
                        else {
                            strcpy(movetext,cmd);
                        }
                        // remove from pending list
                        for (int j=1;j<num_pending;j++) pending[j-1] = pending[j];
                            --num_pending;
                            goto process_move;
                    }
                }
            }
        }
    }
    return true;
}


int CDECL main(int argc, char **argv)
{
    signal(SIGINT,SIG_IGN);

    // Show a message on the console
    cout << "Arasan " Arasan_Version;
#ifdef SMP
    cout << " (SMP build)";
#endif
    cout << " Copyright 1994-2008 by Jon Dart" << endl;
    // Must use unbuffered console
    setbuf(stdin,0);

    init_options(argv[0]);
    if (!init_globals(argv[0],0,1)) {
        cleanup_globals();
        exit(-1);
    }
    atexit(cleanup_globals);

    ecoCoder = new ECO();
    Board board;
    main_board = &board;
    int arg = 1;

    if (argc > 1) {
        while (arg < argc && *(argv[arg]) == '-') {
            char c = *(argv[arg]+1);
            switch (c) {
#ifdef SMP
                case 'c':
                    ++arg;
                        options.search.ncpus = Util::Min(MAX_CPUS,atol(argv[arg]));
                    if (options.search.ncpus<=0) {
                        cerr << "-c parameter must be >=1" << endl;
                            exit(-1);
                    }
                    break;
#endif
                 case 'i':
                    if (strcmp(argv[arg]+1,"ics")==0)
                        ics = true;
                    else {
                        cerr << "Warning: unknown option: " << argv[arg]+1 <<
                            endl;
                    }
                    break;
                 case 'H':
                    ++arg;
                    options.search.hash_table_size = atol(argv[arg]);
                    break;
                 case 't':
                    //++arg;
                    doTrace = true;
                    break;
                 default:
                    cerr << "Warning: unknown option: " << argv[arg]+1 <<
                    endl;
                    break;
            }
            ++arg;
        }
    }
    if (arg < argc) {
        cout << "loading " << argv[arg] << endl;
#if defined (_WIN32) || (defined (__GNUC__) && (__GNUC__ >=3))
        ifstream pos_file( argv[arg], ios::in);
#else
        ifstream pos_file( argv[arg], ios::in | ios::nocreate);
#endif
        if (pos_file.good()) {
            pos_file >> board;
        }
        else {
            cout << "file not found: " << argv[arg] << endl;
                return -1;
        }
    }
    *last_move_image = '\0';

    if (options.store_games) {
        if (options.game_pathname == "") {
	  game_pathname = derivePath("games.pgn");
        }
        else {
          game_pathname = options.game_pathname;
	}
    }
    searcher = new SearchController();
    if (testing) {
        do_test(test_file);
    }
    else {
        searcher->register_command_function(check_command);
            searcher->register_post_function(post_output);

        // Read stdin until we get a quit command
            int nextc;
        while ((nextc = getc(stdin)) != EOF) {
            //         if (doTrace) cout << "(debug) main got : " << (char)nextc << endl;
            if (nextc == '\n') {
                cmd_buf[cmd_buf_count] = '\0';
                    cmd_buf_count = '\0';
                    if (doTrace) cout << "(debug) got cmd (main): " << cmd_buf << endl;
                    if (!do_command(cmd_buf,board))
                    break;
                    if (!do_pending(board))
                    break;
            }
            else {
                if (cmd_buf_count >= CMD_BUF_SIZE-1) {
                    cout << "(debug) warning: command buffer full" << endl;
                        cmd_buf_count = 0;
                }
                cmd_buf[cmd_buf_count++] = nextc;
            }
        }

        // handle termination.

        save_game();
    }
    delete searcher;
    delete ecoCoder;

    if (!testing) cout << "quit" << endl;
    return 0;
}
